/**********************************************************************
 ** Text - is used by the builder port to manage help files, info files, 
 **        and other text files that would be readable from the mud, provide
 **        methods for creating, modifying, and deleting the mud text
 **        files online
 **    
 ** Last reviewed:
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef TEXT_C
#define TEXT_C

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "mudtypes.h"
#include "text.h"
#include "lexer.h"
#include "builder.h"
#include "player.h"
#include "objtype.h"
#include "global.h"
#include "utils.h"
#include "newfuncts.h"
#include "memchk.h"

/***********************************************************************
 ** Text (constructor) - loads text name and attributes
 **
 ** Parameters: text_name - this text name
 **             the_type - the type of text file 
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Text::Text(char *text_name, int the_type)
{
   Strings tmp_name;

   tmp_name.assign_word(text_name, 1);

   if (tmp_name.str_len() > MAX_VERB_LEN)
      tmp_name.truncate(MAX_VERB_LEN);
   set_name(tmp_name.str_show());

   obj_type = OBJ_TYPE_TEXT;
   text_type = the_type;

}


/***********************************************************************
 ** ~Text (destructor) - cleans up the text file
 **
 ** Parameters: None 
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Text::~Text(void)
{
}


/***********************************************************************
 ** set_title - sets the title string for this text file
 **
 ** Parameters: the_string - the string we are setting it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Text::set_title(char *the_string)
{
   if (the_string == NULL)
      return -1;

   title = the_string;
   return 1;
}


/***********************************************************************
 ** get_title - gets the title string for this text file
 **
 ** Parameters: None
 **
 ** Returns: pointer to the title string
 **
 ***********************************************************************/

char *Text::get_title(void)
{
   return title.str_show();
}



/***********************************************************************
 ** set_body - sets the body string for this text file
 **
 ** Parameters: the_string - the string we are setting it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Text::set_body(char *the_string)
{
   if (the_string == NULL)
      return -1;

   body = the_string;
   return 1;
}

/***********************************************************************
 ** get_body - gets the body string for this text file
 **
 ** Parameters: None
 **
 ** Returns: pointer to the format string
 **
 ***********************************************************************/

char *Text::get_body(void)
{
   return body.str_show();
}


/***********************************************************************
 ** load_text - loads a text file from a file into memory
 **
 ** Parameters: None
 **
 ** Returns: pointer to the next action
 **
 ***********************************************************************/

int Text::load_text()
{
   token_record *the_token;
   FILE         *textfile = NULL;
   Strings      the_filename;
   char         *temp_desc;


   if (text_type == TEXT_TYPE_HELP)
   {
      the_filename.sprintf("%s/%s%s%s", the_config.basedir.str_show(),
                                          HELPDIR, get_name(), HELPEXTENTION);
   }
   else if (text_type == TEXT_TYPE_INFO)
   {
      the_filename.sprintf("%s/%s%s%s", the_config.basedir.str_show(),
                                           INFODIR, get_name(), INFOEXTENTION);
   }
   else if (text_type == TEXT_TYPE_BANNER)
   {
      the_filename.sprintf("%s/%s%s%s", the_config.basedir.str_show(),
                                     BANNERDIR, get_name(), BANNEREXTENTION);
   }

   if ((textfile = xfopen(the_filename.str_show(), "r", "text_file")) == NULL)
   {
      mainstruct->log_error("Could not load text file\n", "load_text");

      return -1;
   } 

   if (text_type != TEXT_TYPE_BANNER)
   {
      the_token = get_token(textfile, '^');

      if ((temp_desc = read_desc_type(textfile, mainstruct->get_log(), this)) 
                                                       == NULL)
      {
         xfclose(textfile, "text_file");
         return -1;
      }

      set_title(temp_desc);
      delete temp_desc;

      the_token = get_token(textfile, '^');
   }

   if ((temp_desc = read_desc_type(textfile, mainstruct->get_log(), this)) 
                                                       == NULL)
   {
      xfclose(textfile, "text_file");
      return -1;
   }

   set_body(temp_desc);
   delete temp_desc;

   xfclose(textfile, "text_file");

   return 1;
}


/***********************************************************************
 ** describe - describes the help file to a builder
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void Text::describe(Builder *the_builder)
{
   the_builder->send_bldr("\n&+GText: \t&+M%s&*\n", get_name());
   the_builder->send_bldr("&+GType: \t&+w%s&*\n", texttypename[text_type]);
   if (text_type != TEXT_TYPE_BANNER)
      the_builder->send_bldr("&+GTitle: \n&+w%s&*\n", get_title());
   the_builder->send_bldr("&+GBody: \n&+w%s&*\n", get_body());
   the_builder->send_bldr("\n");
}


/***********************************************************************
 ** describe - we can't describe the text to a player, this is impossible
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void Text::describe(Player *the_player)
{
   the_player->send_plr("Use the info or help commands to view.\n");
}

/***********************************************************************
 ** set_attrib - sets a specified attribute to a specified value
 **
 ** Parameters: the_builder - the builder who is changing this attribute
 **             the_parsed - the parsed structure for this
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
int Text::set_attrib(Builder *the_builder, Parse *the_parsed){

   if (the_parsed->get_target1() == NULL)
   {   the_builder->
        send_bldr("You can set the following attributes on a text.\n"
               "   title, body\n");
       return -1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "title",
                               strlen(the_parsed->get_target1())))
   {
      if (text_type != TEXT_TYPE_BANNER)
      {
         if (the_builder->get_long_input(&title) < 0)
         {
            the_builder->send_bldr("Error reading in input, failed!\n");
            return -1;
         }
         return 1;
      }
      else
      {
         the_builder->send_bldr("This type of text does not have a title.\n");
         return -1;
      }
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "body",
                               strlen(the_parsed->get_target1())))
   {
      if (the_builder->get_long_input(&body) < 0)
      {
         the_builder->send_bldr("Error reading in input, failed!\n");
         return -1;
      }
      return 1;
   }
   the_builder->send_bldr("The attribute '%s' is not a title attribute.\n",
                                           the_parsed->get_target1());
   return -1;
}



/***********************************************************************
 ** write_object - writes the text file to the appropriate location
 **
 ** Parameters:  tmp_file - not used
 **              build_format - not used
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
void Text::write_object(FILE *tmp_file, int build_format)
{
   char divider[85];
   int  i;  
   char dividechar;
   Strings the_filename;
   FILE *the_file = NULL;

   tmp_file = NULL;  // eliminate compile warnings
   build_format = 0;


   if (text_type == TEXT_TYPE_HELP)
   {
      dividechar = '-';
      the_filename.sprintf("%s/%s%s%s", the_config.basedir.str_show(),
                                        HELPDIR, get_name(), HELPEXTENTION);
   }
   else if (text_type == TEXT_TYPE_INFO)
   {
      dividechar = '*';
      the_filename.sprintf("%s/%s%s%s", the_config.basedir.str_show(),
                                          INFODIR, get_name(), INFOEXTENTION);
   }
   else if (text_type == TEXT_TYPE_BANNER)
   {
      dividechar = '*';
      the_filename.sprintf("%s/%s%s%s", the_config.basedir.str_show(),
                                       BANNERDIR, get_name(), BANNEREXTENTION);
   }
   else
      return;

   if ((the_file = xfopen(the_filename.str_show(), "w+", 
                                                    "text_file")) == NULL)
   {
      mainstruct->log_error("Could not create text file '%s'\n", get_name());

      return;
   } 

   if (text_type != TEXT_TYPE_BANNER)
   {
      for (i=0; i<65; i++)
         divider[i] = dividechar;
      divider[65] = '\0';

      fprintf(the_file, "\n&+B%s&*\n^", divider);
      fprintf(the_file, "^\n%s^\n", (get_title() == NULL) ? "" : get_title());
      fprintf(the_file, "&+B%s&*\n\n", divider);
      fprintf(the_file, "^^\n%s^\n\n", 
                                    (get_body() == NULL) ? "" : get_body());
   }
   else
   {
      fprintf(the_file, "^\n%s^\n\n", (get_body() == NULL) ? "" : get_body());
   }

   xfclose(the_file, "text_file");

}



/***********************************************************************
 ** copy_object - copies the text to a text of a different name
 **
 ** Parameters: copy_obj - copy attributes from this object
 **
 ** Returns:  1 if succeeded 
 **           0 if failed
 **
 ***********************************************************************/
int Text::copy_object(Entity *copy_obj)
{
   Text *copy_from;

   if (copy_obj->get_type() != OBJ_TYPE_TEXT)
      return 0;

   copy_from = (Text *) copy_obj;

   /******* set the text attributes *****/   
   set_title(copy_from->get_title());
   set_body(copy_from->get_body());

   return 1;
}



/***********************************************************************
 ** get_text_type - gets the numerical indicator of the text type
 **
 ***********************************************************************/
int Text::get_text_type(void)
{
   return text_type;
}


/***********************************************************************
 ** get_mem_size - gets how much memory this special is taking up
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Text::get_mem_size()
{
   int size = 0;

   size = sizeof(this);
   size += get_mem_size_dynamic();
   return size;
}

/***********************************************************************
 ** get_mem_size_dynamic - gets how much memory is taken up by pointers
 **                        pointing to other objects, not including the
 **                        sizeof(this)
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Text::get_mem_size_dynamic()
{
   int  size = 0;

   size += title.get_mem_size_dynamic();
   size += body.get_mem_size_dynamic();

   size += get_mem_size_entity();

   return size;
}


#endif
