/**********************************************************************
 ** Verb_List - maintains a linked list of verb objects
 **
 **
 **    
 ** Reviewed last: version 0.14
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef VERB_LIST_C
#define VERB_LIST_C

#include "config.h"
#include "sysdep.h"
#include "newfuncts.h"
#include "mudtypes.h"
#include "global.h"
#include "verb_list.h"
#include "objtype.h"
#include "action.h"
#include "readobj.h"
#include "player.h"
#include "chatline.h"
#include "lineflags.h"
#include "memchk.h"
#include "utils.h"

/***********************************************************************
 ** _Verb_List (constructor) - creates the verb list, reading in from the
 **                            verbs.lib file and storing in a linked list
 **                            of verbs
 **
 ** Parameters: None
 **
 ** Returns: None
 **
 **
 ***********************************************************************/

Verb_List::Verb_List(verb_data verbdata[])
{
   int i;               /* used to count */
   Verb *the_list;  /* used to move along verb list */
   Verb *new_verb;  /* holds a new verb */
   
   i = 0;

   num_verbs_loaded = num_lookups = max_lookup = avg_lookup = 0;
   min_lookup = 10000;

   num_executes = max_execute = avg_execute = 0;
   min_execute = 10000000;

   /* load the tables into a linked list of verbs */
   the_list = NULL;
   while (verbdata[i].verbname != NULL)
   {
      /* create a new verb with the string listed as the verb name */
      new_verb = new_Verb(verbdata[i].verbname);

      num_verbs_loaded++;
      
      new_verb->set_verb_format(verbdata[i].verbformat);
      new_verb->set_verb_comflag(verbdata[i].permissions);
      new_verb->set_verb_command(verbdata[i].verbcommands);

      the_verbs.add(new_verb->get_name(), (Entity *) new_verb);

      i++;
   }
}


/***********************************************************************
 ** load_actions - loads all the actions from a file into memory, and
 **                places them into the lextree
 **
 ** Parameters: the_file - the file pointer we read actions from
 **             the_builder - the builder who is loading this
 **             error_log - where to log any errors
 **             is_builder - are we loading this for the builder port?
 **
 ** Returns: a pointer to a verb object or NULL if none found
 **
 **
 ***********************************************************************/

int Verb_List::load_actions(FILE *the_file, Builder *the_builder, 
                                          ErrLog *error_log, int is_builder)
{
   Strings holder;
   token_record *the_token;
   Action *new_action;
   Flags *admflags = NULL;

   if (the_builder != NULL)
      admflags = the_builder->get_adminflags();

   /* first read in the actions header */

   /* first read in the version number and compare, raise an error if
      the area version doesn't match the current version */
   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      holder.sprintf("Invalid format in actions header");
      error_log->log_err(holder.str_show(), "load_actions");
      xfclose(the_file, "load_area_file");
      return 0;
   }
   the_token = get_token(the_file, '^');
   if (STRCASECMP(FULLVERSION, the_token->the_string))
   {
     Strings filename;
     Strings directory;

     xfclose(the_file, "load_area_file");

#ifdef WIN32
	 return -2;
#endif

     directory.sprintf("%s/%s", the_config.basedir.str_show(), DATA_DIR);
     filename.sprintf("actions.dat");

     if (prompt_user_convert(filename.str_show(), directory.str_show()) != 1)
     {
       holder.sprintf(_("Invalid version '%s' for data file 'actions.dat'.  "
                     "Should be version '%s'.\n"), the_token->the_string,
                                                             FULLVERSION);
       error_log->log_err(holder.str_show(), "load_area");
       return -1;
     }
     else
     {
       filename.sprintf("%s/actions.dat", directory.str_show());
       if ((the_file = xfopen(filename.str_show(), "r", "load_area_file")) 
	                                                              == NULL)
       {
         holder.sprintf("Error - could not open actions.dat "
                        "file after convert.\n");
         error_log->log_err(holder.str_show(), "Object_List");
	     return 0;
       }
       return load_actions(the_file, the_builder, error_log, is_builder);
     }
   }

   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      error_log->log_err("Invalid format in actions file", "load_area");
   }

   /* get the readdeny */
   the_token = get_token(the_file, '^');
   //   set_read_deny(the_token->the_string);

   /* read in the writeallow list */
   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      error_log->log_err("Invalid format in actions file", "Object_List");
   }

   /* get the writeallow */
   the_token = get_token(the_file, '^');
   //   set_write_allow(the_token->the_string);

   the_token = get_token(the_file, '\0');
   while (the_token->token_type > 0)
   {
      while (the_token->token_type > 0)
      {
         new_action = new_Action(the_token->the_string);
         if (new_action->load_action(the_file, error_log, is_builder) > 0)
            the_verbs.add(new_action->get_name(), (Entity *) new_action);
 
         the_token = get_token(the_file, '\0');
      }      
   }
   xfclose(the_file, "load_area_file");

   return 1;
}


/***********************************************************************
 ** load_chatlines - loads all the chatlines from a file into memory and
 **                  places them into the lextree
 **
 ** Parameters: the_file - the file pointer we read chatlines from
 **             the_builder - the builder who is loading this
 **             error_log - where to log any errors
 **             is_builder - are we loading this for the builder port?
 **             lineflag - the lineflag to constrain what we load
 **
 ** Returns: a pointer to a verb object or NULL if none found
 **
 **
 ***********************************************************************/

int Verb_List::load_chatlines(FILE *the_file, Builder *the_builder, 
                             ErrLog *error_log, int is_builder, int lineflag)
{
   Strings holder;
   token_record *the_token;
   ChatLine *new_chatline;
   Flags *admflags = NULL;
   Flags *tmp_lineflags;

   if (the_builder != NULL)
      admflags = the_builder->get_adminflags();

   /* first read in the chatlines header */

   /* first read in the version number and compare, raise an error if
      the area version doesn't match the current version */
   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      holder.sprintf("Invalid format in chatlines header");
      error_log->log_err(holder.str_show(), "load_chatlines");
      xfclose(the_file, "load_area_file");
      return -1;
   }
   the_token = get_token(the_file, '^');
   if (STRCASECMP(FULLVERSION, the_token->the_string))
   {
     Strings filename;
     Strings directory;

     xfclose(the_file, "load_area_file");

#ifdef WIN32
	 return -2;
#endif
     directory.sprintf("%s/%s", the_config.basedir.str_show(), DATA_DIR);
     filename.sprintf("chatlines.dat");

     if (prompt_user_convert(filename.str_show(), directory.str_show()) != 1)
     {
       holder.sprintf(_("Invalid version '%s' for data file 'chatlines.dat'.  "
                     "Should be version '%s'.\n"), the_token->the_string,
                                                             FULLVERSION);
       error_log->log_err(holder.str_show(), "load_area");
       return -1;
     }
     else
     {
       filename.sprintf("%s/chatlines.dat", directory.str_show());
       if ((the_file = xfopen(filename.str_show(), "r", "load_area_file")) 
	                                                              == NULL)
       {
         holder.sprintf("Error - could not open chatlines.dat "
                        "file after convert.\n");
         error_log->log_err(holder.str_show(), "Object_List");
	 return 0;
       }
       return load_chatlines(the_file, the_builder, error_log, is_builder, 
			     lineflag);
     }
   }

   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      error_log->log_err("Invalid format in chatlines file", "load_chatlines");
   }

   /* get the readdeny */
   the_token = get_token(the_file, '^');
   //   set_read_deny(the_token->the_string);

   /* read in the writeallow list */
   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      error_log->log_err("Invalid format in chatlines file", "load_chatlines");
   }

   /* get the writeallow */
   the_token = get_token(the_file, '^');
   //   set_write_allow(the_token->the_string);

   the_token = get_token(the_file, '\0');
   while (the_token->token_type > 0)
   {
      the_token = get_token(the_file, '\0');
      new_chatline = new_ChatLine(the_token->the_string);
      if (new_chatline->load_chatline(the_file, error_log, is_builder) > 0)
      {
	 tmp_lineflags = new_chatline->get_lineflags();

         /* if this chatline is meant for this port, add it */
         if (tmp_lineflags->get_flag(lineflag)) 
            the_verbs.add(new_chatline->get_name(), (Entity *) new_chatline);
	 else
	    delete_ChatLine(new_chatline);
      }
      else
         delete_ChatLine(new_chatline);


      the_token = get_token(the_file, '\0');
   }

   xfclose(the_file, "load_area_file");

   return 1;
}


/***********************************************************************
 ** load_abilities - loads all the abilities from a file into memory, and
 **                  places them into the lextree
 **
 ** Parameters: the_file - the file pointer we read actions from
 **             the_builder - the builder who is loading this
 **             error_log - where to log any errors
 **             is_builder - are we loading this for the builder port?
 **
 ** Returns: a pointer to a verb object or NULL if none found
 **
 **
 ***********************************************************************/

int Verb_List::load_abilities(FILE *the_file, Builder *the_builder, 
                                          ErrLog *error_log, int is_builder)
{
   Strings holder;
   token_record *the_token;
   Entity *new_ability;
   Flags *admflags = NULL;

   if (the_builder != NULL)
      admflags = the_builder->get_adminflags();

   /* first read in the abilities header */

   /* first read in the version number and compare, raise an error if
      the area version doesn't match the current version */
   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      holder.sprintf("Invalid format in abilities header");
      error_log->log_err(holder.str_show(), "load_abilities");
      xfclose(the_file, "load_area_file");
      return 0;
   }
   the_token = get_token(the_file, '^');
   if (STRCASECMP(FULLVERSION, the_token->the_string))
   {
     Strings filename;
     Strings directory;

     xfclose(the_file, "load_area_file");

#ifdef WIN32
	 return -2;
#endif

     directory.sprintf("%s/%s", the_config.basedir.str_show(), DATA_DIR);
     filename.sprintf("abilities.dat");

     if (prompt_user_convert(filename.str_show(), directory.str_show()) != 1)
     {
       holder.sprintf(_("Invalid version '%s' for data file 'abilities.dat'.  "
                     "Should be version '%s'.\n"), the_token->the_string,
                                                             FULLVERSION);
       error_log->log_err(holder.str_show(), "load_area");
       return -1;
     }
     else
     {
       filename.sprintf("%s/abilities.dat", directory.str_show());
       if ((the_file = xfopen(filename.str_show(), "r", "load_area_file")) 
	                                                              == NULL)
       {
         holder.sprintf("Error - could not open abilities.dat "
                        "file after convert.\n");
         error_log->log_err(holder.str_show(), "Object_List");
	 return 0;
       }
       return load_abilities(the_file, the_builder, error_log, is_builder);
     }
   }

   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      error_log->log_err("Invalid format in abilities file", 
                                                         "load_abilities");
      xfclose(the_file, "load_area_file");
      return -1;
   }

   /* get the readdeny */
   the_token = get_token(the_file, '^');

   /* read in the writeallow list */
   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      error_log->log_err("Invalid format in abilities file", 
                                                       "load_abilities");
      xfclose(the_file, "load_area_file");
      return -1;
   }

   /* get the writeallow */
   the_token = get_token(the_file, '^');

   the_token = get_token(the_file, '\0');
   while (the_token->token_type > 0)
   {
      if ((new_ability = 
            read_ability(the_file, the_token, error_log, is_builder)) > 0)
         the_verbs.add(new_ability->get_name(), new_ability);

      the_token = get_token(the_file, '\0');      
   }

   xfclose(the_file, "load_area_file");
   return 1;
}


/***********************************************************************
 ** ~_Verb_List (destructor) - deletes all verbs in the list
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 **
 ***********************************************************************/

Verb_List::~Verb_List()
{
}


/***********************************************************************
 ** get_verb - Gets a verb from the list, using the verb name (command) as
 **            the index
 **
 ** Parameters: command - the verb name to search for
 **
 ** Returns: a pointer to a verb object or NULL if none found
 **
 **
 ***********************************************************************/

Entity *Verb_List::get_verb(char *command)
{
   Entity     *tmp_verb;
   long       timespan;


#if defined( AIME_WIN32 )
   FILETIME time_start, time_end;

   GetSystemTimeAsFileTime( &time_start );

#else
   struct timeval time_start, time_end;

   gettimeofday(&time_start, NULL);
#endif


   if (command == NULL)
      return NULL;

   tmp_verb = the_verbs.find_abbrev(command);


#if defined( AIME_WIN32 )
   GetSystemTimeAsFileTime( &time_end );

   if( time_start.dwHighDateTime != time_end.dwHighDateTime )
   {
      timespan = ( long )( ( time_end.dwHighDateTime
                             - time_start.dwHighDateTime )
                           * 4294.967296 );
   }
   else
   {
      timespan = 0;
   }

   timespan += ( time_end.dwLowDateTime
                - time_start.dwLowDateTime ) / 1000000;

#else
	  gettimeofday(&time_start, NULL);

     if (time_start.tv_sec != time_end.tv_sec)
     {
        timespan = ((1000000 - time_start.tv_usec) + 
                   ((time_end.tv_sec - (time_start.tv_sec+1)) * 1000000) +
                   (time_end.tv_usec));
     }
     else
        timespan = time_end.tv_usec - time_start.tv_usec;
#endif


   num_lookups++;
   if (timespan > max_lookup)
      max_lookup = timespan;

   if (timespan < min_lookup)
      min_lookup = timespan;

   avg_lookup = ((avg_lookup*(num_lookups-1)) + timespan) / num_lookups;
  
   return tmp_verb;
}


/***********************************************************************
 ** find_verb (private) - finds the next verb in the list
 **
 ** Parameters: command - the verb to find
 **             the_list - the starting point to look at
 **
 ** Returns: a pointer to a verb object, or NULL if none found
 **
 **
 ***********************************************************************/

Entity *Verb_List::find_verb(char *command)
{
   Entity *temp_list;
   long   timespan;


#if defined( AIME_WIN32 )
   FILETIME time_start, time_end;

   GetSystemTimeAsFileTime( &time_start );

#else
   struct timeval time_start, time_end;

   gettimeofday(&time_start, NULL);
#endif

   if (command == NULL)
      return NULL;

   temp_list = the_verbs.find_abbrev(command);


#if defined( AIME_WIN32 )
   GetSystemTimeAsFileTime( &time_end );

   if( time_start.dwHighDateTime != time_end.dwHighDateTime )
   {
      timespan = ( long )( ( time_end.dwHighDateTime
                             - time_start.dwHighDateTime )
                           * 4294.967296 );
   }
   else
   {
      timespan = 0;
   }

   timespan += ( time_end.dwLowDateTime
                - time_start.dwLowDateTime ) / 1000000;

#else
	  gettimeofday(&time_start, NULL);

     if (time_start.tv_sec != time_end.tv_sec)
     {
        timespan = ((1000000 - time_start.tv_usec) + 
                   ((time_end.tv_sec - (time_start.tv_sec+1)) * 1000000) +
                   (time_end.tv_usec));
     }
     else
        timespan = time_end.tv_usec - time_start.tv_usec;
#endif


   num_lookups++;
   if (timespan > max_lookup)
      max_lookup = timespan;

   if (timespan < min_lookup)
      min_lookup = timespan;

   avg_lookup = ((avg_lookup*(num_lookups-1)) + timespan) / num_lookups;

   return temp_list;
}


/***********************************************************************
 ** add_new_execute - adds new timestats for this command execution
 **
 ** Parameters: the_verb - the verb that is executing
 **             timespan - how many milliseconds it took to execute
 **
 ** Returns: 1 for success
 **
 **
 ***********************************************************************/

int Verb_List::add_new_execute(Verb *the_verb, long timespan)
{
   num_executes++;
   if (timespan > max_execute)
      max_execute = timespan;

   if (timespan < min_execute)
      min_execute = timespan;

   avg_execute = ((avg_execute*(num_executes-1)) + timespan) / num_executes;

   the_verb->add_new_execute(timespan);

   return 1;
}


/***********************************************************************
 ** display_data - displays verb database statistics
 **                
 **
 ** Parameters: the_player - who we send the messages to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Verb_List::display_data(Player *the_player)
{
   the_player->send_plr("\n&+GVerb Database Statistics&*\n");
   the_player->send_plr("&+B-----------------------------&*\n\n");
   the_player->send_plr("Number of objects loaded: %ld\n\n", 
                                                       num_verbs_loaded);
   the_player->send_plr("&+gMax Lookup Time (microseconds)&+b: &+W%ld&*\n", 
                                                                max_lookup);
   the_player->send_plr("&+gAvg Lookup Time (microseconds)&+b: &+W%ld&*\n", 
                                                                avg_lookup);
   the_player->send_plr("&+gMin Lookup Time (microseconds)&+b: &+W%ld&*\n\n",
                                                                min_lookup);

   the_player->send_plr("&+gMax Execute Time (microseconds)&+b: &+W%ld&*\n", 
                                                                max_execute);
   the_player->send_plr("&+gAvg Execute Time (microseconds)&+b: &+W%ld&*\n", 
                                                                avg_execute);
   the_player->send_plr("&+gMin Execute Time (microseconds)&+b: &+W%ld&*\n\n",
                                                                min_execute);

   return 1;
}


/***********************************************************************
 ** display_verb_stats - displays verb statistics
 **                
 ** Parameters: the_player - who we send the messages to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Verb_List::display_verb_stats(Player *the_player)
{
   Entity *tmp_entity;
   int    counter = 0;

   the_player->send_plr(
           "\n&+WVerb Statistics (Min, Max, Avg in microsecs)&*\n\n");
   the_player->send_plr("Verb Name                Used\tMin\tMax\tAvg\n");
   the_player->send_plr(
      "&+B-----------------------------------------------------&*\n\n");

   the_verbs.reset_current();
   tmp_entity = the_verbs.get_next();
   while (tmp_entity != NULL)
   {
      if (tmp_entity->get_type() == OBJ_TYPE_VERB)
      {
         ((Verb *)tmp_entity)->display_data(the_player);
         counter++;
      }
      tmp_entity = the_verbs.get_next();
   }

   the_player->send_plr(
      "&+B-----------------------------------------------------&*\n");
   the_player->send_plr("&+WVerbs displayed: &*%d\n\n", counter);

   return 1;
}


/***********************************************************************
 ** add_init_chatlines - add the initial chatlines that are marked for
 **                      new players to the player's chatlines
 **                
 ** Parameters: the_player - who we add the chatlines to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Verb_List::add_init_chatlines(Player *the_player)
{
   Entity *tmp_entity;
   Strings holder;   

   the_verbs.reset_current();
   tmp_entity = the_verbs.get_next();
   while (tmp_entity != NULL)
   {
      if (tmp_entity->get_type() == OBJ_TYPE_CHATLINE)
      {
	ChatLine *tmp_chatline = (ChatLine *) tmp_entity;
        Flags *tmp_lineflags = tmp_chatline->get_lineflags();

        if (tmp_lineflags->get_flag(LINEFLAG_NEWPLAYER))
	{
	  holder = the_player->get_chatlines();
          holder.str_cat(tmp_chatline->get_name());
          holder.str_cat(" ");
          the_player->set_chatlines(holder.str_show());
	}
      }
      tmp_entity = the_verbs.get_next();
   }
   return 1;
}

#endif







