/**********************************************************************
 ** Strings class: A class to manipulate strings 
 **  
 **
 ** Reviewed through: version 0.10
 **    
 **
 ** Copyright (C) 2000 George Noel (Slate), Ed Boraas
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef STRINGS_C
#define STRINGS_C

#ifndef CONVERTER
#include "sysdep.h"
#include "config.h"
#include "strings.h"
#include "utils.h"
#include "player.h"
#include "individual.h"
#include "newfuncts.h"
#include "objtype.h"
#else
#include "convconfig.h"
#include "../../include/strings.h"
#include "../../include/sysdep.h"
#endif

/***********************************************************************
 ** Strings (constructor) - initializes the string and creates it
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Strings::Strings()
{
   str = NULL;
   length_alloc = 0;
}


/***********************************************************************
 ** Strings (constructor) - initializes the string and creates it
 **
 ** Parameters: the_string - the string to initialize it with
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Strings::Strings(char *the_string)
{
   int the_length;

   length_alloc = 0;
   str = NULL;
   if (the_string == NULL)
      return;

   the_length = strlen(the_string);
   if (the_length <= 0)
   {
      if ((str != NULL) && (strlen(str) >= length_alloc))
      {
         printf("ERROR! String length exceeded allocated space!\n");
         RAISE(1);
      }
      return;
   }

   str = strmalloc(the_length+2);
   length_alloc = the_length+2;
   strncpy(str, the_string, the_length);
   if ((str != NULL) && (strlen(str) >= length_alloc))
   {
      printf("ERROR! String length exceeded allocated space!\n");
      RAISE(1);
   }
   return;
}


/***********************************************************************
 ** ~Strings (destructor) - deallocates the string memory
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Strings::~Strings()
{
   if (str != NULL)
      delete_char(str);
}


/***********************************************************************
 ** strmalloc - allocates memory for strings
 **
 ** Parameters: the_length - the amount to allocate
 **
 ** Returns: pointer to the newly allocated string
 **          NULL if error
 **
 ***********************************************************************/

char *Strings::strmalloc(int the_length)
{
   char *the_str;

   the_str = new_char(the_length);
   BZERO(the_str, the_length);
   return the_str;
}

/***********************************************************************
 ** str_copy - copies another string to this one
 **
 ** Parameters: the_string - the string object to copy from
 **
 ** Returns: number of characters in the new string if successful
 **          -1 if failure
 **
 ***********************************************************************/

int Strings::str_copy(Strings *the_string)
{
   if ((str != NULL) && (strlen(str) >= length_alloc))
   {
      printf("ERROR! String length exceeded allocated space!\n");
      RAISE(1);
   }
   return str_copy(the_string->str_show());
}


/***********************************************************************
 ** str_copy - copies another string to this one
 **
 ** Parameters: the_string - the string object to copy from
 **
 ** Returns: number of characters in the new string if successful
 **          -1 if failure
 **
 ***********************************************************************/

int Strings::str_copy(char *the_string)
{
   int new_len;  /* the length of the_string */

   if (the_string == NULL)
   {
     //      printf("error, passed null string.\n");
      return -1;
   }

   if ((new_len = strlen(the_string)) <= 0)
   {
      if (new_len == 0)
         str = NULL;
      return -1;
   }

   /* if the memory locations are the same, we are copying the same string
      to itself, so just do nothing */
   if (the_string == str)
   {
      if ((str != NULL) && (strlen(str) >= length_alloc))
      {
         printf("ERROR! String length exceeded allocated space!\n");
         RAISE(1);
      }
      return new_len;
   }

   delete_char(str);

   str = strmalloc(new_len+2);
   length_alloc = new_len+2;
   strncpy(str, the_string, new_len); 

   if ((str != NULL) && (strlen(str) >= length_alloc))
   {
      printf("ERROR! String length exceeded allocated space!\n");
      RAISE(1);
   }
   return new_len;
}


/***********************************************************************
 ** str_show - returns the string for displaying
 **
 ** Parameters: None
 **
 ** Returns: a pointer to the string if successful
 **          NULL if unsuccessful
 **
 ***********************************************************************/

char *Strings::str_show(void)
{
   if ((str != NULL) && (strlen(str) >= length_alloc))
   {
      printf("ERROR! String length exceeded allocated space!\n");
      RAISE(1);
   }
   return str;
}


/***********************************************************************
 ** find_letter - finds a letter in the string and points to it
 **
 ** Parameters: the_num - the number of the letter in the string
 **             the_str - the string we are searching
 **
 ** Returns: pointer to the char if found
 **          NULL if failure
 **
 ***********************************************************************/

char *Strings::find_letter(char *the_str, int the_num)
{
   int i;          /* used to count */
   char *tmp_str;  /* used to move along the string */

   tmp_str = the_str;
   if (tmp_str == NULL)
   {
      if ((str != NULL) && (strlen(str) >= length_alloc))
      {
         printf("ERROR! String length exceeded allocated space!\n");
         RAISE(1);
      }
      return NULL;
   }

   for (i=0; i < the_num; i++)
   {
      tmp_str++;
      if (!*tmp_str)
      {
         if ((str != NULL) && (strlen(str) >= length_alloc))
         {
            printf("ERROR! String length exceeded allocated space!\n");
            RAISE(1);
         }
         return NULL;
      }
   }
   return tmp_str;
}

/***********************************************************************
 ** change_letter - allows you to change a letter in the string
 **
 ** Parameters: the_num - the number of the letter in the string
 **             the_letter - the letter to be changed
 **
 ** Returns: 1 if successful
 **          -1 if failure
 **
 ***********************************************************************/

int Strings::change_letter(int the_num, char the_letter)
{
   char *tmp_str;

   tmp_str = find_letter(str, the_num);
   if (tmp_str == NULL)
   {
      if ((str != NULL) && (strlen(str) >= length_alloc))
      {
         printf("ERROR! String length exceeded allocated space!\n");
         RAISE(1);
      }
      return -1;
   }

   *tmp_str = the_letter;
   if ((str != NULL) && (strlen(str) >= length_alloc))
   {
      printf("ERROR! String length exceeded allocated space!\n");
      RAISE(1);
   }
   return 1;
}


/***********************************************************************
 ** get_letter - gets the letter in the string at the number
 **
 ** Parameters: the_num - the number of the letter to get
 **
 ** Returns: the character at that number
 **
 ***********************************************************************/
  
char Strings::get_letter(int the_num)
{
   char *tmp_str;

   tmp_str = find_letter(str, the_num);
   if (tmp_str == NULL)
   {
      if ((str != NULL) && (strlen(str) >= length_alloc))
      {
         printf("ERROR! String length exceeded allocated space!\n");
         RAISE(1);
      }
      return -1;
   }

   if ((str != NULL) && (strlen(str) >= length_alloc))
   {
      printf("ERROR! String length exceeded allocated space!\n");
      RAISE(1);
   }
   return *tmp_str;
}


/***********************************************************************
 ** upper - changes a character to uppercase
 **
 ** Parameters: the_num - the number of the char to change
 **
 ** Returns: 1 if successful
 **          -1 if failure
 **
 ***********************************************************************/

int Strings::upper(int the_num)
{
   char *tmp_str;

   tmp_str = find_letter(str, the_num);
   if (tmp_str == NULL)
      return -1;

   *tmp_str = toupper(*tmp_str);
   if ((str != NULL) && (strlen(str) >= length_alloc))
   {
      printf("ERROR! String length exceeded allocated space!\n");
      RAISE(1);
   }
   return 1;
}


/***********************************************************************
 ** lower - changes a character to lowercase
 **
 ** Parameters: the_num - the number of the char to change
 **
 ** Returns: 1 if successful
 **          -1 if failure
 **
 ***********************************************************************/

int Strings::lower(int the_num)
{
   char *tmp_str;

   tmp_str = find_letter(str, the_num);
   if (tmp_str == NULL)
      return -1;

   *tmp_str = tolower(*tmp_str);
   if ((str != NULL) && (strlen(str) >= length_alloc))
   {
      printf("ERROR! String length exceeded allocated space!\n");
      RAISE(1);
   }
   return 1;
}


/***********************************************************************
 ** uppercase - changes the whole string to uppercase
 **
 ** Parameters: None
 **
 ** Returns: 1 if successful
 **          -1 if failure
 **
 ***********************************************************************/

int Strings::uppercase()
{
   int i;          /* used to count */
   int the_num;
   char *tmp_str;  /* used to move along the string */
 
   tmp_str = str;
   the_num = strlen(tmp_str);
   for (i=0; i <= the_num; i++)
   {
      tmp_str++;
      if (!tmp_str)
         return -1;
      *tmp_str = toupper(*tmp_str);
   }
   if ((str != NULL) && (strlen(str) >= length_alloc))
   {
      printf("ERROR! String length exceeded allocated space!\n");
      RAISE(1);
   }
   return *tmp_str;
}


/***********************************************************************
 ** lowercase - changes the whole string to lowercase
 **
 ** Parameters: None
 **
 ** Returns: 1 if successful
 **          -1 if failure
 **
 ***********************************************************************/

int Strings::lowercase()
{
   int i;          /* used to count */
   int the_num;
   char *tmp_str;  /* used to move along the string */
 
   the_num = strlen(str);
   tmp_str = str;
   for (i=0; i <= the_num; i++)
   {
      if (!tmp_str)
         return -1;
      *tmp_str = tolower(*tmp_str);
      tmp_str++;
   }
   if ((str != NULL) && (strlen(str) >= length_alloc))
   {
      printf("ERROR! String length exceeded allocated space!\n");
      RAISE(1);
   }
   return 1;
}


/***********************************************************************
 ** truncate - truncates a string at a certain character
 **
 ** Parameters: where_at - where along the string to truncate
 **
 ** Returns: 1 if successful
 **          -1 if failure
 **
 ***********************************************************************/

int Strings::truncate(int where_at)
{
   int i;          /* used to count */
   char *tmp_str;  /* used to move along the string */
   char *new_str; 
   int stringlen;

   if ((where_at == 0) && (str == NULL))
      return 1;

   if (str == NULL)
      return -1;

   if (where_at == 0)
   {
      delete_char(str);
      str = NULL;
      if ((str != NULL) && (strlen(str) >= length_alloc))
      {
         printf("ERROR! String length exceeded allocated space!\n");
         RAISE(1);
      }
      return 0;
   }

   tmp_str = str;
   for (i=0; i < where_at; i++)
   {
      tmp_str++;
      if (!(*tmp_str))
         return -1;
   }
   *tmp_str = '\0';

   stringlen = strlen(str);

   new_str = strmalloc(strlen(str) + 2);
   length_alloc = strlen(str) + 2;
   strncpy(new_str, str, strlen(str) + 2);
   delete_char(str);      
   str = new_str;
   if ((str != NULL) && (strlen(str) >= length_alloc))
   {
      printf("ERROR! String length exceeded allocated space!\n");
      RAISE(1);
   }
   return stringlen;
}


/***********************************************************************
 ** str_len - returns the length of the string, counting colorcodes
 **
 ** Parameters: Nothing
 **
 ** Returns: length of the string if successful
 **          -1 if failure
 **
 ***********************************************************************/

int Strings::str_len()
{
   if ((str != NULL) && (strlen(str) >= length_alloc))
   {
      printf("ERROR! String length exceeded allocated space!\n");
      RAISE(1);
   }
   return  (str == NULL) ? 0 : strlen(str);
}


/***********************************************************************
 ** cstr_len - returns the length of the string, excluding colorcodes
 **
 ** Parameters: Nothing
 **
 ** Returns: length of the string if successful
 **          -1 if failure
 **
 ***********************************************************************/

int Strings::cstr_len()
{
   int count = 0;
   int i;
   int string_len;
   char *tmp_str;

   string_len = strlen(str);
   tmp_str = str;
   for (i=0; i<string_len; i++)
   {
      if (*tmp_str == '&')
      {
         tmp_str++;
         if ((*tmp_str) && ((*tmp_str == '+') || (*tmp_str == '-') ||
                            (*tmp_str == '=')))
	 {
            tmp_str++;
            if (*tmp_str)
	    {
               count += 3;
               tmp_str++;
            }
         }
         else if ((*tmp_str) && ((*tmp_str == '&')))
	 {
            count++;
         }
         else if ((*tmp_str) && ((*tmp_str == '*')))
            count += 2;
      }
      tmp_str++;
   }
   if ((str != NULL) && (strlen(str) >= length_alloc))
   {
      printf("ERROR! String length exceeded allocated space!\n");
      RAISE(1);
   }
   return string_len - count;
}


/***********************************************************************
 ** str_cat - adds a string onto another string
 **
 ** Parameters: Nothing
 **
 ** Returns: length of the string if successful
 **          -1 if failure
 **
 ***********************************************************************/

int Strings::str_cat(char *new_str)
{
   int  old_len;
   int  new_len;
   int  tot_len;
   char *the_string;

   if ((new_str == NULL) || (strlen(new_str) <= 0))
      return -1;

   if (str != NULL)
      old_len = strlen(str);
   else
      old_len = 0;

   new_len = strlen(new_str);
   tot_len = old_len + new_len;
   the_string = strmalloc(tot_len + 2);
   length_alloc = tot_len + 2;
   if (str != NULL)
      strncpy(the_string, str, old_len);

   strcat(the_string, new_str);

   delete_char(str);

   str = the_string;
   if ((str != NULL) && (strlen(str) >= length_alloc))
   {
      printf("ERROR! String length exceeded allocated space!\n");
      RAISE(1);
   }
   return tot_len;
}


/***********************************************************************
 ** str_cmp - compares two strings for equality
 **
 ** Parameters: new_str - the string to check against this one
 **
 ** Returns: 1 if equal
 **          0 if not equal
 **
 ***********************************************************************/

int Strings::str_cmp(char *new_str)
{
   if ((str != NULL) && (strlen(str) >= length_alloc))
   {
      printf("ERROR! String length exceeded allocated space!\n");
      RAISE(1);
   }
   return (!STRCASECMP(str, new_str));
}


/***********************************************************************
 ** str_n_cmp - adds a string onto another string, checks only n length
 **
 ** Parameters: new_str - the string to check against this one
 **             the_length - the length of the string to check
 **
 ** Returns: 1 if equal
 **          0 if not equal
 **
 ***********************************************************************/

int Strings::str_n_cmp(char *new_str, int the_length)
{
   if ((str != NULL) && (strlen(str) >= length_alloc))
   {
      printf("ERROR! String length exceeded allocated space!\n");
      RAISE(1);
   }
   return (!STRNCASECMP(str, new_str, the_length));
}


/***********************************************************************
 ** sprintf - does what sprintf does, but with strings
 **
 ** Parameters: new_str - the string to combine
 **             ...     - the vargs
 **
 ** Returns:  length of string if successful
 **          -1 if failure
 **
 ***********************************************************************/

int Strings::sprintf(char *new_str, ... )
{
   int xlen;       /* holds the total characters resulting from vsprinf */
   va_list args;   /* holds the list of arguments */
   char *tmp_str;  /* used to move along the string */
   char *arg_str;  /* used to display the argument if we want to, testing */
   int  arg_int;   /* used to hold an integer argument, for testing */
   char arg_char;  /* used to hold a char argument, for testing */
   int  count = 0;


   /* we have to first estimate the size of a string we need to allocate
      to put this into. This is not an easy task, so we have to search
      along the va_list structs and count the size of each. So far it is
      only set up for strings, integers, and characters. Anything else and
      youre SOL */

   va_start( args, new_str );
   tmp_str = new_str;
   while (*tmp_str)
   {
      if (*tmp_str == '%')
      {
         if ((*(tmp_str+1)) && (*(tmp_str+1) == 's'))
	 {
            arg_str = va_arg(args, char *);
            if (arg_str != NULL)
               count += (strlen(arg_str));
            else
	       count += 6;
         }
         else if ((*(tmp_str+1)) && (*(tmp_str+1) == 'd'))
	 {
            arg_int = va_arg(args, int);
            count += 10;
         }
         else if ((*(tmp_str+1)) && (*(tmp_str+1) == 'c'))
	 {
            arg_char = (char) va_arg(args, int);
            count += 3;
         }
         else if ((*(tmp_str+1)) && !isalpha(*(tmp_str+1)))
	 {
            arg_str = (char *) va_arg(args, void *);
            count += 100;
         }
         else
	 {
            arg_str = (char *) va_arg(args, void *);
	    count += 10;
         }
      }
      tmp_str++;
      count++;
   }

   /* increment two more just for luck */
   count += 2;

   /* now we reset the varg list and create the array, so we can load them
      all into the string */
   va_start( args, new_str );
   tmp_str = new_char(count);
   xlen = vsprintf( tmp_str, new_str, args );
   str_copy(tmp_str);

   /* now clean up after ourselves */
   va_end( args );
   delete_char(tmp_str);
   if ((str != NULL) && (strlen(str) >= length_alloc))
   {
      printf("ERROR! String length exceeded allocated space!\n");
      RAISE(1);
   }
   return xlen;
}


/***********************************************************************
 ** num_char - counts the number of a specific char in a string
 **
 ** Parameters: the_char - the character to count
 **
 ** Returns:  number of chars found
 **          -1 if failure
 **
 ***********************************************************************/

int Strings::num_char(char the_char)
{
   char *str_ptr;     /* used to move along the string */
   int count = 0;     /* number of ampersands */

   if (str == NULL)
   {
      if ((str != NULL) && (strlen(str) >= length_alloc))
      {
         printf("ERROR! String length exceeded allocated space!\n");
         RAISE(1);
      }
      return 0;
   }

   str_ptr = str;
   while (*str_ptr)
   {
      if (*str_ptr == the_char)
         count++;   
      str_ptr++;
   }
   if ((str != NULL) && (strlen(str) >= length_alloc))
   {
      printf("ERROR! String length exceeded allocated space!\n");
      RAISE(1);
   }
   return count;
}


/***********************************************************************
 ** remove_newline - replaces the last \n or \r it comes across with \0
 **
 ** Parameters: None
 **
 ** Returns:  new length of string if successful
 **          -1 if failure
 **
 ***********************************************************************/

int Strings::remove_newline(void)
{
   char *temp_ptr;
   char *newline_holder;

   temp_ptr = str;
 
   if( temp_ptr == NULL )
      return -1;

   while( *temp_ptr != '\0' )
   {
      /* look for the end of the string */
      while(    ( *temp_ptr != '\0' )
             && ( *temp_ptr != '\n' )
             && ( *temp_ptr != '\r' ) )
         temp_ptr++;

      newline_holder = temp_ptr;

      temp_ptr++;
      while(    ( *temp_ptr == '\n' )
             || ( *temp_ptr == '\r' ) )
         temp_ptr++;


      /* make sure there is a \0 at the end of the string */
      if( *temp_ptr == '\0' )
         *newline_holder = '\0';
   }

   if ((str != NULL) && (strlen(str) >= length_alloc))
   {
      printf("ERROR! String length exceeded allocated space!\n");
      RAISE(1);
   }
   return str_len();
}


/***********************************************************************
 ** remove_head_spaces - removes the spaces at the beginning of the string
 **
 ** Parameters: None
 **
 ** Returns:  new length of string if successful
 **          -1 if failure
 **
 ***********************************************************************/

int Strings::remove_head_spaces(void)
{
   char *temp_ptr;
   char *old_str;
   int  counter = 1;

   temp_ptr = str; 
   old_str = str;

   if (temp_ptr == NULL)
      return -1;

   while ((*temp_ptr) && (*temp_ptr == ' '))
   {
      temp_ptr++;
      counter++;
   }

   if (!*temp_ptr)
      return str_len();

   str = NULL;
   str_copy(temp_ptr);

   delete_char(old_str);

   return str_len();
}


/***********************************************************************
 ** remove_tail_spaces - removes the spaces at the end of the string
 **
 ** Parameters: None
 **
 ** Returns:  new length of string if successful
 **          -1 if failure
 **
 ***********************************************************************/

int Strings::remove_tail_spaces(void)
{
   char *temp_ptr;
   char *last_char;

   temp_ptr = str; 

   if (temp_ptr == NULL)
      return -1;

   while (*temp_ptr)
   {
      if (*temp_ptr != ' ')
         last_char = temp_ptr;

      temp_ptr++;
   }
   if (*temp_ptr)
      temp_ptr++;

   *temp_ptr = '\0';

   return str_len();
}


/***********************************************************************
 ** handle_backspaces - finds any backspaces and deletes the chars they
 **                     were meant to delete.  Stupid windows telnet
 **                     client requires this silly setting
 **
 ** Parameters: None
 **
 ** Returns:  new length of string if successful
 **          -1 if failure
 **
 ***********************************************************************/

int Strings::handle_backspaces(void)
{
   char *temp_ptr;
   char *old_str;
   char *new_str;
   char *begin_line;

   if (str == NULL)
      return 0;

   temp_ptr = str; 
   old_str = str;

   str = new_char(strlen(old_str) + 2);
   BZERO(str, strlen(old_str) + 2);

   new_str = str;
   begin_line = new_str;   

   while (*temp_ptr)
   {
      if (*temp_ptr == '\b')
      {
         if (new_str != begin_line)
            new_str--;
      } 
      else
      {
         *new_str = *temp_ptr;
         if (*new_str == '\n')
            begin_line = new_str+1;
         new_str++; 
      }
      temp_ptr++;
   }
   *new_str = '\0';

   length_alloc = strlen(old_str) + 2;
   delete_char(old_str);

   return str_len();
}


/***********************************************************************
 ** remove_returns - removes all \r it comes across
 **
 ** Parameters: None
 **
 ** Returns:  new length of string if successful
 **          -1 if failure
 **
 ***********************************************************************/

int Strings::remove_returns(void)
{
   Strings tmp_str;
   char *tmp_ptr;
   char *rewrite_str;

   tmp_str = str_show();
   tmp_ptr = tmp_str.str_show();

   rewrite_str = tmp_ptr; 
   while ((*tmp_ptr) && (*tmp_ptr != '\0'))
   {
      if (*tmp_ptr == '\r')
         tmp_ptr++;
      if ((*tmp_ptr) && (*tmp_ptr != '\0'))
         *rewrite_str = *tmp_ptr;
      tmp_ptr++;
      rewrite_str++;
   }
   *rewrite_str = '\0';
   str_copy(tmp_str.str_show());
   if ((str != NULL) && (strlen(str) >= length_alloc))
   {
      printf("ERROR! String length exceeded allocated space!\n");
      RAISE(1);
   }
   return str_len();
}


/***********************************************************************
 ** assign_word - assigns a word to the string
 **
 ** Parameters: the_str - the string to search for that word
 **             word_num - the position of the word from beginning of string
 **
 ** Returns:  1 for success
 **          -1 for failure
 **
 ***********************************************************************/

int Strings::assign_word(char *the_str, int word_num)
{
   char *tmp_str;
   char *str_holder;
   int first = 1; /* is this the first word we have tried */
   char *word_str;
   int  i;
   int  is_alnum = 0;

   if ((word_num <= 0) || (the_str == NULL))
   {
      if ((str != NULL) && (strlen(str) >= length_alloc))
      {
         printf("ERROR! String length exceeded allocated space!\n");
         RAISE(1);
      }
      return -1;
   }

   /* since we are going to write to the string, want to not write to the
      original, so we make a copy */
   str_holder = new_char(strlen(the_str) + 2);
   BZERO(str_holder, strlen(the_str) + 2);
   strncpy(str_holder, the_str, strlen(the_str));
   tmp_str = str_holder;

   /* skip all preceeding spaces */
   while ((*tmp_str) && (*tmp_str == ' '))
      tmp_str++;

   /* check if there is no word to get here */
   if (first && ((!*tmp_str) || (*tmp_str == '\n') || (*tmp_str == '\r')))
   {
      delete_char(str_holder);
      if ((str != NULL) && (strlen(str) >= length_alloc))
      {
         printf("ERROR! String length exceeded allocated space!\n");
         RAISE(1);
      }
      return -1;
   }

   /* pass each word that we dont need */
   for (i=1; i<word_num; i++)
   {
      is_alnum = 0;
      if ((!*tmp_str) || (*tmp_str == '\0'))
      {
         delete_char(str_holder);
         if ((str != NULL) && (strlen(str) >= length_alloc))
         {
            printf("ERROR! String length exceeded allocated space!\n");
            RAISE(1);
         }
         return -1;
      }

      /* treat a non-alphanumeric character as one word */
      if (!isalnum(*tmp_str))
      {
         is_alnum = 1;
         tmp_str++;
      }

      /* go to the end of this word */
      while ((!is_alnum) && (*tmp_str) && (*tmp_str != ' ') && 
             ((isalnum(*tmp_str)) || (*tmp_str == '@') || (*tmp_str == '_') ||
              (*tmp_str == '`') || (*tmp_str == '\'')))
         tmp_str++;

      /* remove whitespace */
      while ((*tmp_str) && (*tmp_str == ' '))
         tmp_str++;

      /* if we hit an end of line before we get to the next word */
      if ((!*tmp_str) || (*tmp_str == '\0') || (*tmp_str == '\n') || 
                                                         (*tmp_str == '\r'))
      {
         delete_char(str_holder);
         delete_char(str);
         str = NULL;
         return -1;
      }
 
      first = 0;
   }
   word_str = tmp_str;
   while ((*tmp_str) && (*tmp_str != '\0') && 
          (((isalnum(*tmp_str)) && (*tmp_str != ' ')) || 
                                 (*tmp_str == '@') || (*tmp_str == '_') ||
                                 (*tmp_str == '`') || (*tmp_str == '\'')))

   {
      tmp_str++;
   }
   if ((tmp_str == word_str) && (!isalpha(*tmp_str)))
      tmp_str++;

   if (*tmp_str)
      *tmp_str = '\0';

   if (strlen(word_str) <= 0)
   {
      delete_char(str);
      str = NULL;
      delete_char(str_holder);
      return -1;
   }
   str_copy(word_str);
   delete_char(str_holder);

   if ((str != NULL) && (strlen(str) >= length_alloc))
   {
      printf("ERROR! String length exceeded allocated space!\n");
      RAISE(1);
   }
   return 1;   
}


/***********************************************************************
 ** assign_word_sdl - assigns a word to the string, space delineated...
 **                   meaning that a word in this case is all that is 
 **                   found between two spaces
 **
 ** Parameters: the_str - the string to search for that word
 **             word_num - the position of the word from beginning of string
 **
 ** Returns:  1 for success
 **          -1 for failure
 **
 ***********************************************************************/

int Strings::assign_word_sdl(char *the_str, int word_num)
{
   char *tmp_str;
   char *str_holder;
   int first = 1; /* is this the first word we have tried */
   char *word_str;
   int  i;

   if ((word_num <= 0) || (the_str == NULL))
   {
      if ((str != NULL) && (strlen(str) >= length_alloc))
      {
         printf("ERROR! String length exceeded allocated space!\n");
         RAISE(1);
      }
      return -1;
   }

   /* since we are going to write to the string, want to not write to the
      original, so we make a copy */
   str_holder = new_char(strlen(the_str) + 2);
   BZERO(str_holder, strlen(the_str) + 2);
   strncpy(str_holder, the_str, strlen(the_str));
   tmp_str = str_holder;

   /* skip all preceeding spaces */
   while ((*tmp_str) && (*tmp_str == ' '))
      tmp_str++;

   /* check if there is no word to get here */
   if (first && ((!*tmp_str) || (*tmp_str == '\n') || (*tmp_str == '\r')))
   {
      delete_char(str_holder);
      if ((str != NULL) && (strlen(str) >= length_alloc))
      {
         printf("ERROR! String length exceeded allocated space!\n");
         RAISE(1);
      }
      return -1;
   }

   /* pass each word that we dont need */
   for (i=1; i<word_num; i++)
   {
      if ((!*tmp_str) || (*tmp_str == '\0'))
      {
         delete_char(str_holder);
         if ((str != NULL) && (strlen(str) >= length_alloc))
         {
            printf("ERROR! String length exceeded allocated space!\n");
            RAISE(1);
         }
         return -1;
      }

      /* go to the end of this word */
      while ((*tmp_str) && (*tmp_str != ' '))
         tmp_str++;

      /* remove whitespace */
      while ((*tmp_str) && (*tmp_str == ' '))
         tmp_str++;

      /* if we hit an end of line before we get to the next word */
      if ((!*tmp_str) || (*tmp_str == '\0') || (*tmp_str == '\n') || 
                                                         (*tmp_str == '\r'))
      {
         delete_char(str_holder);
         delete_char(str);
         str = NULL;
         return -1;
      }
 
      first = 0;
   }
   word_str = tmp_str;
   while ((*tmp_str) && (*tmp_str != ' '))
   {
      tmp_str++;
   }

   if (*tmp_str)
      *tmp_str = '\0';

   if (strlen(word_str) <= 0)
   {
      delete_char(str);
      str = NULL;
      delete_char(str_holder);
      return -1;
   }
   str_copy(word_str);
   delete_char(str_holder);

   if ((str != NULL) && (strlen(str) >= length_alloc))
   {
      printf("ERROR! String length exceeded allocated space!\n");
      RAISE(1);
   }
   return 1;   
}



/***********************************************************************
 ** assign_phrase - assigns all words after a certain word
 **
 ** Parameters: the_str - the string to get the phrase from
 **             after_word_num - the word to get all after
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/

int Strings::assign_phrase(char *the_str, int after_word_num)
{

   char *tmp_str;
   char *prev_str;
   int  i;


   tmp_str = the_str;
   if (after_word_num <= 0)
      return -1;

   for (i=1; i<=after_word_num; i++)
   {
      prev_str = tmp_str;
      /* go to end of this word */
      while ((*tmp_str) && (*tmp_str != ' ') && (isalnum(*tmp_str)))
         tmp_str++;

      /* if we havent moved, move forward one char at a min */
      if ((*tmp_str) && (tmp_str == prev_str))
         tmp_str++;

      /* remove whitespace */
      while ((*tmp_str) && (*tmp_str == ' '))
         tmp_str++;

      if ((!*tmp_str) || (*tmp_str == '\0'))
         return -1;

   }
   str_copy(tmp_str);
   if ((str != NULL) && (strlen(str) >= length_alloc))
   {
      printf("ERROR! String length exceeded allocated space!\n");
      RAISE(1);
   }
   return 1;  
}

/***********************************************************************
 ** operator = - does the same thing as str_copy
 **
 ** Parameters: x - the string to copy
 **
 ** Returns:  same as str_copy
 **
 ***********************************************************************/

int Strings::operator = (char *x)
{
   if ((str != NULL) && (strlen(str) >= length_alloc))
   {
      printf("ERROR! String length exceeded allocated space!\n");
      RAISE(1);
   }
   return str_copy(x);
}


/***********************************************************************
 ** operator = - does the same thing as str_copy
 **
 ** Parameters: x - the string to copy
 **
 ** Returns:  same as str_copy
 **
 ***********************************************************************/

int Strings::operator = (Strings *x)
{
   if ((str != NULL) && (strlen(str) >= length_alloc))
   {
      printf("ERROR! String length exceeded allocated space!\n");
      RAISE(1);
   }
   return str_copy(x->str_show());
}


/***********************************************************************
 ** find_in_str - finds a string inside another string divided by ' '
 **
 ** Parameters: find_str - the string to find
 **
 ** Returns: 1 if found
 **          0 if not found
 **
 ***********************************************************************/

int Strings::find_in_str(char *find_str)
{
   char *finder;

   if ((find_str == NULL) || (strlen(find_str) < 1))
      return 0;

   finder = str;
  
   if (finder == NULL)
      return 0;

   while (*finder)
   {
      if (!STRNCASECMP(find_str, finder, strlen(find_str)))
	return 1;
      while ((*finder) && (*finder != ' '))
         finder++;
      finder++;
   }
   if ((str != NULL) && (strlen(str) >= length_alloc))
   {
      printf("ERROR! String length exceeded allocated space!\n");
      RAISE(1);
   }
   return 0;
}


/***********************************************************************
 ** make_newlines_visible - converts newlines from '\n' to '\\n' so they
 **                         will print out the characters and not treat
 **                         it as a newline when sent to the players
 **
 ** Parameters: None
 **
 ** Returns:  new length if success
 **          -1 if failure
 **
 ***********************************************************************/

int Strings::make_newlines_visible(void)
{
   char *old_str;
   char *temp_ptr;
   char *new_str;

   temp_ptr = str;
   int size = 0; 

   if ((str == NULL) || (str_len() == 0))
      return -1;

   while ((*temp_ptr) && (*temp_ptr != '\0'))
   {    
      
      if ((*temp_ptr == '\\') && (*(temp_ptr+1) == 'n'))
      {
         temp_ptr++;
         size += 3;
      }
      else
         size++;
      if (*temp_ptr)
         temp_ptr++;
   }

   old_str = str;
   str = strmalloc(size+2);
   length_alloc = size+2;
   temp_ptr = old_str;
   new_str = str;

   while ((*temp_ptr) && (*temp_ptr != '\0'))
   {
      if ((*temp_ptr == '\\') && (*(temp_ptr+1) == 'n'))
      {
         *new_str++ = '\\';
         *new_str++ = '\\';
         *new_str++ = 'n';
         temp_ptr += 2; 
      }
      else
      {
         *new_str++ = *temp_ptr++;
      }
   }
   *new_str = '\0';

   delete_char(old_str);

   if ((str != NULL) && (strlen(str) >= length_alloc))
   {
      printf("ERROR! String length exceeded allocated space!\n");
      RAISE(1);
   }
   return str_len();
}


/***********************************************************************
 ** format_for_comm - converts the string that is passed in into a format
 **                   designed for communications lines.
 **
 ** Parameters: None
 **
 ** Returns:  new length if success
 **          -1 if failure
 **
 ***********************************************************************/

int Strings::format_for_comm(char *the_string, int indent, int max_len)
{
   char *str_ptr;       // used to move along the string
   char *str_holder;   // holds a temp string we allocate
   char *last_space;   // points to the last space found
   int  is_first = 1;  // is this the first line
   char *spaces;       // holds a string of spaces the size of the indent
   char *start;        // the start of the line we are working on
   char *color_marker = NULL; // marks where the color should go
   char char_holder;
   char char_holder2 = '\0';
   int i;
   int is_colored = 0; // is the text cyan

   if ((the_string == NULL) || (indent < 0) || (max_len <= 0))
      return -1;

   /* if the indent is too long, dont format it so we avoid a crash */
   if (indent >= (max_len-10))
      return -1;

   /* if there is a string already here, destroy it */
   if (str != NULL)
   {
      delete_char(str);
      str = NULL;
      length_alloc = 0;
   }

   /* create a place to hold the passed-in string */
   str_holder = new_char(strlen(the_string) + 2);
   BZERO(str_holder, strlen(the_string) + 2);
   strcpy(str_holder, the_string);

   /* create a string with the initial spaces on consecutive lines */
   spaces = new_char(indent + 2);
   BZERO(spaces, indent + 2);
   str_ptr = spaces;
   for (i=0; i<indent; i++)
   {
      *str_ptr = ' ';
      str_ptr++;
   }
   *str_ptr = '\0';


   str_ptr = str_holder;

   /* first, ignore the spaces at the beginning of the line, we don't
      indent these */

   while ((*str_ptr) && (*str_ptr == ' '))
      str_ptr++;

   while (*str_ptr) 
   {
      last_space = NULL;

      /* if this is not the first line, append with spaces */
      if (!is_first)
      {
         str_cat(spaces);         
      }

      start = str_ptr;
      for (i=0; i<(max_len-indent); i++)
      {
   
         if (!*str_ptr)
            break;

         if (*str_ptr == ' ')
	 {
            last_space = str_ptr;
            if (is_colored)
	    {
	       *color_marker = '\0';
	       str_cat(start);
	       str_cat("&+C");
	       start = color_marker + 1;
               *str_ptr = '\0';
               str_cat(start);
               str_cat(" &+G");
               start = str_ptr + 1;
               is_colored = 0;            
            }
         }

         if (*str_ptr == '$')
	 {
            is_colored = 1;
            color_marker = str_ptr;
         }

         str_ptr++;
      }
      if ((last_space != NULL) && (*str_ptr) && (i == (max_len - indent)))
         str_ptr = last_space;

      if (*str_ptr)
      {
         if (last_space == NULL)
	 {
            char_holder2 = *str_ptr;
	 }
         *str_ptr = '\n';
         str_ptr++;
         char_holder = *str_ptr;
         *str_ptr = '\0';
         str_cat(start);
         *str_ptr = char_holder;
         if (last_space == NULL)
	 {
            str_ptr--;
            *str_ptr = char_holder2;
	 }
         if (is_colored)
            color_marker = str_ptr;
      }
      else
      {
         if (i != (max_len - indent))
	 {
            if (is_colored)
	    {
	       *color_marker = '\0';
	       str_cat(start);
	       str_cat("&+C");
	       start = color_marker + 1;
               is_colored = 0;            
            }
            str_cat(start);
            str_cat("&*");
         }

         /* if this is exactly the max_len - indent long, we have a problem,
            so just write the thing and be done with it */
         else if (last_space != '\0')
	 {
            str_cat(start);
            str_cat("&*");
         }
         else
	 {
            str_cat("\n");
            str_ptr++;
            last_space = str_ptr;
         }
      }
      is_first = 0;
   }
   delete_char(str_holder);
   delete_char(spaces);

   return strlen(str);
}


/***********************************************************************
 ** swap_chars - swaps one char for another
 **
 ** Parameters: char1 - the char we are swapping from
 **             char2 - what we are changing char1 to
 **
 ** Returns:  1 for success, -1 for failure
 **
 ***********************************************************************/

int Strings::swap_chars(char char1, char char2)
{
   char *str_ptr;

   if (str == NULL)
      return -1;

   str_ptr = str;
   while (*str_ptr)
   {
      if (*str_ptr == char1)
         *str_ptr = char2;
      str_ptr++;
   }
   return 1;
}

#ifndef CONVERTER

/***********************************************************************
 ** format_for_actions - formats the string for action execution, putting
 **                      appropriate things in place of the %a and stuff
 **
 ** Parameters: the_string - the string we are formatting
 **             the_user -   the user who is doing the acting
 **             the_target - the target who is getting acted upon
 **
 ** Returns:  1 for success, -1 for failure
 **
 ***********************************************************************/

int Strings::format_for_actions(char *the_string, MudObject *the_user, 
                                                  Individual *the_target)
{
   char *tmp_str;
   char *start_str;
   char *str_holder;
   int  sex = -1;

   if ((the_string == NULL) || (the_user == NULL))
      return -1;

   if ((the_user->get_type() != OBJ_TYPE_PLAYER) && 
       (the_user->get_type() != OBJ_TYPE_BUILDER))
      return -1;
 
   if (str != NULL)
   {
      delete_char(str);
      str = NULL;    
   }

   str_holder = new_char(strlen(the_string) + 2);
   BZERO(str_holder, strlen(the_string) + 2);
   strcpy(str_holder, the_string);
   
   tmp_str = str_holder;
   start_str = tmp_str;
   while (*tmp_str)
   {
      if (*tmp_str == '%')
      {
         if ((*(tmp_str+1)) && (*(tmp_str+1) == 'a'))
	 {
            if (start_str != tmp_str)
	    {
               *tmp_str = '\0';
               str_cat(start_str);
            }
            str_cat(the_user->get_title());
            tmp_str += 2;
            start_str = tmp_str;
         }
         else if ((*(tmp_str+1)) && (*(tmp_str+1) == 't') && 
                                                   (the_target != NULL))
	 {
            if (start_str != tmp_str)
	    {
               *tmp_str = '\0';
               str_cat(start_str);
            }
            str_cat(the_target->get_title());
            tmp_str += 2;
            start_str = tmp_str;
         }
         else if ((*(tmp_str+1)) && 
                         ((*(tmp_str+1) == '[') || (*(tmp_str+1) == '<')) &&
		                     (the_user->get_type() == OBJ_TYPE_PLAYER))
	 {
            sex = -1;
            if (*(tmp_str+1) == '[')
               sex = ((Player *) the_user)->is_male();
            else if (the_target != NULL)
               sex = ((Individual *) the_target)->is_male();
            
            if (sex != -1)
	    {
	       /* first, write what we have so far */ 
               if (start_str != tmp_str)
	       {
                  *tmp_str = '\0';
                  str_cat(start_str);
               }
               tmp_str += 2;
               if (!*tmp_str)
                  break;
               start_str = tmp_str;

               /* go to the end of the first word */
               while ((*tmp_str) && (*tmp_str != '%'))
                  tmp_str++;
               if (!*tmp_str)
                  break;
 
               /* if male, use the first one */
               if (sex)
	       {
                  *tmp_str = '\0';
                  str_cat(start_str);
               }
               tmp_str++;
               if (!*tmp_str)
                  break;
               start_str = tmp_str;

               /* go to the end of the first word */
               while ((*tmp_str) && ((*tmp_str != '>') && (*tmp_str != ']')))
                  tmp_str++;

               if (!*tmp_str)
                  break;
 
               /* if male, use the first one */
               if (!sex)
	       {
                  *tmp_str = '\0';
                  str_cat(start_str);
               }
               tmp_str++;
               if (!*tmp_str)
                  break;
               start_str = tmp_str;
            }
         }
         else if ((*(tmp_str+1)) && (*(tmp_str+1) == '!') && 
		  (the_user->get_type() == OBJ_TYPE_PLAYER))
	 {
            if (start_str != tmp_str)
	    {
               *tmp_str = '\0';
               str_cat(start_str);
            }
            if (((Player *)the_user)->is_male())
               str_cat("his");
            else
               str_cat("her");
            tmp_str += 2;
            start_str = tmp_str;
         }
         else if ((*(tmp_str+1)) && (*(tmp_str+1) == '~') && 
                (the_target != NULL) && (the_user->get_type() == OBJ_TYPE_PLAYER))
	 {
            if (start_str != tmp_str)
	    {
               *tmp_str = '\0';
               str_cat(start_str);
            }
            if (((Player *) the_user)->is_male())
               str_cat("him");
            else
               str_cat("her");
            tmp_str += 2;
            start_str = tmp_str;
         }
         else
            tmp_str++;
      }
      else
         tmp_str++;
   }
   if (start_str != tmp_str)
      str_cat(start_str);
   delete_char(str_holder);
   return 1;
}


/***********************************************************************
 ** format_for_abilities - formats the string for abilities execution, 
 **                        putting appropriate things in place of the 
 **                        %a and stuff
 **
 ** Parameters: the_string - the string we are formatting
 **             the_player - the player who is doing the acting
 **             the_target - the target who is getting acted upon
 **
 ** Returns:  1 for success, -1 for failure
 **
 ***********************************************************************/

int Strings::format_for_abilities(char *the_string, Player *the_player, 
                                                  MudObject *the_target)
{
   char *tmp_str;
   char *start_str;
   char *str_holder;
   int  sex = -1;

   if ((the_string == NULL) || (the_player == NULL))
      return -1;

   if (str != NULL)
   {
      delete_char(str);
      str = NULL;    
   }

   str_holder = new_char(strlen(the_string) + 2);
   BZERO(str_holder, strlen(the_string) + 2);
   strcpy(str_holder, the_string);
   
   tmp_str = str_holder;
   start_str = tmp_str;
   while (*tmp_str)
   {
      if (*tmp_str == '%')
      {
         if ((*(tmp_str+1)) && (*(tmp_str+1) == 'a'))
	 {
            if (start_str != tmp_str)
	    {
               *tmp_str = '\0';
               str_cat(start_str);
            }
            str_cat(the_player->get_title());
            tmp_str += 2;
            start_str = tmp_str;
         }
         else if ((*(tmp_str+1)) && (*(tmp_str+1) == 't') && 
                                                   (the_target != NULL))
	 {
            if (start_str != tmp_str)
	    {
               *tmp_str = '\0';
               str_cat(start_str);
            }

            str_cat(the_target->get_title());
            tmp_str += 2;
            start_str = tmp_str;
         }
         else if ((*(tmp_str+1)) && 
                         ((*(tmp_str+1) == '[') || (*(tmp_str+1) == '<')))
	 {
            sex = -1;
            if (*(tmp_str+1) == '[')
               sex = the_player->is_male();
            else if (the_target != NULL)
               sex = ((Individual *) the_target)->is_male();
            
            if (sex != -1)
	    {
	       /* first, write what we have so far */ 
               if (start_str != tmp_str)
	       {
                  *tmp_str = '\0';
                  str_cat(start_str);
               }
               tmp_str += 2;
               if (!*tmp_str)
                  break;
               start_str = tmp_str;

               /* go to the end of the first word */
               while ((*tmp_str) && (*tmp_str != '%'))
                  tmp_str++;
               if (!*tmp_str)
                  break;
 
               /* if male, use the first one */
               if (sex)
	       {
                  *tmp_str = '\0';
                  str_cat(start_str);
               }
               tmp_str++;
               if (!*tmp_str)
                  break;
               start_str = tmp_str;

               /* go to the end of the first word */
               while ((*tmp_str) && ((*tmp_str != '>') && (*tmp_str != ']')))
                  tmp_str++;

               if (!*tmp_str)
                  break;
 
               /* if male, use the first one */
               if (!sex)
	       {
                  *tmp_str = '\0';
                  str_cat(start_str);
               }
               tmp_str++;
               if (!*tmp_str)
                  break;
               start_str = tmp_str;
            }
         }
         else if ((*(tmp_str+1)) && (*(tmp_str+1) == '!'))
	 {
            if (start_str != tmp_str)
	    {
               *tmp_str = '\0';
               str_cat(start_str);
            }
            if (the_player->is_male())
               str_cat("his");
            else
               str_cat("her");
            tmp_str += 2;
            start_str = tmp_str;
         }
         else if ((*(tmp_str+1)) && (*(tmp_str+1) == '~') && 
                                                   (the_target != NULL))
	 {
            if (start_str != tmp_str)
	    {
               *tmp_str = '\0';
               str_cat(start_str);
            }
            if (the_player->is_male())
               str_cat("him");
            else
               str_cat("her");
            tmp_str += 2;
            start_str = tmp_str;
         }
         else
            tmp_str++;
      }
      else
         tmp_str++;
   }
   if (start_str != tmp_str)
      str_cat(start_str);
   delete_char(str_holder);
   return 1;
}
#endif

/***********************************************************************
 ** copy_lines - copies a set number of lines to this string from a string
 **
 ** Parameters: the_string - the string we are copying from
 **             lines - the number of lines to copy
 **
 ** Returns:  1 for success, -1 for failure
 **
 ***********************************************************************/

int Strings::copy_lines(char *the_string, int lines)
{
   char char_holder;
   char *str_ptr;
   int  i;

   if ((the_string == NULL) || (lines <= 0))
      return -1;

   str_ptr = the_string;
   for(i=0; i<lines; i++)
   {
      while ((*str_ptr) && (*str_ptr != '\n'))
         str_ptr++;

      if (!*str_ptr)
         break;
    
      str_ptr++;
      if ((*str_ptr) && (*str_ptr == '\r'))
         str_ptr++;
   }

   char_holder = *str_ptr;
   *str_ptr = '\0';
   str_copy(the_string);
   *str_ptr = char_holder;
   return 1;
}


/***********************************************************************
 ** del_lines - deletes a specified number of lines from this string
 **
 ** Parameters: lines - the number of lines to delete
 **
 ** Returns:  1 for success, -1 for failure
 **
 ***********************************************************************/

int Strings::del_lines(int lines)
{
   char *str_holder;
   char *str_ptr;
   int  i;
   int  len = 0;  
 

   if ((str == NULL) || (lines <= 0))
      return -1;

   str_ptr = str;
   for(i=0; i<lines; i++)
   {
      while ((*str_ptr) && (*str_ptr != '\n'))
      {
         str_ptr++;
      }

      if (!*str_ptr)
         break;
    
      str_ptr++;
      if ((*str_ptr) && (*str_ptr == '\r'))
      {
         str_ptr++;
      }
   }

   if (!*str_ptr)
   {
      truncate(0);
      return 0;
   }

   len = strlen(str_ptr);
   str_holder = new_char(len + 2);
   BZERO(str_holder, (len + 2));
   strcpy(str_holder, str_ptr);
   str_copy(str_holder);
   delete_char(str_holder);

   return 1;
}


/***********************************************************************
 ** get_mem_size - gets how much memory this special is taking up
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Strings::get_mem_size()
{
   int size = 0;

   size = sizeof(this);
   size += get_mem_size_dynamic();
   return size;
}

/***********************************************************************
 ** get_mem_size_dynamic - gets how much memory is taken up by pointers
 **                        pointing to other objects, not including the
 **                        sizeof(this)
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Strings::get_mem_size_dynamic()
{
   int size = 0;

   if (str != NULL)
      size = sizeof(str);

   return size;   
}


/***********************************************************************
 ** sanitize_input - takes the user's input and ensures we don't have
 **                  anything in it that would be bad, such as %s and
 **                  %d that may confuse the processor.  Basically it
 **                  just converts the % to a %%
 **
 ** Returns: number sanitized if successful
 **          -1 if failure
 **
 ***********************************************************************/

int Strings::sanitize_input()
{
   char *old_str;
   char *start_ptr;
   char *end_ptr;

   if (str == NULL)
   {
      return -1;  
   }

   /* first we save the old string and dereference the str pointer from it */
   old_str = str;
   str = NULL;

   /* now we copy back the old string, making changes where necessary */
   start_ptr = old_str;
   end_ptr = start_ptr;
   while (*end_ptr)
   {
      if (*end_ptr == '%')
      {
         *end_ptr = '\0';
         str_cat(start_ptr);
         str_cat("%%");
         start_ptr = end_ptr+1;
      }
      end_ptr++;       
   }
   str_cat(start_ptr);
   delete_char(old_str);
   return 1;
}

#endif





