/**********************************************************************
 ** Food - a merger object that can be eaten, and supplies some sort of
 **        improvement to the player
 **
 ** Last reviewed: version 0.10
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **    
 **********************************************************************/

#ifndef FOOD_C
#define FOOD_C

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "mudtypes.h"
#include "mudobject.h"
#include "objtype.h"
#include "food.h"
#include "flags.h"
#include "player.h"

/***********************************************************************
 ** Food (constructor)
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Food::Food(void)
{
}


/***********************************************************************
 ** Food (constructor) - creates the food item
 **
 ** Parameters: the_name - the name of the food
 **             the_area - the area the food belongs to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Food::Food(char *the_name, char *the_area)
{
   Strings holding;

   /* this should be the only place that this is set */
   obj_type = OBJ_TYPE_FOOD;
   
   if (the_name != NULL)
      set_name(the_name);
   
   if (the_area != NULL)
      set_area(the_area);

   size = weight = state = 0;  
   number_of = energy = 0;
   
   set_loc_str("none");
}

/***********************************************************************
 ** ~Food (destructor) - destroys it
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Food::~Food()
{
}


/***********************************************************************
 ** write_object - writes the food to a specified file in builder
 **                file format
 **
 ** Parameters: the_file - the file to write to
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
void Food::write_object(FILE *the_file, int build_format)
{
   fprintf(the_file, "\nfood %s\n", get_name());
   if (build_format)
      fprintf(the_file, "%d\n", is_modified());

   write_mudobject_attrib(the_file);
   write_item_attrib(the_file);
   write_moveable_attrib(the_file);
   write_merger_attrib(the_file);
   fprintf(the_file, "%d\n", get_energy());
}

/***********************************************************************
 ** describe - describes the marker to a builder
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void Food::describe(Builder *the_builder)
{
   the_builder->send_bldr("\n&+GFood: \t\t&+M%s&*\n", get_name());
   the_builder->send_bldr("&+GTitle: \t\t&+w%s&*\n", get_title());
   the_builder->send_bldr("&+GAltnames: \t&+g%s&*\n", get_altname());
   the_builder->send_bldr("&+GSpecials: \t&+g%s&*\n", get_special_str());
   the_builder->send_bldr("&+GGuards: \t&+g%s&*\n", get_guards());
   the_builder->send_bldr("&+GLocation: \t&+M%s&*\n", get_location());
   the_builder->send_bldr("&+GSize: \t\t&+w%d&*\n", get_orig_size());
   the_builder->send_bldr("&+GWeight:\t\t&+w%d&*\n", get_orig_weight());
   the_builder->send_bldr("&+GEnergy:\t\t&+w%d&*\n", get_energy());
   the_builder->send_bldr("&+GBrief0:\n&*%s\n", get_brief(0));
   the_builder->send_bldr("&+GBrief1:\n&*%s\n", get_brief(1));
   the_builder->send_bldr("&+GDesc:&*\n%s\n", get_desc());

   the_builder->send_bldr("\n");
}


/***********************************************************************
 ** describe - describes the marker to a player
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void Food::describe(Player *the_player)
{
   MudObject *tmp_container;
   Strings   container_name;

   the_player->send_plr("\n&+GFood: \t\t&+M%s&*\n", get_name());
   the_player->send_plr("&+GTitle: \t\t&+w%s&*\n", get_title());
   the_player->send_plr("&+GAltnames: \t&+g%s&*\n", get_altname());
   the_player->send_plr("&+GSpecials: \t&+g%s&*\n", get_special_str());
   the_player->send_plr("&+GGuards: \t&+g%s&*\n", get_guards());
   the_player->send_plr("&+GStartLoc: \t&+M%s&*\n", get_location());

   tmp_container = get_contained_by();
   if (tmp_container == NULL)
      container_name = "nowhere";
   else
      container_name.sprintf("%s@%s", tmp_container->get_name(), 
                                         tmp_container->get_area());

   the_player->send_plr("&+GCurrentLoc: \t&+M%s&*\n", 
                                                 container_name.str_show());

   the_player->send_plr("&+GSize: \t\t&+w%d&*\n", get_orig_size());
   the_player->send_plr("&+GWeight:\t\t&+w%d&*\n", get_orig_weight());
   the_player->send_plr("&+GEnergy:\t\t&+w%d&*\n", get_energy());
   the_player->send_plr("&+GBrief0:\n&*%s\n", get_brief(0));
   the_player->send_plr("&+GBrief1:\n&*%s\n", get_brief(1));
   the_player->send_plr("&+GDesc:&*\n%s\n", get_desc());

   list_specials(the_player);

   the_player->send_plr("\n");
}


/***********************************************************************
 ** set_attrib - sets a specified attribute to a specified value
 **
 ** Parameters: the_builder - the builder who is changing this attribute
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
int Food::set_attrib(Builder *the_builder, Parse *the_parsed){

   if (the_parsed->get_target1() == NULL)
   {   the_builder->
           send_bldr("You can set the following attributes on food.\n"
       "   title, weight, size, altnames, desc, itemflags, brief0,\n", 
       "   brief1, location, guards, code, specials, and energy\n");
       return -1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "title",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_title(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "weight",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_weight(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "size",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_size(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "altnames",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_altnames(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "guards",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_guard(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "specials", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_special(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "desc", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_desc(the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "brief0", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_brief(the_builder, 0);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "itemflags",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_itemflags(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "brief1", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_brief(the_builder, 1);
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "energy", 
                               strlen(the_parsed->get_target1())))
   {
      
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a number as well.\n");
         return -1;
      }
 
      if (!isdigit(*(the_parsed->get_speech()))) 
      {
         the_builder->send_bldr("You need to specify a number as energy.\n");
         return -1;
      }

      set_energy(atoi(the_parsed->get_speech()));
      the_builder->send_bldr("Energy set to %d on food object %s.\n",
                                           get_energy(), get_name());
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "location", 
                               strlen(the_parsed->get_target1())))
   {
      the_builder->
        send_bldr("Sorry, you can't set the location attribute on a merger.\n"
                  "It must be set to the object holding location.\n");
      return -1;
   }

   the_builder->send_bldr("The attribute '%s' is not a moveable attribute.\n",
                                           the_parsed->get_target1());
   return -1;
}


/***********************************************************************
 ** copy_object - copies the food to an object of a different name
 **
 ** Parameters: copy_obj - copy attributes from this object
 **
 ** Returns:  1 if succeeded 
 **           0 if failed
 **
 ***********************************************************************/
int Food::copy_object(Entity *copy_obj)
{
   Food *copy_from;

   if (copy_obj->get_type() != OBJ_TYPE_FOOD)
      return 0;

   copy_from = (Food *) copy_obj;

   /******* set the mudobject attributes *****/
   copy_mudobject_attrib((MudObject *) copy_from);

   /****** set the item attributes ******/
   copy_item_attrib((Item *) copy_from);

   /****** set the moveable attributes ******/
   copy_moveable_attrib((Moveable *) copy_from);

   /****** set the merger attributes ******/
   copy_merger_attrib((Merger *) copy_from);

   /****** set the food attributes ******/
   set_energy(copy_from->get_energy());

   return 1;
}


/***********************************************************************
 ** operator = - copies an object to this object
 **
 ** Parameters: None
 **
 ** Returns: a pointer to this object copied to
 **
 ***********************************************************************/

Food *Food::operator = (Food *copy_from)
{
   if (!STRCASECMP(copy_from->get_name(), get_name()))
      return NULL;

   copy_object(copy_from);
   return this;
}


/***********************************************************************
 ** set_energy - sets the energy that this food item is worth
 **
 ** Parameters: new_num - the new number to set energy to
 **
 ** Returns:  1 if succeeded 
 **           -1 if failed
 **
 ***********************************************************************/
int Food::set_energy(int new_num)
{
   energy = new_num;
   return 1;
}



/***********************************************************************
 ** get_energy - gets the energy in one of this food item
 **
 **
 ***********************************************************************/

int Food::get_energy() 
{ 
   return energy;
}


/***********************************************************************
 ** read_food_attrib - reads in food attributes from the file
 **
 ** Parameters: read_file - the file to read in from
 **             areaname - the area that we are reading
 **             error_log - the error log to write any errors to
 **
 ** Returns:  1 for successful read
 **          -1 for errors in the read
 **
 ***********************************************************************/

int Food::read_food_attrib(FILE *read_file, ErrLog *error_log)
{
   token_record *the_token;
   Strings      holder;

   /* Set food energy */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute energy in mudobject %s", 
                                                            get_name());
      error_log->log_err(holder.str_show(), "read_food_attrib");
      return -1;
   }
   set_energy(atoi(the_token->the_string));

   return 1;
}


/***********************************************************************
 ** get_mem_size - gets how much memory this special is taking up
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Food::get_mem_size()
{
   int size = 0;

   size = sizeof(this);
   size += get_mem_size_dynamic();
   return size;
}

/***********************************************************************
 ** get_mem_size_dynamic - gets how much memory is taken up by pointers
 **                        pointing to other objects, not including the
 **                        sizeof(this)
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Food::get_mem_size_dynamic()
{
   int  size = 0;
   

   size += get_mem_size_merger();
   size += get_mem_size_moveable();
   size += get_mem_size_item();
   size += get_mem_size_mudobj();
   size += get_mem_size_entity();

   return size;
}

#endif



