/**********************************************************************
 ** aime - functions to convert an aime format to another aime format
 **  
 **
 **    
 **
 **********************************************************************/

#ifndef AIME_C
#define AIME_C

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include "aime.h"
#include "itemflagold.h"
#include "../../include/strings.h"
#include "../../include/lexer.h"
#include "../../include/config.h"
#include "../../include/sysdep.h"
#include "../../include/utils.h"
#include "../../include/ability.h"
#include "../../include/itemflags.h"
#include "../../include/doorflags.h"
#include "../../include/indflags.h"
#include "convconfig.h"

long conv_doorflag;

/***********************************************************************
 ** convert_mudobj_attrib - reads in the mudobject attributes and converts
 **                         them to pseudo-data format
 **
 ** Parameters: read_file - the AIME file we read from
 **             write_file - the AIME file we write to
 **             objname - the name of the object we are converting
 **             version - which version we are loading
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int convert_mudobj_attrib(FILE *read_file, FILE *write_file, char *objname, float version)
{
   token_record *the_token;
   char         *temp_desc;

   fprintf(write_file, "Name ^%s^\n", objname);

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      printf("Invalid format in mudobject %s.\n", objname);
      return -1;
   }

   /* get the title of this location */
   the_token = get_token(read_file, '^');
   fprintf(write_file, "Title ^%s^\n", (the_token->the_string == NULL) ? "" : 
																		the_token->the_string);

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      printf("Invalid format in mudobject %s.\n", objname);
   }

   /* get the altnames */
   the_token = get_token(read_file, '^');
   fprintf(write_file, "Altname ^%s^\n", (the_token->the_string == NULL) ? "" : 
																		the_token->the_string);

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      printf("Invalid format in mudobject %s.\n", objname);
      return -1;
   }

   /* get the clones */
   the_token = get_token(read_file, '^');
   fprintf(write_file, "Clones ^%s^\n", (the_token->the_string == NULL) ? "" : 
																		the_token->the_string);


   /* get the object location */
   the_token = get_token(read_file, '\0');
   fprintf(write_file, "Location ^%s^\n", (the_token->the_string == NULL) ? "" : 
																		the_token->the_string);

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      printf("Invalid format in mudobject %s.\n", objname);
      return -1;
   }

   /* get the specials */
   the_token = get_token(read_file, '^');
   fprintf(write_file, "SpecialStr ^%s^\n", (the_token->the_string == NULL) ? "" : 
																		the_token->the_string);

   /* get the description */
   temp_desc = read_desc_type(read_file, 0, NULL);
   
   if (temp_desc == NULL)
      return -1;
   fprintf(write_file, "Desc ^\n%s^\n", temp_desc);
   delete temp_desc;

   /* Set capacity */
   the_token = get_token(read_file, '\0');

   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in mudobject %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "Capacity %d\n", atoi(the_token->the_string));

   version = 0;
   return 1;
}

/***********************************************************************
 ** convert_itemflag - converts the itemflag values from the old format
 **                     (pre 0.61) to the new itemflag format, plus filling
 **                     the doorflags if applicable
 **
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

long convert_itemflag(long oldflags)
{
   long newflags = 0;

   conv_doorflag = 0;

   newflags |= (((oldflags & (1 << ITEMFLAGOLD_CONTAINER)) ? 1 : 0) << 
		ITEMFLAG_CONTAINER);
   newflags |= (((oldflags & (1 << ITEMFLAGOLD_LIGHTABLE)) ? 1 : 0) << 
		ITEMFLAG_LIGHTABLE);
   newflags |= (((oldflags & (1 << ITEMFLAGOLD_LIT)) ? 1 : 0) << ITEMFLAG_LIT);
   newflags |= (((oldflags & (1 << ITEMFLAGOLD_GLOWING)) ? 1 : 0) << 
		ITEMFLAG_GLOWING); 
   newflags |= (((oldflags & (1 << ITEMFLAGOLD_BOOZE)) ? 1 : 0) << ITEMFLAG_BOOZE);
   conv_doorflag |= (((oldflags & (1 << ITEMFLAGOLD_ROPETIE)) ? 1 : 0) << 
		     DOORFLAG_ROPETIE); 
   newflags |= (((oldflags & (1 << ITEMFLAGOLD_ISTIED)) ? 1 : 0) << 
		ITEMFLAG_ISTIED); 
   conv_doorflag |= (((oldflags & (1 << ITEMFLAGOLD_TIEINSIDE)) ? 1 : 0) << 
		     DOORFLAG_TIEINSIDE); 
   conv_doorflag |= (((oldflags & (1 << ITEMFLAGOLD_TIEOUTSIDE)) ? 1 : 0) << 
		     DOORFLAG_TIEOUTSIDE); 
   conv_doorflag |= (((oldflags & (1 << ITEMFLAGOLD_UNTIEINSIDE)) ? 1 : 0) << 
		     DOORFLAG_UNTIEINSIDE);
   conv_doorflag |= (((oldflags & (1 << ITEMFLAGOLD_UNTIEOUTSIDE)) ? 1 : 0) << 
		     DOORFLAG_UNTIEOUTSIDE);
   conv_doorflag |= (((oldflags & (1 << ITEMFLAGOLD_SPECIALDOOR)) ? 1 : 0) << 
		     DOORFLAG_SPECIALDOOR);
   newflags |= (((oldflags & (1 << ITEMFLAGOLD_LOCKABLE)) ? 1 : 0) << 
		ITEMFLAG_LOCKABLE); 
   newflags |= (((oldflags & (1 << ITEMFLAGOLD_CLOSEABLE)) ? 1 : 0) << 
		ITEMFLAG_CLOSEABLE); 
   newflags |= (((oldflags & (1 << ITEMFLAGOLD_LOCKED)) ? 1 : 0) << 
		ITEMFLAG_LOCKED); 
   newflags |= (((oldflags & (1 << ITEMFLAGOLD_CLOSED)) ? 1 : 0) << 
		ITEMFLAG_CLOSED); 
   newflags |= (((oldflags & (1 << ITEMFLAGOLD_MAILBOX)) ? 1 : 0)
		<< ITEMFLAG_MAILBOX); 
   newflags |= (((oldflags & (1 << ITEMFLAGOLD_WORN)) ? 1 : 0)
		<< ITEMFLAG_WORN); 
   conv_doorflag |= (((oldflags & (1 << ITEMFLAGOLD_OUTSIDEINVIS)) ? 1 : 0) << 
		     DOORFLAG_OUTSIDEINVIS);
   return newflags;
}


/***********************************************************************
 ** convert_item_attrib - reads in the item attributes and converts
 **                       them to pseudo-data format
 **
 ** Parameters: read_file - the AIME file we read from
 **             write_file - the AIME file we write to
 **             objname - the name of the object we are converting
 **             version - which version we are loading
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int convert_item_attrib(FILE *read_file, FILE *write_file, char *objname, float version)
{
   token_record *the_token;

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      printf("Invalid format in item %s.\n", objname);
      return -1;
   }

   /* get the guards */
   the_token = get_token(read_file, '^');
   fprintf(write_file, "Guard ^%s^\n", (the_token->the_string == NULL) ? "" : 
																		the_token->the_string);

   /* Set flag1 */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in item %s.\n", objname);
      return -1;
   }

   if (version <= (float) 0.592)
     fprintf(write_file, "ItemFlag1 %ld\n", 
	                       convert_itemflag(atol(the_token->the_string)));
   else
     fprintf(write_file, "ItemFlag1 %ld\n", atol(the_token->the_string));
   
   /* Set flag2 */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in item %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "ItemFlag2 %ld\n", atol(the_token->the_string));

   version = 0;

   return 1;
}


/***********************************************************************
 ** convert_moveable_attrib - reads in the moveable attributes and converts
 **                           them to pseudo-data format
 **
 ** Parameters: read_file - the AIME file we read from
 **             write_file - the AIME file we write to
 **             objname - the name of the object we are converting
 **             version - which version we are loading
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int convert_moveable_attrib(FILE *read_file, FILE *write_file, char *objname, float version)
{
   token_record *the_token;
   char         *tmp_charholder;

   /* set the brief desc */
   tmp_charholder = read_desc_type(read_file, 0, NULL); 
   fprintf(write_file, "Brief1 ^\n%s^\n", tmp_charholder);
   delete tmp_charholder;

   /* set the brief desc */
   tmp_charholder = read_desc_type(read_file, 0, NULL); 
   fprintf(write_file, "Brief2 ^\n%s^\n", tmp_charholder);
   delete tmp_charholder;


   /* Set size */
   the_token = get_token(read_file, '\0');
     
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in moveable %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "Size %d\n", atoi(the_token->the_string));

   /* Set weight */
   the_token = get_token(read_file, '\0');
     
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in moveable %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "Weight %d\n", atoi(the_token->the_string));

   if (version > 0.57)
   {
      /* Set num_allowed */
      the_token = get_token(read_file, '\0');
     
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in moveable %s.\n", objname);
         return -1;
      }
      fprintf(write_file, "NumAllowed %d\n", atoi(the_token->the_string));

      /* Set percent_allowed */
      the_token = get_token(read_file, '\0');
     
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in moveable %s.\n", objname);
         return -1;
      }
      fprintf(write_file, "PercentAllowed %d\n", atoi(the_token->the_string));

   }

   version = 0;
   return 1;
}


/***********************************************************************
 ** convert_mobile_attrib - reads in the mobile attributes and converts
 **                         them to pseudo-data format
 **
 ** Parameters: read_file - the AIME file we read from
 **             write_file - the AIME file we write to
 **             objname - the name of the object we are converting
 **             version - which version we are loading
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int convert_mobile_attrib(FILE *read_file, FILE *write_file, char *objname, float version)
{
   token_record *the_token;
   char         *temp_desc;
   bool         sex;
   int          flag_offset;
   int          which_flag;
   int          flag;

   temp_desc = read_desc_type(read_file, 0, NULL);
   if (temp_desc == NULL)
      return -1;
   fprintf(write_file, "Brief ^\n%s^\n", temp_desc);
   delete temp_desc;


   /* set the subareas */
   the_token = get_token(read_file,'\0');
   if (the_token->token_type != T_CARROT)
   {
      printf("Invalid format in mobile %s.\n", objname);
      return -1;
   } 
   the_token = get_token(read_file, '^');
   fprintf(write_file, "SubArea ^%s^\n", (the_token->the_string == NULL) ? "" : 
																		the_token->the_string);

   /* get the start wield left */
   the_token = get_token(read_file, '\0');
   fprintf(write_file, "StartWieldLeft ^%s^\n", (the_token->the_string == NULL) ? "" : 
																		the_token->the_string);

   /* get the start wield right */
   the_token = get_token(read_file, '\0');
   fprintf(write_file, "StartWieldRight ^%s^\n", (the_token->the_string == NULL) ? "" : 
																		the_token->the_string);

   /* get the next item, it should be as '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      printf("Invalid format in startwear, mobile %s.\n", objname);
      return -1;
   }
   /* get the start wear objects */
   the_token = get_token(read_file, '^');
   fprintf(write_file, "StartWear ^%s^\n", (the_token->the_string == NULL) ? "" : 
																		the_token->the_string);

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      printf("Invalid format in bareweapon, mobile %s.\n", objname);
      return -1;
   }
   /* get the BareWeapon of this mobile */
   the_token = get_token(read_file, '^');
   fprintf(write_file, "BareWeapon ^%s^\n", (the_token->the_string == NULL) ? "" : 
																		the_token->the_string);

   /* Set strength */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in strength, mobile %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "Strength %d\n", atoi(the_token->the_string));

   /* Set dexterity */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in dexterity, mobile %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "Dexterity %d\n", atoi(the_token->the_string));

   /* Set constitution */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in constitution, mobile %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "Constitution %d\n", atoi(the_token->the_string));

   /* Set intel */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in intel, mobile %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "Intel %d\n", atoi(the_token->the_string));

   if (version > 0.58)
   {
      /* Set wisdom */
      the_token = get_token(read_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in wisdom, mobile %s.\n", objname);
         return -1;
      }
      fprintf(write_file, "Wisdom %d\n", atoi(the_token->the_string));

      /* Set charisma */
      the_token = get_token(read_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in charisma, mobile %s.\n", objname);
         return -1;
      }
      fprintf(write_file, "Charisma %d\n", atoi(the_token->the_string));
   }

   /* Set flag1 */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in mobflag1, mobile %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "MobFlag1 %ld\n", atol(the_token->the_string));

   flag = atol(the_token->the_string);

   /* Set flag2 */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in mobflag2, mobile %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "MobFlag2 %ld\n", atol(the_token->the_string));

   if (version < 0.602)
   {
       flag_offset = INDFLAG_MALE / 32;
       which_flag = INDFLAG_MALE - (flag_offset * 32);
       ((flag & (1 << which_flag)) != 0) ? sex = 0 : sex = 1;
       fprintf(write_file, "Sex %d\n", sex);
   }
   else
   {
       // Set sex
       the_token = get_token(read_file, '\0');
       if (the_token->token_type != T_NUMERICAL)
       {
           printf("Invalid format in sex, mobile %s.\n", objname);
           return -1;
       }
       fprintf(write_file, "Sex %d\n", atoi(the_token->the_string));
   }

   if (version > 0.602)
   {
       // Set vislvl
       the_token = get_token(read_file, '\0');
       if (the_token->token_type != T_NUMERICAL)
       {
           printf("Invalid format in vislvl, mobile %s\n", objname);
           return -1;
       }
       fprintf(write_file, "VisLvl %d\n", atoi(the_token->the_string));
   }

   /* Set speed */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in speed, mobile %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "Speed %d\n", atoi(the_token->the_string));

   /* Set aggression */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in aggression, mobile %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "Aggression %d\n", atoi(the_token->the_string));

   if (version > 0.591)
   {
     /* Set wimpy */
     the_token = get_token(read_file, '\0');
     if (the_token->token_type != T_NUMERICAL)
     {
       printf("Invalid format in wimpy, mobile %s.\n", objname);
       return -1;
     }
     fprintf(write_file, "Wimpy %d\n", atoi(the_token->the_string));
     
     /* Set maxmagic */
     the_token = get_token(read_file, '\0');
     if (the_token->token_type != T_NUMERICAL)
     {
       printf("Invalid format in maxmagic, mobile %s.\n", objname);
       return -1;
     }
     fprintf(write_file, "MaxMagic %d\n", atoi(the_token->the_string));

     /* Set maxendurance */
     the_token = get_token(read_file, '\0');
     if (the_token->token_type != T_NUMERICAL)
     {
       printf("Invalid format in maxendurance, mobile %s.\n", objname);
       return -1;
     }
     fprintf(write_file, "MaxEndurance %d\n", atoi(the_token->the_string));


     /* get the talk structs */
     fprintf(write_file, "StartRankList\n");
     the_token = get_token(read_file, '\0');
     while (the_token->token_type != T_POUND)
     {       
       fprintf(write_file, "StartRank\n");

       /* first we get the rank */
       the_token = get_token(read_file, '\0');
       if (the_token->token_type != T_NUMERICAL)
       {
	 printf("Invalid format in rank, mobile %s.\n", objname);
	 return -1;
       }
       fprintf(write_file, "Rank %d\n", atoi(the_token->the_string));

       /* now we get experience */
       the_token = get_token(read_file, '\0');
       if (the_token->token_type != T_NUMERICAL)
       {
	 printf("Invalid format in experience, mobile %s.\n", objname);
	 return -1;
       }
       fprintf(write_file, "Experience %d\n", atoi(the_token->the_string));
       fprintf(write_file, "EndRank\n");
       the_token = get_token(read_file,'\0');
     }
     fprintf(write_file, "EndRankList\n");
   }

   /* get the talk structs */
   fprintf(write_file, "StartTellReplies\n");
   the_token = get_token(read_file, '\0');
   while (the_token->token_type != T_POUND)
   {
      if (the_token->token_type != T_PLUS)
      {
         printf("Invalid format in mobile %s, expected '+' not found.\n", 
		objname);
         return -1;
      }

      fprintf(write_file, "StartReply\n");

      /* first we get the tell keylist */
      the_token = get_token(read_file, '\0');
      if (the_token->token_type != T_CARROT)
      {
         printf("Invalid format in keywords, mobile %s.\n", objname);
         return -1;
      }
      the_token = get_token(read_file, '^');
      fprintf(write_file, "Keywords ^%s^\n", 
	      (the_token->the_string == NULL) ? "" : the_token->the_string);
      
      temp_desc = read_desc_type(read_file, 0, NULL);
      if (temp_desc == NULL)
         return -1;
      fprintf(write_file, "Reply ^\n%s^\n", 
	      (the_token->the_string == NULL) ? "" : the_token->the_string);
      delete temp_desc;
      
      fprintf(write_file, "EndReply\n");
      the_token = get_token(read_file,'\0');
   }
   fprintf(write_file, "EndTellReplies\n");

   /* get the shop structs */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_POUND)
   {
	  fprintf(write_file, "StartShop\n");
      if (the_token->token_type != T_CARROT)
      {
         printf("Invalid format in startshop, mobile %s.\n", objname);
         return -1;
      }

      if (convert_shop_attrib(read_file, write_file, objname, version) <= 0)
      {
         return -1;
      }
      fprintf(write_file, "EndShop\n");
   }
   return 1;
}



/***********************************************************************
 ** convert_shop_attrib - reads in the shop attributes and converts
 **                       them to pseudo-data format
 **
 ** Parameters: read_file - the AIME file we read from
 **             write_file - the AIME file we write to
 **             objname - the name of the object we are converting
 **             version - which version we are loading
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int convert_shop_attrib(FILE *read_file, FILE *write_file, char *objname, float version)
{
   token_record *the_token;
   char         *temp_desc;

   /* read in the shop title */
   the_token = get_token(read_file, '^');
   fprintf(write_file, "Name ^%s^\n", (the_token->the_string == NULL) ? "" : 
																		the_token->the_string);

   /* read in the type of currency this shop will use */
   the_token = get_token(read_file, '\0');
   fprintf(write_file, "Currency ^%s^\n", (the_token->the_string == NULL) ? "" : 
																		the_token->the_string);

   the_token = get_token(read_file, '\0');
   fprintf(write_file, "StartShopItems\n");
   while (the_token->token_type == T_PLUS)
   {
	  fprintf(write_file, "StartItem\n");
      /* first we get the object name */
      the_token = get_token(read_file,'\0');
	  fprintf(write_file, "ItemName ^%s^\n", (the_token->the_string == NULL) ? "" : 
																		the_token->the_string);
      /* next we get the alias name */
      the_token = get_token(read_file, '\0');
	  fprintf(write_file, "AliasName ^%s^\n", (the_token->the_string == NULL) ? "" : 
																		the_token->the_string);

      /* Read in the value */
      the_token = get_token(read_file, '\0');
     
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in mobile %s's shop.\n", objname);
         return -1;
      }
	  fprintf(write_file, "Value %d\n", atoi(the_token->the_string));

      /* Read in the type */
      the_token = get_token(read_file, '\0');
     
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in mobile %s's shop.\n", objname);
         return -1;
      }
	  fprintf(write_file, "ItemType %d\n", atoi(the_token->the_string));

      /* Read in the number of */
      the_token = get_token(read_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in mobile %s's shop.\n", objname);
         return -1;
      }
	  fprintf(write_file, "NumberOf %d\n", atoi(the_token->the_string));

      /* read in the item description */
      temp_desc = read_desc_type(read_file, 0, NULL);
      if (temp_desc == NULL)
         return -1;
	  fprintf(write_file, "Desc ^\n%s^\n", temp_desc);
      delete temp_desc;
	  fprintf(write_file, "EndItem\n");

      the_token = get_token(read_file, '\0');
   }

   fprintf(write_file, "EndShopItems\n");

   if (the_token->token_type != T_POUND)
   {
      printf("Invalid format in mobile %s's shop.\n", objname);
      return -1;
   }
   version = 0;
   return 1;
}



/***********************************************************************
 ** convert_loc_attrib - reads in the location attributes and converts
 **                      them to pseudo-data format
 **
 ** Parameters: read_file - the AIME file we read from
 **             write_file - the AIME file we write to
 **             objname - the name of the object we are converting
 **             version - which version we are loading
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int convert_loc_attrib(FILE *read_file, FILE *write_file, char *objname, float version)
{
   token_record *the_token;
   int           i;           /* used to loop */
   char          *temp_desc;

   char *dir[10] = {"North", "South", "East", "West", "Up", "Down", "Northeast", "Northwest", 
	   "Southeast", "Southwest"};

   /* loop through all directions, setting their exits. If we find an @
      after we get the exit, make sure we add the area to the exit */
   for (i = 0; i < 10; i++)
   {
      the_token = get_token(read_file, '\0');
	  fprintf(write_file, "%s ^%s^\n", dir[i], (the_token->the_string == NULL) ? "" : 
																		the_token->the_string);
   }

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      printf("Invalid format in location %s.\n", objname);
      return -1;
   }

   /* get the extradir */
   the_token = get_token(read_file, '^');
   fprintf(write_file, "ExtraDir ^%s^\n", (the_token->the_string == NULL) ? "" : 
																		the_token->the_string);
   /* set the listen desc */
   temp_desc = read_desc_type(read_file, 0, NULL);
   
   if (temp_desc == NULL)
      return -1;
   fprintf(write_file, "Listen ^\n%s^\n", the_token->the_string);
   delete temp_desc;


   /* set the smell desc */
   temp_desc = read_desc_type(read_file, 0, NULL);
   
   if (temp_desc == NULL)
      return -1;
   fprintf(write_file, "Smell ^\n%s^\n", the_token->the_string);
   delete temp_desc;

   /* set the first extra desc */
   temp_desc = read_desc_type(read_file, 0, NULL);
   
   if (temp_desc == NULL)
      return -1;
   fprintf(write_file, "ExtraDesc1 ^\n%s^\n", the_token->the_string);
   delete temp_desc;

   /* set the second extra desc */
   temp_desc = read_desc_type(read_file, 0, NULL);
   
   if (temp_desc == NULL)
      return -1;
   fprintf(write_file, "ExtraDesc2 ^\n%s^\n", the_token->the_string);
   delete temp_desc;


   /* This is where the room size, terrain, and lighting are read in but
      since they don't exist in previous versions to 0.56, we set them to default
      and not read them */
   if (version >= 0.56)
   {
      /* Set room size */
      the_token = get_token(read_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in location %s.\n", objname);
         return -1;
      }
	  fprintf(write_file, "HorizSize %d\n", atoi(the_token->the_string));
      
      /* Set terrain */
      the_token = get_token(read_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in location %s.\n", objname);
         return -1;
      }
	  fprintf(write_file, "Terrain %d\n", atoi(the_token->the_string));
 
      /* Set lighting */
      the_token = get_token(read_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in location %s.\n", objname);
         return -1;
      }
	  fprintf(write_file, "Lighting %d\n", atoi(the_token->the_string));
   }

   /* Set flag1 */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in location %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "LocFlag1 %ld\n", atol(the_token->the_string));

   /* Set flag2 */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in location %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "LocFlag2 %ld\n", atol(the_token->the_string));

   return 1;
}


/***********************************************************************
 ** convert_marker_attrib - reads in the marker attributes and converts
 **                         them to pseudo-data format
 **
 ** Parameters: read_file - the AIME file we read from
 **             write_file - the AIME file we write to
 **             objname - the name of the object we are converting
 **             version - which version we are loading
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int convert_marker_attrib(FILE *read_file, FILE *write_file, char *objname, float version)
{
  /* do nothing here */

   read_file = NULL;
   version = 0;
   return 1;
}


/***********************************************************************
 ** convert_door_attrib - reads in the door attributes and converts
 **                       them to pseudo-data format
 **
 ** Parameters: read_file - the AIME file we read from
 **             write_file - the AIME file we write to
 **             objname - the name of the object we are converting
 **             version - which version we are loading
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int convert_door_attrib(FILE *read_file, FILE *write_file, char *objname, float version)
{
   token_record *the_token;
   int          i;
   char         *temp_desc;

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      printf("Invalid format in door '%s'.\n", objname);
      return -1;
   }

   /* get the keynames */
   the_token = get_token(read_file, '^');
   fprintf(write_file, "KeyNames ^%s^\n", (the_token->the_string == NULL) ? "" : 
																		the_token->the_string);
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in door '%s'.\n", objname);
      return -1;
   }
   fprintf(write_file, "DoorState %d\n", atoi(the_token->the_string));

   for (i=0; i<= DOOR_MAXSTATE; i++)
   {
      /* set the inside briefs */
      temp_desc = read_desc_type(read_file, 0, NULL); 
      if (temp_desc == NULL)
         return -1;
      fprintf(write_file, "InsideBrief%d ^\n%s^\n", i, temp_desc);
      delete temp_desc;
   }

   /* get the object outside location */
   the_token = get_token(read_file, '\0');
   fprintf(write_file, "OutsideLoc ^%s^\n", (the_token->the_string == NULL) ? "" : 
																		the_token->the_string);

   /* set the outside desc */
   temp_desc = read_desc_type(read_file, 0, NULL); 
   if (temp_desc == NULL)
      return -1;
   fprintf(write_file, "OutsideDesc ^\n%s^\n", the_token->the_string);
   delete temp_desc;

   for (i=0; i<= DOOR_MAXSTATE; i++)
   {
      /* set the outside briefs */
      temp_desc = read_desc_type(read_file, 0, NULL); 
      if (temp_desc == NULL)
         return -1;
      fprintf(write_file, "OutsideBrief%d ^\n%s^\n", i, temp_desc);
      delete temp_desc;
   }

   /* Set distance */
   the_token = get_token(read_file, '\0');
     
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in door '%s'.\n", objname);
      return -1;
   }
   fprintf(write_file, "Distance %d\n", atoi(the_token->the_string));

   /* get the rope name */
   the_token = get_token(read_file, '\0');
   fprintf(write_file, "RopeName ^%s^\n", (the_token->the_string == NULL) ? "" : 
																		the_token->the_string);

   if (version >= 0.601)
   {
      /* Set flag1 */
      the_token = get_token(read_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in door %s.\n", objname);
         return -1;
      }

      fprintf(write_file, "DoorFlag1 %ld\n", atol(the_token->the_string));
   
      /* Set flag2 */
      the_token = get_token(read_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in door %s.\n", objname);
         return -1;
      }
      fprintf(write_file, "DoorFlag2 %ld\n", atol(the_token->the_string));
   }
   else
   {
        fprintf(write_file, "DoorFlag1 %ld\n", conv_doorflag);
	fprintf(write_file, "DoorFlag2 0\n");
   }

   return 1;
}



/***********************************************************************
 ** convert_book_attrib - reads in the book attributes and converts
 **                       them to pseudo-data format
 **
 ** Parameters: read_file - the AIME file we read from
 **             write_file - the AIME file we write to
 **             objname - the name of the object we are converting
 **             version - which version we are loading
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int convert_book_attrib(FILE *read_file, FILE *write_file, char *objname, float version)
{
   token_record *the_token;

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      printf("Invalid format in location %s.\n", objname);
      return -1;
   }

   /* get the subject */
   the_token = get_token(read_file, '^');
   fprintf(write_file, "Subject ^%s^\n", (the_token->the_string == NULL) ? 
	                                         "" : the_token->the_string);
   version = 0;
   return 1;
}


/***********************************************************************
 ** convert_weapon_attrib - reads in the weapon attributes and converts
 **                       them to pseudo-data format
 **
 ** Parameters: read_file - the AIME file we read from
 **             write_file - the AIME file we write to
 **             objname - the name of the object we are converting
 **             version - which version we are loading
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int convert_weapon_attrib(FILE *read_file, FILE *write_file, char *objname, float version)
{
   token_record *the_token;

   /* get the weapon class */
   the_token = get_token(read_file, '\0');
   fprintf(write_file, "WeaponClass ^%s^\n", (the_token->the_string == NULL) ? "" : 
																	the_token->the_string);
   
   /* Set damage */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in weapon %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "Damage %d\n", atoi(the_token->the_string));

   /* Set dep strength */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in weapon %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "DepStrength %d\n", atoi(the_token->the_string));

   /* Set dep dexterity */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in weapon %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "DepDexterity %d\n", atoi(the_token->the_string));

   /* Set wield type */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in weapon %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "WieldType %d\n", atoi(the_token->the_string));

   version = 0;
   return 1;
}


/***********************************************************************
 ** convert_wearable_attrib - reads in the wearable attributes and converts
 **                           them to pseudo-data format
 **
 ** Parameters: read_file - the AIME file we read from
 **             write_file - the AIME file we write to
 **             objname - the name of the object we are converting
 **             version - which version we are loading
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int convert_wearable_attrib(FILE *read_file, FILE *write_file, char *objname, float version)
{
   token_record *the_token;

   /* Set armor */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in wearable %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "Armor %d\n", atoi(the_token->the_string));

   /* Set flag */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in location %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "WearFlag1 %ld\n", atol(the_token->the_string));

   version = 0;
   return 1;
}


/***********************************************************************
 ** convert_merger_attrib - reads in the merger attributes and converts
 **                         them to pseudo-data format
 **
 ** Parameters: read_file - the AIME file we read from
 **             write_file - the AIME file we write to
 **             objname - the name of the object we are converting
 **             version - which version we are loading
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int convert_merger_attrib(FILE *read_file, FILE *write_file, char *objname, float version)
{
   token_record *the_token;
   
   /* Set number_of */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in merger %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "NumberOf %d\n", atoi(the_token->the_string));
  
   version = 0;
   return 1;
}




/***********************************************************************
 ** convert_food_attrib - reads in the food attributes and converts
 **                       them to pseudo-data format
 **
 ** Parameters: read_file - the AIME file we read from
 **             write_file - the AIME file we write to
 **             objname - the name of the object we are converting
 **             version - which version we are loading
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int convert_food_attrib(FILE *read_file, FILE *write_file, char *objname, float version)
{
   token_record *the_token;
   
   /* Set the attribute */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in food %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "Energy %d\n", atoi(the_token->the_string));
  
   version = 0;
   return 1;
}



/***********************************************************************
 ** convert_money_attrib - reads in the money attributes and converts
 **                        them to pseudo-data format
 **
 ** Parameters: read_file - the AIME file we read from
 **             write_file - the AIME file we write to
 **             objname - the name of the object we are converting
 **             version - which version we are loading
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int convert_money_attrib(FILE *read_file, FILE *write_file, char *objname, float version)
{
   token_record *the_token;
   
   /* Set the attribute */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in money %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "MoneyIndex %d\n", atoi(the_token->the_string));
  
   version = 0;
   return 1;
}


/***********************************************************************
 ** convert_specials_attrib - reads in the specials attributes and converts
 **                           them to pseudo-data format
 **
 ** Parameters: read_file - the AIME file we read from
 **             write_file - the AIME file we write to
 **             objname - the name of the object we are converting
 **             version - which version we are loading
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int convert_specials_attrib(FILE *read_file, FILE *write_file, char *objname, float version)
{
   token_record *the_token;
   char         *tmp_str;
   char         *temp_desc;
    
   fprintf(write_file, "Name ^%s^\n", objname);

  /* get the trigger of this special */
   the_token = get_token(read_file, '\0');
   fprintf(write_file, "Trigger ^%s^\n", (the_token->the_string == NULL) ? "" : 
																	the_token->the_string);

   /* get activation time */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in special %s.\n", objname);
      return -1;
   }

   fprintf(write_file, "Activation %u\n", strtoul(the_token->the_string, &tmp_str, 0));
   
   /* get interval between activations */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in special %s.\n", objname);
      return -1;
   }

   fprintf(write_file, "Interval %u\n", strtoul(the_token->the_string, &tmp_str, 0));

   /* get num of times this should activate */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in special %s.\n", objname);
      return -1;
   }

   fprintf(write_file, "NumTimes %d\n", atoi(the_token->the_string));

   /* get the counter value */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in special %s.\n", objname);
      return -1;
   }

   fprintf(write_file, "Counter %ld\n", atol(the_token->the_string));

   /* set the code */
   temp_desc = read_desc_type(read_file, 0, NULL); 
   if (temp_desc == NULL)
      return -1;
   fprintf(write_file, "Code ^\n%s^\n", temp_desc);
   delete temp_desc;

   if (version > 0.57)
   {
      /* set the comments */
      temp_desc = read_desc_type(read_file, 0, NULL); 
      if (temp_desc == NULL)
         return -1;
      fprintf(write_file, "Comments ^\n%s^\n", temp_desc);
      delete temp_desc;
   }

   version = 0;
   return 1;
}


/***********************************************************************
 ** read_ability_attrib - reads in the ability attributes of the object
 **
 ** Parameters: read_file - the AIME file we read from
 **             version - which version we are loading
 **
 ** Returns: 1 for success
 **
 ***********************************************************************/

int convert_ability_attrib(FILE *read_file, FILE *write_file, char *objname, float version)
{
   token_record *the_token;
   char         *temp_desc;
       
   fprintf(write_file, "Name ^%s^\n", objname);

   /* get the acting string */
   temp_desc = read_desc_type(read_file, 0, NULL);  
   if (temp_desc == NULL)
      return -1;

   fprintf(write_file, "ActingActor ^%s^\n", temp_desc);
   delete temp_desc;

   /* get the acting string */
   temp_desc = read_desc_type(read_file, 0, NULL);  
   if (temp_desc == NULL)
      return -1;
   fprintf(write_file, "ActingTarget ^%s^\n", temp_desc);
   delete temp_desc;

   /* get the acting room string */
   temp_desc = read_desc_type(read_file, 0, NULL);  
   if (temp_desc == NULL)
      return -1;
   fprintf(write_file, "ActingRoom ^%s^\n", temp_desc);
   delete temp_desc;

   /* get the success string */
   temp_desc = read_desc_type(read_file, 0, NULL);  
   if (temp_desc == NULL)
      return -1;
   fprintf(write_file, "SuccessActor ^%s^\n", temp_desc);
   delete temp_desc;

   /* get the success string */
   temp_desc = read_desc_type(read_file, 0, NULL);  
   if (temp_desc == NULL)
      return -1;
   fprintf(write_file, "SuccessTarget ^%s^\n", temp_desc);
   delete temp_desc;

   /* get the success string */
   temp_desc = read_desc_type(read_file, 0, NULL);  
   if (temp_desc == NULL)
      return -1;
   fprintf(write_file, "SuccessRoom ^%s^\n", temp_desc);
   delete temp_desc;

   /* get the failure string */
   temp_desc = read_desc_type(read_file, 0, NULL);  
   if (temp_desc == NULL)
      return -1;
   fprintf(write_file, "FailureActor ^%s^\n", temp_desc);
   delete temp_desc;

   /* get the failure string */
   temp_desc = read_desc_type(read_file, 0, NULL);  
   if (temp_desc == NULL)
      return -1;
   fprintf(write_file, "FailureTarget ^%s^\n", temp_desc);
   delete temp_desc;

   /* get the failure string */
   temp_desc = read_desc_type(read_file, 0, NULL);  
   if (temp_desc == NULL)
      return -1;
   fprintf(write_file, "FailureRoom ^%s^\n", temp_desc);
   delete temp_desc;

   // This has been put into the dependencies list so we don't use it anymore
   if (version <= 0.59)
   {
      /* get the next item, it should be a '^', if not, raise error */
     the_token = get_token(read_file, '\0');
     if (the_token->token_type != T_CARROT)
     {
       printf("Invalid format in itemsneeded, ability %s.\n", objname);
       return -1;
     }

     /* get the items needed string */
     the_token = get_token(read_file, '^');
     // fprintf(write_file, "ItemsNeeded ^%s^\n", the_token->the_string);
   }

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      printf("Invalid format in specialname '%s', ability %s.\n", 
	     the_token->the_string, objname);
      return -1;
   }

   /* get the special name string */
   the_token = get_token(read_file, '^');
   fprintf(write_file, "SpecialName ^%s^\n", the_token->the_string);

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      printf("Invalid format in succtrig, ability %s.\n", objname);
      return -1;
   }

   /* get the succtrig string */
   the_token = get_token(read_file, '^');
   fprintf(write_file, "SuccessTrigger ^%s^\n", the_token->the_string);

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      printf("Invalid format in failtrig, ability %s.\n", objname);
      return -1;
   }

   /* get the failtrig string */
   the_token = get_token(read_file, '^');
   fprintf(write_file, "FailTrigger ^%s^\n", the_token->the_string);

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      printf("Invalid format in attempttrig '%s', ability %s.\n", 
	     the_token->the_string, objname);
      return -1;
   }

   /* get the attempttrig string */
   the_token = get_token(read_file, '^');
   fprintf(write_file, "AttemptTrigger ^%s^\n", the_token->the_string);

   /* Set drain value */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in drain, ability %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "Drain %s\n", the_token->the_string);

   /* get the dependency values if we have them */
   if (version > 0.58)
   {
      fprintf(write_file, "StartDependencyList\n");

      // Now read in the dependencies
      the_token = get_token(read_file, '\0');     
      while (the_token->token_type != T_POUND)
      {
	fprintf(write_file, "StartDependency\n");
	if (the_token->token_type != T_PLUS)
	{
	  printf("Invalid format in startde, ability %s\n", objname);
	  return -1;
	}

	/* Set deptype value */
	the_token = get_token(read_file, '\0');     
	if (the_token->token_type != T_NUMERICAL)
	{
	  printf("Invalid format in deptype, ability %s\n", objname);
	  return -1;
	}
	fprintf(write_file, "DependencyType %d\n", atoi(the_token->the_string));

	/* get the next item, it should be a '^', if not, raise error */
	the_token = get_token(read_file, '\0');
	if (the_token->token_type != T_CARROT)
	{
	  printf("Invalid format in depstrtype, ability %s\n", objname);
	  return -1;
	}

	/* get the depstrtype string */
	the_token = get_token(read_file, '^');
	fprintf(write_file, "DependencyString ^%s^\n", the_token->the_string);
	
	/* Set deptype value */
	the_token = get_token(read_file, '\0');     
	if (the_token->token_type != T_NUMERICAL)
	{
	  printf("Invalid format in dependencyint, ability %s\n", objname);
	  return -1;
	}
	fprintf(write_file, "DependencyInt %d\n", atoi(the_token->the_string));
	
	fprintf(write_file, "EndDependency\n");
	the_token = get_token(read_file, '\0');
      }
      
      fprintf(write_file, "EndDependencyList\n");
   }

   version = 0;

   return 1;
}


/***********************************************************************
 ** read_spell_attrib - reads in the spell attributes of the object
 **
 ** Parameters: read_file - the AIME file we read from
 **             version - which version we are loading
 **
 ** Returns: 1 for success
 **
 ***********************************************************************/

int convert_spell_attrib(FILE *read_file, FILE *write_file, char *objname, float version)
{
   token_record *the_token;

   if (version <= 0.58)
   {
	  fprintf(write_file, "StartDependencyList\n");

      /* Set min_intel value */
      the_token = get_token(read_file, '\0');     
      if (the_token->token_type != T_NUMERICAL)
	  {
         printf("Invalid format in spell %s.\n", objname);
         return -1;
	  }

	  if (atoi(the_token->the_string) > 0)
	  {
		  fprintf(write_file, "StartDependency\n");
		  fprintf(write_file, "DependencyType %d\n", (int) ReqIntel);
		  fprintf(write_file, "DependencyInt %d\n", atoi(the_token->the_string));
		  fprintf(write_file, "EndDependency\n");
	  }

	  fprintf(write_file, "EndDependencyList\n");
   }

   /* Set flag */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in spell %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "SpellFlag1 %ld\n", atol(the_token->the_string));

   return 1;
}


/***********************************************************************
 ** read_skill_attrib - reads in the skill attributes of the object
 **
 ** Parameters: read_file - the AIME file we read from
 **             version - which version we are loading
 **
 ** Returns: 1 for success
 **
 ***********************************************************************/

int convert_skill_attrib(FILE *read_file, FILE *write_file, char *objname, float version)
{
   token_record *the_token;

   if (version <= 0.58)
   {
      fprintf(write_file, "StartDependencyList\n");

	  /* Set min_strength value */
	  the_token = get_token(read_file, '\0');     
	  if (the_token->token_type != T_NUMERICAL)
	  {
	     printf("Invalid format in skill %s.\n", objname);
	     return -1;
	  }

	  if (atoi(the_token->the_string) > 0)
	  {
		  fprintf(write_file, "StartDependency\n");
		  fprintf(write_file, "DependencyType %d\n", (int) ReqStrength);
		  fprintf(write_file, "DependencyInt %d\n", atoi(the_token->the_string));
		  fprintf(write_file, "EndDependency\n");
	  }


	  /* Set min_dexterity value */
	  the_token = get_token(read_file, '\0');     
	  if (the_token->token_type != T_NUMERICAL)
	  {
		 printf("Invalid format in skill %s.\n", objname);
         return -1;
	  }

	  if (atoi(the_token->the_string) > 0)
	  {
		  fprintf(write_file, "StartDependency\n");
		  fprintf(write_file, "DependencyType %d\n", (int) ReqDexterity);
		  fprintf(write_file, "DependencyInt %d\n", atoi(the_token->the_string));
		  fprintf(write_file, "EndDependency\n");
	  }
	  
	  fprintf(write_file, "EndDependencyList\n");
   }

   /* Set flag */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in skill %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "SkillFlag1 %ld\n", atol(the_token->the_string));

   return 1;
}


/***********************************************************************
 ** read_action_attrib - reads in the action attributes of the object
 **
 ** Parameters: read_file - the AIME file we read from
 **             version - which version we are loading
 **
 ** Returns: 1 for success
 **
 ***********************************************************************/

int convert_action_attrib(FILE *read_file, FILE *write_file, char *objname, float version)
{
   token_record *the_token;
   char         *temp_desc;

   fprintf(write_file, "Name ^%s^\n", objname);

   /* get the actor string */
   temp_desc = read_desc_type(read_file, 0, NULL);  
   if (temp_desc == NULL)
      return -1;
   fprintf(write_file, "Actor ^%s^\n", temp_desc);
   delete temp_desc;

   /* get the crowd string */
   temp_desc = read_desc_type(read_file, 0, NULL);  
   if (temp_desc == NULL)
      return -1;
   fprintf(write_file, "Crowd ^%s^\n", temp_desc);
   delete temp_desc;

   /* get the target string */
   temp_desc = read_desc_type(read_file, 0, NULL);  
   if (temp_desc == NULL)
      return -1;
   fprintf(write_file, "Target ^%s^\n", temp_desc);
   delete temp_desc;

   /* get the sender string */
   temp_desc = read_desc_type(read_file, 0, NULL);  
   if (temp_desc == NULL)
      return -1;
   fprintf(write_file, "Sender ^%s^\n", temp_desc);
   delete temp_desc;

   /* get the bystander string */
   temp_desc = read_desc_type(read_file, 0, NULL);  
   if (temp_desc == NULL)
      return -1;
   fprintf(write_file, "Bystander ^%s^\n", temp_desc);
   delete temp_desc;

   /* Set flag */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in skill %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "ActFlag1 %ld\n", atol(the_token->the_string));

   version = 0;
   return 1;
}


/***********************************************************************
 ** read_chatline_attrib - reads in the chatline attributes of the object
 **
 ** Parameters: read_file - the AIME file we read from
 **             version - which version we are loading
 **
 ** Returns: 1 for success
 **
 ***********************************************************************/

int convert_chatline_attrib(FILE *read_file, FILE *write_file, char *objname, float version)
{
   token_record *the_token;

   fprintf(write_file, "Name ^%s^\n", objname);

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      printf("Invalid format in chatline %s lead_in_actor.\n", objname);
      return -1;
   }

   /* get the lead_in_actor of this location */
   the_token = get_token(read_file, '^');
   fprintf(write_file, "LeadInActor ^%s^\n", 
	   (the_token->the_string == NULL) ? "" : the_token->the_string);

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      printf("Invalid format in chatline %s lead_in_crowd.\n", objname);
      return -1;
   }

   /* get the lead_in_crowd of this location */
   the_token = get_token(read_file, '^');
   fprintf(write_file, "LeadInCrowd ^%s^\n", 
	   (the_token->the_string == NULL) ? "" : the_token->the_string);

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      printf("Invalid format in chatline %s text_color.\n", objname);
      return -1;
   }

   /* get the text_color of this location */
   the_token = get_token(read_file, '^');
   fprintf(write_file, "TextColor ^%s^\n", 
	   (the_token->the_string == NULL) ? "" : the_token->the_string);

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      printf("Invalid format in chatline %s border.\n", objname);
      return -1;
   }

   /* get the border of this location */
   the_token = get_token(read_file, '^');
   fprintf(write_file, "Border ^%s^\n", 
	   (the_token->the_string == NULL) ? "" : the_token->the_string);

   /* Set flag */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in chatline %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "ChatFlag1 %ld\n", atol(the_token->the_string));

   version = 0;
   return 1;
}



/***********************************************************************
 ** read_race_attrib - reads in the race attributes of the object
 **
 ** Parameters: read_file - the AIME file we read from
 **             version - which version we are loading
 **
 ** Returns: 1 for success
 **
 ***********************************************************************/

int convert_race_attrib(FILE *read_file, FILE *write_file, char *objname, float version)
{
   token_record *the_token;
   char         *temp_desc;

   fprintf(write_file, "Name ^%s^\n", objname);

   /* get the description string */
   temp_desc = read_desc_type(read_file, 0, NULL);  
   if (temp_desc == NULL)
      return -1;
   fprintf(write_file, "Description ^%s^\n", temp_desc);
   delete temp_desc;

   /* get the init_desc string */
   temp_desc = read_desc_type(read_file, 0, NULL);  
   if (temp_desc == NULL)
      return -1;
   fprintf(write_file, "InitialDesc ^%s^\n", temp_desc);
   delete temp_desc;

   /* get the init_brief string */
   temp_desc = read_desc_type(read_file, 0, NULL);  
   if (temp_desc == NULL)
      return -1;
   fprintf(write_file, "InitialBrief ^%s^\n", temp_desc);
   delete temp_desc;
 
   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      printf("Invalid format in race %s initability.\n", objname);
      return -1;
   }

   /* get the initability of this location */
   the_token = get_token(read_file, '^');
   fprintf(write_file, "InitialAbility ^%s^\n", (the_token->the_string == NULL) ? "" : 
																		the_token->the_string);

   if (version > 0.55)
   {
      /* get the init_loc left */
      the_token = get_token(read_file, '\0');
      fprintf(write_file, "InitialLocation ^%s^\n", (the_token->the_string == NULL) ? "" : 
																		the_token->the_string);
   }

   if (version > 0.56)
   {
      /* get the death_location left */
      the_token = get_token(read_file, '\0');
      fprintf(write_file, "DeathLocation ^%s^\n", (the_token->the_string == NULL) ? "" : 
																		the_token->the_string);

      /* get the death_text string */
      temp_desc = read_desc_type(read_file, 0, NULL);  
      if (temp_desc == NULL)
         return -1;
      fprintf(write_file, "DeathText ^%s^\n", (the_token->the_string == NULL) ? "" : 
																		the_token->the_string);
	  delete temp_desc;
   }


   if (version > 0.58)
   {
     /* get the next item, it should be a '^', if not, raise error */
     the_token = get_token(read_file, '\0');
     if (the_token->token_type != T_CARROT)
     {
        printf("Invalid format in allow_incl for race %s.\n", 
                                                    objname);
     }

     /* get the allowed inclinations */
     the_token = get_token(read_file, '^');
     fprintf(write_file, "AllowInclinations ^%s^\n", the_token->the_string);

     /* get the next item, it should be a '^', if not, raise error */
     the_token = get_token(read_file, '\0');
     if (the_token->token_type != T_CARROT)
     {
        printf("Invalid format in allow talents for race %s.\n", 
                                                    objname);
     }

     /* get the allowed talents */
     the_token = get_token(read_file, '^');
     fprintf(write_file, "AllowTalents ^%s^\n", the_token->the_string);

   }

   /* Set init_str value */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in race %s init_str, got %s.\n", 
        objname, the_token->the_string);
      return -1;
   }
   fprintf(write_file, "InitialStrength %d\n", atoi(the_token->the_string));

   /* Set init_dex value */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in race %s init_dex.\n", objname);
      return -1;
   }
   fprintf(write_file, "InitialDexterity %d\n", atoi(the_token->the_string));

   /* Set init_intel value */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in race %s init_intel.\n", objname);
      return -1;
   }
   fprintf(write_file, "InitialIntel %d\n", atoi(the_token->the_string));

   if (version > 0.58)
   {
      /* Set init_con value */
      the_token = get_token(read_file, '\0');     
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in race %s init_con.\n", objname);
         return -1;
      }
      fprintf(write_file, "InitialCon %d\n", atoi(the_token->the_string));

      /* Set init_wis value */
      the_token = get_token(read_file, '\0');     
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in race %s init_wis.\n", objname);
         return -1;
      }
      fprintf(write_file, "InitialWis %d\n", atoi(the_token->the_string));

      /* Set init_cha value */
      the_token = get_token(read_file, '\0');     
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in race %s init_cha.\n", objname);
         return -1;
      }
      fprintf(write_file, "InitialCha %d\n", atoi(the_token->the_string));

   }


   if (version > 0.56)
   {
	  fprintf(write_file, "StartTutorialList\n");
      the_token = get_token(read_file, '\0');
      while (the_token->token_type == T_PLUS)
      {

	     fprintf(write_file, "StartTutorial\n");

         /* first we get the tutorial name */
         the_token = get_token(read_file,'\0');
		 fprintf(write_file, "Name ^%s^\n", the_token->the_string);

         /* Read in the sequence */
         the_token = get_token(read_file, '\0');
     
         if (the_token->token_type != T_NUMERICAL)
         {
            printf("Invalid format for tutorial sequence in race %s.\n", 
                                                  objname);
            return -1;
         }
		 fprintf(write_file, "Sequence %d\n", atoi(the_token->the_string));

         /* get the next item, it should be a '^', if not, raise error */
         the_token = get_token(read_file, '\0');
         if (the_token->token_type != T_CARROT)
         {
            printf("Invalid format for tutorial prompt in race %s.\n", 
                                               objname);
            return -1;
         }

         /* get the prompt of this tutorial */
         the_token = get_token(read_file, '^');
		 fprintf(write_file, "Prompt ^%s^\n", the_token->the_string);

         /* Set tutorial_special value */
         the_token = get_token(read_file, '\0');     
		 fprintf(write_file, "TutorialSpecial ^%s^\n", (the_token->the_string == NULL) ? "" : 
																	the_token->the_string);

         /* Read in if the tutorial is mandatory */
         the_token = get_token(read_file, '\0');
     
         if (the_token->token_type != T_NUMERICAL)
         {
            printf("Invalid format for tutorial mandatory in race %s.\n", 
                                                 objname);
            return -1;
         }
		 fprintf(write_file, "Mandatory %d\n", atoi(the_token->the_string));

         the_token = get_token(read_file, '\0');

		 fprintf(write_file, "EndTutorial\n");
      }

      if (the_token->token_type != T_POUND)
      {
         printf("Invalid format in race %s's tutorial.\n", objname);
         return -1;
      }
	  fprintf(write_file, "EndTutorialList\n");

   }
   return 1;
}



/***********************************************************************
 ** read_quest_attrib - reads in the quest attributes of the object
 **
 ** Parameters: read_file - the AIME file we read from
 **             version - which version we are loading
 **
 ** Returns: 1 for success
 **
 ***********************************************************************/

int convert_quest_attrib(FILE *read_file, FILE *write_file, char *objname, float version)
{
   token_record *the_token;
   char         *temp_desc;

   fprintf(write_file, "Name ^%s^\n", objname);

   /* get the description string */
   temp_desc = read_desc_type(read_file, 0, NULL);  
   if (temp_desc == NULL)
      return -1;
   fprintf(write_file, "Description ^%s^\n", temp_desc);
   delete temp_desc;

   /* Set difficulty value */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in quest %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "Difficulty %d\n", atoi(the_token->the_string));

   /* Set puzzles value */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in quest %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "Puzzles %d\n", atoi(the_token->the_string));

   /* Set combat value */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in quest %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "Combat %d\n", atoi(the_token->the_string));

   /* Set qpoints value */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in quest %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "QuestPoints %d\n", atoi(the_token->the_string));

   /* Set required value */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in quest %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "Required %d\n", atoi(the_token->the_string));

   version = 0;
   return 1;
}



/***********************************************************************
 ** read_level_attrib - reads in the level attributes of the object
 **
 ** Parameters: read_file - the AIME file we read from
 **             version - which version we are loading
 **
 ** Returns: 1 for success
 **
 ***********************************************************************/

int convert_level_attrib(FILE *read_file, FILE *write_file, char *objname, float version)
{
   token_record *the_token;
   char         *temp_desc;

   fprintf(write_file, "Name ^%s^\n", objname);

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      printf("Invalid format in chain name for level %s.\n", 
                                                    objname);
   }

   /* get the chain name */
   the_token = get_token(read_file, '^');
   fprintf(write_file, "ChainName ^%s^\n", the_token->the_string);

   /* get the award string string */
   temp_desc = read_desc_type(read_file, 0, NULL);  
   if (temp_desc == NULL)
      return -1;
   fprintf(write_file, "AwardString ^%s^\n", (temp_desc == NULL) ? "" : temp_desc);
   delete temp_desc;

   /* get the level string */
   temp_desc = read_desc_type(read_file, 0, NULL);  
   if (temp_desc == NULL)
      return -1;
   fprintf(write_file, "LevelString ^%s^\n", (temp_desc == NULL) ? "" : temp_desc);
   delete temp_desc;

   /* Set lvl_num value */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in lvl_num for level %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "LevelNumber %d\n", atoi(the_token->the_string));

   /* Set min_str value */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in min_str for level %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "MinStrength %d\n", atoi(the_token->the_string));

   /* Set min_dex value */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in min_dex for level %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "MinDexterity %d\n", atoi(the_token->the_string));

   /* Set min_intel value */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in min_intel for level %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "MinIntel %d\n", atoi(the_token->the_string));

   /* Set min_exp value */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in min_exp for level %s.\n", objname);
      return -1;
   }

   fprintf(write_file, "MinExperience %d\n", atoi(the_token->the_string));

   if (version > 0.58)
   {
      /* Set min_con value */
      the_token = get_token(read_file, '\0');     
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in min_con for level %s.\n", objname);
         return -1;
      }

      fprintf(write_file, "MinConstitution %d\n", atoi(the_token->the_string));

      /* Set min_wis value */
      the_token = get_token(read_file, '\0');     
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in min_wis for level %s.\n", objname);
         return -1;
      }

      fprintf(write_file, "MinWisdom %d\n", atoi(the_token->the_string));

      /* Set min_cha value */
      the_token = get_token(read_file, '\0');     
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in min_cha for level %s.\n", objname);
         return -1;
      }

      fprintf(write_file, "MinCharisma %d\n", atoi(the_token->the_string));
   }

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      printf("Invalid format in req_abilities for level %s.\n", 
                                                       objname);
      return -1;
   }

   /* get the req_abilities */
   the_token = get_token(read_file, '^');
   fprintf(write_file, "RequiredAbilities ^%s^\n", (the_token->the_string == NULL) ? "" : 
															the_token->the_string);

   /* get the special req str string */
   temp_desc = read_desc_type(read_file, 0, NULL);  
   if (temp_desc == NULL)
      return -1;
   fprintf(write_file, "SpecialRequirements ^%s^\n", (temp_desc == NULL) ? "" : temp_desc);
   delete temp_desc;

   /* get the when awarded str string */
   temp_desc = read_desc_type(read_file, 0, NULL);  
   if (temp_desc == NULL)
      return -1;
   fprintf(write_file, "WhenAwarded ^%s^\n", (temp_desc == NULL) ? "" : temp_desc);
   delete temp_desc;

   /* Set flag */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in skill %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "LevelFlag1 %ld\n", atol(the_token->the_string));
  
   version = 0;
   return 1;
}



/***********************************************************************
 ** read_mask_attrib - reads in the mask attributes of the object
 **
 ** Parameters: read_file - the AIME file we read from
 **             version - which version we are loading
 **
 ** Returns: 1 for success
 **
 ***********************************************************************/

int convert_mask_attrib(FILE *read_file, FILE *write_file, char *objname, float version)
{
   token_record *the_token;
   int sign = 1;

   fprintf(write_file, "Name ^%s^\n", objname);

   /* Set flag */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type == T_MINUS)
   {
	  sign = -1;
	  the_token = get_token(read_file, '\0');
   }
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in skill %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "ComFlag1 %ld\n", (atol(the_token->the_string) * sign));

    /* Set flag */
   the_token = get_token(read_file, '\0');
   sign = 1;
   if (the_token->token_type == T_MINUS)
   {
	  sign = -1;
	  the_token = get_token(read_file, '\0');
   }
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in skill %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "ComFlag2 %ld\n", (atol(the_token->the_string) * sign));

    /* Set flag */
   the_token = get_token(read_file, '\0');
   sign = 1;
   if (the_token->token_type == T_MINUS)
   {
	  sign = -1;
	  the_token = get_token(read_file, '\0');
   }
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in skill %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "ComFlag3 %ld\n", (atol(the_token->the_string) * sign));

    /* Set flag */
   the_token = get_token(read_file, '\0');
   sign = 1;
   if (the_token->token_type == T_MINUS)
   {
	  sign = -1;
	  the_token = get_token(read_file, '\0');
   }
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in skill %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "ComFlag4 %ld\n", (atol(the_token->the_string) * sign));

    /* Set flag */
   the_token = get_token(read_file, '\0');
   sign = 1;
   if (the_token->token_type == T_MINUS)
   {
	  sign = -1;
	  the_token = get_token(read_file, '\0');
   }
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in skill %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "ComFlag5 %ld\n", (atol(the_token->the_string) * sign));

    /* Set flag */
   the_token = get_token(read_file, '\0');
   sign = 1;
   if (the_token->token_type == T_MINUS)
   {
	  sign = -1;
	  the_token = get_token(read_file, '\0');
   }
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in skill %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "ComFlag6 %ld\n", (atol(the_token->the_string) * sign));
   
   /* Set flag */
   the_token = get_token(read_file, '\0');
   sign = 1;
   if (the_token->token_type == T_MINUS)
   {
	  sign = -1;
	  the_token = get_token(read_file, '\0');
   }
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in skill %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "AdminFlag1 %ld\n", (atol(the_token->the_string) * sign));

   /* Set flag */
   the_token = get_token(read_file, '\0');
   sign = 1;
   if (the_token->token_type == T_MINUS)
   {
	  sign = -1;
	  the_token = get_token(read_file, '\0');
   }
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in skill %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "AdminFlag2 %ld\n", (atol(the_token->the_string) * sign));

   /* Set flag */
   the_token = get_token(read_file, '\0');
   sign = 1;
   if (the_token->token_type == T_MINUS)
   {
	  sign = -1;
	  the_token = get_token(read_file, '\0');
   }
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in skill %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "AdminFlag3 %ld\n", (atol(the_token->the_string) * sign));

   /* Set flag */
   the_token = get_token(read_file, '\0');
   sign = 1;
   if (the_token->token_type == T_MINUS)
   {
	  sign = -1;
	  the_token = get_token(read_file, '\0');
   }
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in skill %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "AdminFlag4 %ld\n", (atol(the_token->the_string) * sign));

   /* Set flag */
   the_token = get_token(read_file, '\0');
   sign = 1;
   if (the_token->token_type == T_MINUS)
   {
	  sign = -1;
	  the_token = get_token(read_file, '\0');
   }
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in skill %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "AdminFlag5 %ld\n", (atol(the_token->the_string) * sign));

   /* Set flag */
   the_token = get_token(read_file, '\0');
   sign = 1;
   if (the_token->token_type == T_MINUS)
   {
	  sign = -1;
	  the_token = get_token(read_file, '\0');
   }
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in skill %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "AdminFlag6 %ld\n", (atol(the_token->the_string) * sign));

   version = 0;

   return 1;
}



/***********************************************************************
 ** convert_boat_attrib - reads in the boats attributes and converts
 **                       them to pseudo-data format
 **
 ** Parameters: read_file - the AIME file we read from
 **             write_file - the AIME file we write to
 **             objname - the name of the object we are converting
 **             version - which version we are loading
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int convert_boat_attrib(FILE *read_file, FILE *write_file, char *objname, float version)
{
   token_record *the_token;

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      printf("Invalid format in boat %s.\n", objname);
      return -1;
   }

   /* get the move_str string */
   the_token = get_token(read_file, '^');
   fprintf(write_file, "MoveStr ^%s^\n", (the_token->the_string == NULL) ? "" : 
																		the_token->the_string);

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      printf("Invalid format in boat %s.\n", objname);
      return -1;
   }

   /* get the move_str string */
   the_token = get_token(read_file, '^');
   fprintf(write_file, "WaterBrief ^%s^\n", (the_token->the_string == NULL) ? "" : 
																		the_token->the_string);
   version = 0;
   return 1;
}


/***********************************************************************
 ** convert_rope_attrib - reads in the rope attributes and converts
 **                       them to pseudo-data format
 **
 ** Parameters: read_file - the AIME file we read from
 **             write_file - the AIME file we write to
 **             objname - the name of the object we are converting
 **             version - which version we are loading
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int convert_rope_attrib(FILE *read_file, FILE *write_file, char *objname, float version)
{
   token_record *the_token;

   /* Set rope length */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in boat %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "RopeLength %d\n", atoi(the_token->the_string));

   version = 0;
   return 1;
}



/***********************************************************************
 ** read_inclination_attrib - reads in the inclination attributes of the object
 **
 ** Parameters: read_file - the AIME file we read from
 **             version - which version we are loading
 **
 ** Returns: 1 for success
 **
 ***********************************************************************/

int convert_inclination_attrib(FILE *read_file, FILE *write_file, char *objname, float version)
{
   token_record *the_token;
   char         *temp_desc;
   int          is_neg = 1;

   fprintf(write_file, "Name ^%s^\n", objname);

   /* get the description string */
   temp_desc = read_desc_type(read_file, 0, NULL);  
    if (temp_desc == NULL)
      return -1;
   fprintf(write_file, "Description ^%s^\n", (temp_desc == NULL) ? "" : temp_desc);
   delete temp_desc;

   /* get the special */
   the_token = get_token(read_file, '\0');
   fprintf(write_file, "Special ^%s^\n", (the_token->the_string == NULL) ? "" : 
																		the_token->the_string);

   /* Set str_offset value */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type == T_MINUS)
   {
      is_neg = 1;
      the_token = get_token(read_file, '\0');
   }

   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in str_offset for inclination %s.\n", 
                                                        objname);
      return -1;
   }
   fprintf(write_file, "StrengthOffset %d\n", (is_neg * atoi(the_token->the_string)));

   is_neg = 1;

   /* Set dex_offset value */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type == T_MINUS)
   {
      is_neg = 1;
      the_token = get_token(read_file, '\0');
   }

   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in dex_offset for inclination %s.\n", 
                                                        objname);
      return -1;
   }
   fprintf(write_file, "DexterityOffset %d\n", (is_neg * atoi(the_token->the_string)));
   is_neg = 1;

   /* Set con_offset value */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type == T_MINUS)
   {
      is_neg = 1;
      the_token = get_token(read_file, '\0');
   }

   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in con_offset for inclination %s.\n", 
                                                        objname);
      return -1;
   }
   fprintf(write_file, "ConstitutionOffset %d\n", (is_neg * atoi(the_token->the_string)));
   is_neg = 1;

   /* Set intel_offset value */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type == T_MINUS)
   {
      is_neg = 1;
      the_token = get_token(read_file, '\0');
   }

   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in intel_offset for inclination %s.\n", 
                                                        objname);
      return -1;
   }
   fprintf(write_file, "IntelOffset %d\n", (is_neg * atoi(the_token->the_string)));

   if (version > 0.58)
   {
      /* Set wis_offset value */
      the_token = get_token(read_file, '\0');     
      if (the_token->token_type == T_MINUS)
      {
         is_neg = 1;
         the_token = get_token(read_file, '\0');
      }

      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in wisdom_offset for inclination %s.\n", 
                                                        objname);
         return -1;
      }
      fprintf(write_file, "WisdomOffset %d\n", (is_neg * atoi(the_token->the_string)));
      is_neg = 1;

      /* Set cha_offset value */
      the_token = get_token(read_file, '\0');     
      if (the_token->token_type == T_MINUS)
      {
         is_neg = 1;
         the_token = get_token(read_file, '\0');
      }

      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in charisma_offset for inclination %s.\n", 
                                                         objname);
         return -1;
      }
      fprintf(write_file, "CharismaOffset %d\n", (is_neg * atoi(the_token->the_string)));
   }

   version = 0;
   return 1;
}



/***********************************************************************
 ** read_talent_attrib - reads in the talent attributes of the object
 **
 ** Parameters: read_file - the AIME file we read from
 **             version - which version we are loading
 **
 ** Returns: 1 for success
 **
 ***********************************************************************/

int convert_talent_attrib(FILE *read_file, FILE *write_file, char *objname, float version)
{
   token_record *the_token;
   char         *temp_desc;
   int          is_neg = 1;

   fprintf(write_file, "Name ^%s^\n", objname);

   /* get the description string */
   temp_desc = read_desc_type(read_file, 0, NULL);  
   if (temp_desc == NULL)
      return -1;
   fprintf(write_file, "Description ^%s^\n", (temp_desc == NULL) ? "" : temp_desc);
   delete temp_desc;

   /* get the special */
   the_token = get_token(read_file, '\0');
   fprintf(write_file, "Special ^%s^\n", (the_token->the_string == NULL) ? "" : 
																		the_token->the_string);

   /* Set cost value */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type == T_MINUS)
   {
      is_neg = -1;
      the_token = get_token(read_file, '\0');
   }

   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in cost for talent %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "Cost %d\n", (is_neg * atoi(the_token->the_string)));
   is_neg = 1;

   /* Set allowed value */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      printf("Invalid format in allowed for talent %s.\n", objname);
      return -1;
   }
   fprintf(write_file, "Allowed %d\n", atoi(the_token->the_string));

   version = 0;
   return 1;
}


/***********************************************************************
 ** aime_to_pseudo - converts aime area to pseudo-data
 **
 ** Parameters: read_file - the AIME file we read from
 **             write_file - the AIME file we write to
 **             filename - the name of the file we are reading in
 **             version - the version of this aime file
 **
 ** Returns: 1 for success
 **
 ***********************************************************************/

int aime_to_pseudo(FILE *read_file, FILE *write_file, char *filename, 
                                                                float version)
{
   Strings      objtype;
   token_record *the_token;
   Strings      objname;
   Strings      tmp_filename;
   char         *tmp_char;
   char         *last_slash = NULL;
   int          short_header = 0;
   int          is_action = 0;
   int          is_race = 0;
   int          is_mask = 0;
   int          is_level = 0;
   int          is_quest = 0;
   int          is_inclination = 0;
   int          is_talent = 0;
   int          is_chatline = 0;

   /* break out the filename */
   tmp_filename = filename;
   tmp_char = tmp_filename.str_show();
   while ((*tmp_char) && (*tmp_char != '.'))
   {
#ifdef WIN32
      if (*tmp_char == '\\')
#else
      if (*tmp_char == '/')
#endif
         last_slash = tmp_char;
      tmp_char++;
   }
   if (!*tmp_char)
      tmp_char = tmp_filename.str_show();
   else
      *tmp_char = '\0';
   
   if (last_slash == NULL)
      tmp_char = tmp_filename.str_show();
   else
      tmp_char = last_slash+1;

   if ((!STRCASECMP(tmp_char, "abilities")) ||
       (!STRCASECMP(tmp_char, "actions")) ||
       (!STRCASECMP(tmp_char, "races")) ||
       (!STRCASECMP(tmp_char, "levels")) ||
       (!STRCASECMP(tmp_char, "quests")) ||
       (!STRCASECMP(tmp_char, "masks")) ||
       (!STRCASECMP(tmp_char, "chatlines")))
      short_header = 1;

   if (!STRCASECMP(tmp_char, "actions"))
      is_action = 1;
   else if (!STRCASECMP(tmp_char, "races"))
      is_race = 1;
   else if (!STRCASECMP(tmp_char, "masks"))
      is_mask = 1;
   else if (!STRCASECMP(tmp_char, "levels"))
      is_level = 1;
   else if (!STRCASECMP(tmp_char, "quests"))
      is_quest = 1;
   else if (!STRCASECMP(tmp_char, "inclinations"))
      is_inclination = 1;
   else if (!STRCASECMP(tmp_char, "talents"))
      is_talent = 1;
   else if (!STRCASECMP(tmp_char, "chatlines"))
      is_chatline = 1;


   /* write the header information first */
   fprintf(write_file, "StartHeader\n");

   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      printf("Invalid format in area file '%s' header.\n", filename);
      return 0;
   }
   the_token = get_token(read_file, '^');
   fprintf(write_file, "ReadDeny ^%s^\n", (the_token->the_string == NULL) ? "" : 
                                            the_token->the_string);

   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      printf("Invalid format in area file '%s' header.\n", filename);
      return 0;
   }
   the_token = get_token(read_file, '^');
   fprintf(write_file, "WriteAllow ^%s^\n", (the_token->the_string == NULL) ? "" : 
                                            the_token->the_string);

   if (!short_header)
   {
      /* get the area flags */
      the_token = get_token(read_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in area file '%s' header.\n", filename);
         return 0;
      }
      fprintf(write_file, "Flag1 %ld\n", atol(the_token->the_string));

      the_token = get_token(read_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in area file '%s' header.\n", filename);
         return 0;
      }
      fprintf(write_file, "Flag2 %ld\n", atol(the_token->the_string));


      /* now write the reload interval */
      the_token = get_token(read_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in area file '%s' header.\n", filename);
         return 0;
      }
      fprintf(write_file, "ReloadSecs %s\n", the_token->the_string);
   }
   fprintf(write_file, "EndHeader\n\n");


   the_token = get_token(read_file, '\0');
   if (is_action)
   {
      while (the_token->token_type > 0)
      {
         objname = the_token->the_string;
	 fprintf(write_file, "StartAction\n");
         if (convert_action_attrib(read_file, write_file, objname.str_show(), 
				   version) <= 0)
	 {
	   fclose(write_file);
	   return -1;
	 }
	 fprintf(write_file, "EndAction\n\n");
         the_token = get_token(read_file, '\0');
      }
      return 1;
   }
   if (is_chatline)
   {
      while (the_token->token_type > 0)
      {
         the_token = get_token(read_file, '\0');
         objname = the_token->the_string;
		 fprintf(write_file, "StartChatLine\n");
         if (convert_chatline_attrib(read_file, write_file, 
				          objname.str_show(), version) <= 0)
	 {
	    fclose(write_file);
            return -1;
	 }
	 fprintf(write_file, "EndChatLine\n\n");
         the_token = get_token(read_file, '\0');
      }
      return 1;
   }

   else if (is_race)
   {
      while (the_token->token_type > 0)
      {
         if (version > 0.55)
            the_token = get_token(read_file, '\0');

         objname = the_token->the_string;

		 fprintf(write_file, "StartRace\n");
         if (convert_race_attrib(read_file, write_file, objname.str_show(), version) <= 0)
         {
			 fclose(write_file);
			 return -1;
		 }

	 fprintf(write_file, "EndRace\n\n");
         the_token = get_token(read_file, '\0');
      }
      return 1;
   }  
   else if (is_quest)
   {
      while (the_token->token_type > 0)
      {
         if (version > 0.55)
            the_token = get_token(read_file, '\0');

         objname = the_token->the_string;

	 printf("name: %s\n", objname.str_show());
	 fprintf(write_file, "StartQuest\n");
         if (convert_quest_attrib(read_file, write_file, objname.str_show(), 
				  version) <= 0)
         {
	   fclose(write_file);
	   return -1;
	 }

	 fprintf(write_file, "EndQuest\n\n");
         the_token = get_token(read_file, '\0');
      }
      return 1;
   }  
   else if (is_level)
   {
      while (the_token->token_type > 0)
      {
         if (version > 0.55)
            the_token = get_token(read_file, '\0');

         objname = the_token->the_string;

		 fprintf(write_file, "StartLevel\n");
         if (convert_level_attrib(read_file, write_file, objname.str_show(), version) <= 0)
         {
			 fclose(write_file);
			 return -1;
		 }

		 fprintf(write_file, "EndLevel\n\n");
         the_token = get_token(read_file, '\0');
      }
      return 1;
   }  
   else if (is_mask)
   {
      while (the_token->token_type > 0)
      {
//         if (version > 0.55)
//            the_token = get_token(read_file, '\0');

         objname = the_token->the_string;

		 fprintf(write_file, "StartMask\n");
         if (convert_mask_attrib(read_file, write_file, objname.str_show(), version) <= 0)
         {
			 fclose(write_file);
			 return -1;
		 }

		 fprintf(write_file, "EndMask\n\n");
         the_token = get_token(read_file, '\0');
      }
      return 1;
   }  
   else if (is_inclination)
   {
      while (the_token->token_type > 0)
      {
         if (version > 0.55)
            the_token = get_token(read_file, '\0');

         objname = the_token->the_string;

		 fprintf(write_file, "StartInclination\n");
         if (convert_inclination_attrib(read_file, write_file, objname.str_show(), version) <= 0)
         {
			 fclose(write_file);
			 return -1;
		 }

		 fprintf(write_file, "EndInclination\n\n");
         the_token = get_token(read_file, '\0');
      }
      return 1;
   }  

   else if (is_talent)
   {
      while (the_token->token_type > 0)
      {
         if (version > 0.55)
            the_token = get_token(read_file, '\0');

         objname = the_token->the_string;

		 fprintf(write_file, "StartTalent\n");
         if (convert_talent_attrib(read_file, write_file, objname.str_show(), version) <= 0)
         {
			 fclose(write_file);
			 return -1;
		 }

		 fprintf(write_file, "EndTalent\n\n");
         the_token = get_token(read_file, '\0');
      }
      return 1;
   }  
   while (the_token->token_type > 0)
   {
      objtype = the_token->the_string;
      the_token = get_token(read_file, '\0');
      objname = the_token->the_string;

      printf("Converting: %s\n", objname.str_show());
      fflush(stdout);
      if (!STRCASECMP(objtype.str_show(), "loc"))
      {
         fprintf(write_file, "StartLocation\n");
         if ((convert_mudobj_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_loc_attrib(read_file, write_file, objname.str_show(), version) < 0))
            return -1;
		 fprintf(write_file, "EndLocation\n\n");

      }
      else if (!STRCASECMP(objtype.str_show(), "marker"))
      {
         fprintf(write_file, "StartMarker\n");
         if ((convert_mudobj_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_item_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_marker_attrib(read_file, write_file, objname.str_show(), version) < 0))
            return -1;
		 fprintf(write_file, "EndMarker\n\n");
      }
      else if (!STRCASECMP(objtype.str_show(), "door"))
      {
	 conv_doorflag = 0;
         fprintf(write_file, "StartDoor\n");
         if ((convert_mudobj_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_item_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_door_attrib(read_file, write_file, objname.str_show(), version) < 0))
            return -1;
		 fprintf(write_file, "EndDoor\n\n");
      }
      else if (!STRCASECMP(objtype.str_show(), "key"))
      {
         fprintf(write_file, "StartKey\n");
         if ((convert_mudobj_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_item_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_moveable_attrib(read_file, write_file, objname.str_show(), version) < 0))
            return -1;
		 fprintf(write_file, "EndKey\n\n");

      }
      else if (!STRCASECMP(objtype.str_show(), "book"))
      {
         fprintf(write_file, "StartBook\n");
         if ((convert_mudobj_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_item_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_moveable_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_book_attrib(read_file, write_file, objname.str_show(), version) < 0))
            return -1;
		 fprintf(write_file, "EndBook\n\n");
      }
      else if (!STRCASECMP(objtype.str_show(), "mob"))
      {
         fprintf(write_file, "StartMobile\n");
         if ((convert_mudobj_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_mobile_attrib(read_file, write_file, objname.str_show(), version) < 0))
            return -1;
		 fprintf(write_file, "EndMobile\n\n");
      }
      else if (!STRCASECMP(objtype.str_show(), "weapon"))
      {
         fprintf(write_file, "StartWeapon\n");
         if ((convert_mudobj_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_item_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_moveable_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_weapon_attrib(read_file, write_file, objname.str_show(), version) < 0))
            return -1;
		 fprintf(write_file, "EndWeapon\n\n");
      }
      else if (!STRCASECMP(objtype.str_show(), "moveable"))
      {
         fprintf(write_file, "StartMoveable\n");
         if ((convert_mudobj_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_item_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_moveable_attrib(read_file, write_file, objname.str_show(), version) < 0))
            return -1;
		 fprintf(write_file, "EndMoveable\n\n");
      }
      else if (!STRCASECMP(objtype.str_show(), "wearable"))
      {
         fprintf(write_file, "StartWearable\n");
         if ((convert_mudobj_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_item_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_moveable_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_wearable_attrib(read_file, write_file, objname.str_show(), version) < 0))
            return -1;
		 fprintf(write_file, "EndWearable\n\n");
      }
      else if (!STRCASECMP(objtype.str_show(), "food"))
      {
         fprintf(write_file, "StartFood\n");
         if ((convert_mudobj_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_item_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_moveable_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_merger_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_food_attrib(read_file, write_file, objname.str_show(), version) < 0))
            return -1;
		 fprintf(write_file, "EndFood\n\n");
      }
      else if (!STRCASECMP(objtype.str_show(), "merger"))
      {
         fprintf(write_file, "StartMerger\n");
         if ((convert_mudobj_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_item_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_moveable_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_merger_attrib(read_file, write_file, objname.str_show(), version) < 0))
            return -1;
		 fprintf(write_file, "EndMerger\n\n");

      }
      else if (!STRCASECMP(objtype.str_show(), "money"))
      {
         fprintf(write_file, "StartMoney\n");
         if ((convert_mudobj_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_item_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_moveable_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_merger_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_money_attrib(read_file, write_file, objname.str_show(), version) < 0))
            return -1;
		 fprintf(write_file, "EndMoney\n\n");
      }
      else if (!STRCASECMP(objtype.str_show(), "boat"))
      {
         fprintf(write_file, "StartBoat\n");
         if ((convert_mudobj_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_item_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_moveable_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_boat_attrib(read_file, write_file, objname.str_show(), version) < 0))
            return -1;
		 fprintf(write_file, "EndBoat\n\n");
      }
      else if (!STRCASECMP(objtype.str_show(), "rope"))
      {
         fprintf(write_file, "StartRope\n");
         if ((convert_mudobj_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_item_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_moveable_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_rope_attrib(read_file, write_file, objname.str_show(), version) < 0))
            return -1;
		 fprintf(write_file, "EndRope\n\n");
      }

      else if (!STRCASECMP(objtype.str_show(), "special"))
      {
         fprintf(write_file, "StartSpecial\n");
		 
         if (convert_specials_attrib(read_file, write_file, objname.str_show(), version) < 0)
            return -1;
		 fprintf(write_file, "EndSpecial\n\n");

      }
      else if (!STRCASECMP(objtype.str_show(), "spell"))
      {
         fprintf(write_file, "StartSpell\n");
         if ((convert_ability_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_spell_attrib(read_file, write_file, objname.str_show(), version) < 0))
            return -1;
		 fprintf(write_file, "EndSpell\n\n");
      }
      else if (!STRCASECMP(objtype.str_show(), "skill"))
      {
         fprintf(write_file, "StartSkill\n");
         if ((convert_ability_attrib(read_file, write_file, objname.str_show(), version) < 0) ||
             (convert_skill_attrib(read_file, write_file, objname.str_show(), version) < 0))
            return -1;
		 fprintf(write_file, "EndSkill\n\n");
      }
 
      else
      {
         printf("Invalid object type '%s' in area zone file '%s'.\n", 
                                objtype.str_show(), filename);
         return 0;
      }

      fprintf(write_file, "\n\n");
      the_token = get_token(read_file, '\0');
   }
   return 1;
}


#endif





