/*
    This file is part of AirSnort.

    AirSnort is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    AirSnort is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with AirSnort; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <signal.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <linux/wireless.h>

#include "capture.h"
#include "crack.h"
#include "PacketSource.h"
#include "bssidlist.h"

//#define WEP_FLAG_WORKS

#define ADDR1(x) (x)->raw + 148
#define ADDR2(x) (x)->raw + 154
#define ADDR3(x) (x)->raw + 160

extern int scan;
extern int chan;
extern int doCapture;

extern char cmd[80];
int count = 0;
int lasttime = 0;
int useOrinoco = 0;

void quick_message(char *title, char *message);

//initialize data in four capture records according to TO_DS,
//FROM_DS flags.  This saves time during packet decoding
void initCapRecs(CaptureRec *cr, PacketInfo *pi) {
   int i;
   for (i = 0; i < 4; i++) {
      cr[i].pInfo = pi;
      cr[i].iv = pi->raw + 168;
      switch (i) {
         case 0:  //STA to STA, or management and control
            cr[0].bssid = ADDR3(pi);
            cr[0].sta =  ADDR2(pi);
            break;
         case 2:  //AP to STA data
            cr[2].bssid = ADDR2(pi);
            cr[2].sta = ADDR1(pi); 
            break;
         case 3:  //AP to AP 
            cr[3].iv += 6;
         case 1:  //STA to AP data
            cr[i].bssid = ADDR1(pi);
            cr[i].sta = ADDR2(pi); 
            break;
      }
   }
}

//the thread function for the packet capture thread.  It opens a packet 
//source and collects packets until instructed to stop. Packets are 
//parsed for interesting info and placed on the packet queue for consumption
//by the cracking thread
void *capture(void *arg) {
   int len, namelen, whichType, channelLoc;
   PacketInfo pi;
   CaptureRec cap[4], *thePacket;
   BssidList *temp;
   unsigned char packet[3000];
   PacketSource ps;
   const unsigned char *MAC_ADDR3 = packet + 160;

   pi.raw = packet;
   initCapRecs(cap, &pi);

   CreatePacketSource(&ps);
   if (openPacketSource(&ps) < 0) {
     quick_message("Error", "Unable to open packet source");
     doCapture = 0;
     pthread_exit(NULL);
   }

   while (doCapture) {
      pi.pack = NULL;

      //this method of cycling through the channels is not the greatest
      //but it gets the job done.  I would rather see it cycle at a fixed
      //interval like 0.1 seconds
      if (scan) {
        if (count == 5) {
           count = 0;
           chan = (chan % 11) + 1;
           setChannel(chan);
        }
        count++;
      }
      len = getPacket(&ps, (char *) packet, 3000, 10);
      pi.channel = chan;
      if (len < 0) {
        if( len == ERR_TIMEOUT ) {
	  continue;
        }
      
        if( errno == ENOBUFS ) continue;
        quick_message("Error", "Unable to get packet"); 
        break;
      }

#ifndef WEP_FLAG_WORKS
//could just process all beacon frames and add a WEP column in the display
//then this becomes a mini prismstubler of sorts, as all received APs 
//would be displayed
      if ((packet[178] & 0x10) && ((packet[144] == 0x80) || (packet[144] == 0x50))) {
         if (packet[144] == 0x80 && useOrinoco) {
            continue;  //orinoco's seem to mangle beacon frames.
         }
         //beacon or probe request frame w/ WEP
         namelen = packet[181];
         temp = bssidFind(MAC_ADDR3);
         if (temp) {
            if (temp->name) {
               if (packet[144] == 0x80) { //don't take name more than once from a beacon
                  continue;
               }
               free(temp->name);
            }
            temp->name = (char*) malloc(namelen + 1);
            strncpy(temp->name, packet + 182, namelen);
            (temp->name)[namelen] = 0;
            continue;
         }
       //might want to check elementid at position 180 == 0 as well
         pi.name = (char*) malloc(namelen + 1);
         strncpy(pi.name, packet + 182, namelen);
         pi.name[namelen] = 0;
         if (temp) {
            temp->name = pi.name;
         }
         else {
            channelLoc = 182 + namelen + 2;
            channelLoc += packet[channelLoc - 1];
            if (packet[channelLoc] == 3) {
               pi.channel = packet[channelLoc + 2];
            }
            addBssidFromBeacon(cap); 
         }
         continue;  //done
      }
#endif

      if ((packet[144] & 0x48) != 0x08) continue;  //not a data frame
      whichType = packet[145] & 0x03;
      thePacket = &(cap[whichType]);
      len -= whichType == 3 ? 178 : 172;  //adjust len to size of frame body

      //this is a sloppy way to detect unencrypted packets, but since 0xAA is not
      //an IV byte of interest anyway, its probably not a big deal
      if (thePacket->iv[0] == 0xAA && !(packet[145] & 0x40)) continue;

      if (isResolved(thePacket->iv)) {
        pi.pack = (Packet*) malloc(sizeof(Packet));
        pi.pack->len = len;  //i.e. 802.11b "Frame Body"
        pi.pack->buf = (unsigned char*) malloc(len);
        memcpy(pi.pack->buf, thePacket->iv, len);
        pi.pack->next = NULL;
      }

      addBssid(thePacket); 
   }

   DestroyPacketSource(&ps);
   resetMAC();

   pthread_exit(NULL);
}

/* 
* The setChannel function was lifted from the wlanctl source file:
*
*src/wlanctl/wlanctl.c
*
* user utility for the wlan card
*
* Copyright (C) 1999 AbsoluteValue Systems, Inc.  All Rights Reserved.
* --------------------------------------------------------------------
*
* linux-wlan
*
    linux-wlan is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    linux-wlan is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with linux-wlan; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*
* --------------------------------------------------------------------
*
* Inquiries regarding the linux-wlan Open Source project can be
* made directly to:
*
* AbsoluteValue Systems Inc.
* info@linux-wlan.com
* http://www.linux-wlan.com
*
* --------------------------------------------------------------------
*/

char dev[WLAN_DEVNAMELEN_MAX] = "wlan0";

/*
 * Set the NIC into promiscuous mode on the indicated channel
 * This code was distilled from wlanctl.c which is part of 
 * the linux-wlan-ng package.  This is the only wlanctl functionality
 * needed in airsnort, so I put it here rather than shelling out
 * to invoke wlanctl-ng every time I wanted to change channels. CSE.
 */

#define MAKE_SYS_CALL
/* 
 * as opposed to making ioctl calls directly.  We can make ioctls 
 * directly if we can tie into the linux-wlan-ng header files
 */

int setChannel( unsigned char channel )
{
   p80211msg_lnxreq_wlansniff_t sniff;

   int result = -1;
   int fd;
   struct iwreq ireq;  //for Orinoco

   memset(&sniff, 0, sizeof(p80211msg_lnxreq_wlansniff_t));
   sniff.msgcode = DIDmsg_lnxreq_wlansniff;
   sniff.msglen = sizeof(p80211msg_lnxreq_wlansniff_t);
   strcpy((char*) sniff.devname, dev);

   sniff.enable.did = DIDmsg_lnxreq_wlansniff_enable;
   sniff.enable.len = 4;
   sniff.enable.data = P80211ENUM_truth_true;

   sniff.channel.did = DIDmsg_lnxreq_wlansniff_channel;
   sniff.channel.len = 4;
   sniff.channel.data = channel;

   sniff.resultcode.did = DIDmsg_lnxreq_wlansniff_resultcode;
   sniff.resultcode.status = P80211ENUM_msgitem_status_no_value;
   sniff.resultcode.len = 4;

   if (useOrinoco) {  //assumes patched pcmcia-cs-3.1.31 and using orinoco_cs
      /* get a socket */
      fd = socket(AF_INET, SOCK_STREAM, 0);

      if ( fd == -1 ) {
         return result;
      }
      ireq.u.data.pointer = (caddr_t) &sniff;   
      strcpy(ireq.ifr_ifrn.ifrn_name, dev);
      result = ioctl( fd, SIOCDEVPRIVATE + 0x8, &ireq);
      close(fd);
   }
   else {
#ifndef MAKE_SYS_CALL
      result = do_ioctl((uint8_t*) &sniff);
#else
      char cmd[128];
      static char *parms = "keepwepflags=false prismheader=true";
      sprintf(cmd, "wlanctl-ng %s lnxreq_wlansniff enable=true channel=%d %s > /dev/null", 
                    dev, channel, use_PF_PACKET ? parms : "");
      result = system(cmd);
#endif
   }
   return result;
}

int resetMAC()
{
   p80211msg_lnxreq_wlansniff_t sniff;
   int result = -1;
   int fd;
   struct iwreq ireq;  //for Orinoco

   memset(&sniff, 0, sizeof(p80211msg_lnxreq_wlansniff_t));
   sniff.msgcode = DIDmsg_lnxreq_wlansniff;
   sniff.msglen = sizeof(p80211msg_lnxreq_wlansniff_t);
   strcpy((char*) sniff.devname, dev);

   sniff.enable.did = DIDmsg_lnxreq_wlansniff_enable;
   sniff.enable.len = 4;
   sniff.enable.data = P80211ENUM_truth_false;

   sniff.channel.did = DIDmsg_lnxreq_wlansniff_channel;
   sniff.channel.status = P80211ENUM_msgitem_status_no_value;
   sniff.channel.len = 4;

   sniff.resultcode.did = DIDmsg_lnxreq_wlansniff_resultcode;
   sniff.resultcode.status = P80211ENUM_msgitem_status_no_value;
   sniff.resultcode.len = 4;

   if (useOrinoco) {  //assumes patched pcmcia-cs-3.1.31 and using orinoco_cs
      /* get a socket */
      fd = socket(AF_INET, SOCK_STREAM, 0);

      if ( fd == -1 ) {
         return result;
      }
      ireq.u.data.pointer = (caddr_t) &sniff;   
      strcpy(ireq.ifr_ifrn.ifrn_name, dev);
      result = ioctl( fd, SIOCDEVPRIVATE + 0x8, &ireq);
      close(fd);
   }
   else {
#ifndef MAKE_SYS_CALL
      result = do_ioctl((uint8_t*) &sniff);
#else
      char cmd[80];
      sprintf(cmd, "wlanctl-ng %s lnxreq_wlansniff enable=false > /dev/null", dev);
      result = system(cmd);
#endif
   }
   return result;
}

int do_ioctl( unsigned char *userMsg )
{
   int result = -1;
   int fd;
   p80211ioctl_req_t req;
   p80211msg_t *msg = (p80211msg_t *) userMsg;

   strcpy(msg->devname, dev);

   /* set the magic */
   req.magic = P80211_IOCTL_MAGIC;

   /* get a socket */
   fd = socket(AF_INET, SOCK_STREAM, 0);

   if ( fd == -1 ) {
      return result;
   }

   req.len = msg->msglen;
   req.data = msg;   
   strcpy( req.name, dev);
   req.result = 0;

   result = ioctl( fd, P80211_IFREQ, &req);
   close(fd);
   return result;

}
