//==============================================
//  copyright            : (C) 2003-2005 by Will Stokes
//==============================================
//  This program is free software; you can redistribute it
//  and/or modify it under the terms of the GNU General
//  Public License as published by the Free Software
//  Foundation; either version 2 of the License, or
//  (at your option) any later version.
//==============================================

//Systemwide includes
#include <qlayout.h>
#include <qlabel.h>
#include <qfont.h>
#include <qpixmap.h>
#include <qimage.h>
#include <qpushbutton.h>
#include <qfileinfo.h>

#include <qtoolbutton.h>

//Projectwide includes
#include "albumStatistics.h"
#include "../../config.h"
#include "../../backend/album.h"
#include "../../backend/subalbum.h"
#include "../../backend/photo.h"
#include "../../backend/tools/imageTools.h"

//==============================================
AlbumStatistics::AlbumStatistics( Album* album,
                                             QWidget* parent,
                                             const char* name ) :
                                             QDialog(parent,name)
{
  //--------------------------------------------------------------
  QColor white(255, 255, 255);
  QColor darkBlue(35, 75, 139);
  //--------------------------------------------------------------
  //this album pointer
  this->album = album;
  //--
  //compute size on disk
  int albumSize = 0;
  Subalbum* curSubalbum = album->getFirstSubalbum();
  QFileInfo info;
  while(curSubalbum != NULL)
  {
    Photo* curPhoto = curSubalbum->getFirst();
    while(curPhoto != NULL)
    {
      info.setFile( curPhoto->getImageFilename() );
      albumSize+=info.size();

      info.setFile( curPhoto->getSlideshowFilename() );
      albumSize+=info.size();

      curPhoto = curPhoto->getNext();
    }
    curSubalbum = curSubalbum->getNext();
  }
  //--
  //set window title
  setCaption( tr("Album Statistics"));
  //--
  //create title
  titleMessage = new QLabel( tr("Album Statistics:"), this);

  QFont titleFont = titleMessage->font();
  titleFont.setWeight(QFont::Bold);
  titleFont.setPointSize( titleFont.pointSize() + 2 );

  QFont statsFont = titleMessage->font();
  statsFont.setWeight(QFont::Bold);

  titleMessage->setFont( titleFont );
  //--
  //create stats
  //--
  numSubalbums = new QLabel( tr("Collections:"), this);
  numSubalbums->setFont( statsFont );
  numSubalbumsVal = new QLabel(this);
  numSubalbumsVal->setText( QString("%1").arg(album->getNumSubalbums()) );
  numSubalbumsVal->setFont( statsFont );
  //--
  numPhotos = new QLabel( tr("Photos:"), this);
  numPhotos->setFont( statsFont );
  numPhotosVal = new QLabel(this);
  numPhotosVal->setText( QString("%1").arg(album->getNumPhotos()) );
  numPhotosVal->setFont( statsFont );
  //--
  sizeOnDisk = new QLabel( tr("Size:"), this);
  sizeOnDisk->setFont( statsFont );
  sizeOnDiskVal = new QLabel(this);
  sizeOnDiskVal->setFont( statsFont );
  if(albumSize < 1024)
    sizeOnDiskVal->setText( QString(tr("~%1 Bytes")).arg(albumSize) );
  else if( albumSize/1024 < 1024)
    sizeOnDiskVal->setText( QString(tr("~%1 Kb")).arg( ((float)albumSize)/1024 ) );
  else if( albumSize/(1024*1024) < 1024)
    sizeOnDiskVal->setText( QString(tr("~%1 Mb")).arg( ((float)albumSize)/(1024*1024) ) );
  else
    sizeOnDiskVal->setText( QString(tr("~%1 Gigs")).arg( ((float)albumSize)/(1024*1024*1024) ) );
  //--
  QString months[] = { tr("January"),
  tr("February"),
  tr("March"),
  tr("April"),
  tr("May"),
  tr("June"),
  tr("July"),
  tr("August"),
  tr("September"),
  tr("October"),
  tr("November"),
  tr("December") };

  created = new QLabel( tr("Created:"), this);
  created->setFont( statsFont );
  QString cVal = QString("%1 %2").arg(months[album->getCreationMonth()-1]).arg(album->getCreationDay());
  if(album->getCreationDay() == 1 ||
     album->getCreationDay() == 21 ||
     album->getCreationDay() == 31)
     cVal = cVal + "st";
  else if(album->getCreationDay() == 2 ||
     album->getCreationDay() == 22)
     cVal = cVal + "nd";
  else if(album->getCreationDay() == 3 ||
     album->getCreationDay() == 23)
     cVal = cVal + "rd";
  else
     cVal = cVal + "th";
  cVal = QString("%1, %2").arg(cVal).arg(album->getCreationYear());

  createdVal = new QLabel( cVal, this );
  createdVal->setFont( statsFont );

  modified = new QLabel( tr("Modified:"), this);
  modified->setFont( statsFont );
  QString mVal = QString("%1 %2").arg(months[album->getModificationMonth()-1]).arg(album->getModificationDay());

  if(album->getModificationDay() == 1 ||
     album->getModificationDay() == 21 ||
     album->getModificationDay() == 31)
     mVal = mVal + "st";
  else if(album->getModificationDay() == 2 ||
     album->getModificationDay() == 22)
     mVal = mVal + "nd";
  else if(album->getModificationDay() == 3 ||
     album->getModificationDay() == 23)
     mVal = mVal + "rd";
  else
     mVal = mVal + "th";
  mVal = QString("%1, %2").arg(mVal).arg(album->getModificationYear());
  modifiedVal = new QLabel( mVal, this );
  modifiedVal->setFont( statsFont );
    //--
  //create album image and title labels
  albumPreview = new QFrame( this );
  albumIcon = new QLabel( albumPreview );

  //if no rep image use small version
  if(album->getRepresentativeImage(LARGE) != NULL)
  {
    QImage tImage = album->getRepresentativeImage( LARGE )->convertToImage();
    int newWidth, newHeight;
    calcScaledImageDimensions( tImage.width(), tImage.height(), 
                               300, 300, 
                               newWidth, newHeight);
    QImage tImage2 = tImage.smoothScale( newWidth, newHeight );
    albumImage = new QPixmap( newWidth, newHeight );
    albumImage->convertFromImage( tImage2 );
    albumIcon->setPixmap( *albumImage );
  }

  albumTitle = new QLabel( albumPreview );
  if(album->getName().compare("") != 0)
  {
    albumTitle->setText( "\"" + album->getName() + "\"" );
  }
  albumTitle->setFont( statsFont );
  //--
  //create close button
  closeButton = new QPushButton( tr("Close"), this );
  closeButton->setSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed );
  closeButton->setDefault(true);
  closeButton->setFocus();
  connect( closeButton, SIGNAL(clicked()), SLOT(close()) );
  //--
  setPaletteBackgroundColor( darkBlue );
  titleMessage->setPaletteForegroundColor( white );
  titleMessage->setPaletteBackgroundColor( darkBlue );
  numSubalbums->setPaletteForegroundColor( white );
  numSubalbums->setPaletteBackgroundColor( darkBlue );
  numSubalbumsVal->setPaletteForegroundColor( white );
  numSubalbumsVal->setPaletteBackgroundColor( darkBlue );
  numPhotos->setPaletteForegroundColor( white );
  numPhotos->setPaletteBackgroundColor( darkBlue );
  numPhotosVal->setPaletteForegroundColor( white );
  numPhotosVal->setPaletteBackgroundColor( darkBlue );
  sizeOnDisk->setPaletteForegroundColor( white );
  sizeOnDisk->setPaletteBackgroundColor( darkBlue );
  sizeOnDiskVal->setPaletteForegroundColor( white );
  sizeOnDiskVal->setPaletteBackgroundColor( darkBlue );
  created->setPaletteForegroundColor( white );
  created->setPaletteBackgroundColor( darkBlue );
  createdVal->setPaletteForegroundColor( white );
  createdVal->setPaletteBackgroundColor( darkBlue );
  modified->setPaletteForegroundColor( white );
  modified->setPaletteBackgroundColor( darkBlue );
  modifiedVal->setPaletteForegroundColor( white );
  modifiedVal->setPaletteBackgroundColor( darkBlue );
  albumTitle->setPaletteForegroundColor( white );
  albumTitle->setPaletteBackgroundColor( darkBlue );
  albumPreview->setPaletteBackgroundColor( darkBlue );
  closeButton->setEraseColor(darkBlue);
  //--
  //place widgets in grid
  grid = new QGridLayout( this, 10, 3, 0);
  grid->setMargin(WIDGET_SPACING);
  grid->setSpacing(WIDGET_SPACING);

  grid->addRowSpacing( 0, 10 );
  grid->setRowStretch( 0, 1 );

  //add statistics text
  grid->addMultiCellWidget( titleMessage, 1, 1, 0, 1, Qt::AlignCenter);

    //add space between "Album Statistics" text and actual statistics
  grid->addRowSpacing( 2, 10 );
  grid->setRowStretch( 2, 1 );

  grid->addWidget( numSubalbums, 3, 0 );
  grid->addWidget( numSubalbumsVal, 3, 1, Qt::AlignRight );
  grid->addWidget( numPhotos, 4, 0 );
  grid->addWidget( numPhotosVal, 4, 1, Qt::AlignRight );
  grid->addWidget( sizeOnDisk, 5, 0 );
  grid->addWidget( sizeOnDiskVal, 5, 1, Qt::AlignRight );
  grid->addWidget( created, 6,0 );
  grid->addWidget( createdVal, 6, 1, Qt::AlignRight );
  grid->addWidget( modified, 7,0 );
  grid->addWidget( modifiedVal, 7, 1, Qt::AlignRight );

  grid->setRowStretch( 8, 1 );


  //add album image and name
  grid2 = new QGridLayout( albumPreview, 2, 1, 0 );
  grid2->setSpacing(WIDGET_SPACING);

  grid2->addWidget( albumIcon, 0, 0, Qt::AlignCenter );
  grid2->addWidget( albumTitle, 1, 0, Qt::AlignCenter );
  grid->addMultiCellWidget( albumPreview, 0,8, 2, 2, Qt::AlignCenter );

  //add ok button
  grid->addMultiCellWidget( closeButton, 9,9, 0, 2, Qt::AlignCenter );
    //--
  //set window to not be resizeable
  show();
  setFixedSize(size());
  //-------------------------------
}
//==============================================
void AlbumStatistics::setCreationDate()
{

}
//==============================================
void AlbumStatistics::closeEvent( QCloseEvent* e)
{
  QWidget::closeEvent( e );
  emit closed();
}
//==============================================
void AlbumStatistics::reject()
{
  QDialog::reject();
  emit closed();
}
//==============================================
