/*****************************************************************************
*
* ALPS Project: Algorithms and Libraries for Physics Simulations
*
* ALPS Light Libraries
*
* Copyright (C) 1994-2003 by Matthias Troyer <troyer@itp.phys.ethz.ch>,
*                            Synge Todo <wistaria@comp-phys.org>
*
* This software is part of the "ALPS Light" Libraries, public-domain
* part of the ALPS Libraries. If you need the full functionality of
* the ALPS Libraries, such as Lattice, Model, Scheduler, etc, please
* use the full version of ALPS Libraries, which is available from
* http://alps.comp-phys.org/.
*
* Permission is hereby granted, free of charge, to any person or organization
* obtaining a copy of the software and accompanying documentation covered by
* this license (the "Software") to use, reproduce, display, distribute,
* execute, and transmit the Software, and to prepare derivative works of the
* Software, and to permit third-parties to whom the Software is furnished to
* do so, all subject to the following:
*
* The copyright notices in the Software and this entire statement, including
* the above license grant, this restriction and the following disclaimer,
* must be included in all copies of the Software, in whole or in part, and
* all derivative works of the Software, unless such copies or derivative
* works are solely in the form of machine-executable object code generated by
* a source language processor.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
* FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT 
* SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE 
* FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE, 
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
* DEALINGS IN THE SOFTWARE.
*
*****************************************************************************/

/* $Id: pseudo_des.h,v 1.5 2004/07/08 04:57:30 wistaria Exp $ */

#ifndef ALPS_RANDOM_PSEUDO_DES_HPP
#define ALPS_RANDOM_PSEUDO_DES_HPP

#include <alps/config.h>
#include <iostream>
#include <algorithm>  // std::swap
#include <boost/config.hpp>
#include <boost/limits.hpp>
#include <boost/integer_traits.hpp>

namespace alps {

class pseudo_des
{
public:
  typedef uint32_t result_type;
  
#ifndef BOOST_NO_INCLASS_MEMBER_INITIALIZATION
  static const bool has_fixed_range = true;
  static const result_type min_value = boost::integer_traits<result_type>::const_min;
  static const result_type max_value = boost::integer_traits<result_type>::const_max;
#else
  BOOST_STATIC_CONSTANT(bool, has_fixed_range = false);
#endif

  BOOST_STATIC_CONSTANT(uint32_t, default_seed = 4357);

  result_type min() const { return std::numeric_limits<result_type>::min(); }
  result_type max() const { return std::numeric_limits<result_type>::max(); }

  pseudo_des() : seed_(default_seed), state_(1) {}
  explicit pseudo_des(uint32_t s) : seed_(s), state_(1) {}

  // compiler-generated copy ctor and assignment operator are fine

  void seed(uint32_t s=default_seed) {
    seed_ = s;
    state_ = 1;
  }

  result_type operator()() {
    return hash(seed_, state_++);
  }

  void operator+=(uint32_t skip) { state_ += skip; }

  std::ostream& write(std::ostream& os) const {
    return os << seed_ << ' ' << state_;
  }
  std::istream& read(std::istream& is) {
    return is >> seed_ >> state_;
  }
  bool operator==(const pseudo_des& rhs) const {
    return seed_ == rhs.seed_ && state_ == rhs.state_;
  }

protected:
  uint32_t low_bits(uint32_t d) const {
    return d & 0xffff;
  }
  uint32_t high_bits(uint32_t d) const {
    return d >> 16;
  }
  uint32_t swap_bits(uint32_t d) const {
    return high_bits(d) | (low_bits(d) << 16);
  }

  uint32_t hash(uint32_t w0, uint32_t w1) const {
    const int num_iter = 4;
    const uint32_t g0[num_iter] = {
      0xbaa96887L, 0x1e17d32cL, 0x03bcdc3cL, 0x0f33d1b2L
    };
    const uint32_t g1[num_iter] = {
      0x4b0f3b58L, 0xe874f0c3L, 0x6955c5a6L, 0x55a7ca46L
    };
    for (int i = 0; i < num_iter; ++i) {
      std::swap(w0, w1);
      uint32_t mask = w0 ^ g0[i];
      uint32_t lbm = low_bits(mask);
      uint32_t hbm = high_bits(mask);
      w1 ^= (swap_bits(lbm * lbm + ~(hbm * hbm)) ^ g1[i]) + lbm * hbm;
    }
    return w1;
  }

private:
  uint32_t seed_;
  uint32_t state_;
};

} // end namespace alps

#endif // RANDOM_PSEUDO_DES_HPP
