/*
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *  GUS's memory access via proc filesystem
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "../../include/driver.h"
#include "../../include/gus.h"
#include "../../include/info.h"

struct proc_private {
	int rom;		/* data are in ROM */
	unsigned int address;
	unsigned int size;
	snd_gus_card_t * gus;
};

static long snd_gf1_mem_proc_dump(void *private_data, void *file_private_data,
			          struct file *file, char *buf, long count)
{
	long size;
	struct proc_private *priv = (struct proc_private *)private_data;
	snd_gus_card_t *gus = priv->gus;
	int err;

	size = count;
	if (file->f_pos + size > priv->size)
		size = (long)priv->size - file->f_pos;
	if (size > 0) {
		if ((err = snd_gus_dram_read(gus, buf, file->f_pos, size, priv->rom)) < 0)
			return err;
		file->f_pos += size;
		return size;
	}
	return 0;
}			

static long long snd_gf1_mem_proc_lseek(void *private_data,
					void *private_file_data,
					struct file *file,
					long long offset,
					int orig)
{
	struct proc_private *priv = (struct proc_private *)private_data;

	switch (orig) {
	case 0:	/* SEEK_SET */
		file->f_pos = offset;
		break;
	case 1:	/* SEEK_CUR */
		file->f_pos += offset;
		break;
	case 2: /* SEEK_END */
		file->f_pos = priv->size - offset;
		break;
	default:
		return -EINVAL;
	}
	if (file->f_pos > priv->size)
		file->f_pos = priv->size;
	return file->f_pos;
}

static void snd_gf1_mem_proc_free(void *private_data)
{
	snd_kfree(private_data);
}

int snd_gf1_mem_proc_init(snd_gus_card_t * gus)
{
	int idx;
	char name[16];
	struct proc_private *priv;
	snd_info_entry_t *entry;

	memset(&gus->gf1.rom_entries, 0, sizeof(gus->gf1.rom_entries));
	memset(&gus->gf1.ram_entries, 0, sizeof(gus->gf1.ram_entries));
	for (idx = 0; idx < 4; idx++) {
		if (gus->gf1.mem_alloc.banks_8[idx].size > 0) {
			priv = (struct proc_private *)snd_kcalloc(sizeof(*priv), GFP_KERNEL);
			if (priv == NULL) {
				snd_gf1_mem_proc_done(gus);
				return -ENOMEM;
			}
			priv->gus = gus;
			sprintf(name, "gus-ram-%i", idx);
			entry = snd_info_create_entry(gus->card, name);
			if (entry) {
				entry->type = SND_INFO_ENTRY_DATA;
				entry->private_data = priv;
				entry->private_free = snd_gf1_mem_proc_free;
				entry->t.data.read = snd_gf1_mem_proc_dump;
				entry->t.data.lseek = snd_gf1_mem_proc_lseek;
				priv->address = gus->gf1.mem_alloc.banks_8[idx].address;
				priv->size = entry->size = gus->gf1.mem_alloc.banks_8[idx].size;
				if (snd_info_register(entry) < 0) {
					snd_info_free_entry(entry);
					entry = NULL;
				}
			}
			gus->gf1.ram_entries[idx] = entry;
		}
	}
	for (idx = 0; idx < 4; idx++) {
		if (gus->gf1.rom_present & (1 << idx)) {
			priv = (struct proc_private *)snd_kcalloc(sizeof(*priv), GFP_KERNEL);
			if (priv == NULL) {
				snd_gf1_mem_proc_done(gus);
				return -ENOMEM;
			}
			priv->rom = 1;
			priv->gus = gus;
			sprintf(name, "gus-rom-%i", idx);
			entry = snd_info_create_entry(gus->card, name);
			if (entry) {
				entry->type = SND_INFO_ENTRY_DATA;
				entry->private_data = priv;
				entry->private_free = snd_gf1_mem_proc_free;
				entry->t.data.read = snd_gf1_mem_proc_dump;
				entry->t.data.lseek = snd_gf1_mem_proc_lseek;
				priv->address = idx * 4096 * 1024;
				priv->size = entry->size = gus->gf1.rom_memory;
				if (snd_info_register(entry) < 0) {
					snd_info_free_entry(entry);
					entry = NULL;
				}
			}
			gus->gf1.rom_entries[idx] = entry;
		}
	}
	return 0;
}

int snd_gf1_mem_proc_done(snd_gus_card_t * gus)
{
	int idx;

	for (idx = 0; idx < 4; idx++) {
		if (gus->gf1.ram_entries[idx])
			snd_info_unregister(gus->gf1.ram_entries[idx]);
	}
	for (idx = 0; idx < 4; idx++) {
		if (gus->gf1.rom_entries[idx])
			snd_info_unregister(gus->gf1.rom_entries[idx]);
	}
	return 0;
}
