/*
 *  Driver for ESS Solo-1 (ES1938, ES1946) soundcard
 *  Copyright (c) by Jaromir Koutek <miri@punknet.cz>,
 *                   Jaroslav Kysela <perex@suse.cz>,
 *                   Thomas Sailer <sailer@ife.ee.ethz.ch>,
 *                   Abramo Bagnara <abramo@alsa-project.org>,
 *                   Markus Gruber <gruber@eikon.tum.de>
 * 
 * Rewritten from sonicvibes.c source.
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define SND_MAIN_OBJECT_FILE
#include "../include/driver.h"
#include "../include/es1938.h"
#include "../include/opl3.h"
#include "../include/initval.h"

EXPORT_NO_SYMBOLS;
MODULE_DESCRIPTION("\
Driver: ESS Solo-1\n\
Card: Generic ES1938\n\
Card: TerraTec 128i PCI\n\
PCI: 0x125d=0x1969\n\
");
MODULE_LICENSE("GPL");

#ifndef PCI_VENDOR_ID_ESS
#define PCI_VENDOR_ID_ESS		0x125d
#endif
#ifndef PCI_DEVICE_ID_ESS_ES1938
#define PCI_DEVICE_ID_ESS_ES1938	0x1969
#endif

int snd_index[SND_CARDS] = SND_DEFAULT_IDX;	/* Index 0-MAX */
char *snd_id[SND_CARDS] = SND_DEFAULT_STR;	/* ID for this card */
MODULE_PARM(snd_index, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_index, "Index value for ESS Solo-1 soundcard.");
MODULE_PARM(snd_id, "1-" __MODULE_STRING(SND_CARDS) "s");
MODULE_PARM_DESC(snd_id, "ID string for ESS Solo-1 soundcard.");

struct snd_solo {
	struct pci_dev *pci;
	snd_irq_t *irqptr;
	snd_dma_t *dma1ptr;
	snd_dma_t *dma2ptr;
	snd_card_t *card;
	es1938_t *solo;
	snd_pcm_t *pcm;
	snd_kmixer_t *mixer;
#ifdef USE_MIDI
	snd_rawmidi_t *midi_uart;
#endif
	snd_hwdep_t *synth;
};

static struct snd_solo *snd_solo_cards[SND_CARDS] = SND_DEFAULT_PTR;

static void snd_solo_use_inc(snd_card_t * card)
{
	MOD_INC_USE_COUNT;
}

static void snd_solo_use_dec(snd_card_t * card)
{
	MOD_DEC_USE_COUNT;
}

static int __init snd_solo_detect(snd_card_t * card, struct snd_solo *scard)
{
	int idx;

	if ((scard->pci = pci_find_device(PCI_VENDOR_ID_ESS,
					  PCI_DEVICE_ID_ESS_ES1938,
					  scard->pci)) == NULL)
		return -ENODEV;
	for (idx = 0; idx < 5; idx++) {
		if (pci_resource_start(scard->pci, idx) == 0 ||
		    !(pci_resource_flags(scard->pci, idx) & IORESOURCE_IO))
			return -ENODEV;
	}
	if (snd_register_ioport(card, pci_resource_start(scard->pci, 1), 0x10, "ESS Solo-1 SB", NULL) < 0)
		goto __nodev;
	if (snd_register_ioport(card, pci_resource_start(scard->pci, 2), 0x10, "ESS Solo-1 VC (DMA)", NULL) < 0)
		goto __nodev;
	if (snd_register_ioport(card, pci_resource_start(scard->pci, 3), 4, "ESS Solo-1 MIDI", NULL) < 0)
		goto __nodev;
	if (snd_register_ioport(card, pci_resource_start(scard->pci, 4), 4, "ESS Solo-1 Game port", NULL) < 0)
		goto __nodev;
	return 0;
      __nodev:
	snd_unregister_ioports(card);
	return -ENODEV;
}

static void snd_solo_interrupt2(int irq, void *dev_id, struct pt_regs *regs)
{
	struct snd_solo *scard = (struct snd_solo *) dev_id;

	if (scard == NULL || scard->solo == NULL)
		return;
	snd_solo_interrupt(scard->solo);
}

#define SOLO_DMA_SIZE 64

static int __init snd_solo_resources(snd_card_t * card, struct snd_solo *scard, int dev)
{
	int err;

	if ((err = snd_register_interrupt(card, "ESS Solo-1", scard->pci->irq,
			    SND_IRQ_TYPE_PCI, snd_solo_interrupt2, scard,
					  NULL, &scard->irqptr)) < 0)
		return err;
	if ((err = snd_register_dma_channel(card, "ESS Solo-1 DMA 1", 0,
					    SND_DMA_TYPE_PCI_16MB,
			      SOLO_DMA_SIZE, NULL, &scard->dma1ptr)) < 0)
		return err;
	if ((err = snd_register_dma_channel(card, "ESS Solo-1 DMA 2", 1,
					    SND_DMA_TYPE_PCI,
			      SOLO_DMA_SIZE, NULL, &scard->dma2ptr)) < 0)
		return err;
	return 0;
}

static int __init snd_solo_probe(int dev, struct snd_solo *scard)
{
	snd_card_t *card;
	snd_pcm_t *pcm = NULL;
	snd_kmixer_t *mixer = NULL;
#ifdef USE_MIDI
	snd_rawmidi_t *midi_uart = NULL;
#endif
	snd_hwdep_t *synth = NULL;

	card = snd_card_new(snd_index[dev], snd_id[dev],
			    snd_solo_use_inc, snd_solo_use_dec);
	if (card == NULL)
		return -ENOMEM;
	card->type = SND_CARD_TYPE_ESS_ES1938;
	scard->pci = NULL;
	do {
		if (!snd_solo_detect(card, scard))
			break;
	} while (scard->pci);
	if (scard->pci == NULL) {
		snd_card_free(card);
		return -ENODEV;
	}
	if (snd_solo_resources(card, scard, dev) < 0) {
		snd_card_free(card);
		return -ENODEV;
	}
	if (snd_solo_create(card, scard->pci, scard->dma1ptr,
			    scard->dma2ptr, scard->irqptr, 0, 0,
			    &scard->solo) < 0)
		goto __nodev;
	if (snd_solo_new_pcm(scard->solo, 0, &pcm) < 0)
		goto __nodev;
	if (snd_solo_new_mixer(scard->solo, 0, pcm, &mixer) < 0)
		goto __nodev;

	strcpy(card->abbreviation, "Solo-1");
	strcpy(card->shortname, "ESS Solo-1");
	sprintf(card->longname, "%s rev %i, irq %li",
		card->shortname,
		scard->solo->revision,
		scard->irqptr->irq);

	if (!snd_card_register(card)) {
		scard->card = card;
		scard->mixer = mixer;
		scard->pcm = pcm;
#ifdef USE_MIDI
		scard->midi_uart = scard->solo->rmidi = midi_uart;
#endif
		scard->synth = synth;
		return 0;
	}
	goto __nodev;

      __nodev:
	snd_card_free(card);
	return -ENXIO;
}

#ifdef MODULE

static int __exit snd_solo_free2(int dev)
{
	struct snd_solo *scard;

	scard = snd_solo_cards[dev];
	snd_solo_cards[dev] = NULL;
	if (scard) {
		snd_card_unregister(scard->card);
		snd_kfree(scard);
	}
	return 0;
}

#endif

static int __init alsa_card_es1938_init(void)
{
	int dev, cards;
	struct snd_solo *scard;

	for (dev = cards = 0; dev < SND_CARDS; dev++) {
		scard = (struct snd_solo *) snd_kcalloc(sizeof(struct snd_solo), GFP_KERNEL);
		if (scard == NULL)
			continue;
		if (snd_solo_probe(dev, scard) < 0) {
			snd_kfree(scard);
			break;
		}
		snd_solo_cards[dev] = scard;
		cards++;
	}
	if (!cards) {
#ifdef MODULE
		snd_printk("ESS Solo-1 soundcard #%i not found or device busy\n", dev + 1);
#endif
		return -ENODEV;
	}
	return 0;
}

static void __exit alsa_card_es1938_exit(void)
{
	int dev;

	for (dev = 0; dev < SND_CARDS; dev++)
		snd_solo_free2(dev);
}

module_init(alsa_card_es1938_init)
module_exit(alsa_card_es1938_exit)
