// ========================================================================
// copyright (C) 1999-2003 by Tobias Erbsland <te@profzone.ch>
// ------------------------------------------------------------------------
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2 of the License, or (at your
// option) any later version.
// ========================================================================

#include "htmlreport.h"
#include <iostream>
#include <fstream>

htmlReport::htmlReport() :
		strPrefix( "html" ),
		strName( "HTML Report" ),
		strVersion( "V0.2" ),
		strAuthor( "Tobias Erbsland <te@profzone.ch>" ),
		strHelp( "Create output in plain html." ),
		boolTrOpen( false )  // sure false
{
	addParam( "tableborder", "Set the size of the Tableborder", "Size equal or greater 0.",
		Parameter::INTEGER, "0" );
	addParam( "disable-hr", "Disable the -hr- tag after every report", "This is a flag.",
		Parameter::FLAG, "" );
	addParam( "header-file", "Use a custom file as header for the output", "The full path to the file",
		Parameter::STRING, "" );
	addParam( "footer-file", "Use a custom file as footer for the output", "The full path to the file",
		Parameter::STRING, "" );
}

void htmlReport::startNewReport( const std::string &title, const reportType_e reportType )
{
	Report::startNewReport( title, reportType );
}

void htmlReport::printHeader( void )
{
	if ( col_list.size() == 0 ) return ; // Fehler.
	(*os) << "<!-- new report begin -->" << std::endl;

	if ( Type == TABLE )
	{
		(*os) << "<table border=\"" << getIntParam( "tableborder" ) << "\">\n"
			<< "\t<caption align=\"top\">" << Title << "</caption>\n"
			<< "\t<thead>\n"
			<< "\t\t<tr>\n";

		std::list< col_t >::iterator i;

		for ( i = col_list.begin(); i != col_list.end(); i++ )
		{
			// Clear sum
			(*i).floatSum = 0.0;
			(*i).longSum = 0;

			(*os) << "\t\t\t<th";
			switch ( (*i).Align )
			{
				case LEFT: (*os) << " align=\"left\""; break;
				case RIGHT: (*os) << " align=\"right\""; break;
			};
			(*os) << ">" << (*i).Title << "</th>\n";
		}
		(*os) << "\t\t</tr>\n"
			<< "\t</thead>\n";

		// Set pointer to first column.
		ci = col_list.begin();
		// Reset Linecounter.
		lc = 1;
		// Reset Tr Flag
		boolTrOpen = false;
	}

}

void htmlReport::printField( const std::string &field )
{
	if ( col_list.size() == 0 ) return ; // Fehler.
	checkMissingTrTag();
	checkAutoField();
	switch ( (*ci).Align )
	{
		default:
		case LEFT:
			(*os) << "\t\t\t<td align=\"left\">";
			break;
		case RIGHT:
			(*os) << "\t\t\t<td align=\"right\">";
			break;
	};
	switch ( (*ci).Style )
	{
		default:
		case NONE:
			break;
		case ROUND:
			(*os).put( '(' );
			break;
		case RECT:
			(*os).put( '[' );
			break;
		case QUOTE:
			(*os).put( '"' );
			break;
	};
	(*os) << field; //.substr(0,(*ci).Size - subwidth).c_str();
	if ( (*ci).Typ == PERCENT ) ( *os ).put( '%' );
	switch ( (*ci).Style )
	{
		default:
		case NONE:
			break;
		case ROUND:
			(*os).put( ')' );
			break;
		case RECT:
			(*os).put( ']' );
			break;
		case QUOTE:
			(*os).put( '"' );
			break;
	};
	(*os) << "</td>\n";
	nextField();
}

void htmlReport::printField( const long field )
{
	if ( col_list.size() == 0 ) return ; // Error
	checkMissingTrTag();
	checkAutoField();
	switch ( (*ci).Align )
	{
		default:
		case LEFT:
			(*os) << "\t\t\t<td align=\"left\">";
			break;
		case RIGHT:
			(*os) << "\t\t\t<td align=\"right\">";
			break;
	}
	switch ( ( *ci ).Style )
	{
		default:
		case NONE:
			break;
		case ROUND:
			(*os).put( '(' );
			break;
		case RECT:
			(*os).put( '[' );
			break;
		case QUOTE:
			(*os).put( '"' );
			break;
	};
	if ( (*ci).Typ == DATE )
	{
		(*os) << getDateStr( field );
	}
	else
	{
		(*os) << field;
	};
	if ( (*ci).Typ == PERCENT ) ( *os ).put( '%' );
	switch ( (*ci).Style )
	{
		default:
		case NONE:
			break;
		case ROUND:
			(*os).put( ')' );
			break;
		case RECT:
			(*os).put( ']' );
			break;
		case QUOTE:
			(*os).put( '"' );
			break;
	};
	// Is this a sum?
	if ( (*ci).Sum ) ( *ci ).longSum += field;
	(*os) << "</td>\n";
	nextField();
}

void htmlReport::printField( const double field )
{
	if ( col_list.size() == 0 ) return ; // Error.
	checkMissingTrTag();
	checkAutoField();
	(*os).precision( 1 ); // Nachkommastellen.
	(*os).setf( std::ios::fixed );
	switch ( ( *ci ).Align )
	{
		default:
		case LEFT:
			(*os) << "\t\t\t<td align=\"left\">";
			break;
		case RIGHT:
			(*os) << "\t\t\t<td align=\"right\">";
			break;
	};
	switch ( (*ci).Style )
	{
		default:
		case NONE:
			break;
		case ROUND:
			(*os).put( '(' );
			break;
		case RECT:
			(*os).put( '[' );
			break;
		case QUOTE:
			(*os).put( '"' );
			break;
	};
	(*os) << field;
	if ( (*ci).Typ == PERCENT ) ( *os ).put( '%' );
	switch ( ( *ci ).Style )
	{
		default:
		case NONE:
			break;
		case ROUND:
			(*os).put( ')' );
			break;
		case RECT:
			(*os).put( ']' );
			break;
		case QUOTE:
			(*os).put( '"' );
			break;
	};
	// Testen, ob Summe
	if ( (*ci).Sum ) ( *ci ).floatSum += field;
	(*os) << "</td>\n";
	nextField();
}

void htmlReport::skipField( const bool realline )
{
	if ( col_list.size() == 0 ) return ; // Fehler.
	if ( realline ) checkAutoField();
	checkMissingTrTag();
	(*os) << "\t\t\t<td></td>\n";
	if ( nextField() ) lc--; // Line wieder zuruecksetzen.
}

void htmlReport::printFooter( void )
{
	bool sf = false; // Summe bilden?

	if ( Type == TABLE )
	{
		if ( col_list.size() == 0 ) return ; // Fehler.
		for ( ci = col_list.begin(); ci != col_list.end(); ci++ )
		{
			if ( (*ci).Sum )
			{
				sf = true;
			};
		}
		if ( sf )
		{
			(*os) << "\t<tfoot>\n";
			ci = col_list.begin();
			do
			{
				if ( ( *ci ).Sum )
				{
					switch ( ( *ci ).Typ )
					{
						case FLOAT:
							printField( ( *ci ).floatSum );
							break;
						case INTEGER:
							printField( ( *ci ).longSum );
							break;
						default:
							skipField( false );
					}
				}
				else
				{
					skipField( false );
				}
			}
			while ( ci != col_list.begin() );
			(*os) << "\t</tfoot>\n";
		}
		(*os) << "</table>\n";
	}
	if ( !getFlagParam( "disable-hr" ) )
	{
		// xhtml hr tag
		(*os) << "<hr />\n";
	}
}

/** Erzeugt ggf. alle Automatischen Felder. */
void htmlReport::checkAutoField( void )
{
	while ( true )
	{
		if ( ( *ci ).Typ == COUNTER )
		{
			checkMissingTrTag();
			(*os) << "\t\t\t<td align=right>" << lc << "</td>\n";
			if ( nextField() )
			{
				return ;
			}
		}
		else
		{
			return ;
		}
	}
}

/** Springt zum nchsten feld und mach ggf. einen Zeilenumbruch. */
bool htmlReport::nextField( void )
{
	if ( ++ci == col_list.end() )
	{
		(*os) << "\t\t</tr>\n";
		boolTrOpen = false; // Reset Tr Flag
		ci = col_list.begin();
		lc++; // Linecounter.
		return true;
	}
	else
	{
		return false;
	}
}

/** Check if a tr tag necessary to open the current line. */
void htmlReport::checkMissingTrTag( void )
{
	if ( !boolTrOpen )
	{
		(*os) << "\t\t<tr>\n";
		boolTrOpen = true;
	}
}

void htmlReport::printPageHeader( void )
{
	if ( !os ) return ; // Make sure only print on a stream.

	std::string header = getStringParam( "header-file" );
	if( header != "" )
	{
		std::ifstream in( header.c_str() );
		if( in )
		{
			// Copy the file to the outstream
			(*os) << in.rdbuf();
		}
		else
		{
			std::cerr << "Error: can't open \"" << header << "\"." << std::endl;
		}
	}
	else
	{
		// Print the default header
		(*os) << "<html>\n"
			<< "<head>\n"
			<< "\t<title>Anteater V" VERSION " - Report</title>\n"
			<< "</head>\n"
			<< "<body>\n";
	}
}

void htmlReport::printPageFooter( void )
{
	if ( !os ) return ; // Make sure only print on a stream.

	(*os) << "<p><small>generated with <a href=\"http://anteater.sourceforge.net/\">"
	"Anteater V" VERSION "</a></small></p>\n";

	std::string footer = getStringParam( "footer-file" );
	if( footer != "" )
	{
		std::ifstream in( footer.c_str() );
		if( in )
		{
			// Copy the file to the outstream
			(*os) << in.rdbuf();
		}
		else
		{
			std::cerr << "Error: can't open \"" << footer << "\"." << std::endl;
		}
	}
	else
	{
		// Print the default footer
		(*os) << "</body>\n"
			<< "</html>" << std::endl;
	}
}
