/* Copyright (C) 2002, 2003, 2004 Jan Wedekind.
   This file is part of the recipe database application AnyMeal.

   AnyMeal is free software; you can redistribute it and/or modify it under
   the terms of the GNU GENERAL PUBLIC LICENSE as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   AnyMeal is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTIBILITY or FITNESS
   FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
   details.

   You should have received a copy of the GNU General Public License along
   with AnyMeal; if not, contact one of the authors of this software. */
#include <cassert>
#include <sqlext.h>
#include "odbcDatabase.hpp"
#include "odbcStatement.hpp"
#include "odbcWrap.hpp"

using namespace std;

ODBCDatabase::ODBCDatabase( const std::string &databaseName,
                            const std::string &userName,
                            const std::string &passWord ) throw (Error):
  Database( userName ), odbcEnvironment( NULL ), connection( NULL )
{
  
  bool connected = false;

  try {
    // 1. allocate Environment handle and register version 
    odbcWrap( SQLAllocHandle( SQL_HANDLE_ENV, SQL_NULL_HANDLE,
                              &odbcEnvironment ), SQL_NULL_HANDLE, NULL );
    
    odbcWrap( SQLSetEnvAttr( odbcEnvironment, SQL_ATTR_ODBC_VERSION,
                             (void*)SQL_OV_ODBC3, 0 ),
              SQL_HANDLE_ENV, odbcEnvironment );

    odbcWrap( SQLAllocHandle( SQL_HANDLE_DBC, odbcEnvironment, &connection ),
              SQL_HANDLE_ENV, odbcEnvironment );

    // 2. allocate connection handle, set timeout
    odbcWrap( SQLSetConnectAttr( connection, SQL_LOGIN_TIMEOUT,
                                 (SQLPOINTER *)5, 0 ),
              SQL_HANDLE_DBC, connection );

    // 3. Connect to the datasource "web" 
    odbcWrap( SQLConnect( connection, (SQLCHAR*)databaseName.c_str(), SQL_NTS,
                          (SQLCHAR*)userName.c_str(), SQL_NTS,
                          (SQLCHAR*)passWord.c_str(), SQL_NTS ),
              SQL_HANDLE_DBC, connection );

    connected = true;

  } catch ( Error &e ) {

    if ( connected ) {
      assert( connection );
      SQLDisconnect( connection );
    };
    if ( connection )
      SQLFreeHandle( SQL_HANDLE_DBC, connection );
    if ( odbcEnvironment )
      SQLFreeHandle( SQL_HANDLE_ENV, odbcEnvironment );
    throw e;

  };

};

ODBCDatabase::~ODBCDatabase(void)
{
  assert( connection );
  SQLDisconnect( connection );
  SQLFreeHandle( SQL_HANDLE_DBC, connection );
  assert( odbcEnvironment );
  SQLFreeHandle( SQL_HANDLE_ENV, odbcEnvironment );
}

StatementPtr ODBCDatabase::execQuery( const std::string &query ) throw (Error)
{
  StatementPtr retVal( new ODBCStatement( connection, query ) );
  return retVal;
}
