/*
 *  apcaction.c -- Actions taken when something is happen to UPS.
 *
 *  apcupsd.c	-- Simple Daemon to catch power failure signals from a
 *		   BackUPS, BackUPS Pro, or SmartUPS (from APCC).
 *		-- Now SmartMode support for SmartUPS and BackUPS Pro.
 *
 *  Copyright (C) 1996-99 Andre M. Hedrick <andre@suse.com>
 *  Copyright (C) 1999-2000 Riccardo Facchetti <riccardo@master.oasi.gpa.it>
 *  All rights reserved.
 *
 */

/*
 *		       GNU GENERAL PUBLIC LICENSE
 *			  Version 2, June 1991
 *
 *  Copyright (C) 1989, 1991 Free Software Foundation, Inc.
 *			     675 Mass Ave, Cambridge, MA 02139, USA
 *  Everyone is permitted to copy and distribute verbatim copies
 *  of this license document, but changing it is not allowed.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 *  IN NO EVENT SHALL ANY AND ALL PERSONS INVOLVED IN THE DEVELOPMENT OF THIS
 *  PACKAGE, NOW REFERRED TO AS "APCUPSD-Team" BE LIABLE TO ANY PARTY FOR
 *  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING
 *  OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF ANY OR ALL
 *  OF THE "APCUPSD-Team" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  THE "APCUPSD-Team" SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 *  ON AN "AS IS" BASIS, AND THE "APCUPSD-Team" HAS NO OBLIGATION TO PROVIDE
 *  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 *  THE "APCUPSD-Team" HAS ABSOLUTELY NO CONNECTION WITH THE COMPANY
 *  AMERICAN POWER CONVERSION, "APCC".  THE "APCUPSD-Team" DID NOT AND
 *  HAS NOT SIGNED ANY NON-DISCLOSURE AGREEMENTS WITH "APCC".  ANY AND ALL
 *  OF THE LOOK-A-LIKE ( UPSlink(tm) Language ) WAS DERIVED FROM THE
 *  SOURCES LISTED BELOW.
 *
 */

#include "apc.h"

extern int kill_on_powerfail;
static void do_shutdown(UPSINFO *ups, int cmdtype);

/*
 * These are the commands understood by the apccontrol shell script.
 * You _must_ keep the the commands[] array in sync with the defines in
 * include/apc_defines.h
 */

UPSCOMMANDS cmd[] = {
    {"powerout",      0},
    {"onbattery",     0},
    {"failing",       0},
    {"timeout",       0},
    {"loadlimit",     0},
    {"runlimit",      0},
    {"doreboot",      0},
    {"doshutdown",    0},
    {"mainsback",     0},
    {"annoyme",       0},
    {"emergency",     0},
    {"changeme",      0},
    {"remotedown",    0},
    {"restartme",     0},
    {"commfailure",   0},
    {"commok",        0},
    {"startselftest", 0},
    {"endselftest",   0}
};

/*
 * These messages must be kept in sync with the above array
 * and the defines in include/apc_defines.h 
 */
UPSCMDMSG cmd_msg[] = {
    {LOG_CRIT,    "Power failure."},
    {LOG_CRIT,    "Running on UPS batteries."},
    {LOG_ALERT,   "Battery power exhausted."},
    {LOG_ALERT,   "Reached run time limit on batteries."},
    {LOG_ALERT,   "Battery charge below low limit."},
    {LOG_ALERT,   "Reached remaining time percentage limit on batteries."},
    {LOG_ALERT,   "Failed to kill the power! Attempting a REBOOT!"}, 
    {LOG_ALERT,   "Initiating system shutdown!"},
    {LOG_ALERT,   "Power is back. UPS running on mains."},
    {LOG_ALERT,   "Users requested to logoff."},
    {LOG_ALERT,   "Battery failure. Emergency."},
    {LOG_CRIT,    "UPS battery must be replaced."},
    {LOG_CRIT,    "Remote shutdown requested"},
    {LOG_CRIT,    "Too Many Errors. Restarting UPS daemon"},
    {LOG_WARNING, "Serial communications with UPS lost."},
    {LOG_WARNING, "Serial communications with UPS restored."},
    {LOG_ALERT,   "UPS Self Test switch to battery."},
    {LOG_ALERT,   "UPS Self Test completed."}
};

void generate_event(UPSINFO *ups, int event)
{
    /* Log message and execute script for this event */
    log_event(ups, cmd_msg[event].level, cmd_msg[event].msg);
    execute_command(ups, cmd[event]);

    /*
     * Additional possible actions. For certain, we now do a
     * shutdown   
     */
    switch (event) {
    /*
     * For the following, in addition to the basic,
     * message logged and executed above, we do a 
     * system shutdown.
     */
    case CMDFAILING:
    case CMDTIMEOUT:
    case CMDRUNLIMIT:
    case CMDLOADLIMIT:
    case CMDEMERGENCY:
    case CMDREMOTEDOWN:
       log_event(ups, cmd_msg[CMDDOSHUTDOWN].level, cmd_msg[CMDDOSHUTDOWN].msg);
       do_shutdown(ups, CMDDOSHUTDOWN);
       break;

    case CMDDOREBOOT:
       /* This should be deprecated. */
       do_shutdown(ups, event);
       break;

    /* For the following, everything is already done. */
    case CMDSTARTSELFTEST:
    case CMDENDSELFTEST:
    case CMDCOMMFAILURE:
    case CMDCOMMOK:
    case CMDRESTARTME:
    case CMDCHANGEME:
    case CMDANNOYME:
    case CMDMAINSBACK:
    case CMDDOSHUTDOWN:               /* Already shutdown, don't recall */
    case CMDPOWEROUT:
    case CMDONBATTERY:
    default:
	break;

   }
}

/*********************************************************************/
void powerfail (int ok)
{
    /*	If apcupsd terminates here, it will never get a chance to
     *	report the event of returning mains-power.
     *	I think apcupsd has no need to force terminate() by itself.
     *	It will receive a SIGTERM from init, when system goes down.
     *  This signal is trapped and will trigger apcupsd's terminate()
     *	function.
     *	(Was already reported/fixed in my 2.7.1 diff,)
     *	(so I do it again now) w.p.
     *
     *	     if (ok == 2) terminate(0);
     *
     *	Closes procfile and logfile to preserve information.
     *
     *	 ok = 1  => power is back
     *	 ok = 2  => power failure
     *	 ok = 3  => remote shutdown
     */

    if (ok == 2) {
	clear_files();
	if (terminate_on_powerfail)
	    terminate(0);
    }

    /*
     *	The network slaves apcupsd needs to terminate here for now.
     *	This sloppy, but it works. If you are networked, then the
     *	master must fall also. This is required so that the UPS
     *	can reboot the slaves.
     */

    if (ok == 3)
	terminate(0);
}
	
/********************************************************************* 
 * if called with zero, prevent users from logging in. 
 * if called with 1   , allow users to login.
 */
void logonfail (int ok)
{
    int lgnfd;

    unlink(NOLOGIN);
    if (ok == 0 && ((lgnfd = open(NOLOGIN, O_CREAT|O_WRONLY, 0644)) >= 0)) {
        write(lgnfd, POWERFAIL "\n", sizeof(POWERFAIL));
	close(lgnfd);
    }
}

static void prohibit_logins(UPSINFO *ups)
{
    if (ups->nologin_file)
	return; 		      /* already done */
    make_file(ups, NOLOGIN);
    ups->nologin_file = TRUE;
    logonfail(0);
    log_event(ups, LOG_ALERT, _("User logins prohibited"));
}

static void do_shutdown(UPSINFO *ups, int cmdtype)
{
    if (ups->ShutDown) {
	return; 		      /* already done */
    }
    ups->ShutDown = time(NULL);
    delete_lockfile(ups);
    ups->FastPoll = TRUE;	      /* speed up polling */
    make_file(ups, PWRFAIL);
    prohibit_logins(ups);

    if (ups->fd != -1) {
	/*
	 * Note, try avoid using this option if at all possible
	 * as it will shutoff the UPS power, and you cannot
	 * be guaranteed that the shutdown command will have
	 * succeeded. This PROBABLY should be executed AFTER
	 * the shutdown command is given (the execute_command below).
	 */
	if (kill_on_powerfail) {
	    kill_power(ups);
	}
    } 

    /* Now execute the shutdown command */
    execute_command(ups, cmd[cmdtype]);

    /* On some systems we may stop on the previous
     * line if a SIGTERM signal is sent to us.	
     */

    if (cmdtype == CMDREMOTEDOWN) 
	powerfail(3);
    else
	powerfail(2);
}

/*
 * These are the different "states" that the UPS can be
 * in.
 */
enum a_state {
    st_PowerFailure,
    st_SelfTest,
    st_OnBattery,
    st_MainsBack,
    st_OnMains 
};

/*
 * Figure out what "state" the UPS is in and
 * return it for use in do_action()
 */
static enum a_state get_state(UPSINFO *ups, time_t now)
{
    enum a_state state;

    if (ups->OnBatt) {
	if (ups->OldOnBatt) {	      /* if already detected on battery */
	   if (ups->SelfTest) {       /* see if UPS is doing self test */
	      state = st_SelfTest;    /*   yes */
	   } else {
	      state = st_OnBattery;   /* No, this must be real power failure */
	   }
	} else {
	   state = st_PowerFailure;   /* Power failure just detected */
	}
    } else {
	if (ups->OldOnBatt) {	      /* if we were on batteries */
	    state = st_MainsBack;     /* then we just got power back */
	} else {
	    state = st_OnMains;       /* Solid on mains, normal condition */
	}
    }
    return state;
}

/*
 * Get last Self Test result and edit into string
 *  This routine assumes the UPS is connected !
 */
static char *str_self_test_results(UPSINFO *ups)
{
    /* Get results of last self test */
    ups->X[0] = 0;
    if (ups->UPS_Cap[CI_ST_STAT]) {
	strncpy(ups->X,apc_chat(ups->UPS_Cmd[CI_ST_STAT], ups), 
		sizeof(ups->X));
    }
    /*
     * Responses are:
     * "OK" - good battery, 
     * "BT" - failed due to insufficient capacity, 
     * "NG" - failed due to overload, 
     * "NO" - no results available (no test performed in last 5 minutes) 
     */
    if (ups->X[0] == 'O' && ups->X[1] == 'K') {
       return "Battery OK";
    }	   
    if (ups->X[0] == 'B' && ups->X[1] == 'T') {
       return "Test failed -- insufficient battery capacity";
    }	   
    if (ups->X[0] == 'N' && ups->X[1] == 'G') {
       return "Test failed -- battery overloaded";
    }	   
    return "No test results available";
}

/*********************************************************************/
void do_action(UPSINFO *ups, int ups_connected)
{
    time_t now;
    static int requested_logoff = 0; /* asked user to logoff */
    static int first = 1;
    enum a_state state;

    time(&now); 		  /* get current time */
    if (first) {
	ups->last_time_nologon = ups->last_time_annoy = now;
	ups->last_time_on_line = now;
	ups->OldOnBatt = 0;
	ups->OldBattLow = 0;
	first = 0;
    }

    if (read_andlock_shmarea(ups)) {
	/*
	 * If failed to acquire shm lock, return and try again later.
	 */
        log_event(ups, LOG_CRIT, _("Failed to acquire shm lock in do_action."));
	return;
    }

    if (ups->ChangeBatt) {		      /* Replace battery */
	/* Complain every 9 hours, this causes the complaint to
	 * cycle around the clock and hopefully be more noticable
	 * without being too annoying.	Also, ignore all change battery
	 * indications for the first 10 minutes of running time to
	 * prevent false alerts.
	 * Finally, issue the event 5 times, then clear the counter
	 * to silence false alarms. If the battery is really dead, the
	 * counter will be reset in apcsmart.c
	 */
#if AVERSION==4
	if (now - gcfg.start_time < 60 * 10 || ups->ChangeBatt > 5) {
#else
	if (now - ups->start_time < 60 * 10 || ups->ChangeBatt > 5) {
#endif
	    ups->ChangeBatt = 0;
	} else if (now - ups->last_time_changeme > 60 * 60 * 9) {
	    generate_event(ups, CMDCHANGEME);
	    ups->last_time_changeme = now;
	    ups->ChangeBatt++;
	}
    }

    if (ups->RestartDaemon) {
	generate_event(ups, CMDRESTARTME);
    }

    /*
     *	Must SHUTDOWN Remote System Calls
     */
    if (ups->remotedown) {
	ups->BatteryUp = 0;
	generate_event(ups, CMDREMOTEDOWN);
	return;
    }

    state = get_state(ups, now);
    switch (state) {
    case st_OnMains:
	/*
	 * If power is good, update the timers.
	 */
	ups->last_time_nologon = ups->last_time_annoy = now;
	ups->last_time_on_line = now;
	ups->FastPoll = FALSE;					       
	break;

    case st_PowerFailure:
       /*
	*  This is our first indication of a power problem
	*/
	ups->FastPoll = TRUE;	      /* speed up polling */

	/* See if this is a self test rather than power failure */
	if (ups_connected && ups->UPS_Cap[CI_WHY_BATT]) {
	    ups->G[0] = 0;
	    strncpy(ups->G, apc_chat(ups->UPS_Cmd[CI_WHY_BATT], ups), 
		    sizeof(ups->G));
            if (ups->G[0] == 'S') {
		ups->SelfTest = now;  /* set Self Test start time */
	    }
	}

	if (ups->SelfTest) {
	    generate_event(ups, CMDSTARTSELFTEST);
	} else {
	    generate_event(ups, CMDPOWEROUT);
	}
	ups->last_time_nologon = ups->last_time_annoy = now;
	ups->last_time_on_line = now;
	ups->last_onbatt_time = now;
	ups->num_xfers++;

	/*
	 *  Now enable the DTR for the CUSTOM_SIMPLE cable
	 *  Note: this enables the the CTS bit, which allows
	 *	   us to detect the BattLow condition!!!!
	 */
	if (ups->cable.type == CUSTOM_SIMPLE)
	    (void)ioctl(ups->fd, TIOCMBIS, &dtr_bit);
	break;

    case st_SelfTest:
       /* allow 12 seconds max for selftest */
       if (now - ups->SelfTest < 12 && !ups->BattLow)
	   break;
       /* Cancel self test, announce power failure */
       ups->SelfTest = 0;
       /* FALL-THRU to st_OnBattery */
    case st_OnBattery:
	/*
	 *  Did the second test verify the power is failing?
	 */
	if (!ups->BatteryUp) {
	    ups->BatteryUp = 1;   /* it is confirmed, we are on batteries */
	    generate_event(ups, CMDONBATTERY);
	    ups->last_time_nologon = ups->last_time_annoy = now;
	    ups->last_time_on_line = now;
	    break;
	} 

	if (ups->ShutDown) {	      /* shutdown requested but still running */
	   if (ups->killdelay && now - ups->ShutDown >= ups->killdelay) {
	       if (ups_connected)
		   kill_power(ups);
	       ups->ShutDown = now;   /* wait a bit before doing again */
	   }
	} else {		      /* not shutdown yet */
	    /*
	     *	Did BattLow bit go high? Then the battery power is failing.
	     *	Normal Power down during Power Failure
	     */
	    if (!ups->OldBattLow && ups->BattLow) {
	       ups->BatteryUp = 0;
	       generate_event(ups, CMDFAILING);
	       break;
	    }

	    /*
	     *	Did MaxTimeOnBattery Expire?  (TIMEOUT in apcupsd.conf)
	     *	Normal Power down during Power Failure
	     */
	    if ((ups->maxtime > 0) && 
		((now - ups->last_time_on_line) > ups->maxtime)) {
		ups->timedout = TRUE;
		generate_event(ups, CMDTIMEOUT);
		break;
	    }
	    /*
	     *	Did Battery Charge or Runtime go below percent cutoff?
	     *	Normal Power down during Power Failure
	     */
	    if ((ups->mode.type >= NBKPRO)) {
		if (ups->UPS_Cap[CI_BATTLEV] && ups->BattChg <= ups->percent) {
		    ups->load = TRUE;
		    generate_event(ups, CMDLOADLIMIT);
		    break;
		} else if (ups->UPS_Cap[CI_RUNTIM] && ups->TimeLeft <= ups->runtime) {
		    ups->timelout = TRUE;
		    generate_event(ups, CMDRUNLIMIT);
		    break;
		}
	    }

	    /*
	     *	We are on batteries, the battery is low, and the power is not
	     *	down ==> the battery is dead.  KES Sept 2000
	     *
	     *	Then the battery has failed!!!
	     *	Must do Emergency Shutdown NOW
	     *
	     */
	    if (ups->BattLow && !ups->LineDown) {
		ups->BatteryUp = 0;
		ups->emergencydown = TRUE;
		generate_event(ups, CMDEMERGENCY);
	    }

	    /*
	     *	Announce to LogOff, with initial delay ....
	     */
	    if (((now - ups->last_time_on_line) > ups->annoydelay) &&
		((now - ups->last_time_annoy) > ups->annoy) &&
		  ups->nologin_file) {
		    if (!requested_logoff) {
			/* generate log message once */
			generate_event(ups, CMDANNOYME);
		    } else {
			/* but execute script every time */
			execute_command(ups, cmd[CMDANNOYME]);
		    }
		    time(&ups->last_time_annoy);
		    requested_logoff = TRUE;
	    }
	    /*
	     *	Delay NoLogons....
	     */
	    if (!ups->nologin_file) {
		switch(ups->nologin.type) {
		case NEVER:
		    break;
		case TIMEOUT:
		    if ((now - ups->last_time_nologon) > ups->nologin_time) {
			prohibit_logins(ups);
		    }
		    break;
		case PERCENT:
		    if (ups->UPS_Cap[CI_BATTLEV] && ups->nologin_time >= ups->BattChg) {
			prohibit_logins(ups);
		    }
		    break;
		case MINUTES:
		    if (ups->UPS_Cap[CI_RUNTIM] && ups->nologin_time >= ups->TimeLeft) {
			prohibit_logins(ups);
		    }
		    break;
		case ALWAYS:
		default:
		    prohibit_logins(ups);
		    break;
		}
	    }
	}      
	break;

    case st_MainsBack:
	/*
	 *  The the power is back after a power failure or a self test	 
	 */
	ups->BatteryUp = 0;
	if (ups->ShutDown) {
	    /*
	     * If we have a shutdown to cancel, do it now.
	     */
	    ups->ShutDown = 0;
	    powerfail(1);
	    unlink(PWRFAIL);
            log_event(ups, LOG_ALERT, _("Cancelling shutdown"));
	}

	if (ups->SelfTest) {
	    ups->LastSelfTest = ups->SelfTest;
	    ups->SelfTest = 0;
            log_event(ups, LOG_ALERT, _("UPS Self Test completed: %s"),
		str_self_test_results(ups));
	    execute_command(ups, cmd[CMDENDSELFTEST]);
	} else {
	    generate_event(ups, CMDMAINSBACK);
	}

	if (ups->nologin_file) 
            log_event(ups, LOG_ALERT, _("Allowing logins"));
	logonfail(1);
	ups->nologin_file = FALSE;
	requested_logoff = FALSE;
	if (ups->cable.type == CUSTOM_SIMPLE) {
	    /*
	     *	Clearing DTR and TxD (ST).
	     */
	    (void)ioctl(ups->fd, TIOCMBIC, &dtr_bit);
	    (void)ioctl(ups->fd, TIOCMBIC, &st_bit);
	} 
	ups->last_offbatt_time = now;
	/* Sanity check. Sometimes only first power problem trips    
	 * thus last_onbatt_time is not set when we get here */
	if (ups->last_onbatt_time <= 0)
	   ups->last_onbatt_time = ups->last_offbatt_time;
	ups->cum_time_on_batt += (ups->last_offbatt_time - ups->last_onbatt_time);
	break;

    default:
	break;
    }

    /* Do a non-blocking wait on any exec()ed children */
    if (ups->num_execed_children > 0) {
	while (waitpid(-1, NULL, WNOHANG) > 0) {
	    ups->num_execed_children--;
	}
    }

    /*
     *	Remember status
     */

    ups->OldOnBatt = ups->OnBatt;
    ups->OldBattLow = ups->BattLow;

    write_andunlock_shmarea(ups);
}
