/*
 * upsfetch - utility program to retrieve data from apcupsd.status
 *
 * Original Author: Russell Kroll <rkroll@exploits.org>
 *
 * This source creates a handy object file that can be linked into other
 * programs for easy retrieval of common UPS parameters from the apcupsd
 * status file.
 *
 * Modified: Jonathan Benson <jbenson@technologist.com>
 *	   19/6/98 to suit apcupsd
 *	   23/6/98 added more graphs and menu options
 *
 * Modified: Kern Sibbald <kern@sibbald.com>
 *	   2 Nov 99 to work with apcupsd named pipes
 *	   5 Nov 99 added more graphs	      
 *	  11 Nov 99 to work with apcnetd networking
 *		    also modified it to be a bit closer 
 *		    to the original version.
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

#include "apcnetlib.h"

int fill_buffer(int sockfd);


#define SERV_TCP_PORT 7000
#define STATFILE "/etc/apcupsd/apcupsd.status"

char statbuf[4096];
int  statlen = 0;
char last_host[256];
char connect_host[256];

/* List of variables that can be read by getupsvar()   
 * First field is that name given to getupsvar(),
 * Second field is our internal name as produced by the STATUS 
 *   output from apcupsd.
 * Third field, if 0 returns everything to the end of the
 *    line, and if 1 returns only to first space (e.g. integers,
 *    and floating point values.
 */
static struct { 	
   char *request;
   char *upskeyword;
   int nfields;
} cmdtrans[] = {
   {"MODEL",      "MODEL",    0},
   {"UPSMODEL",   "UPSMODEL", 0},
   {"DATE",       "DATE",     0},
   {"BATTCAP",    "BCHARGE",  1},
   {"MBATTCHG",   "MBATTCHG", 1},
   {"BATTVOLT",   "BATTV",    1},
   {"NOMBATTV",   "NOMBATTV", 1},
   {"UTILITY",    "LINEV",    1},
   {"UPSLOAD",    "LOADPCT",  1},
   {"LOADPCT",    "LOADPCT",  1},
   {"OUTPUTV",    "OUTPUTV",  1},
   {"STATUS",     "STATUS",   0},
   {"STATFLAG",   "STATFLAG", 1},
   {"LINEMIN",    "MINLINEV", 1},
   {"LINEMAX",    "MAXLINEV", 1},
   {"UPSTEMP",    "ITEMP",    1},
   {"OUTPUTFREQ", "LINEFREQ", 1},
   {"TRANSLO",    "LOTRANS",  1},
   {"TRANSHI",    "HITRANS",  1},
   {"RUNTIME",    "TIMELEFT", 1},
   {"MINTIMEL",   "MINTIMEL", 1},
   {"RETPCT",     "RETPCT",   1},          /* min batt to turn on UPS */
   {"SENSE",      "SENSE",    1},
   {"HOSTNAME",   "HOSTNAME", 1},
   {"BATTDATE",   "BATTDATE", 1},
   {"SERIALNO",   "SERIALNO", 1},
   {"LASTXFER",   "LASTXFER", 0},          /* reason for last xfer to batteries */
   {"SELFTEST",   "SELFTEST", 1},          /* results of last self test */
   {"RELEASE",    "RELEASE",  1},
   {"UPSNAME",    "UPSNAME",  1},
   {"LOWBATT",    "DLOWBATT", 1},          /* low battery power off delay */
   {"BATTPCT",    "BCHARGE",  1},
   {"HIGHXFER",   "HITRANS",  1},
   {"LOWXFER",    "LOTRANS",  1},
   {"CABLE",      "CABLE",    0},
   {"FIRMWARE",   "FIRMWARE", 0},
   {NULL, NULL}
};

void closeupsfd(int fd)
{  
   return;
}

char *upsstrerror(char *file, int line, int err) 
{
   static char buf[1000];
   char *rtnbuf;

   sprintf(buf, "Unknown error in %s:%d\n", file, line);
   rtnbuf = malloc(strlen(buf) + 1);
   strcpy(rtnbuf, buf);
   return rtnbuf;
}

int fetch_data(char *host); /* forward reference */


int upsconnect(char *host)
{
   strcpy(connect_host, host);
   return fetch_data(host);
}

int getupsvarlist(char *host, char *buf, int buflen) 
{
   int len;
   int i;

   *buf = 0;
   for (i=0; cmdtrans[i].request; i++) {
      strcat(buf, cmdtrans[i].request);
      strcat(buf, " ");
   }
   len = strlen(buf);
   buf[len-1] = 0;
   return 1;
}

int getupsvarlistfd(int fd, char *upsname, char *buf, int buflen)
{
   return getupsvarlist("xxx", buf, buflen);
}


/*
 * Read data into memory buffer to be used by getupsvar()
 * Returns 0 on error
 * Returns 1 if data fetched
 */
int fetch_data(char *host)
{
   int sockfd;
   int stat;

   if (statlen != 0 && (strcmp(last_host, host) == 0))
       return 1;		      /* alread have data this host */
   strncpy(last_host, host, sizeof(last_host)); 
   last_host[sizeof(last_host) - 1] = 0;
   statlen = 0;
   if ((sockfd = net_open(host, NULL, SERV_TCP_PORT)) < 0) {
      net_errmsg = "upsfetch: tcp_open failed";
      return 0;
   }

   stat = fill_buffer(sockfd);		     /* fill statbuf */
   if (stat == 0) {
      *last_host = 0;
      statlen = 0;
   }
   net_close(sockfd);
   return stat;

#ifdef READ_FROM_FILE /* old way of doing it, deprecated */
       static int readfd = -1;

       if (readfd < 0) {
	 if ((readfd = open(STATFILE, O_RDONLY)) < 0) {
            sprintf (net_errbuf, "ERROR: Could not open status file %s\n", STATFILE);
	    net_errmsg = net_errbuf;
	    return 0;
	 }
	 if ((statlen = read(readfd, statbuf, sizeof(statbuf)-1)) < 100) {
            sprintf (net_errbuf, "ERROR: read of status file %s returned %d\n", STATFILE, statlen);
	    net_errmsg = net_errbuf;
	    return 0;
	 }
	 statbuf[statlen] = 0;	       /* string terminator */
       }
       return 1;
#endif
} 

/*
 * Read data into memory buffer to be used by getupsvar()
 * Returns 0 on error
 * Returns 1 if data fetched
 */
int fetch_events(char *host)
{
   char buf[500];
   int sockfd;
   int n, stat = 1;

   statlen = 0;
   statbuf[0] = 0;
   if ((sockfd = net_open(host, NULL, SERV_TCP_PORT)) < 0) {
      net_errmsg = "upsfetch: tcp_open failed";
      printf(net_errmsg);
      return 0;
   }

   if (net_send(sockfd, "events", 6) != 6) {
      net_errmsg = "fill_buffer: writen error on socket\n";
      printf(net_errmsg);
      return 0;
   }
   /*
    * Now read the events
    */
   while ((n = net_recv(sockfd, buf, sizeof(buf)-1)) > 0) {
      buf[n] = 0;
      strcat(statbuf, buf);
   }
   if (n < 0)
      stat = 0;
   *last_host = 0;
   net_close(sockfd);
   return stat;

} 


/* In our version, we have prefetched all the data, so the
 * host argument is ignored here.
 * Returns 1 if var found
 *   answer has var
 * Returns 0 if variable name not found
 *   answer has "Not found" is variable name not found
 *   answer may have "N/A" if the UPS does not support this
 *	 feature
 * Returns -1 if network problem
 *   answer has "N/A" if host is not available or network error
 */
int getupsvar(char *host, char *request, char *answer, int anslen) {

       int i;
       char *stat_match = NULL;
       char *find;
       int nfields = 0;
	
       if (!fetch_data(host)) {
               strcpy(answer, "N/A");
	       return -1;
       }

       for (i=0; cmdtrans[i].request; i++) 
	       if (!(strcmp(cmdtrans[i].request, request))) {
		    stat_match = cmdtrans[i].upskeyword;
		    nfields = cmdtrans[i].nfields;
	       }

       if (stat_match != NULL) {
	       if ((find=strstr(statbuf, stat_match)) != NULL) {
			if (nfields == 1)  /* get one field */
                              sscanf (find, "%*s %*s %s", answer);
			else {		   /* get everything to eol */
			      i = 0;
			      find += 11;  /* skip label */
                              while (*find != '\n')
				 answer[i++] = *find++;
			      answer[i] = 0;
			}
                        if (strcmp(answer, "N/A") == 0)
				return 0;
			return 1;
	       }
       }

       strcpy(answer, "Not found");
       return 0;
}

int getupsvarfd (int fd, char *upsname, char *varname, char *buf, int buflen)
{
   if (getupsvar(connect_host, varname, buf, buflen) == 1) {
      return 1;
   }
   printf("UPS VAR: %s not found\n", varname);
   return -1;
}
      

#define MAXLINE 512

/* Fill buffer with data from UPS network daemon   
 * Returns 0 on error
 * Returns 1 if OK
 */
int fill_buffer(int sockfd)
{
   int n, stat = 1; 
   char buf[1000];

   statbuf[0] = 0;
   statlen = 0;
   if (net_send(sockfd, "status", 6) != 6) {
      net_errmsg = "fill_buffer: writen error on socket\n";
      return 0;
   }

   while ((n = net_recv(sockfd, buf, sizeof(buf)-1)) > 0) {
      buf[n] = 0;
      strcat(statbuf, buf);
   }
   if (n < 0)
      stat = 0;

   statlen = strlen(statbuf);
   return stat;

}
