/*
 *  apclock.c -- Lock file managing functions
 *
 *  Copyright (C) 1998-99 Brian Schau <bsc@fleggaard.dk>
 *		  1998-99-2000 Riccardo Facchetti <riccardo@master.oasi.gpa.it>
 *
 *  apcupsd.c -- Simple Daemon to catch power failure signals from a
 *		 BackUPS, BackUPS Pro, or SmartUPS (from APCC).
 *	      -- Now SmartMode support for SmartUPS and BackUPS Pro.
 *
 *  Copyright (C) 1996-99 Andre M. Hedrick <andre@suse.com>
 *  All rights reserved.
 *
 */

/*
 *		       GNU GENERAL PUBLIC LICENSE
 *			  Version 2, June 1991
 *
 *  Copyright (C) 1989, 1991 Free Software Foundation, Inc.
 *			     675 Mass Ave, Cambridge, MA 02139, USA
 *  Everyone is permitted to copy and distribute verbatim copies
 *  of this license document, but changing it is not allowed.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 *  IN NO EVENT SHALL ANY AND ALL PERSONS INVOLVED IN THE DEVELOPMENT OF THIS
 *  PACKAGE, NOW REFERRED TO AS "APCUPSD-Team" BE LIABLE TO ANY PARTY FOR
 *  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING
 *  OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF ANY OR ALL
 *  OF THE "APCUPSD-Team" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  THE "APCUPSD-Team" SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 *  ON AN "AS IS" BASIS, AND THE "APCUPSD-Team" HAS NO OBLIGATION TO PROVIDE
 *  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 *  THE "APCUPSD-Team" HAS ABSOLUTELY NO CONNECTION WITH THE COMPANY
 *  AMERICAN POWER CONVERSION, "APCC".  THE "APCUPSD-Team" DID NOT AND
 *  HAS NOT SIGNED ANY NON-DISCLOSURE AGREEMENTS WITH "APCC".  ANY AND ALL
 *  OF THE LOOK-A-LIKE ( UPSlink(tm) Language ) WAS DERIVED FROM THE
 *  SOURCES LISTED BELOW.
 *
 */

/*
 *  Contributed by Facchetti Riccardo <riccardo@master.oasi.gpa.it>
 *
 *  Initial lock functions by BSC (see apcupsd.c)
 */

#include "apc.h"

/* check to see if a serial port lockfile exists.
 * If so, and the process is no longer running,
 * blow away the lockfile.  
 */
static int check_stale_lockfile(UPSINFO *ups)
{
    char pidbuffer[12];
    int size;
    int stalepid;
    int error;

    /*
     * If the daemon is talking with APC_NET, the lock file is not
     * needed.
     */
    if (ups->cable.type == APC_NET)
	return LCKNOLOCK;

    if (ups->lockpath[0] == '\0') {
	/*
         * If there's no lockfile configured, return error.
	 * This is a _must_. See my comment in apcconfig.c
	 */
        log_event(ups, LOG_ERR, "No lock path configured.\n");
	return LCKERROR;
    }

    errno = 0;
    if ((ups->lockfile = open(ups->lockpath, O_RDONLY)) < 0) {
	/*
         * Cannot open the file (may be it doesn't exist and that's okay
	 * for us so return success).
	 */
	if (errno == ENOENT)
	    return LCKSUCCESS;

	/*
	 * On any other error, return error.
	 */
        log_event(ups, LOG_ERR, "Lock file open error. ERR=%s\n", 
	   strerror(errno));
	return LCKERROR;
    }

    if ((size = read(ups->lockfile, &pidbuffer, 11)) == -1) {
	/*
	 * If we can not read from file, close it and return error:
	 * the file exist but we can not check for stale.
	 */
	error = LCKERROR;
        log_event(ups, LOG_ERR, "Lock file read error. ERR=%s\n",           
	   strerror(errno));
	goto out;
    }

    if (size == 0 || (sscanf(pidbuffer, "%d", &stalepid) != 1)) {
	/*
         * If there's no data in the file or the data written is wrong
	 * we have a process that:
	 * 1 - running but failed to write the lock file
	 * 2 - not running and failed to write the lock file
	 *
	 * Anyway we assume the worst case (1) and return error.
	 */
	error = LCKERROR;
        log_event(ups, LOG_ERR, "Lock file data error: %s\n", pidbuffer);
	goto out;
    }

    /* Check if it is our current pid or the pid of our parent */
    if (stalepid == getpid() || stalepid == getppid()) {
	/*
	 * We are us (may be a crash of the machine ... same pid
	 * because same boot sequence ... leave it alone and go run)
	 */
	error = LCKEXIST;
	goto out;
    }

    /*
     * Okay, now we have a stalepid to check.
     * kill(pid,0) checks to see if the process is still running.
     */
    if (kill(stalepid, 0) == -1 && errno == ESRCH) {
	/*
	 * Okay this is a stale lock:
	 * we can unlink even before closing it.
	 */
	if (unlink(ups->lockpath) < 0) {
	    log_event(ups, LOG_ERR, 
                   "Unable to unlink old lock file %s because %s\n",
		   ups->lockpath, strerror(errno));
	    error = LCKERROR;
	} else {
	    error = LCKSUCCESS;
	}
	goto out;
    }

    /*
     * We have unfortunately found a perfectly valid lock file.
     * Don't touch it.
     */
    log_event(ups, LOG_ERR, "Valid lock file for pid=%d, but not ours pid=%d\n", stalepid, getpid());
    error = LCKERROR;

out:
    close(ups->lockfile);
    ups->lockfile = -1;
    return error;
}

/* 
 * Create serial port lock file   
 */
int create_lockfile(UPSINFO *ups)
{
    char pidbuffer[12];
    int error;

    switch (error = check_stale_lockfile(ups)) {
    case LCKNOLOCK:
	/*
	 * Lockfile not needed: return success.
	 */
    case LCKEXIST:
	/*
	 * Lockfile exists and contains our pid.
	 */
	return LCKSUCCESS;

    case LCKERROR:
	/*
	 * Lockfile exists and is not stale.
	 */
	return LCKERROR;

    case LCKSUCCESS:
	/*
	 * Lockfile does not exist _now_.
	 */
	break;
    }
       
    /* If this is a powerkill execution, we are probably running
     * with the filesystems read-only, so don't try to create
     * the lock file. 
     */
#if AVERSION==3
    if (kill_ups_power)
	return error;
#endif

    /*
     * Now the file does not exist any more.
     * Open it for creation and don't accept any kind of error.
     */
    errno = 0;
    if ((ups->lockfile = open(ups->lockpath,
			  O_CREAT|O_EXCL|O_RDWR, 0644)) < 0) {
	/*
	 * Okay there is some problem with the lock path or
	 * something like that.
	 */
	log_event(ups, LOG_ERR, 
           "Cannot create %s serial port lock file: ERR=%s\n", 
	    ups->lockpath, strerror(errno));
	return LCKERROR;
    }

    if (sprintf(pidbuffer, "%010ld", (long)getpid()) <= 0) {
	/*
	 * Problems with sprintf
	 */
	error = LCKERROR;
        log_event(ups, LOG_ERR, "Lock file sprintf error.\n");
	goto out;
    }

    if (write(ups->lockfile, pidbuffer, strlen(pidbuffer)+1) != strlen(pidbuffer)+1) {
	/*
	 * Problems with write.
	 */
	error = LCKERROR;
        log_event(ups, LOG_ERR, "Lock file %s write failure. ERR=%s\n",
	   ups->lockpath, strerror(errno));
	goto out;
    }

    /*
     * Done it.
     */
    error = LCKSUCCESS;

out:
    close(ups->lockfile);
    ups->lockfile = -1;
    return error;
}

void delete_lockfile(UPSINFO *ups)
{
    if (ups->lockpath[0] != '\0') {
	/*
	 * If lockfile is ours, close it and delete it,
	 * otherwise do nothing.
	 */
	if (check_stale_lockfile(ups) == LCKEXIST) {
	    if (ups->lockfile != -1) {
		close(ups->lockfile);
		ups->lockfile = -1;
	    }
	    unlink(ups->lockpath);
	}
	/*
         * Now ups->lockfile is == -1 so there's no need to
	 * blank ups->lockfile too.
	 */
    }
}
