
// make sure the right function strerror_r is used
#define _XOPEN_SOURCE 600
#include <string.h>

#define LOCAL_DEBUG
#include "debug.h"

#include "config.h"
#include "fileitem.h"
#include "header.h"
#include "acfg.h"
#include "acbuf.h"

#include <iostream>
#include <sstream>
#include <utime.h>
#include <errno.h>

using namespace MYSTD;

header const * fileitem::GetHeader()
{
	return &m_head;
}

fileitem::fileitem(MYSTD::string sPath) :
	condition(),
	m_bCheckFreshness(true),
	status(FIST_FRESH),
	m_nIncommingCount(0),
	m_nSizeChecked(0),
	m_nSizeSeen(0),	
	m_sPath(acfg::cachedir+sPathSep+sPath),
	m_sKey(sPath),
	m_filefd(-1)
{
   ldbg("item created, m_sPath: "<< m_sPath);
}

uint64_t fileitem::GetTransferCount()
{
	setLockGuard;
	uint64_t ret=m_nIncommingCount;
	m_nIncommingCount=0;
	return ret;
}

int fileitem::GetFileFd() {
	setLockGuard;
	if(status<FIST_DLGOTHEAD)
		return -1;
	ldbg("Opening " << m_sPath);
	int fd=open(m_sPath.c_str(), O_RDONLY);
	
#ifdef HAS_ADVISE
	// optional, experimental
	if(status==FIST_COMPLETE)
		posix_fadvise(fd, 0, m_nSizeChecked, POSIX_FADV_SEQUENTIAL);
#endif

	return fd;
}


FiStatus fileitem::Setup(bool bCheckFreshness) {
	
	setLockGuard;
	if(status>FIST_FRESH)
		return status;
	
	status=FIST_INITED;
	
	m_bCheckFreshness = bCheckFreshness;
	
	if(m_head.LoadFromFile(m_sPath+".head") >0 && m_head.type==header::ANSWER )
	{
		if(200 != m_head.getStatus())
			goto error_clean;
		
		struct stat stbuf;
		if (0==::stat(m_sPath.c_str(), &stbuf))
			m_nSizeSeen=stbuf.st_size;
		
		// some plausibility checks
		if(m_bCheckFreshness)
		{
			const char *p=m_head.h[header::LAST_MODIFIED];
			if(!p)
				goto error_clean; // suspicious, cannot use it
		}
		else
		{ // non-volatile files, so could accept the length, do some checks first
			
			const char *pContLen=m_head.h[header::CONTENT_LENGTH];
			if(pContLen)
			{
				off_t nContLen=atol(pContLen); // if it's 0 then we assume it's 0
				
				// file larger than it could ever be?
				if(nContLen < m_nSizeSeen)
					goto error_clean;
				
				// accept length for non-volatile files...
				m_nSizeChecked=m_nSizeSeen;
				// ... are they even complete? a 0 value also looks weird, try to continue
				if(m_nSizeSeen == nContLen && nContLen>0)
					status=FIST_COMPLETE;	
			}
			else
			{
				// no content length known, assume it's ok
				m_nSizeChecked=m_nSizeSeen;
			}				
		}
	}
	return status;

	error_clean:
			unlink((m_sPath+".head").c_str());
			m_head.clear();
			m_nSizeSeen=0;
			status=FIST_INITED;
			return status; // unuseable, to be redownloaded
}

void _LogWithErrno(const char *msg, const string & sFile)
{
	char buf[32];
	buf[0]=buf[31]=0x0;
	const char *szErr=&buf[0];

#ifdef _GNU_SOURCE
#warning GLIBC BUG DETECTED -- _GNU_SOURCE is set in C++ mode
	szErr=strerror_r(errno, buf, sizeof(buf)-1);
#else
	strerror_r(errno, buf, sizeof(buf)-1);
#endif
	
	aclog::err( (sFile+" storage error ["+msg+"]: "+szErr).c_str());
}

bool fileitem::StoreHeader(const header & h)
{
	setLockGuard;

	ldbg("StoreHeader for " << m_sKey << ", current status: " << status);
	
	if(status > FIST_DLPENDING) // already started? error? whatever
		return false;
	
	m_nIncommingCount+=h.m_nEstimLength;
	
	// status will change, most likely... ie. BOUNCE action
	notifyAll();

#define SETERROR(x) { m_head.frontLine="HTTP/1.1 "; m_head.frontLine+=x; status=FIST_ERROR; _LogWithErrno(x, m_sPath); }
#define SETERRORKILLFILE(x) { SETERROR(x); goto kill_file; }
#define BOUNCE(x) { SETERROR(x); _LogWithErrno(x, m_sPath); return false; }
	
	int code=h.getStatus();
	ldbg("http code: "<< code);
	bool bStoreData(false);
	string sHeadPath=m_sPath+".head";
	
	if(200 == code)
	{
		bStoreData=true;
		m_nSizeChecked=0;
		m_head=h;
	}
	else if(206 == code)
	{
		if(m_nSizeSeen<=0)
		{
			// wtf? Cannot have requested partial content
			BOUNCE("500 Unexpected Partial Response");
		}
		/*
		 * Range: bytes=453291-
		 * ...
		 * Content-Length: 7271829
		 * Content-Range: bytes 453291-7725119/7725120
		 */
		const char *p=h.h[header::CONTENT_RANGE];
		if(!p)
			BOUNCE("500 Missing Content-Range in Partial Response");
		long myfrom, myto, mylen;
		int n=sscanf(p, "bytes %ld-%ld/%ld", &myfrom, &myto, &mylen);
		if(n<=0)
			n=sscanf(p, "bytes=%ld-%ld/%ld", &myfrom, &myto, &mylen);
		
		ldbg("resuming? n: "<< n << " und myfrom: " <<myfrom << 
				" und myto: " << myto << " und mylen: " << mylen);
		if(n!=3  // check for nonsense
				|| myfrom != m_nSizeSeen-1
				|| myfrom<0 || mylen<0)
			BOUNCE("500 Server reports illegal range");
	
		m_nSizeChecked=myfrom;
		bStoreData=true;
		
		m_head=h;
		m_head.frontLine="HTTP/1.1 200 OK";
		m_head.del(header::CONTENT_RANGE);
		m_head.set(header::CONTENT_LENGTH, mylen);
	}
	else if(416 == code) // that's always bad; it cannot be complete (-1 trick) -> kill cached file ASAP
	{
		SETERRORKILLFILE("503 Server disagrees on file size, cleaning up");
	}
	else
	{
		bStoreData=false;
		// have a clear one with just the error message
		m_head.frontLine=h.frontLine;
		m_head.type=header::ANSWER;
		
	}
			
	if(bStoreData)
	{
		// using adaptive Delete-Or-Replace-Or-CopyOnWrite strategy
		
		// First opening the file first to be sure that it can be written. Header storage is the critical point,
		// every error after that leads to full cleanup to not risk inconsistent file contents 
		
		int flags = O_WRONLY | O_CREAT;
		struct stat stbuf;
				
		mkbasedir(m_sPath);
		m_filefd=open(m_sPath.c_str(), flags, 00644);
		ldbg("file opened?! returned: " << m_filefd);
		
		// self-recovery from cache poisoned with files with wrong permissions
		if (m_filefd<0)
		{
			if(m_nSizeChecked>0) // OOOH CRAP! CANNOT APPEND HERE! Do what's still possible.
			{
				string temp=m_sPath+".tmp";
				if(FileCopy(m_sPath, temp) && 0==unlink(m_sPath.c_str()) )
				{
					if(0!=rename(temp.c_str(), m_sPath.c_str()))
						BOUNCE("503 Cannot rename files");
					
					// be sure about that
					if(0!=stat(m_sPath.c_str(), &stbuf) || stbuf.st_size!=m_nSizeSeen)
						BOUNCE("503 Cannot copy file parts, filesystem full?");
					
					m_filefd=open(m_sPath.c_str(), flags, 00644);
					ldbg("file opened after copying around: ");
				}
				else
					BOUNCE("503 Cannot store or remove files");
			}
			else
			{
				unlink(m_sPath.c_str());
				m_filefd=open(m_sPath.c_str(), flags, 00644);
				ldbg("file force-opened?! returned: " << m_filefd);
			}
		}
		
		if (m_filefd<0)
			BOUNCE(errno==ENOSPC ? "503 OUT OF DISK SPACE" : 
			"503 Cache storage error while opening data file");
		
		if(0!=fstat(m_filefd, &stbuf) || !S_ISREG(stbuf.st_mode))
			SETERRORKILLFILE("503 Not a regular file");
		
		// crop, but only if the new size is smaller. MUST NEVER become larger (would fill with zeros)
		if(m_nSizeChecked < m_nSizeSeen)
		{
			if(0!=ftruncate(m_filefd, m_nSizeChecked))
				SETERRORKILLFILE("503 Cannot change file size");
		}
		else if(m_nSizeChecked>m_nSizeSeen) // should never happen and caught by the checks above
			SETERRORKILLFILE("503 Internal error on size checking");
		
		ldbg("Opening "+sHeadPath);
		int count=m_head.StoreToFile(sHeadPath);
		
		// unlink and retry
		if(count<0)
			unlink(sHeadPath.c_str());
		count=m_head.StoreToFile(sHeadPath);

		if(count<0)
			SETERRORKILLFILE( (-count!=ENOSPC) ? "503 Cache storage error" : "503 OUT OF DISK SPACE");
			
		// double-check the sane state
		if(0!=fstat(m_filefd, &stbuf) || stbuf.st_size!=m_nSizeChecked)
			SETERRORKILLFILE("503 Inconsistent file state");
			
		if(m_nSizeChecked!=lseek(m_filefd, m_nSizeChecked, SEEK_SET))
			SETERRORKILLFILE("503 IO error, positioning");
	}
	
	status=FIST_DLGOTHEAD;
	return true;

	kill_file:
	if(m_filefd>=0)
	{
		close(m_filefd);
		m_filefd=-1;
	}
	unlink(m_sPath.c_str());
	unlink(sHeadPath.c_str());
	
	status=FIST_ERROR;
	return false;
}

bool fileitem::StoreFileData(const char *data, unsigned int size) {
	
	setLockGuard;

	// something might care, most likely... also about BOUNCE action
	notifyAll();
	
	m_nIncommingCount+=size;
	
	if(status >= FIST_ERROR || status < FIST_DLGOTHEAD)
		return false;
	
	if (size==0)
	{
		status=FIST_COMPLETE;

		// we are done! Fix header from chunked transfers?
		if (m_filefd>=0 && ! m_head.h[header::CONTENT_LENGTH])
		{
			m_head.set(header::CONTENT_LENGTH, m_nSizeChecked);
			m_head.StoreToFile(m_sPath+".head");
		}
	}
	else
	{
		status = FIST_DLRECEIVING;

		if (m_filefd>=0)
		{
			while(size>0)
			{
				int r=write(m_filefd, data, size);
				if (r<0) 
				{
					if(EINTR==errno || EAGAIN==errno)
						continue;
					if(ENOSPC==errno)
						BOUNCE("503 OUT OF DISK SPACE");
					BOUNCE("503 General storage error");
				}
				m_nSizeChecked+=r;
				size-=r;
				data+=r;
			}
			
		}
	}
	return true;
}

void fileitem::SetFailureMode(const MYSTD::string & message, FiStatus fist)
{
	ldbg("SetFailureMode: " << fist << ", " << message);
	setLockGuard;
	dbgline;
	if(m_filefd>=0)
	{
		::close(m_filefd);
		m_filefd=-1;
	}
	// maybe was already set internally before
	if(status<FIST_ERROR)
	{
		m_head.clear();
		m_head.frontLine=string("HTTP/1.1 ")+message;
		m_head.type=header::ANSWER;
		status=fist;
		notifyAll();
	}
}

fileitem::~fileitem() {

	ldbg("Destroying fitem, " << m_sKey);
	setLockGuard;
	
	m_head.clear();
	
	if(m_filefd>=0)
	{
		close(m_filefd);
	}
	ldbg("Fitem destroyed, " << m_sKey);
}





struct tFileRefEntry
{
	int nRefCount;
	tFileItemPtr ref;
	tFileRefEntry() : nRefCount(0) { };
};
static map<string, tFileRefEntry> mapItems;
//static pthread_mutex_t mapLck = PTHREAD_MUTEX_INITIALIZER;
lockable mapLck;

void DelUser(string sKey) {
   ldbg("fileitem::DelUser()");

   lockguard managementLock(mapLck); 
	
   tFileRefEntry & entry = mapItems[sKey];
   entry.nRefCount--;
   if(entry.nRefCount<=0)
   {
	   ASSERT(entry.ref.get());
	   // atomic and quick-and-dirty
	   lockguard g(*(entry.ref));
	   entry.ref->status=FIST_ERRNOUSER;
	   mapItems.erase(sKey);
   }
}

tFileItemPtr GetFileItem(MYSTD::string sPathKey)
{
	tFileItemPtr p;
	MYTRY
	{
		lockguard lockGlobalMap(mapLck);

		map<string, tFileRefEntry>::iterator it=mapItems.find(sPathKey);
		if(it!=mapItems.end())
		{
			it->second.nRefCount++;
			return it->second.ref;
		}
		p.reset(new fileitem(sPathKey));
		tFileRefEntry & entry = mapItems[sPathKey];
		if(!entry.ref)
		{
			entry.ref=p;
			entry.nRefCount++;
		}
		return entry.ref;
	}
	MYCATCH(std::bad_alloc)
	{
		if(p)
			p.reset();
		return p;
	}

}

#ifdef DEBUG

void DumpItems()
{
	lockguard lockGlobalMap(mapLck);

	map<string, tFileRefEntry>::iterator it=mapItems.begin();
	cout << "Remaining file items:\n";
	for(;it!=mapItems.end(); it++)
	{
		cout << it->second.ref->status << ": " << it->first <<endl;
	}
}
#endif
