# 
# putdir.sh - create a new directory in an archive
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 
errname=putdir

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "*+create a new directory in an archive from a template\\n"
		printf "usage: putdir [options] source-dir dest-dir\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " -R --root root                specify the local archive root\\n"
		printf " -A --archive archive          specify the archive name\\n"
		printf " -e --errname name             program name for error messages\\n"
		printf "\\n"
		printf "Copy SOURCE-DIR to the indicated archive, creating DEST-DIR\\n"
		printf "which must not already exist.\\n"
		printf "\\n"
		printf "For information about how the archive to operate on is selected,\\n"
		printf "try \"larch my-default-archive --help\"\\n."
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=

while test $# -ne 0 ; do

  case "$1" in 

    -R|--root)		shift
    			if test $# -eq 0 ; then
			  printf "putdir: -R and --root require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archroot="$1"
			shift
			;;

    -A|--archive)	shift
    			if test $# -eq 0 ; then
			  printf "putdir: -A and --archive require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archive="$1"
			shift
			;;

    -e|--errname)	shift
    			if test $# -eq 0 ; then
			  printf "putdir: -e and --errname require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			errname="$1"
			shift
			;;

    -*)			printf "putdir: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ne 2 ; then
    printf "usage: putdir [options] source-dir end-dir\\n" 1>&2
    printf "try --help\\n" 1>&2
    exit 1
fi

source="$1"
dest="$2"

################################################################
# Sanity Check and Process Defaults
# 

archive=`larch my-default-archive -e "$errname" -R "$root" -A "$archive"`

cd "$source"
source="`pwd`"

################################################################
# Create a Temp Directory In the Archive
# 
# This currently handles only local archives.
# 

tmpnam=",,new-dir.`larch my-id -u`.$$"

wftp-cd "`dirname $dest`"
wftp-mkdir "$tmpnam"
wftp-cd "$tmpnam"
there="`wftp-pwd`"

if test ! -z "$debug" ; then
  larch heading "putdir\\n" 1>&2
  larch heading --sub "there: %s\\n" "$there" 1>&2
  larch heading --sub "source: %s\\n" "$source" 1>&2
fi

for f in `ls "$source"` ; do

  wftp-cd "$there"

  if test ! -d "$source/$f" ; then

    if test ! -z "$debug" ; then
      larch heading --sub "file: %s\\n" "$f" 1>&2
    fi
    wftp-put "$f" < "$source/$f"

  else

    if test ! -z "$debug" ; then
      larch heading --sub "dir: %s\\n" "$f" 1>&2
    fi
    larch nested larch nested larch putdir -R "$root" -A "$archive" "$f" "$f"

  fi

done

wftp-cd "$there"

if test ! -z "$debug" ; then
  larch heading --sub "back to: %s\\n" "$there" 1>&2
fi

################################################################
# Put it in place
# 

wftp-cd ..

if test ! -z "$debug" ; then
  larch heading --sub "up to %s\\n" "`wftp-pwd`" 1>&2
  larch heading --sub "rename: %s %s\\n" "$tmpnam" "`basename \"$dest\"`" 1>&2
fi

wftp-rename "$tmpnam" "`basename \"$dest\"`"

if test ! -z "$debug" ; then
  larch heading --sub "renamed\\n" 1>&2
fi


