# 
# make-category.sh - create a new category within an archive
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "create a new category within an archive\\n"
		printf "usage: make-category [options] category\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " -R --root root                specify the local archive root\\n"
		printf " -A --archive archive          specify the archive name\\n"
		printf "\\n"
		printf " -r --readme file              save FILE as the README\\n"
	        printf "                               for this category\\n"
		printf "\\n"
		printf "Create CATEGORY as a category in the indicated archive.\\n"
		printf "The category must not already exist.\\n"
		printf "\\n"
		printf "For information about how the archive to operate on is selected,\\n"
		printf "try \"larch my-default-archive --help\".\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=
readme=

while test $# -ne 0 ; do

  case "$1" in 

    -R|--root)		shift
    			if test $# -eq 0 ; then
			  printf "make-category: -R and --root require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archroot="$1"
			shift
			;;

    -A|--archive)	shift
    			if test $# -eq 0 ; then
			  printf "make-category: -A and --archive require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archive="$1"
			shift
			;;


    -r|--readme)	shift
			if test $# -eq 0 ; then
			  printf "make-category: -R and --readme require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			readme="$1"
			shift
			;;

    -*)			printf "make-category: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ne 1 ; then
    printf "usage: make-category [options] category\\n" 1>&2
    printf "try --help\\n" 1>&2
    exit 1
fi

category="$1"

################################################################
# Sanity Check 
# 

larch valid-package-name -e make-category -b -- "$category"
idstr="`larch my-id -e make-category`"

archive="`larch parse-package-name -R \"$archroot\" -A \"$archive\" --arch \"$category\"`"
category="`larch parse-package-name --basename \"$category\"`"

################################################################
# Get a Temp Dir
# 

tmpdir="`larch make-cache-temp-dir new-category-$$`"


################################################################
# Create the Category Directory (in the Temp Dir)
# 

tmpcat="$tmpdir/$category"
mkdir "$tmpcat"


################################################################
# Make the =README file
# 

printf "Creator: %s\\nDate: %s\\n\\n" "$idstr" "`date`" > "$tmpcat/=README" 


if test ! -z "$readme" ; then

  cat "$readme" >> "$tmpcat/=README" 

fi




################################################################
# Try to install it in the archive
# 

cd "$tmpdir"

status=0

if ! larch with-archive -R "$archroot" -A "$archive" larch putdir -e make-category -R "$archroot" -A "$archive"  "$category" "$category" ; then
  printf "make-category: unable to create category\\n" 1>&2
  status=1
fi

if larch my-notifier > /dev/null 2>&1 ; then
  notifier="`larch my-notifier`"
  ( cd "$notifier" ; \
    larch without-archive as-daemon $ARCH_LOG_OPTION --null-output larch notify "$notifier" "$archive/$category" )
fi


################################################################
# Remove the Cache Temp Directory
# 
cd ..
rm -rf "$tmpdir"

exit "$status"

