# 
# revisions.sh - print a list of revisions in an archive version
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "list the revisions in an archive version\\n"
		printf "usage: revisions [options] [[archive/]version [patch...]]\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " -R --root root                specify the local archive root\\n"
		printf " -A --archive archive          specify the archive name\\n"
		printf "\\n"
		printf " -r --reverse                  sort from newest to oldest\\n"
		printf " -s --summary                  print a summary of each patch\\n"
		printf " -c --creator                  print the creator id of each patch\\n"
		printf " -D --date                     print the date of each patch\\n"
		printf " -f --full                     print full names of patch levels\\n"
		printf "\\n"
		printf " --silent-error                no error message if the indicated\\n"
		printf "                                 version does not exist (exit status\\n"
		printf "                                 only)\\n"
		printf "\\n"
		printf "Print a list of revisions within an archive version.\\n"
		printf "\\n"
		printf "The list is ordinarilly sorted from oldest to newest,\\n"
		printf "but the order can be changed with -r (--reverse).\\n"
		printf "\\n"
		printf "The output format is:\\n"
		printf "\\n"
		printf "	\"%%s\\\\n\" \"\$patchlevel\" \\n"
		printf "\\n"
		printf "With optional arguments specifying patches, list only those\\n"
		printf "patches, if they exist.  If a listed patch does not exist,\\n"
		printf "exit with status 1.  The -r (--reverse) flag has no effect\\n"
		printf "with optional arguments.\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=
reverse=
summary=
creator=
date=
full=
debug_opt=

silent_error=

while test $# -ne 0 ; do

  case "$1" in 

    --debug)		shift
    			debug_opt=--debug
			printf "\n" 1>&2
			printf "revisions: DEBUGGING ACTIVATED\n" 1>&2
			printf "\n" 1>&2
			set -x
			;;
			
    --silent-error)	shift
    			silent_error=--silent_error
			;;

    -R|--root)		shift
    			if test $# -eq 0 ; then
			  printf "make-category: -R and --root require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archroot="$1"
			shift
			;;

    -A|--archive)	shift
    			if test $# -eq 0 ; then
			  printf "make-category: -A and --archive require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archive="$1"
			shift
			;;


    -r|--reverse)	shift
    			reverse=-r
			;;

    -s|--summary)	shift
			summary=--summary
			;;

    -c|--creator)	shift
    			creator=--creator
			;;

    -D|--date)		shift
    			date=--date
			;;

    -f|--full)		shift
			full=--full
			;;

    --)			shift
			break
			;;

    -*)			printf "revisions: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;

  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -lt 0 ; then
  printf "usage: revisions [options] [[archive/]version [patch...]]\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

if test $# -ne 0 ; then
  archive_version="$1"
  shift
else
  archive_version=
fi

################################################################
# Sanity Check and Process Defaults
# 

if test -z "$archive_version" ; then
  archive_version="`larch tree-version .`"
  if test ! -z "$archive" ; then 
    archive_revision="`larch parse-package-name --non-arch \"$archive_version\"`"
  fi
fi

larch valid-package-name -e revisions --tolerant -- "$archive_version"

archive="`larch parse-package-name -R \"$archroot\" -A \"$archive\" --arch \"$archive_version\"`"
spec="`larch parse-package-name -R \"$archroot\" -A \"$archive\" --non-arch \"$archive_version\"`"
category="`larch parse-package-name -b \"$archive_version\"`"
branch="`larch parse-package-name -p \"$archive_version\"`"

patches_wanted=
for p in "$@" ; do
  larch valid-patch-level-name -e revisions -- "$p"
  patches_wanted="$patches_wanted $p"
done

################################################################
# Ensure that We Have an Archive Connection 
# 

if test "$WITHARCHIVE" != "$archive" ; then
  exec larch with-archive -R "$archroot" -A "$archive"  \
    larch revisions $debug_opt -R "$archroot" -A "$archive" $reverse $summary $creator $date $full $silent_error "$archive/$spec" "$@"
fi

################################################################
# What Version?
# 

if larch valid-package-name --vsn -- "$spec" ; then
  version="`larch parse-package-name --package-version \"$spec\"`"
else
  version="`larch versions $debug_opt --reverse \"$archive/$branch\" | head -1`"
  if test -z "$version" ; then
    printf "\\n" 1>&2
    printf "revisions: no version found\\n" 1>&2
    printf "  archive: %s\\n" "$archive" 1>&2
    printf "  branch: %s\n" "$branch" 1>&2
    printf "\\n" 1>&2
    exit 1
  fi
fi


################################################################
# Print the List
# 

patchlvlre="base-0|patch-[0-9]+|version-0|versionfix-[0-9]+"

if test -z "$summary" ; then

  if test -z "$patches_wanted" ; then

    if test -z "$full" ; then
      full_filter=cat
    else
      full_filter="sed -e s,^,$archive/$version--,"
    fi

    wftp-home
    if ! wftp-cd "$category/$branch/$version" ; then
      if test -z "$silent_error" ; then
        printf "\\n" 1>&2
        printf "revisions: no such version\\n" 1>&2
        printf "  archive: %s\\n" "$archive" 1>&2
        printf "  version: %s\\n" "$version" 1>&2
        printf "\\n" 1>&2
      fi
      exit 1
    fi

    wftp-ls \
      | grep -E "^($patchlvlre)$" \
      | sort -t- -k "1,1"${reverse#-} -k "2,2"${reverse#-}n \
      | $full_filter

    wftp-home

  else

    wftp-cd $category/$branch/$version

    for p in $patches_wanted ; do 
      wftp-cd $p
      wftp-cd ..
      if test -z "$full" ; then
        printf "%s\\n" $p
      else
        printf "%s/%s--%s\\n" $archive $version $p
      fi
    done

    wftp-home

  fi

else

  wftp-home

  if test -z "$patches_wanted" ; then
    wftp-cd "$category/$branch/$version"
    patches_wanted="`wftp-ls \
		     | grep -E \"^($patchlvlre)\$\" \
		     | sort -t- -k \"1,1\"${reverse#-} -k \"2,2\"${reverse#-}n`"
    wftp-home
  fi

  for patch in $patches_wanted  ; do

    if test -z "$full" ; then
      printf "%s\\n" $patch
    else
      printf "%s/%s--%s\\n" $archive $version $patch
    fi

    wftp-get "$category/$branch/$version/$patch/log" \
    | awk -v summary="$summary" \
	  -v creator="$creator" \
	  -v date="$date" \
	  -f "$ARCH_SUBCMD_ROOT/patch-logs/patch-list-description.awk"

  done

fi

