# 
# log-for-merge.sh: generate a log entry body for a merge
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#       --version | -V
#       --help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
                printf "generate a log entry bodyfor a merge\\n"
                printf "usage: log-for-merge [options] [[archive/]version]\\n"
                printf "\\n"
                printf " -V --version                  print version info\\n"
                printf " -h --help                     display help\\n"
                printf "\\n"
                printf " -R --root root                specify the local archive root\\n"
                printf " -A --archive archive          specify the archive name\\n"
                printf " -D --dir DIR                  use project tree containing DIR\\n"
                printf "\\n"
                printf "Generate a log entry body describing a merge into VERSION.\\n"
                printf "\\n"
                exit 0
                ;;

      *)
                ;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=
dir=.

while test $# -ne 0 ; do

  case "$1" in 

    -D|--dir)           shift
                        if test $# -eq 0 ; then
                          printf "log-for-merge: -D and --dir require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        dir="$1"
                        shift
                        ;;

    -R|--root)          shift
                        if test $# -eq 0 ; then
                          printf "log-for-merge: -R and --root require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archroot="$1"
                        shift
                        ;;

    -A|--archive)       shift
                        if test $# -eq 0 ; then
                          printf "log-for-merge: -A and --archive require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archive="$1"
                        shift
                        ;;

    --)			shift
    			break
			;;
			
    -*)                 printf "log-for-merge: unrecognized option (%s)\\n" "$1" 1>&2
                        printf "try --help\\n" 1>&2
                        exit 1
                        ;;

    *)                  break
                        ;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -gt 1 ; then
  printf "usage: log-for-merge [options] [[archive/]revision]\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

if test $# -ne 0 ; then
  version="$1"
  shift
else
  version=
fi

################################################################
# Sanity Check and Process Defaults
# 

here="`pwd`"
cd "$dir"
dir="`pwd`"
wdroot="`larch tree-root --accurate`"

cd "$wdroot"

if test -z "$version" ; then
  version=`larch tree-version`
fi

larch valid-package-name -e log-for-merge --vsn "$version"

archive="`larch parse-package-name -R \"$archroot\" -A \"$archive\" --arch "$version"`"
version="`larch parse-package-name -R \"$archroot\" -A \"$archive\" --non-arch "$version"`"
category="`larch parse-package-name --basename \"$version\"`"
branch="`larch parse-package-name \"$version\"`"
vsn="`larch parse-package-name --vsn \"$version\"`"

################################################################
# Print the log entry body
# 
# 

printf "Patches applied:\\n"

larch new-on-branch --reverse "$archive/$version" \
| xargs need-args 'larch log-ls --full --summary' \
| awk 'match($0, "^[^ \t]") { printf("\n  * %s\n", $0); }
       match($0, "^[ \t]")  { printf(" %s\n", $0); }'

printf "\\n"


# tag: Tom Lord Wed Jan  9 01:53:56 2002 (patch-logs/log-for-merge.sh)
#
