# remove-log.sh: remove a version patch log from a project tree
# 
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "remove a version patch log from a project tree\\n"
		printf "usage: remove-log [options] [archive/]version\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " -R --root root                specify the local archive root\\n"
		printf " -A --archive archive          specify the archive name\\n"
		printf "\\n"
		printf " -d --dir DIR                  cd to DIR first\\n"
		printf "\\n"
		printf "Remove the patch log for VERSION from the project tree containing DIR\\n"
		printf "(or the current directory).\\n"
		printf "\\n"
		printf "Use this command with caution -- it erases revision history from\\n"
		printf "the project tree!\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=
dir=.

while test $# -ne 0 ; do

  case "$1" in 

    -d|--dir)		shift
			if test $# -eq 0 ; then
			  printf "remove-log: -d and --dir require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			dir="$1"
			shift
			;;

    -R|--root)		shift
    			if test $# -eq 0 ; then
			  printf "remove-log: -R and --root require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archroot="$1"
			shift
			;;

    -A|--archive)	shift
    			if test $# -eq 0 ; then
			  printf "remove-log: -A and --archive require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archive="$1"
			shift
			;;


    -*)			printf "remove-log: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ne 1 ; then
  printf "usage: remove-log [options] [archive/]version\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

arch_version="$1"

################################################################
# Sanity Check
# 

cd "$dir"
dir="`pwd`"

larch valid-package-name -e "remove-log" --vsn -- "$arch_version"

archive=`larch parse-package-name -R "$archroot" -A "$archive" --arch "$arch_version"`
version=`larch parse-package-name -R "$archroot" -A "$archive" --package-version "$arch_version"`
category=`larch parse-package-name -b $version`
branch=`larch parse-package-name $version`

cd "$dir"
wdroot="`larch tree-root --accurate`"

arch_logdir="$wdroot/{arch}/$category/$branch/$version/$archive"

if test ! -e "$arch_logdir" ; then
  printf "remove-log: patch log not found\\n" 1>&2
  printf "  source tree: %s\\n" "$wdroot" 1>&2
  printf "  version: %s/%s\\n" $archive $version 1>&2
  printf "\\n" 1>&2
  printf "try larch logs --help\\n" 1>&2
  printf "\\n" 1>&2
  exit 1
fi


################################################################
# Do It
# 

rm -rf "$arch_logdir"

# tag: Tom Lord Thu Feb 14 16:39:21 2002 (patch-logs/remove-log.sh)
#
