/* pipe-friendly.c -
 *
 ****************************************************************
 * Copyright (C) 2002  Tom Lord
 * 
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "hackerlab/os/errno.h"
#include "hackerlab/cmd/main.h"



static t_uchar * program_name = "pipe-friendly";
static t_uchar * usage = "pipe-friendly";
static t_uchar * version_string = (cfg__std__package " from regexps.com\n"
				   "\n"
				   "Copyright 2002 Tom Lord\n"
				   "\n"
				   "This is free software; see the source for copying conditions.\n"
				   "There is NO warranty; not even for MERCHANTABILITY or FITNESS FOR A\n"
				   "PARTICULAR PURPOSE.\n"
				   "\n"
				   "Report bugs to <lord@regexps.com>.\n"
				   "\n");

#define OPTS(OP, OP2) \
  OP (opt_help_msg, "h", "help", 0, \
      "Display a help message and exit.") \
  OP (opt_long_help, "H", 0, 0, \
      "Display a verbose help message and exit.") \
  OP (opt_version, "V", "version", 0, \
      "Display a release identifier string") \
  OP2 (opt_version, 0, 0, 0, "and exit.")

static t_uchar long_help[] = ("copy stdin to stdout, ignoring EPIPE\n"
			      "Some shell program implementations insist on yacking to"
			      "stderr in the event of EPIPE on stdout, even though this"
			      "is a perfectly reasonable condition.  Pipe such programs"
			      "into pipe-friendly, and such error messages will be"
			      "supressed, though at the cost of not terminating the"
			      "program generating stdin to pipe-friendly on EPIPE.\n");

enum options
{
  OPTS (OPT_ENUM, OPT_IGN)  
};

struct opt_desc opts[] = 
{
  OPTS (OPT_DESC, OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
main (int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, version_string, long_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
	break;
      switch (o)
	{
	default:
	  safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
	  panic ("internal error parsing arguments");

	usage_error:
	  opt_usage (2, argv[0], program_name, usage, 1);
	  exit (1);

	/* bogus_arg: */
	  safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
	  goto usage_error;
	}
    }

  {
    char buf[4096];
    ssize_t amt;

    /* do we need to mask out SIGPIPE to work around buggy shells?
     * Bug reports will tell...  
     */

    while (1)
      {
	int errn;

	amt = vu_read_retry (&errn, 0, buf, sizeof (buf));

	if (amt < 0)
	  {
	    safe_printfmt (2, "\n");
	    safe_printfmt (2, "pipe-friendly: I/O error on input\n");
	    safe_printfmt (2, "    error: %s (%d)\n", errno_to_string (errn), errn);
	    safe_printfmt (2, "\n");
	    exit (2);
	  }

	if (amt == 0)
	  break;

	if (amt != vu_write_retry (&errn, 1, buf, amt))
	  {
	    if (errn == EPIPE)
	      continue;

	    safe_printfmt (2, "\n");
	    safe_printfmt (2, "pipe-friendly: I/O error on output\n");
	    safe_printfmt (2, "    error: %s (%d)\n", errno_to_string (errn), errn);
	    safe_printfmt (2, "\n");
	    exit (2);
	  }
      }
  }
  return 0;
}

