
/*
 * Argus Client Software.  Tools to read, analyze and manage Argus data.
 * Copyright (C) 2000-2001 QoSient, LLC.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*
 * rasort - sort argus records.
 *
 * written by Carter Bullard
 * QoSient, LLC
 *
 */

#include <argus_client.h>
#include <rasort.h>


int RaSortRoutine (const void *, const void *);

void
ArgusClientInit ()
{
   int i, x;

   if (!(RaInitialized)) {
      RaWriteOut = 0;

      if ((RaArgusRecordQueue = RaNewQueue()) == NULL)
         ArgusLog (LOG_ERR, "ArgusClientInit: RaNewQueue error %s\n", strerror(errno));

      bzero ((char *) RaSortAlgorithms, sizeof(RaSortAlgorithms));
      RaSortAlgorithms[0] = RaSortAlgorithmTable[0];

      for (i = 0; i < ARGUS_MAX_SORT_ALG; i++) {
         if (RaSortAlgorithmStrings[i] != NULL) {
            for (x = 0; x < MAX_SORT_ALG_TYPES; x++) {
               if (!strncmp (RaSortKeyWords[x], RaSortAlgorithmStrings[i], strlen(RaSortKeyWords[x]))) {
                  RaSortAlgorithms[i] = RaSortAlgorithmTable[x];
                  break;
               }
            }
            if (x == MAX_SORT_ALG_TYPES)
               ArgusLog (LOG_ERR, "sort syntax error. \'%s\' not supported", RaSortAlgorithmStrings[i]);
         } else
            break;
      }

      RaInitialized++;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusClientInit () returning\n");
#endif
}

int RaParseCompleting = 0;

void
RaParseComplete (int sig)
{
   int i = 0;

   if (!RaParseCompleting) {
      RaParseCompleting++;

      if (Nflag == 0)
         Nflag = RaArgusRecordQueue->count;

      if (Nflag > 0) {
         RaSortQueue (RaArgusRecordQueue);

         for (i = 0; i < Nflag; i++)
            RaSendArgusRecord ((struct ArgusRecordStore *) RaArgusRecordQueue->array[i]);
      }
   }
}

void
ArgusClientTimeout ()
{
}

void
parse_arg (int argc, char**argv)
{ 
}


void
usage ()
{
   extern char version[];

   fprintf (stderr, "Rasort Version %s\n", version);
   fprintf (stderr, "usage: %s [[-s sortfield] [-s sortfield] ...] [ra-options] [- filter-expression]\n", ArgusProgramName);
   fprintf (stderr, "options:    -s <sortfield> specify the <sortfield>(s) in order.\n");
   fprintf (stderr, "                           valid sorfields are:\n");
   fprintf (stderr, "                              startime, lasttime, duration,\n");
   fprintf (stderr, "                              srcaddr, dstaddr, proto, sport, dport,\n");
   fprintf (stderr, "                              stos, dtos, sttl, dttl,\n");
   fprintf (stderr, "                              bytes, srcbytes, dstbytes,\n");
   fprintf (stderr, "                              packets, srcpackets, dstpackets.\n");
   fprintf (stderr, "\n");
   fprintf (stderr, "ra-options: -b             dump packet-matching code.\n");
   fprintf (stderr, "            -C             treat the remote source as a Cisco Netflow source.\n");
   fprintf (stderr, "            -D <level>     specify debug level\n");
   fprintf (stderr, "            -F <conffile>  read configuration from <conffile>.\n");
   fprintf (stderr, "            -h             print help.\n");
   fprintf (stderr, "            -P <portnum>   specify remote argus <portnum> (tcp/561).\n");
   fprintf (stderr, "            -r <file>      read argus data <file>. '-' denotes stdin.\n");
   fprintf (stderr, "            -S <host>      specify remote argus <host>.\n");
   fprintf (stderr, "            -t <timerange> specify <timerange> for reading records.\n");
   fprintf (stderr, "                  format:  timeSpecification[-timeSpecification]\n");
   fprintf (stderr, "                           timeSpecification: [mm/dd[/yy].]hh[:mm[:ss]]\n");
   fprintf (stderr, "                                               mm/dd[/yy]\n");
   fprintf (stderr, "            -T <secs>      attach to remote server for T seconds.\n");
#ifdef ARGUS_SASL
   fprintf (stderr, "            -U <user/auth> specify <user/auth> authentication information.\n");
#endif
   fprintf (stderr, "            -w <file>      write output to <file>. '-' denotes stdout.\n");
   exit(1);
}


void
RaProcessRecord (struct ArgusRecord *argus)
{
   struct ArgusRecordStore *obj = NULL;
   
   if (argus != NULL) {
      if ((obj = RaNewArgusStore(argus)) != NULL) {
         if ((obj->data[0] = RaNewArgusData(argus)) != NULL)
            RaAddToQueue(RaArgusRecordQueue, &obj->qhdr);
         else
            ArgusLog (LOG_ERR, "RaProcessRecord: RaNewArgusData(0x%x) %s\n",
                            argus, strerror(errno));
      } else
         ArgusLog (LOG_ERR, "RaProcessRecord: RaNewArgusStore(0x%x) %s\n",
                            argus, strerror(errno));
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaProcessRecord(0x%x) done.\n", argus);
#endif
}


u_char ArgusRecordBuffer[1024];
int ArgusLastRecord = 0;

void
process_man (struct ArgusRecord *argus)
{
   if ((argus->ahdr.cause & ARGUS_STOP)) {
      bcopy ((char *) ArgusOriginal, ArgusRecordBuffer, argus->ahdr.length);
      ArgusLastRecord++;
   }
}


void
process_tcp (struct ArgusRecord *argus)
{
   RaProcessRecord (argus);
}


void
process_icmp (struct ArgusRecord *argus)
{
   RaProcessRecord (argus);
}


void
process_udp (struct ArgusRecord *argus)
{
   RaProcessRecord (argus);
}


void
process_ip (struct ArgusRecord *argus)
{
   RaProcessRecord (argus);
}


void
process_arp (struct ArgusRecord *argus)
{
   RaProcessRecord (argus);
}


void
process_non_ip (struct ArgusRecord *argus)
{
   RaProcessRecord (argus);
}


void
RaSendArgusRecord(struct ArgusRecordStore *store)
{
   struct ArgusRecord *argus = store->data[0]->argus;

   if (wfile) {

#ifdef _LITTLE_ENDIAN
      ArgusHtoN(argus);
#endif

      ArgusWriteNewLogfile (wfile, argus);

#ifdef _LITTLE_ENDIAN
      ArgusNtoH(argus);
#endif

   } else {

      if (argus->ahdr.type & ARGUS_MAR)
         printf ("%s\n", get_man_string (argus));
         
      else {
         ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);

         switch (argus->ahdr.status & 0xFFFF) {
            case ETHERTYPE_IP:
               switch (argus->argus_far.flow.ip_flow.ip_p) {
                  case IPPROTO_TCP:              
                     printf ("%s\n", get_tcp_string (argus));
                     break;
   
                  case IPPROTO_ICMP:              
                     printf ("%s\n", get_icmp_string (argus));
                     break;
   
                  default:
                     printf ("%s\n", get_ip_string (argus));
                     break;
               }
               break;
   
            case ETHERTYPE_ARP:
            case ETHERTYPE_REVARP:
               printf ("%s\n", get_arp_string (argus));
               break;
   
            default:
               printf ("%s\n", get_nonip_string (argus));
               break;
         }
      }
      fflush (stdout);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaSendArgusRecord(0x%x) done.\n", store);
#endif
}

