/*
 * Copyright (c) 2000 QoSient, LLC
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of QoSient not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * QOSIENT, LLC DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS, IN NO EVENT SHALL QOSIENT, LLC BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

/*
 * Copyright (c) 1990, 1991, 1992, 1993
 *   The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from the Stanford/CMU enet packet filter,
 * (net/enet.c) distributed as part of 4.3BSD, and code contributed
 * to Berkeley by Steven McCanne and Van Jacobson both of Lawrence 
 * Berkeley Laboratory.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *   This product includes software developed by the University of
 *   California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *   @(#)bpf.c   7.5 (Berkeley) 7/15/91
 *
 */

#ifndef ArgusFilter
#define ArgusFilter
#endif

#include <stdlib.h>
#include <unistd.h>

#include <errno.h>
#include <string.h>

#include <sys/file.h>
#include <sys/stat.h>
#include <sys/ioctl.h>

#if defined(HAVE_SOLARIS)
#include <fcntl.h>
#endif

#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>

#include <compat.h>
#include <pcap.h>
#include <interface.h>

#include <argus_parse.h>
#include <argus_out.h>

#include <sys/param.h>
#include <sys/time.h>
#include <net/bpf.h>

#include <argus_filter.h>

#ifdef sun
#include <netinet/in.h>
#endif

#include <compat.h>
#include <ethertype.h>


#if defined(sparc) || defined(mips) || defined(ibm032) || defined(__alpha)\
    || defined(AIX)
#define BPF_ALIGN
#endif

#ifndef BPF_ALIGN
#define EXTRACT_SHORT(p)   ((arg_uint16)ntohs(*(arg_uint16 *)p))
#define EXTRACT_LONG(p)      (ntohl(*(unsigned int *)p))
#else
#define EXTRACT_SHORT(p)\
   ((arg_uint16)\
      ((arg_uint16)*((u_char *)p+0)<<8|\
       (arg_uint16)*((u_char *)p+1)<<0))
#define EXTRACT_LONG(p)\
      ((unsigned int)*((u_char *)p+0)<<24|\
       (unsigned int)*((u_char *)p+1)<<16|\
       (unsigned int)*((u_char *)p+2)<<8|\
       (unsigned int)*((u_char *)p+3)<<0)
#endif

#include <ctype.h>

/* Hex digit to integer. */
static inline int
xdtoi(int c)
{
   if (isdigit(c))
      return c - '0';
   else if (islower(c))
      return c - 'a' + 10;
   else
      return c - 'A' + 10;
}

/*
 * Execute the filter program starting at pc on the packet p
 * wirelen is the length of the original packet
 * buflen is the amount of data present
 */


unsigned int
argus_filter (struct bpf_insn *pc, u_char *p)
{
   return argus_filter_orig (pc, p, sizeof(struct ArgusCanonicalRecord), sizeof(struct ArgusCanonicalRecord));
}

unsigned int
argus_filter_orig (pc, p, wirelen, buflen) 
struct bpf_insn *pc;
u_char *p;
int wirelen, buflen;
{
   unsigned int A = 0, X = 0;
   int k;
   int mem [BPF_MEMWORDS];

   if (pc == 0)
      /*
       * No filter means accept all.
       */
      return (unsigned int) -1;

   --pc;
   while (1) {
      ++pc;
      switch (pc->code) {

      default:
#ifdef KERNEL
         return 0;
#else
         abort();
#endif         
      case BPF_RET|BPF_K:
         return (unsigned int)pc->k;

      case BPF_RET|BPF_A:
         return (unsigned int)A;

      case BPF_LD|BPF_W|BPF_ABS:
         k = pc->k;
         if (k + sizeof(int) > buflen) {
#ifdef KERNEL
            int merr;

            if (buflen != 0)
               return 0;
            A = m_xword((struct mbuf *)p, k, &merr);
            if (merr != 0)
               return 0;
            continue;
#else
            return 0;
#endif
         }
         A = EXTRACT_LONG(&p[k]);
         continue;

      case BPF_LD|BPF_H|BPF_ABS:
         k = pc->k;
         if (k + sizeof(short) > buflen) {
#ifdef KERNEL
            int merr;

            if (buflen != 0)
               return 0;
            A = m_xhalf((struct mbuf *)p, k, &merr);
            continue;
#else
            return 0;
#endif
         }
         A = EXTRACT_SHORT(&p[k]);
         continue;

      case BPF_LD|BPF_B|BPF_ABS:
         k = pc->k;
         if (k >= buflen) {
#ifdef KERNEL
            struct mbuf *m;

            if (buflen != 0)
               return 0;
            m = (struct mbuf *)p;
            MINDEX(m, k);
            A = mtod(m, u_char *)[k];
            continue;
#else
            return 0;
#endif
         }
         A = p[k];
         continue;

      case BPF_LD|BPF_W|BPF_LEN:
         A = wirelen;
         continue;

      case BPF_LDX|BPF_W|BPF_LEN:
         X = wirelen;
         continue;

      case BPF_LD|BPF_W|BPF_IND:
         k = X + pc->k;
         if (k + sizeof(int) > buflen) {
#ifdef KERNEL
            int merr;

            if (buflen != 0)
               return 0;
            A = m_xword((struct mbuf *)p, k, &merr);
            if (merr != 0)
               return 0;
            continue;
#else
            return 0;
#endif
         }
         A = EXTRACT_LONG(&p[k]);
         continue;

      case BPF_LD|BPF_H|BPF_IND:
         k = X + pc->k;
         if (k + sizeof(short) > buflen) {
#ifdef KERNEL
            int merr;

            if (buflen != 0)
               return 0;
            A = m_xhalf((struct mbuf *)p, k, &merr);
            if (merr != 0)
               return 0;
            continue;
#else
            return 0;
#endif
         }
         A = EXTRACT_SHORT(&p[k]);
         continue;

      case BPF_LD|BPF_B|BPF_IND:
         k = X + pc->k;
         if (k >= buflen) {
#ifdef KERNEL
            struct mbuf *m;

            if (buflen != 0)
               return 0;
            m = (struct mbuf *)p;
            MINDEX(m, k);
            A = mtod(m, char *)[k];
            continue;
#else
            return 0;
#endif
         }
         A = p[k];
         continue;

      case BPF_LDX|BPF_MSH|BPF_B:
         k = pc->k;
         if (k >= buflen) {
#ifdef KERNEL
            struct mbuf *m;

            if (buflen != 0)
               return 0;
            m = (struct mbuf *)p;
            MINDEX(m, k);
            X = (mtod(m, char *)[k] & 0xf) << 2;
            continue;
#else
            return 0;
#endif
         }
         X = (p[pc->k] & 0xf) << 2;
         continue;

      case BPF_LD|BPF_IMM:
         A = pc->k;
         continue;

      case BPF_LDX|BPF_IMM:
         X = pc->k;
         continue;

      case BPF_LD|BPF_MEM:
         A = mem[pc->k];
         continue;
         
      case BPF_LDX|BPF_MEM:
         X = mem[pc->k];
         continue;

      case BPF_ST:
         mem[pc->k] = A;
         continue;

      case BPF_STX:
         mem[pc->k] = X;
         continue;

      case BPF_JMP|BPF_JA:
         pc += pc->k;
         continue;

      case BPF_JMP|BPF_JGT|BPF_K:
         pc += (A > pc->k) ? pc->jt : pc->jf;
         continue;

      case BPF_JMP|BPF_JGE|BPF_K:
         pc += (A >= pc->k) ? pc->jt : pc->jf;
         continue;

      case BPF_JMP|BPF_JEQ|BPF_K:
         pc += (A == pc->k) ? pc->jt : pc->jf;
         continue;

      case BPF_JMP|BPF_JSET|BPF_K:
         pc += (A & pc->k) ? pc->jt : pc->jf;
         continue;

      case BPF_JMP|BPF_JGT|BPF_X:
         pc += (A > X) ? pc->jt : pc->jf;
         continue;

      case BPF_JMP|BPF_JGE|BPF_X:
         pc += (A >= X) ? pc->jt : pc->jf;
         continue;

      case BPF_JMP|BPF_JEQ|BPF_X:
         pc += (A == X) ? pc->jt : pc->jf;
         continue;

      case BPF_JMP|BPF_JSET|BPF_X:
         pc += (A & X) ? pc->jt : pc->jf;
         continue;

      case BPF_ALU|BPF_ADD|BPF_X:
         A += X;
         continue;
         
      case BPF_ALU|BPF_SUB|BPF_X:
         A -= X;
         continue;
         
      case BPF_ALU|BPF_MUL|BPF_X:
         A *= X;
         continue;
         
      case BPF_ALU|BPF_DIV|BPF_X:
         if (X == 0)
            return 0;
         A /= X;
         continue;
         
      case BPF_ALU|BPF_AND|BPF_X:
         A &= X;
         continue;
         
      case BPF_ALU|BPF_OR|BPF_X:
         A |= X;
         continue;

      case BPF_ALU|BPF_LSH|BPF_X:
         A <<= X;
         continue;

      case BPF_ALU|BPF_RSH|BPF_X:
         A >>= X;
         continue;

      case BPF_ALU|BPF_ADD|BPF_K:
         A += pc->k;
         continue;
         
      case BPF_ALU|BPF_SUB|BPF_K:
         A -= pc->k;
         continue;
         
      case BPF_ALU|BPF_MUL|BPF_K:
         A *= pc->k;
         continue;
         
      case BPF_ALU|BPF_DIV|BPF_K:
         A /= pc->k;
         continue;
         
      case BPF_ALU|BPF_AND|BPF_K:
         A &= pc->k;
         continue;
         
      case BPF_ALU|BPF_OR|BPF_K:
         A |= pc->k;
         continue;

      case BPF_ALU|BPF_LSH|BPF_K:
         A <<= pc->k;
         continue;

      case BPF_ALU|BPF_RSH|BPF_K:
         A >>= pc->k;
         continue;

      case BPF_ALU|BPF_NEG:
         A = -A;
         continue;

      case BPF_MISC|BPF_TAX:
         X = A;
         continue;

      case BPF_MISC|BPF_TXA:
         A = X;
         continue;
      }
   }
}



#ifdef KERNEL
/*
 * Return true if the 'fcode' is a valid filter program.
 * The constraints are that each jump be forward and to a valid
 * code.  The code must terminate with either an accept or reject. 
 * 'valid' is an array for use by the routine (it must be at least
 * 'len' bytes long).  
 *
 * The kernel needs to be able to verify an application's filter code.
 * Otherwise, a bogus program could easily crash the system.
 */
int
bpf_validate(f, len)
struct bpf_insn *f;
int len;
{
   int i;
   struct bpf_insn *p;

   for (i = 0; i < len; ++i) {
      /*
       * Check that that jumps are forward, and within 
       * the code block.
       */
      p = &f[i];
      if (BPF_CLASS(p->code) == BPF_JMP) {
         int from = i + 1;

         if (BPF_OP(p->code) == BPF_JA) {
            if (from + p->k >= len)
               return 0;
         }
         else if (from + p->jt >= len || from + p->jf >= len)
            return 0;
      }
      /*
       * Check that memory operations use valid addresses.
       */
      if ((BPF_CLASS(p->code) == BPF_ST ||
           (BPF_CLASS(p->code) == BPF_LD && 
            (p->code & 0xe0) == BPF_MEM)) &&
          (p->k >= BPF_MEMWORDS || p->k < 0))
         return 0;
      /*
       * Check for constant division by 0.
       */
      if (p->code == (BPF_ALU|BPF_DIV|BPF_K) && p->k == 0)
         return 0;
   }
   return BPF_CLASS(f[len - 1].code) == BPF_RET;
}
#endif



/*
 * Copyright (c) 1990, 1993, 1994
 *   The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code distributions
 * retain the above copyright notice and this paragraph in its entirety, (2)
 * distributions including binary code include the above copyright notice and
 * this paragraph in its entirety in the documentation or other materials
 * provided with the distribution, and (3) all advertising materials mentioning
 * features or use of this software display the following acknowledgement:
 * ``This product includes software developed by the University of California,
 * Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
 * the University nor the names of its contributors may be used to endorse
 * or promote products derived from this software without specific prior
 * written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */


/*
#include <argus-namedb.h>
*/

#ifndef __GNUC__
#define inline
#endif

static inline int skip_space(FILE *);
static inline int skip_line(FILE *);

static inline int
skip_space(f)
FILE *f;
{
   int c;

   do {
      c = getc(f);
   } while (isspace(c) && c != '\n');

   return c;
}

static inline int
skip_line(f)
   FILE *f;
{
   int c;

   do
      c = getc(f);
   while (c != '\n' && c != EOF);

   return c;
}


#include <argus-namedb.h>

struct argus_etherent *
argus_next_etherent(FILE *fp)
{
   int c, d, i;
   char *bp;
   static struct argus_etherent e;
   static int nline = 1;
 top:
   while (nline) {
      /* Find addr */
      c = skip_space(fp);
      if (c == '\n')
         continue;
      /* If this is a comment, or first thing on line
         cannot be ethernet address, skip the line. */
      else if (!isxdigit(c))
         c = skip_line(fp);
      else {
         /* must be the start of an address */
         for (i = 0; i < 6; i += 1) {
            d = xdtoi(c);
            c = getc(fp);
            if (c != ':') {
               d <<= 4;
               d |= xdtoi(c);
               c = getc(fp);
            }
            e.addr[i] = d;
            if (c != ':')
               break;
            c = getc(fp);
         }
         nline = 0;
      }
      if (c == EOF)
         return 0;
   }

   /* If we started a new line, 'c' holds the char past the ether addr,
      which we assume is white space.  If we are continuing a line,
      'c' is garbage.  In either case, we can throw it away. */

   c = skip_space(fp);
   if (c == '\n') {
      nline = 1;
      goto top;
   }
   else if (c == '#') {
      (void)skip_line(fp);
      nline = 1;
      goto top;
   }
   else if (c == EOF)
      return 0;

   /* Must be a name. */
   bp = e.name;
   /* Use 'd' to prevent argus_strbuffer overflow. */
   d = sizeof(e.name) - 1;
   do {
      *bp++ = c;
      c = getc(fp);
   } while (!isspace(c) && c != EOF && --d > 0);
   *bp = '\0';
   if (c == '\n')
      nline = 1;

   return &e;
}


/*
 * Copyright (c) 1994
 *   The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *   This product includes software developed by the Computer Systems
 *   Engineering Group at Lawrence Berkeley Laboratory.
 * 4. Neither the name of the University nor of the Laboratory may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */


#ifdef HAVE_SOLARIS
#include <sys/sockio.h>
#endif

/* Not all systems have IFF_LOOPBACK */
#ifdef IFF_LOOPBACK
#define ISLOOPBACK(p) ((p)->ifr_flags & IFF_LOOPBACK)
#else
#define ISLOOPBACK(p) (strcmp((p)->ifr_name, "lo0") == 0)
#endif

#if !defined(__OpenBSD__)
#include <net/if.h>
#endif

/*
 * Return the name of a network interface attached to the system, or NULL
 * if none can be found.  The interface must be configured up; the
 * lowest unit number is preferred; loopback is ignored.
 */
char *
argus_lookupdev(ebuf)
char *ebuf;
{
   int fd, minunit, n;
   char *cp;
   struct ifreq *ifrp, *ifend, *ifnext, *mp;
   struct ifconf ifc;
   struct ifreq ibuf[16], ifr;
   static char device[sizeof(ifrp->ifr_name) + 1];

   fd = socket(AF_INET, SOCK_DGRAM, 0);
   if (fd < 0) {
      (void)sprintf(ebuf, "socket: %s", argus_strerror(errno));
      return (NULL);
   }
   ifc.ifc_len = sizeof ibuf;
   ifc.ifc_buf = (caddr_t)ibuf;

   if (ioctl(fd, SIOCGIFCONF, (char *)&ifc) < 0 ||
       ifc.ifc_len < sizeof(struct ifreq)) {
      (void)sprintf(ebuf, "SIOCGIFCONF: %s", argus_strerror(errno));
      (void)close(fd);
      return (NULL);
   }
   ifrp = ibuf;
   ifend = (struct ifreq *)((char *)ibuf + ifc.ifc_len);

   mp = NULL;
   minunit = 666;
   for (; ifrp < ifend; ifrp = ifnext) {
#if BSD - 0 >= 199006
      n = ifrp->ifr_addr.sa_len + sizeof(ifrp->ifr_name);
      if (n < sizeof(*ifrp))
         ifnext = ifrp + 1;
      else
         ifnext = (struct ifreq *)((char *)ifrp + n);
      if (ifrp->ifr_addr.sa_family != AF_INET)
         continue;
#else
      ifnext = ifrp + 1;
#endif
      /*
       * Need a template to preserve address info that is
       * used below to locate the next entry.  (Otherwise,
       * SIOCGIFFLAGS stomps over it because the requests
       * are returned in a union.)
       */
      strncpy(ifr.ifr_name, ifrp->ifr_name, sizeof(ifr.ifr_name));
      if (ioctl(fd, SIOCGIFFLAGS, (char *)&ifr) < 0) {
         (void)sprintf(ebuf, "SIOCGIFFLAGS: %s",
             argus_strerror(errno));
         (void)close(fd);
         return (NULL);
      }

      /* Must be up and not the loopback */
      if ((ifr.ifr_flags & IFF_UP) == 0 || ISLOOPBACK(&ifr))
         continue;

      for (cp = ifrp->ifr_name; !isdigit((int)*cp); ++cp)
         continue;
      n = atoi(cp);
      if (n < minunit) {
         minunit = n;
         mp = ifrp;
      }
   }
   (void)close(fd);
   if (mp == NULL) {
      (void)strcpy(ebuf, "no suitable device found");
      return (NULL);
   }

   (void)strncpy(device, mp->ifr_name, sizeof(device) - 1);
   device[sizeof(device) - 1] = '\0';
   return (device);
}

int
argus_lookupnet(char *device, unsigned int *netp, unsigned int *maskp, char *ebuf)
{
   int fd;
   struct sockaddr_in *sin;
   struct ifreq ifr;

   fd = socket(AF_INET, SOCK_DGRAM, 0);
   if (fd < 0) {
      (void)sprintf(ebuf, "socket: %s", argus_strerror(errno));
      return (-1);
   }
   (void)strncpy(ifr.ifr_name, device, sizeof(ifr.ifr_name));
   if (ioctl(fd, SIOCGIFADDR, (char *)&ifr) < 0) {
      (void)sprintf(ebuf, "SIOCGIFADDR: %s: %s",
          device, argus_strerror(errno));
      (void)close(fd);
      return (-1);
   }
   sin = (struct sockaddr_in *)&ifr.ifr_addr;
   *netp = sin->sin_addr.s_addr;
   if (ioctl(fd, SIOCGIFNETMASK, (char *)&ifr) < 0) {
      (void)sprintf(ebuf, "SIOCGIFNETMASK: %s: %s",
          device, argus_strerror(errno));
      (void)close(fd);
      return (-1);
   }
   (void)close(fd);
   *maskp = sin->sin_addr.s_addr;
   if (*maskp == 0) {
      if (IN_CLASSA(*netp))
         *maskp = IN_CLASSA_NET;
      else if (IN_CLASSB(*netp))
         *maskp = IN_CLASSB_NET;
      else if (IN_CLASSC(*netp))
         *maskp = IN_CLASSC_NET;
      else {
         (void)sprintf(ebuf, "inet class for 0x%x unknown",
             *netp);
         return (-1);
      }
   }
   *netp &= *maskp;
   return (0);
}

#ifndef HAVE_SYS_ERRLIST
static char argus_error[20];
#endif


char *
argus_strerror(int errnum)
{
#ifndef HAVE_SYS_ERRLIST
   extern int sys_nerr;
   extern char *sys_errlist[];

   if ((unsigned int)errnum < sys_nerr)
      return ((char *) sys_errlist[errnum]);

   (void)sprintf(argus_error, "Unknown error: %d", errnum);
   return (argus_error);
#else
   return (strerror(errnum));
#endif
}


/*
 * Copyright (c) 1988, 1989, 1990, 1991, 1993, 1994
 *   The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code distributions
 * retain the above copyright notice and this paragraph in its entirety, (2)
 * distributions including binary code include the above copyright notice and
 * this paragraph in its entirety in the documentation or other materials
 * provided with the distribution, and (3) all advertising materials mentioning
 * features or use of this software display the following acknowledgement:
 * ``This product includes software developed by the University of California,
 * Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
 * the University nor the names of its contributors may be used to endorse
 * or promote products derived from this software without specific prior
 * written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 *  Optimization module for tcpdump intermediate representation.
 */


#ifdef __osf__
#include <malloc.h>
#endif
#include <memory.h>


#ifdef linux
#include <values.h>
#endif

#define BITS_PER_WORD (8*sizeof(unsigned int))

#define A_ATOM BPF_MEMWORDS
#define X_ATOM (BPF_MEMWORDS+1)

#define NOP -1

/*
 * This define is used to represent *both* the accumulator and
 * x register in use-def computations.
 * Currently, the use-def code assumes only one definition per instruction.
 */
#define AX_ATOM N_ATOMS

/*
 * A flag to indicate that further optimization is needed.
 * Iterative passes are continued until a given pass yields no
 * branch movement.
 */
static int done;

/*
 * A block is marked if only if its mark equals the current mark.
 * Rather than traverse the code array, marking each item, 'cur_mark' is
 * incremented.  This automatically makes each element unmarked.
 */
static int cur_mark;
#define isMarked(p) ((p)->mark == cur_mark)
#define unMarkAll() cur_mark += 1
#define Mark(p) ((p)->mark = cur_mark)

static void opt_init(struct block *);
static void opt_cleanup(void);

static void make_marks(struct block *);
static void mark_code(struct block *);

static void intern_blocks(struct block *);

static int eq_slist(struct slist *, struct slist *);

static void find_levels_r(struct block *);

static void find_levels(struct block *);
static void find_dom(struct block *);
static void propedom(struct edge *);
static void find_edom(struct block *);
static void find_closure(struct block *);
static int atomuse(struct stmt *);
static int atomdef(struct stmt *);
static void compute_local_ud(struct block *);
static void find_ud(struct block *);
static void init_val(void);
static int F(int, int, int);
static inline void vstore(struct stmt *, int *, int, int);
static void opt_blk(struct block *, int);
static int use_conflict(struct block *, struct block *);
static void opt_j(struct edge *);
static void or_pullup(struct block *);
static void and_pullup(struct block *);
static void opt_blks(struct block *, int);
static inline void link_inedge(struct edge *, struct block *);
static void find_inedges(struct block *);
static void opt_root(struct block **);
static void opt_loop(struct block *, int);
static void fold_op(struct stmt *, int, int);
static inline struct slist *this_op(struct slist *);
static void opt_not(struct block *);
static void opt_peep(struct block *);
static void opt_stmt(struct stmt *, int[], int);
static void deadstmt(struct stmt *, struct stmt *[]);
static void opt_deadstores(struct block *);
static void opt_blk(struct block *, int);
static int use_conflict(struct block *, struct block *);
static void opt_j(struct edge *);
static struct block *fold_edge(struct block *, struct edge *);
static inline int eq_blk(struct block *, struct block *);
static int slength(struct slist *);
static int count_blocks(struct block *);
static void number_blks_r(struct block *);
static int count_stmts(struct block *);
static int convert_code_r(struct block *);

static int n_blocks;
struct block **blocks;
static int n_edges;
struct edge **edges;

/*
 * A bit vector set representation of the dominators.
 * We round up the set size to the next power of two.
 */
static int nodewords;
static int edgewords;
struct block **levels;
unsigned int *space;

/*
 * True if a is in uset {p}
 */
#define SET_MEMBER(p, a) \
((p)[(a) / BITS_PER_WORD] & (1 << ((a) % BITS_PER_WORD)))

/*
 * Add 'a' to uset p.
 */
#define SET_INSERT(p, a) \
(p)[(a) / BITS_PER_WORD] |= (1 << ((a) % BITS_PER_WORD))

/*
 * Delete 'a' from uset p.
 */
#define SET_DELETE(p, a) \
(p)[(a) / BITS_PER_WORD] &= ~(1 << ((a) % BITS_PER_WORD))

/*
 * a := a intersect b
 */
#define SET_INTERSECT(a, b, n)\
{\
   register unsigned int *_x = a, *_y = b;\
   register int _n = n;\
   while (--_n >= 0) *_x++ &= *_y++;\
}

/*
 * a := a - b
 */
#define SET_SUBTRACT(a, b, n)\
{\
   register unsigned int *_x = a, *_y = b;\
   register int _n = n;\
   while (--_n >= 0) *_x++ &=~ *_y++;\
}

/*
 * a := a union b
 */
#define SET_UNION(a, b, n)\
{\
   register unsigned int *_x = a, *_y = b;\
   register int _n = n;\
   while (--_n >= 0) *_x++ |= *_y++;\
}

static uset all_dom_sets;
static uset all_closure_sets;
static uset all_edge_sets;

#ifndef MAX
#define MAX(a,b) ((a)>(b)?(a):(b))
#endif

static void
find_levels_r(b)
   struct block *b;
{
   int level;

   if (isMarked(b))
      return;

   Mark(b);
   b->link = 0;

   if (JT(b)) {
      find_levels_r(JT(b));
      find_levels_r(JF(b));
      level = MAX(JT(b)->level, JF(b)->level) + 1;
   } else
      level = 0;
   b->level = level;
   b->link = levels[level];
   levels[level] = b;
}

/*
 * Level graph.  The levels go from 0 at the leaves to
 * N_LEVELS at the root.  The levels[] array points to the
 * first node of the level list, whose elements are linked
 * with the 'link' field of the struct block.
 */
static void
find_levels(root)
   struct block *root;
{
   memset((char *)levels, 0, n_blocks * sizeof(*levels));
   unMarkAll();
   find_levels_r(root);
}

/*
 * Find dominator relationships.
 * Assumes graph has been leveled.
 */
static void
find_dom(root)
struct block *root;
{
   int i;
   struct block *b;
   unsigned int *x;

   /*
    * Initialize sets to contain all nodes.
    */
   x = all_dom_sets;
   i = n_blocks * nodewords;
   while (--i >= 0)
      *x++ = ~0;
   /* Root starts off empty. */
   for (i = nodewords; --i >= 0;)
      root->dom[i] = 0;

   /* root->level is the highest level no found. */
   for (i = root->level; i >= 0; --i) {
      for (b = levels[i]; b; b = b->link) {
         SET_INSERT(b->dom, b->id);
         if (JT(b) == 0)
            continue;
         SET_INTERSECT(JT(b)->dom, b->dom, nodewords);
         SET_INTERSECT(JF(b)->dom, b->dom, nodewords);
      }
   }
}

static void
propedom(ep)
   struct edge *ep;
{
   SET_INSERT(ep->edom, ep->id);
   if (ep->succ) {
      SET_INTERSECT(ep->succ->et.edom, ep->edom, edgewords);
      SET_INTERSECT(ep->succ->ef.edom, ep->edom, edgewords);
   }
}

/*
 * Compute edge dominators.
 * Assumes graph has been leveled and predecessors established.
 */
static void
find_edom(root)
struct block *root;
{
   int i;
   uset x;
   struct block *b;

   x = all_edge_sets;
   for (i = n_edges * edgewords; --i >= 0; )
      x[i] = ~0;

   /* root->level is the highest level no found. */
   memset(root->et.edom, 0, edgewords * sizeof(*(uset)0));
   memset(root->ef.edom, 0, edgewords * sizeof(*(uset)0));
   for (i = root->level; i >= 0; --i) {
      for (b = levels[i]; b != 0; b = b->link) {
         propedom(&b->et);
         propedom(&b->ef);
      }
   }
}

/*
 * Find the backwards transitive closure of the flow graph.  These sets
 * are backwards in the sense that we find the set of nodes that reach
 * a given node, not the set of nodes that can be reached by a node.
 *
 * Assumes graph has been leveled.
 */
static void
find_closure(root)
struct block *root;
{
   int i;
   struct block *b;

   /*
    * Initialize sets to contain no nodes.
    */
   memset((char *)all_closure_sets, 0,
         n_blocks * nodewords * sizeof(*all_closure_sets));

   /* root->level is the highest level no found. */
   for (i = root->level; i >= 0; --i) {
      for (b = levels[i]; b; b = b->link) {
         SET_INSERT(b->closure, b->id);
         if (JT(b) == 0)
            continue;
         SET_UNION(JT(b)->closure, b->closure, nodewords);
         SET_UNION(JF(b)->closure, b->closure, nodewords);
      }
   }
}

/*
 * Return the register number that is used by s.  If A and X are both
 * used, return AX_ATOM.  If no register is used, return -1.
 *
 * The implementation should probably change to an array access.
 */
static int
atomuse(s)
   struct stmt *s;
{
   register int c = s->code;

   if (c == NOP)
      return -1;

   switch (BPF_CLASS(c)) {

   case BPF_RET:
      return (BPF_RVAL(c) == BPF_A) ? A_ATOM :
         (BPF_RVAL(c) == BPF_X) ? X_ATOM : -1;

   case BPF_LD:
   case BPF_LDX:
      return (BPF_MODE(c) == BPF_IND) ? X_ATOM :
         (BPF_MODE(c) == BPF_MEM) ? s->k : -1;

   case BPF_ST:
      return A_ATOM;

   case BPF_STX:
      return X_ATOM;

   case BPF_JMP:
   case BPF_ALU:
      if (BPF_SRC(c) == BPF_X)
         return AX_ATOM;
      return A_ATOM;

   case BPF_MISC:
      return BPF_MISCOP(c) == BPF_TXA ? X_ATOM : A_ATOM;
   }
   abort();
   /* NOTREACHED */
}

/*
 * Return the register number that is defined by 's'.  We assume that
 * a single stmt cannot define more than one register.  If no register
 * is defined, return -1.
 *
 * The implementation should probably change to an array access.
 */
static int
atomdef(s)
   struct stmt *s;
{
   if (s->code == NOP)
      return -1;

   switch (BPF_CLASS(s->code)) {

   case BPF_LD:
   case BPF_ALU:
      return A_ATOM;

   case BPF_LDX:
      return X_ATOM;

   case BPF_ST:
   case BPF_STX:
      return s->k;

   case BPF_MISC:
      return BPF_MISCOP(s->code) == BPF_TAX ? X_ATOM : A_ATOM;
   }
   return -1;
}

static void
compute_local_ud(b)
   struct block *b;
{
   struct slist *s;
   atomset def = 0, use = 0, kill = 0;
   int atom;

   for (s = b->stmts; s; s = s->next) {
      if (s->s.code == NOP)
         continue;
      atom = atomuse(&s->s);
      if (atom >= 0) {
         if (atom == AX_ATOM) {
            if (!ATOMELEM(def, X_ATOM))
               use |= ATOMMASK(X_ATOM);
            if (!ATOMELEM(def, A_ATOM))
               use |= ATOMMASK(A_ATOM);
         }
         else if (atom < N_ATOMS) {
            if (!ATOMELEM(def, atom))
               use |= ATOMMASK(atom);
         }
         else
            abort();
      }
      atom = atomdef(&s->s);
      if (atom >= 0) {
         if (!ATOMELEM(use, atom))
            kill |= ATOMMASK(atom);
         def |= ATOMMASK(atom);
      }
   }
   if (!ATOMELEM(def, A_ATOM) && BPF_CLASS(b->s.code) == BPF_JMP)
      use |= ATOMMASK(A_ATOM);

   b->def = def;
   b->kill = kill;
   b->in_use = use;
}

/*
 * Assume graph is already leveled.
 */
static void
find_ud(root)
   struct block *root;
{
   int i, maxlevel;
   struct block *p;

   /*
    * root->level is the highest level no found;
    * count down from there.
    */
   maxlevel = root->level;
   for (i = maxlevel; i >= 0; --i)
      for (p = levels[i]; p; p = p->link) {
         compute_local_ud(p);
         p->out_use = 0;
      }

   for (i = 1; i <= maxlevel; ++i) {
      for (p = levels[i]; p; p = p->link) {
         p->out_use |= JT(p)->in_use | JF(p)->in_use;
         p->in_use |= p->out_use &~ p->kill;
      }
   }
}

/*
 * These data structures are used in a Cocke and Shwarz style
 * value numbering scheme.  Since the flowgraph is acyclic,
 * exit values can be propagated from a node's predecessors
 * provided it is uniquely defined.
 */
struct valnode {
   int code;
   int v0, v1;
   int val;
   struct valnode *next;
};

#define MODULUS 213
static struct valnode *hashtbl[MODULUS];
static int curval;
static int maxval;

/* Integer constants mapped with the load immediate opcode. */
#define K(i) F(BPF_LD|BPF_IMM|BPF_W, i, 0L)

struct vmapinfo {
   int is_const;
   int const_val;
};

struct vmapinfo *vmap;
struct valnode *vnode_base;
struct valnode *next_vnode;

static void
init_val()
{
   curval = 0;
   next_vnode = vnode_base;
   memset((char *)vmap, 0, maxval * sizeof(*vmap));
   memset((char *)hashtbl, 0, sizeof hashtbl);
}

/* Because we really don't have an IR, this stuff is a little messy. */
static int
F(code, v0, v1)
int code;
int v0, v1;
{
   u_int hash;
   int val;
   struct valnode *p;

   hash = (u_int)code ^ (v0 << 4) ^ (v1 << 8);
   hash %= MODULUS;

   for (p = hashtbl[hash]; p; p = p->next)
      if (p->code == code && p->v0 == v0 && p->v1 == v1)
         return p->val;

   val = ++curval;
   if (BPF_MODE(code) == BPF_IMM &&
       (BPF_CLASS(code) == BPF_LD || BPF_CLASS(code) == BPF_LDX)) {
      vmap[val].const_val = v0;
      vmap[val].is_const = 1;
   }
   p = next_vnode++;
   p->val = val;
   p->code = code;
   p->v0 = v0;
   p->v1 = v1;
   p->next = hashtbl[hash];
   hashtbl[hash] = p;

   return val;
}

static inline void
vstore(s, valp, newval, alter)
struct stmt *s;
int *valp;
int newval;
int alter;
{
   if (alter && *valp == newval)
      s->code = NOP;
   else
      *valp = newval;
}

static void
fold_op(s, v0, v1)
struct stmt *s;
int v0, v1;
{
   int a, b;

   a = vmap[v0].const_val;
   b = vmap[v1].const_val;

   switch (BPF_OP(s->code)) {
   case BPF_ADD:
      a += b;
      break;

   case BPF_SUB:
      a -= b;
      break;

   case BPF_MUL:
      a *= b;
      break;

   case BPF_DIV:
      if (b == 0)
         Argus_error("division by zero");
      a /= b;
      break;

   case BPF_AND:
      a &= b;
      break;

   case BPF_OR:
      a |= b;
      break;

   case BPF_LSH:
      a <<= b;
      break;

   case BPF_RSH:
      a >>= b;
      break;

   case BPF_NEG:
      a = -a;
      break;

   default:
      abort();
   }
   s->k = a;
   s->code = BPF_LD|BPF_IMM;
   done = 0;
}

static inline struct slist *
this_op(s)
   struct slist *s;
{
   while (s != 0 && s->s.code == NOP)
      s = s->next;
   return s;
}

static void
opt_not(b)
   struct block *b;
{
   struct block *tmp = JT(b);

   JT(b) = JF(b);
   JF(b) = tmp;
}

static void
opt_peep(b)
struct block *b;
{
   struct slist *s;
   struct slist *next, *last;
   int val;
   int v;

   s = b->stmts;
   if (s == 0)
      return;

   last = s;
   while (1) {
      s = this_op(s);
      if (s == 0)
         break;
      next = this_op(s->next);
      if (next == 0)
         break;
      last = next;

      /*
       * st  M[k]   -->   st  M[k]
       * ldx M[k]      tax
       */
      if (s->s.code == BPF_ST &&
          next->s.code == (BPF_LDX|BPF_MEM) &&
          s->s.k == next->s.k) {
         done = 0;
         next->s.code = BPF_MISC|BPF_TAX;
      }
      /*
       * ld  #k   -->   ldx  #k
       * tax         txa
       */
      if (s->s.code == (BPF_LD|BPF_IMM) &&
          next->s.code == (BPF_MISC|BPF_TAX)) {
         s->s.code = BPF_LDX|BPF_IMM;
         next->s.code = BPF_MISC|BPF_TXA;
         done = 0;
      }
      /*
       * This is an ugly special case, but it happens
       * when you say tcp[k] or udp[k] where k is a constant.
       */
      if (s->s.code == (BPF_LD|BPF_IMM)) {
         struct slist *add, *tax, *ild;

         /*
          * Check that X isn't used on exit from this
          * block (which the optimizer might cause).
          * We know the code generator won't generate
          * any local dependencies.
          */
         if (ATOMELEM(b->out_use, X_ATOM))
            break;

         if (next->s.code != (BPF_LDX|BPF_MSH|BPF_B))
            add = next;
         else
            add = this_op(next->next);
         if (add == 0 || add->s.code != (BPF_ALU|BPF_ADD|BPF_X))
            break;

         tax = this_op(add->next);
         if (tax == 0 || tax->s.code != (BPF_MISC|BPF_TAX))
            break;

         ild = this_op(tax->next);
         if (ild == 0 || BPF_CLASS(ild->s.code) != BPF_LD ||
             BPF_MODE(ild->s.code) != BPF_IND)
            break;
         /*
          * XXX We need to check that X is not
          * subsequently used.  We know we can eliminate the
          * accumulator modifications since it is defined
          * by the last stmt of this sequence.
          *
          * We want to turn this sequence:
          *
          * (004) ldi     #0x2      {s}
          * (005) ldxms   [14]      {next}  -- optional
          * (006) addx         {add}
          * (007) tax         {tax}
          * (008) ild     [x+0]      {ild}
          *
          * into this sequence:
          *
          * (004) nop
          * (005) ldxms   [14]
          * (006) nop
          * (007) nop
          * (008) ild     [x+2]
          *
          */
         ild->s.k += s->s.k;
         s->s.code = NOP;
         add->s.code = NOP;
         tax->s.code = NOP;
         done = 0;
      }
      s = next;
   }
   /*
    * If we have a subtract to do a comparison, and the X register
    * is a known constant, we can merge this value into the
    * comparison.
    */
   if (last->s.code == (BPF_ALU|BPF_SUB|BPF_X) &&
       !ATOMELEM(b->out_use, A_ATOM)) {
      val = b->val[X_ATOM];
      if (vmap[val].is_const) {
         b->s.k += vmap[val].const_val;
         last->s.code = NOP;
         done = 0;
      } else if (b->s.k == 0) {
         /*
          * sub x  ->   nop
          * j  #0   j  x
          */
         last->s.code = NOP;
         b->s.code = BPF_CLASS(b->s.code) | BPF_OP(b->s.code) |
            BPF_X;
         done = 0;
      }
   }
   /*
    * Likewise, a constant subtract can be simplified.
    */
   else if (last->s.code == (BPF_ALU|BPF_SUB|BPF_K) &&
       !ATOMELEM(b->out_use, A_ATOM)) {
      b->s.k += last->s.k;
      last->s.code = NOP;
      done = 0;
   }
   /*
    * and #k   nop
    * jeq #0  ->   jset #k
    */
   if (last->s.code == (BPF_ALU|BPF_AND|BPF_K) &&
       !ATOMELEM(b->out_use, A_ATOM) && b->s.k == 0) {
      b->s.k = last->s.k;
      b->s.code = BPF_JMP|BPF_K|BPF_JSET;
      last->s.code = NOP;
      done = 0;
      opt_not(b);
   }
   /*
    * If the accumulator is a known constant, we can compute the
    * comparison result.
    */
   val = b->val[A_ATOM];
   if (vmap[val].is_const && BPF_SRC(b->s.code) == BPF_K) {
      v = vmap[val].const_val;
      switch (BPF_OP(b->s.code)) {

      case BPF_JEQ:
         v = v == b->s.k;
         break;

      case BPF_JGT:
         v = v > b->s.k;
         break;

      case BPF_JGE:
         v = v >= b->s.k;
         break;

      case BPF_JSET:
         v &= b->s.k;
         break;

      default:
         abort();
      }
      if (JF(b) != JT(b))
         done = 0;
      if (v)
         JF(b) = JT(b);
      else
         JT(b) = JF(b);
   }
}

/*
 * Compute the symbolic value of expression of 's', and update
 * anything it defines in the value table 'val'.  If 'alter' is true,
 * do various optimizations.  This code would be cleaner if symbolic
 * evaluation and code transformations weren't folded together.
 */
static void
opt_stmt(s, val, alter)
struct stmt *s;
int val[];
int alter;
{
   int op;
   int v;

   switch (s->code) {

   case BPF_LD|BPF_ABS|BPF_W:
   case BPF_LD|BPF_ABS|BPF_H:
   case BPF_LD|BPF_ABS|BPF_B:
      v = F(s->code, s->k, 0L);
      vstore(s, &val[A_ATOM], v, alter);
      break;

   case BPF_LD|BPF_IND|BPF_W:
   case BPF_LD|BPF_IND|BPF_H:
   case BPF_LD|BPF_IND|BPF_B:
      v = val[X_ATOM];
      if (alter && vmap[v].is_const) {
         s->code = BPF_LD|BPF_ABS|BPF_SIZE(s->code);
         s->k += vmap[v].const_val;
         v = F(s->code, s->k, 0L);
         done = 0;
      }
      else
         v = F(s->code, s->k, v);
      vstore(s, &val[A_ATOM], v, alter);
      break;

   case BPF_LD|BPF_LEN:
      v = F(s->code, 0L, 0L);
      vstore(s, &val[A_ATOM], v, alter);
      break;

   case BPF_LD|BPF_IMM:
      v = K(s->k);
      vstore(s, &val[A_ATOM], v, alter);
      break;

   case BPF_LDX|BPF_IMM:
      v = K(s->k);
      vstore(s, &val[X_ATOM], v, alter);
      break;

   case BPF_LDX|BPF_MSH|BPF_B:
      v = F(s->code, s->k, 0L);
      vstore(s, &val[X_ATOM], v, alter);
      break;

   case BPF_ALU|BPF_NEG:
      if (alter && vmap[val[A_ATOM]].is_const) {
         s->code = BPF_LD|BPF_IMM;
         s->k = -vmap[val[A_ATOM]].const_val;
         val[A_ATOM] = K(s->k);
      }
      else
         val[A_ATOM] = F(s->code, val[A_ATOM], 0L);
      break;

   case BPF_ALU|BPF_ADD|BPF_K:
   case BPF_ALU|BPF_SUB|BPF_K:
   case BPF_ALU|BPF_MUL|BPF_K:
   case BPF_ALU|BPF_DIV|BPF_K:
   case BPF_ALU|BPF_AND|BPF_K:
   case BPF_ALU|BPF_OR|BPF_K:
   case BPF_ALU|BPF_LSH|BPF_K:
   case BPF_ALU|BPF_RSH|BPF_K:
      op = BPF_OP(s->code);
      if (alter) {
         if (s->k == 0) {
            if (op == BPF_ADD || op == BPF_SUB ||
                op == BPF_LSH || op == BPF_RSH ||
                op == BPF_OR) {
               s->code = NOP;
               break;
            }
            if (op == BPF_MUL || op == BPF_AND) {
               s->code = BPF_LD|BPF_IMM;
               val[A_ATOM] = K(s->k);
               break;
            }
         }
         if (vmap[val[A_ATOM]].is_const) {
            fold_op(s, val[A_ATOM], K(s->k));
            val[A_ATOM] = K(s->k);
            break;
         }
      }
      val[A_ATOM] = F(s->code, val[A_ATOM], K(s->k));
      break;

   case BPF_ALU|BPF_ADD|BPF_X:
   case BPF_ALU|BPF_SUB|BPF_X:
   case BPF_ALU|BPF_MUL|BPF_X:
   case BPF_ALU|BPF_DIV|BPF_X:
   case BPF_ALU|BPF_AND|BPF_X:
   case BPF_ALU|BPF_OR|BPF_X:
   case BPF_ALU|BPF_LSH|BPF_X:
   case BPF_ALU|BPF_RSH|BPF_X:
      op = BPF_OP(s->code);
      if (alter && vmap[val[X_ATOM]].is_const) {
         if (vmap[val[A_ATOM]].is_const) {
            fold_op(s, val[A_ATOM], val[X_ATOM]);
            val[A_ATOM] = K(s->k);
         }
         else {
            s->code = BPF_ALU|BPF_K|op;
            s->k = vmap[val[X_ATOM]].const_val;
            done = 0;
            val[A_ATOM] =
               F(s->code, val[A_ATOM], K(s->k));
         }
         break;
      }
      /*
       * Check if we're doing something to an accumulator
       * that is 0, and simplify.  This may not seem like
       * much of a simplification but it could open up further
       * optimizations.
       * XXX We could also check for mul by 1, and -1, etc.
       */
      if (alter && vmap[val[A_ATOM]].is_const
          && vmap[val[A_ATOM]].const_val == 0) {
         if (op == BPF_ADD || op == BPF_OR ||
             op == BPF_LSH || op == BPF_RSH || op == BPF_SUB) {
            s->code = BPF_MISC|BPF_TXA;
            vstore(s, &val[A_ATOM], val[X_ATOM], alter);
            break;
         }
         else if (op == BPF_MUL || op == BPF_DIV ||
             op == BPF_AND) {
            s->code = BPF_LD|BPF_IMM;
            s->k = 0;
            vstore(s, &val[A_ATOM], K(s->k), alter);
            break;
         }
         else if (op == BPF_NEG) {
            s->code = NOP;
            break;
         }
      }
      val[A_ATOM] = F(s->code, val[A_ATOM], val[X_ATOM]);
      break;

   case BPF_MISC|BPF_TXA:
      vstore(s, &val[A_ATOM], val[X_ATOM], alter);
      break;

   case BPF_LD|BPF_MEM:
      v = val[s->k];
      if (alter && vmap[v].is_const) {
         s->code = BPF_LD|BPF_IMM;
         s->k = vmap[v].const_val;
         done = 0;
      }
      vstore(s, &val[A_ATOM], v, alter);
      break;

   case BPF_MISC|BPF_TAX:
      vstore(s, &val[X_ATOM], val[A_ATOM], alter);
      break;

   case BPF_LDX|BPF_MEM:
      v = val[s->k];
      if (alter && vmap[v].is_const) {
         s->code = BPF_LDX|BPF_IMM;
         s->k = vmap[v].const_val;
         done = 0;
      }
      vstore(s, &val[X_ATOM], v, alter);
      break;

   case BPF_ST:
      vstore(s, &val[s->k], val[A_ATOM], alter);
      break;

   case BPF_STX:
      vstore(s, &val[s->k], val[X_ATOM], alter);
      break;
   }
}

static void
deadstmt(s, last)
   register struct stmt *s;
   register struct stmt *last[];
{
   register int atom;

   atom = atomuse(s);
   if (atom >= 0) {
      if (atom == AX_ATOM) {
         last[X_ATOM] = 0;
         last[A_ATOM] = 0;
      }
      else
         last[atom] = 0;
   }
   atom = atomdef(s);
   if (atom >= 0) {
      if (last[atom]) {
         done = 0;
         last[atom]->code = NOP;
      }
      last[atom] = s;
   }
}

static void
opt_deadstores(b)
   register struct block *b;
{
   register struct slist *s;
   register int atom;
   struct stmt *last[N_ATOMS];

   memset((char *)last, 0, sizeof last);

   for (s = b->stmts; s != 0; s = s->next)
      deadstmt(&s->s, last);
   deadstmt(&b->s, last);

   for (atom = 0; atom < N_ATOMS; ++atom)
      if (last[atom] && !ATOMELEM(b->out_use, atom)) {
         last[atom]->code = NOP;
         done = 0;
      }
}

static void
opt_blk(b, do_stmts)
struct block *b;
int do_stmts;
{
   struct slist *s;
   struct edge *p;
   int i;
   int aval;

   /*
    * Initialize the atom values.
    * If we have no predecessors, everything is undefined.
    * Otherwise, we inherent our values from our predecessors.
    * If any register has an ambiguous value (i.e. control paths are
    * merging) give it the undefined value of 0.
    */
   p = b->in_edges;
   if (p == 0)
      memset((char *)b->val, 0, sizeof(b->val));
   else {
      memcpy((char *)b->val, (char *)p->pred->val, sizeof(b->val));
      while ((p = p->next) != NULL) {
         for (i = 0; i < N_ATOMS; ++i)
            if (b->val[i] != p->pred->val[i])
               b->val[i] = 0;
      }
   }
   aval = b->val[A_ATOM];
   for (s = b->stmts; s; s = s->next)
      opt_stmt(&s->s, b->val, do_stmts);

   /*
    * This is a special case: if we don't use anything from this
    * block, and we load the accumulator with value that is
    * already there, eliminate all the statements.
    */
   if (do_stmts && b->out_use == 0 && aval != 0 &&
       b->val[A_ATOM] == aval)
      b->stmts = 0;
   else {
      opt_peep(b);
      opt_deadstores(b);
   }
   /*
    * Set up values for branch optimizer.
    */
   if (BPF_SRC(b->s.code) == BPF_K)
      b->oval = K(b->s.k);
   else
      b->oval = b->val[X_ATOM];
   b->et.code = b->s.code;
   b->ef.code = -b->s.code;
}

/*
 * Return true if any register that is used on exit from 'succ', has
 * an exit value that is different from the corresponding exit value
 * from 'b'.
 */
static int
use_conflict(b, succ)
   struct block *b, *succ;
{
   int atom;
   atomset use = succ->out_use;

   if (use == 0)
      return 0;

   for (atom = 0; atom < N_ATOMS; ++atom)
      if (ATOMELEM(use, atom))
         if (b->val[atom] != succ->val[atom])
            return 1;
   return 0;
}

static struct block *
fold_edge(child, ep)
   struct block *child;
   struct edge *ep;
{
   int sense;
   int aval0, aval1, oval0, oval1;
   int code = ep->code;

   if (code < 0) {
      code = -code;
      sense = 0;
   } else
      sense = 1;

   if (child->s.code != code)
      return 0;

   aval0 = child->val[A_ATOM];
   oval0 = child->oval;
   aval1 = ep->pred->val[A_ATOM];
   oval1 = ep->pred->oval;

   if (aval0 != aval1)
      return 0;

   if (oval0 == oval1)
      /*
       * The operands are identical, so the
       * result is true if a true branch was
       * taken to get here, otherwise false.
       */
      return sense ? JT(child) : JF(child);

   if (sense && code == (BPF_JMP|BPF_JEQ|BPF_K))
      /*
       * At this point, we only know the comparison if we
       * came down the true branch, and it was an equality
       * comparison with a constant.  We rely on the fact that
       * distinct constants have distinct value numbers.
       */
      return JF(child);

   return 0;
}

static void
opt_j(ep)
   struct edge *ep;
{
   register int i, k;
   register struct block *target;

   if (JT(ep->succ) == 0)
      return;

   if (JT(ep->succ) == JF(ep->succ)) {
      /*
       * Common branch targets can be eliminated, provided
       * there is no data dependency.
       */
      if (!use_conflict(ep->pred, ep->succ->et.succ)) {
         done = 0;
         ep->succ = JT(ep->succ);
      }
   }
   /*
    * For each edge dominator that matches the successor of this
    * edge, promote the edge successor to the its grandchild.
    *
    * XXX We violate the set abstraction here in favor a reasonably
    * efficient loop.
    */
 top:
   for (i = 0; i < edgewords; ++i) {
      register unsigned int x = ep->edom[i];

      while (x != 0) {
         k = ffs(x) - 1;
         x &=~ (1 << k);
         k += i * BITS_PER_WORD;

         target = fold_edge(ep->succ, edges[k]);
         /*
          * Check that there is no data dependency between
          * nodes that will be violated if we move the edge.
          */
         if (target != 0 && !use_conflict(ep->pred, target)) {
            done = 0;
            ep->succ = target;
            if (JT(target) != 0)
               /*
                * Start over unless we hit a leaf.
                */
               goto top;
            return;
         }
      }
   }
}


static void
or_pullup(b)
   struct block *b;
{
   int val, at_top;
   struct block *pull;
   struct block **diffp, **samep;
   struct edge *ep;

   ep = b->in_edges;
   if (ep == 0)
      return;

   /*
    * Make sure each predecessor loads the same value.
    * XXX why?
    */
   val = ep->pred->val[A_ATOM];
   for (ep = ep->next; ep != 0; ep = ep->next)
      if (val != ep->pred->val[A_ATOM])
         return;

   if (JT(b->in_edges->pred) == b)
      diffp = &JT(b->in_edges->pred);
   else
      diffp = &JF(b->in_edges->pred);

   at_top = 1;
   while (1) {
      if (*diffp == 0)
         return;

      if (JT(*diffp) != JT(b))
         return;

      if (!SET_MEMBER((*diffp)->dom, b->id))
         return;

      if ((*diffp)->val[A_ATOM] != val)
         break;

      diffp = &JF(*diffp);
      at_top = 0;
   }
   samep = &JF(*diffp);
   while (1) {
      if (*samep == 0)
         return;

      if (JT(*samep) != JT(b))
         return;

      if (!SET_MEMBER((*samep)->dom, b->id))
         return;

      if ((*samep)->val[A_ATOM] == val)
         break;

      /* XXX Need to check that there are no data dependencies
         between dp0 and dp1.  Currently, the code generator
         will not produce such dependencies. */
      samep = &JF(*samep);
   }
#ifdef notdef
   /* XXX This doesn't cover everything. */
   for (i = 0; i < N_ATOMS; ++i)
      if ((*samep)->val[i] != pred->val[i])
         return;
#endif
   /* Pull up the node. */
   pull = *samep;
   *samep = JF(pull);
   JF(pull) = *diffp;

   /*
    * At the top of the chain, each predecessor needs to point at the
    * pulled up node.  Inside the chain, there is only one predecessor
    * to worry about.
    */
   if (at_top) {
      for (ep = b->in_edges; ep != 0; ep = ep->next) {
         if (JT(ep->pred) == b)
            JT(ep->pred) = pull;
         else
            JF(ep->pred) = pull;
      }
   }
   else
      *diffp = pull;

   done = 0;
}

static void
and_pullup(b)
   struct block *b;
{
   int val, at_top;
   struct block *pull;
   struct block **diffp, **samep;
   struct edge *ep;

   ep = b->in_edges;
   if (ep == 0)
      return;

   /*
    * Make sure each predecessor loads the same value.
    */
   val = ep->pred->val[A_ATOM];
   for (ep = ep->next; ep != 0; ep = ep->next)
      if (val != ep->pred->val[A_ATOM])
         return;

   if (JT(b->in_edges->pred) == b)
      diffp = &JT(b->in_edges->pred);
   else
      diffp = &JF(b->in_edges->pred);

   at_top = 1;
   while (1) {
      if (*diffp == 0)
         return;

      if (JF(*diffp) != JF(b))
         return;

      if (!SET_MEMBER((*diffp)->dom, b->id))
         return;

      if ((*diffp)->val[A_ATOM] != val)
         break;

      diffp = &JT(*diffp);
      at_top = 0;
   }
   samep = &JT(*diffp);
   while (1) {
      if (*samep == 0)
         return;

      if (JF(*samep) != JF(b))
         return;

      if (!SET_MEMBER((*samep)->dom, b->id))
         return;

      if ((*samep)->val[A_ATOM] == val)
         break;

      /* XXX Need to check that there are no data dependencies
         between diffp and samep.  Currently, the code generator
         will not produce such dependencies. */
      samep = &JT(*samep);
   }
#ifdef notdef
   /* XXX This doesn't cover everything. */
   for (i = 0; i < N_ATOMS; ++i)
      if ((*samep)->val[i] != pred->val[i])
         return;
#endif
   /* Pull up the node. */
   pull = *samep;
   *samep = JT(pull);
   JT(pull) = *diffp;

   /*
    * At the top of the chain, each predecessor needs to point at the
    * pulled up node.  Inside the chain, there is only one predecessor
    * to worry about.
    */
   if (at_top) {
      for (ep = b->in_edges; ep != 0; ep = ep->next) {
         if (JT(ep->pred) == b)
            JT(ep->pred) = pull;
         else
            JF(ep->pred) = pull;
      }
   }
   else
      *diffp = pull;

   done = 0;
}

static void
opt_blks(root, do_stmts)
   struct block *root;
   int do_stmts;
{
   int i, maxlevel;
   struct block *p;

   init_val();
   maxlevel = root->level;
   for (i = maxlevel; i >= 0; --i)
      for (p = levels[i]; p; p = p->link)
         opt_blk(p, do_stmts);

   if (do_stmts)
      /*
       * No point trying to move branches; it can't possibly
       * make a difference at this point.
       */
      return;

   for (i = 1; i <= maxlevel; ++i) {
      for (p = levels[i]; p; p = p->link) {
         opt_j(&p->et);
         opt_j(&p->ef);
      }
   }
   for (i = 1; i <= maxlevel; ++i) {
      for (p = levels[i]; p; p = p->link) {
         or_pullup(p);
         and_pullup(p);
      }
   }
}

static inline void
link_inedge(parent, child)
   struct edge *parent;
   struct block *child;
{
   parent->next = child->in_edges;
   child->in_edges = parent;
}

static void
find_inedges(root)
   struct block *root;
{
   int i;
   struct block *b;

   for (i = 0; i < n_blocks; ++i)
      blocks[i]->in_edges = 0;

   /*
    * Traverse the graph, adding each edge to the predecessor
    * list of its successors.  Skip the leaves (i.e. level 0).
    */
   for (i = root->level; i > 0; --i) {
      for (b = levels[i]; b != 0; b = b->link) {
         link_inedge(&b->et, JT(b));
         link_inedge(&b->ef, JF(b));
      }
   }
}

static void
opt_root(b)
   struct block **b;
{
   struct slist *tmp, *s;

   s = (*b)->stmts;
   (*b)->stmts = 0;
   while (BPF_CLASS((*b)->s.code) == BPF_JMP && JT(*b) == JF(*b))
      *b = JT(*b);

   tmp = (*b)->stmts;
   if (tmp != 0)
      Argussappend(s, tmp);
   (*b)->stmts = s;
}

static void
opt_loop(root, do_stmts)
   struct block *root;
   int do_stmts;
{

#ifdef BDEBUG
   if (dflag > 1)
      opt_dump(root);
#endif
   do {
      done = 1;
      find_levels(root);
      find_dom(root);
      find_closure(root);
      find_inedges(root);
      find_ud(root);
      find_edom(root);
      opt_blks(root, do_stmts);
#ifdef BDEBUG
      if (dflag > 1)
         opt_dump(root);
#endif
   } while (!done);
}

/*
 * Optimize the filter code in its dag representation.
 */
void
Argusbpf_optimize(rootp)
   struct block **rootp;
{
   struct block *root;

   root = *rootp;

   opt_init(root);
   opt_loop(root, 0);
   opt_loop(root, 1);
   intern_blocks(root);
   opt_root(rootp);
   opt_cleanup();
}

static void
make_marks(p)
   struct block *p;
{
   if (!isMarked(p)) {
      Mark(p);
      if (BPF_CLASS(p->s.code) != BPF_RET) {
         make_marks(JT(p));
         make_marks(JF(p));
      }
   }
}

/*
 * Mark code array such that isMarked(i) is true
 * only for nodes that are alive.
 */
static void
mark_code(p)
   struct block *p;
{
   cur_mark += 1;
   make_marks(p);
}

/*
 * True iff the two stmt lists load the same value from the packet into
 * the accumulator.
 */
static int
eq_slist(x, y)
   struct slist *x, *y;
{
   while (1) {
      while (x && x->s.code == NOP)
         x = x->next;
      while (y && y->s.code == NOP)
         y = y->next;
      if (x == 0)
         return y == 0;
      if (y == 0)
         return x == 0;
      if (x->s.code != y->s.code || x->s.k != y->s.k)
         return 0;
      x = x->next;
      y = y->next;
   }
}

static inline int
eq_blk(b0, b1)
   struct block *b0, *b1;
{
   if (b0->s.code == b1->s.code &&
       b0->s.k == b1->s.k &&
       b0->et.succ == b1->et.succ &&
       b0->ef.succ == b1->ef.succ)
      return eq_slist(b0->stmts, b1->stmts);
   return 0;
}

static void
intern_blocks(root)
   struct block *root;
{
   struct block *p;
   int i, j;
   int done;
 top:
   done = 1;
   for (i = 0; i < n_blocks; ++i)
      blocks[i]->link = 0;

   mark_code(root);

   for (i = n_blocks - 1; --i >= 0; ) {
      if (!isMarked(blocks[i]))
         continue;
      for (j = i + 1; j < n_blocks; ++j) {
         if (!isMarked(blocks[j]))
            continue;
         if (eq_blk(blocks[i], blocks[j])) {
            blocks[i]->link = blocks[j]->link ?
               blocks[j]->link : blocks[j];
            break;
         }
      }
   }
   for (i = 0; i < n_blocks; ++i) {
      p = blocks[i];
      if (JT(p) == 0)
         continue;
      if (JT(p)->link) {
         done = 0;
         JT(p) = JT(p)->link;
      }
      if (JF(p)->link) {
         done = 0;
         JF(p) = JF(p)->link;
      }
   }
   if (!done)
      goto top;
}

static void
opt_cleanup()
{
   free((void *)vnode_base);
   free((void *)vmap);
   free((void *)edges);
   free((void *)space);
   free((void *)levels);
   free((void *)blocks);
}

/*
 * Return the number of stmts in 's'.
 */
static int
slength(s)
   struct slist *s;
{
   int n = 0;

   for (; s; s = s->next)
      if (s->s.code != NOP)
         ++n;
   return n;
}

/*
 * Return the number of nodes reachable by 'p'.
 * All nodes should be initially unmarked.
 */
static int
count_blocks(p)
   struct block *p;
{
   if (p == 0 || isMarked(p))
      return 0;
   Mark(p);
   return count_blocks(JT(p)) + count_blocks(JF(p)) + 1;
}

/*
 * Do a depth first search on the flow graph, numbering the
 * the basic blocks, and entering them into the 'blocks' array.`
 */
static void
number_blks_r(p)
   struct block *p;
{
   int n;

   if (p == 0 || isMarked(p))
      return;

   Mark(p);
   n = n_blocks++;
   p->id = n;
   blocks[n] = p;

   number_blks_r(JT(p));
   number_blks_r(JF(p));
}

/*
 * Return the number of stmts in the flowgraph reachable by 'p'.
 * The nodes should be unmarked before calling.
 */
static int
count_stmts(p)
   struct block *p;
{
   int n;

   if (p == 0 || isMarked(p))
      return 0;
   Mark(p);
   n = count_stmts(JT(p)) + count_stmts(JF(p));
   return slength(p->stmts) + n + 1;
}

/*
 * Allocate memory.  All allocation is done before optimization
 * is begun.  A linear bound on the size of all data structures is computed
 * from the total number of blocks and/or statements.
 */
static void
opt_init(root)
   struct block *root;
{
   unsigned int *p;
   int i, n, max_stmts;

   /*
    * First, count the blocks, so we can malloc an array to map
    * block number to block.  Then, put the blocks into the array.
    */
   unMarkAll();
   n = count_blocks(root);
   blocks = (struct block **)ArgusMalloc(n * sizeof(*blocks));
   unMarkAll();
   n_blocks = 0;
   number_blks_r(root);

   n_edges = 2 * n_blocks;
   edges = (struct edge **)ArgusMalloc(n_edges * sizeof(*edges));

   /*
    * The number of levels is bounded by the number of nodes.
    */
   levels = (struct block **)ArgusMalloc(n_blocks * sizeof(*levels));

   edgewords = n_edges / (8 * sizeof(unsigned int)) + 1;
   nodewords = n_blocks / (8 * sizeof(unsigned int)) + 1;

   /* XXX */
   space = (unsigned int *)ArgusMalloc(2 * n_blocks * nodewords * sizeof(*space)
             + n_edges * edgewords * sizeof(*space));
   p = space;
   all_dom_sets = p;
   for (i = 0; i < n; ++i) {
      blocks[i]->dom = p;
      p += nodewords;
   }
   all_closure_sets = p;
   for (i = 0; i < n; ++i) {
      blocks[i]->closure = p;
      p += nodewords;
   }
   all_edge_sets = p;
   for (i = 0; i < n; ++i) {
      register struct block *b = blocks[i];

      b->et.edom = p;
      p += edgewords;
      b->ef.edom = p;
      p += edgewords;
      b->et.id = i;
      edges[i] = &b->et;
      b->ef.id = n_blocks + i;
      edges[n_blocks + i] = &b->ef;
      b->et.pred = b;
      b->ef.pred = b;
   }
   max_stmts = 0;
   for (i = 0; i < n; ++i)
      max_stmts += slength(blocks[i]->stmts) + 1;
   /*
    * We allocate at most 3 value numbers per statement,
    * so this is an upper bound on the number of valnodes
    * we'll need.
    */
   maxval = 3 * max_stmts;
   vmap = (struct vmapinfo *)ArgusMalloc(maxval * sizeof(*vmap));
   vnode_base = (struct valnode *)ArgusMalloc(maxval * sizeof(*vmap));
}

/*
 * Some pointers used to convert the basic block form of the code,
 * into the array form that BPF requires.  'fstart' will point to
 * the malloc'd array while 'ftail' is used during the recursive traversal.
 */
static struct bpf_insn *fstart;
static struct bpf_insn *ftail;

extern void ArgusLog (int, char *, ...);

#ifdef BDEBUG
int bids[1000];
#endif

static int
convert_code_r(p)
   struct block *p;
{
   struct bpf_insn *dst;
   struct slist *src;
   int slen;
   u_int off;
   int extrajmps;      /* number of extra jumps inserted */
   struct slist **offset = NULL;

   if (p == 0 || isMarked(p))
      return (1);
   Mark(p);

   if (convert_code_r(JF(p)) == 0)
      return (0);
   if (convert_code_r(JT(p)) == 0)
      return (0);

   slen = slength(p->stmts);
   dst = ftail -= (slen + 1 + p->longjt + p->longjf);
      /* inflate length by any extra jumps */

   p->offset = dst - fstart;

   /* generate offset[] for convenience  */
   if (slen) {
      offset = (struct slist **)calloc(sizeof(struct slist *), slen);
      if (!offset) {
         ArgusLog (LOG_ERR, "not enough core");
         /*NOTREACHED*/
      }
   }
   src = p->stmts;
   for (off = 0; off < slen && src; off++) {
#if 0
      printf("off=%d src=%x\n", off, src);
#endif
      offset[off] = src;
      src = src->next;
   }

   off = 0;
   for (src = p->stmts; src; src = src->next) {
      if (src->s.code == NOP)
         continue;
      dst->code = (u_short)src->s.code;
      dst->k = src->s.k;

      /* fill block-local relative jump */
      if ((BPF_CLASS(src->s.code) != BPF_JMP) || (src->s.code == (BPF_JMP|BPF_JA))) {
#if 0
         if (src->s.jt || src->s.jf) {
            ArgusLog (LOG_ERR, "illegal jmp destination");
            /*NOTREACHED*/
         }
#endif
         goto filled;
      }
      if (off == slen - 2)   /*???*/
         goto filled;

       {
      int i;
      int jt, jf;
      char *ljerr = "%s for block-local relative jump: off=%d";

#if 0
      printf("code=%x off=%d %x %x\n", src->s.code,
         off, src->s.jt, src->s.jf);
#endif

      if (!src->s.jt || !src->s.jf) {
         ArgusLog (LOG_ERR, ljerr, "no jmp destination", off);
         /*NOTREACHED*/
      }

      jt = jf = 0;
      for (i = 0; i < slen; i++) {
         if (offset[i] == src->s.jt) {
            if (jt) {
               ArgusLog (LOG_ERR, ljerr, "multiple matches", off);
               /*NOTREACHED*/
            }

            dst->jt = i - off - 1;
            jt++;
         }
         if (offset[i] == src->s.jf) {
            if (jf) {
               ArgusLog (LOG_ERR, ljerr, "multiple matches", off);
               /*NOTREACHED*/
            }
            dst->jf = i - off - 1;
            jf++;
         }
      }
      if (!jt || !jf) {
         ArgusLog (LOG_ERR, ljerr, "no destination found", off);
         /*NOTREACHED*/
      }
       }
filled:
      ++dst;
      ++off;
   }
   if (offset)
      free(offset);

#ifdef BDEBUG
   bids[dst - fstart] = p->id + 1;
#endif
   dst->code = (u_short)p->s.code;
   dst->k = p->s.k;
   if (JT(p)) {
      extrajmps = 0;
      off = JT(p)->offset - (p->offset + slen) - 1;
      if (off >= 256) {
          /* offset too large for branch, must add a jump */
          if (p->longjt == 0) {
             /* mark this instruction and retry */
         p->longjt++;
         return(0);
          }
          /* branch if T to following jump */
          dst->jt = extrajmps;
          extrajmps++;
          dst[extrajmps].code = BPF_JMP|BPF_JA;
          dst[extrajmps].k = off - extrajmps;
      }
      else
          dst->jt = off;
      off = JF(p)->offset - (p->offset + slen) - 1;
      if (off >= 256) {
          /* offset too large for branch, must add a jump */
          if (p->longjf == 0) {
             /* mark this instruction and retry */
         p->longjf++;
         return(0);
          }
          /* branch if F to following jump */
          /* if two jumps are inserted, F goes to second one */
          dst->jf = extrajmps;
          extrajmps++;
          dst[extrajmps].code = BPF_JMP|BPF_JA;
          dst[extrajmps].k = off - extrajmps;
      }
      else
          dst->jf = off;
   }
   return (1);
}


/*
 * Convert flowgraph intermediate representation to the
 * BPF array representation.  Set *lenp to the number of instructions.
 */
struct bpf_insn *
Argusicode_to_fcode(root, lenp)
struct block *root;
int *lenp;
{
   int n;
   struct bpf_insn *fp;

   /*
    * Loop doing convert_codr_r() until no branches remain
    * with too-large offsets.
    */
   while (1) {
       unMarkAll();
       n = *lenp = count_stmts(root);
    
       fp = (struct bpf_insn *)ArgusMalloc(sizeof(*fp) * n);
       memset((char *)fp, 0, sizeof(*fp) * n);
       fstart = fp;
       ftail = fp + n;
    
       unMarkAll();
       if (convert_code_r(root))
      break;
       free(fp);
   }

   return fp;
}

#ifdef BDEBUG
opt_dump(root)
   struct block *root;
{
   struct bpf_program f;

   memset(bids, 0, sizeof bids);
   f.bf_insns = Argusicode_to_fcode(root, &f.bf_len);
   bpf_dump(&f, 1);
   putchar('\n');
   free((char *)f.bf_insns);
}
#endif


/*
 * Copyright (c) 1990, 1991, 1992, 1994
 *   The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code distributions
 * retain the above copyright notice and this paragraph in its entirety, (2)
 * distributions including binary code include the above copyright notice and
 * this paragraph in its entirety in the documentation or other materials
 * provided with the distribution, and (3) all advertising materials mentioning
 * features or use of this software display the following acknowledgement:
 * ``This product includes software developed by the University of California,
 * Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
 * the University nor the names of its contributors may be used to endorse
 * or promote products derived from this software without specific prior
 * written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */


static char *bpf_image(struct bpf_insn *, int);

static char *
bpf_image(p, n)
struct bpf_insn *p;
int n;
{
   int v;
   char *fmt, *op;
   static char image[256];
   char operand[64];

   v = p->k;
   switch (p->code) {

   default:
      op = "unimp";
      fmt = "0x%x";
      v = p->code;
      break;

   case BPF_RET|BPF_K:
      op = "ret";
      fmt = "#%d";
      break;

   case BPF_RET|BPF_A:
      op = "ret";
      fmt = "";
      break;

   case BPF_LD|BPF_W|BPF_ABS:
      op = "ld";
      fmt = "[%d]";
      break;

   case BPF_LD|BPF_H|BPF_ABS:
      op = "ldh";
      fmt = "[%d]";
      break;

   case BPF_LD|BPF_B|BPF_ABS:
      op = "ldb";
      fmt = "[%d]";
      break;

   case BPF_LD|BPF_W|BPF_LEN:
      op = "ld";
      fmt = "#pktlen";
      break;

   case BPF_LD|BPF_W|BPF_IND:
      op = "ld";
      fmt = "[x + %d]";
      break;

   case BPF_LD|BPF_H|BPF_IND:
      op = "ldh";
      fmt = "[x + %d]";
      break;

   case BPF_LD|BPF_B|BPF_IND:
      op = "ldb";
      fmt = "[x + %d]";
      break;

   case BPF_LD|BPF_IMM:
      op = "ld";
      fmt = "#0x%x";
      break;

   case BPF_LDX|BPF_IMM:
      op = "ldx";
      fmt = "#0x%x";
      break;

   case BPF_LDX|BPF_MSH|BPF_B:
      op = "ldxb";
      fmt = "4*([%d]&0xf)";
      break;

   case BPF_LD|BPF_MEM:
      op = "ld";
      fmt = "M[%d]";
      break;

   case BPF_LDX|BPF_MEM:
      op = "ldx";
      fmt = "M[%d]";
      break;

   case BPF_ST:
      op = "st";
      fmt = "M[%d]";
      break;

   case BPF_STX:
      op = "stx";
      fmt = "M[%d]";
      break;

   case BPF_JMP|BPF_JA:
      op = "ja";
      fmt = "%d";
      v = n + p->k;
      break;

   case BPF_JMP|BPF_JGT|BPF_K:
      op = "jgt";
      fmt = "#0x%x";
      break;

   case BPF_JMP|BPF_JGE|BPF_K:
      op = "jge";
      fmt = "#0x%x";
      break;

   case BPF_JMP|BPF_JEQ|BPF_K:
      op = "jeq";
      fmt = "#0x%x";
      break;

   case BPF_JMP|BPF_JSET|BPF_K:
      op = "jset";
      fmt = "#0x%x";
      break;

   case BPF_JMP|BPF_JGT|BPF_X:
      op = "jgt";
      fmt = "x";
      break;

   case BPF_JMP|BPF_JGE|BPF_X:
      op = "jge";
      fmt = "x";
      break;

   case BPF_JMP|BPF_JEQ|BPF_X:
      op = "jeq";
      fmt = "x";
      break;

   case BPF_JMP|BPF_JSET|BPF_X:
      op = "jset";
      fmt = "x";
      break;

   case BPF_ALU|BPF_ADD|BPF_X:
      op = "add";
      fmt = "x";
      break;

   case BPF_ALU|BPF_SUB|BPF_X:
      op = "sub";
      fmt = "x";
      break;

   case BPF_ALU|BPF_MUL|BPF_X:
      op = "mul";
      fmt = "x";
      break;

   case BPF_ALU|BPF_DIV|BPF_X:
      op = "div";
      fmt = "x";
      break;

   case BPF_ALU|BPF_AND|BPF_X:
      op = "and";
      fmt = "x";
      break;

   case BPF_ALU|BPF_OR|BPF_X:
      op = "or";
      fmt = "x";
      break;

   case BPF_ALU|BPF_LSH|BPF_X:
      op = "lsh";
      fmt = "x";
      break;

   case BPF_ALU|BPF_RSH|BPF_X:
      op = "rsh";
      fmt = "x";
      break;

   case BPF_ALU|BPF_ADD|BPF_K:
      op = "add";
      fmt = "#%d";
      break;

   case BPF_ALU|BPF_SUB|BPF_K:
      op = "sub";
      fmt = "#%d";
      break;

   case BPF_ALU|BPF_MUL|BPF_K:
      op = "mul";
      fmt = "#%d";
      break;

   case BPF_ALU|BPF_DIV|BPF_K:
      op = "div";
      fmt = "#%d";
      break;

   case BPF_ALU|BPF_AND|BPF_K:
      op = "and";
      fmt = "#%d";
      break;

   case BPF_ALU|BPF_OR|BPF_K:
      op = "or";
      fmt = "#%d";
      break;

   case BPF_ALU|BPF_LSH|BPF_K:
      op = "lsh";
      fmt = "#%d";
      break;

   case BPF_ALU|BPF_RSH|BPF_K:
      op = "rsh";
      fmt = "#%d";
      break;

   case BPF_ALU|BPF_NEG:
      op = "neg";
      fmt = "";
      break;

   case BPF_MISC|BPF_TAX:
      op = "tax";
      fmt = "";
      break;

   case BPF_MISC|BPF_TXA:
      op = "txa";
      fmt = "";
      break;
   }
   (void)sprintf(operand, fmt, v);
   (void)sprintf(image,
            (BPF_CLASS(p->code) == BPF_JMP &&
             BPF_OP(p->code) != BPF_JA) ?
            "(%03d) %-8s %-16s jt %d\tjf %d"
            : "(%03d) %-8s %s",
            n, op, operand, n + 1 + p->jt, n + 1 + p->jf);
   return image;
}



/*
 * Convert string to integer.  Just like atoi(), but checks for 
 * preceding 0x or 0 and uses hex or octal instead of decimal.
 */
int
stoi(s)
   char *s;
{
   int base = 10;
   int n = 0;

   if (*s == '0') {
      if (s[1] == 'x' || s[1] == 'X') {
         s += 2;
         base = 16;
      }
      else {
         base = 8;
         s += 1;
      }
   }
   while (*s)
      n = n * base + xdtoi(*s++);

   return n;
}


#ifdef NOVFPRINTF
/*
 * Stock 4.3 doesn't have vfprintf. 
 * This routine is due to Chris Torek.
 */
vfprintf(f, fmt, args)
   FILE *f;
   char *fmt;
   va_list args;
{
   int ret;

   if ((f->_flag & _IOWRT) == 0) {
      if (f->_flag & _IORW)
         f->_flag |= _IOWRT;
      else
         return EOF;
   }
   ret = _doprnt(fmt, args, f);
   return ferror(f) ? EOF : ret;
}
#endif

/* VARARGS */
/* VARARGS */
void
#if __STDC__
error(const char *fmt, ...)
#else
error(fmt, va_alist)
   const char *fmt;
   va_dcl
#endif
{
   va_list ap;

#if __STDC__
   va_start(ap, fmt);
#else
   va_start(ap);
#endif

   (void)vfprintf(stderr, fmt, ap);
   va_end(ap);
   if (*fmt) {
      fmt += strlen(fmt);
      if (fmt[-1] != '\n')
         (void)fputc('\n', stderr);
   }
   exit(1);
   /* NOTREACHED */
}


/* A replacement for strdup() that cuts down on ArgusMalloc() overhead */
char *
savestr(const char *str)
{
        u_int size;
        char *p;
        static char *strptr = NULL;
        static u_int strsize = 0;

        size = strlen(str) + 1;
        if (size > strsize) {
                strsize = 1024;
                if (strsize < size)
                        strsize = size;
                strptr = (char *) ArgusMalloc(strsize);
                if (strptr == NULL)
                        ArgusLog(LOG_ERR, "savestr: ArgusMalloc");
        }
        (void)strcpy(strptr, str);
        p = strptr;
        strptr += size;
        strsize -= size;
        return (p);
}



/*
 * Copy arg vector into a new argus_strbuffer, concatenating arguments with spaces.
 */
char *
copy_argv(argv)
char **argv;
{
   char **p;
   int len = 0;
   char *argus_strbuf;
   char *src, *dst;

   p = argv;
   if (*p == 0)
      return 0;

   while (*p)
      len += strlen(*p++) + 1;

   argus_strbuf = (char *) ArgusMalloc (len);

   p = argv;
   dst = argus_strbuf;
   while ((src = *p++) != NULL) {
      while ((*dst++ = *src++) != '\0')
         ;
      dst[-1] = ' ';
   }
   dst[-1] = '\0';

   return argus_strbuf;
}

char *
read_infile(char *fname)
{
   struct stat argus_strbuf;
   int fd;
   char *p;

   fd = open(fname, O_RDONLY);
   if (fd < 0)
      error("can't open '%s'", fname);

   if (fstat(fd, &argus_strbuf) < 0)
      error("can't state '%s'", fname);

   p = (char *) calloc(1, (unsigned)argus_strbuf.st_size + 1);
   if (read(fd, p, (int)argus_strbuf.st_size) != argus_strbuf.st_size)
      error("problem reading '%s'", fname);
   
   return p;
}

/*
 * Left justify 'addr' and return its resulting network mask.

unsigned int
net_mask(addr)
unsigned int *addr;
{
   unsigned int m = 0xffffffff;

   if (*addr)
      while ((*addr & 0xff000000) == 0)
         *addr <<= 8, m <<= 8;

   return m;
}
 */

unsigned int
ipaddrtonetmask(addr)
unsigned int addr;
{
   if (IN_CLASSA (addr)) return IN_CLASSA_NET;
   if (IN_CLASSB (addr)) return IN_CLASSB_NET;
   if (IN_CLASSC (addr)) return IN_CLASSC_NET;
   else return 0;
}


unsigned int
getnetnumber(addr)
unsigned int addr;
{
   if (IN_CLASSA (addr)) return (addr >> 24 );
   if (IN_CLASSB (addr)) return (addr >> 16 );
   if (IN_CLASSC (addr)) return (addr >>  8 );
   else return 0;
}







/*
 * Copyright (c) 1992, 1993, 1994
 *   The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code distributions
 * retain the above copyright notice and this paragraph in its entirety, (2)
 * distributions including binary code include the above copyright notice and
 * this paragraph in its entirety in the documentation or other materials
 * provided with the distribution, and (3) all advertising materials mentioning
 * features or use of this software display the following acknowledgement:
 * ``This product includes software developed by the University of California,
 * Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
 * the University nor the names of its contributors may be used to endorse
 * or promote products derived from this software without specific prior
 * written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#include <sys/types.h>

#include <stdio.h>

extern void bpf_dump(struct bpf_program *, int);
static char *bpf_image(struct bpf_insn *p, int n);

void
bpf_dump(struct bpf_program *p, int option)
{
   struct bpf_insn *insn;
   int i;
   int n = p->bf_len;

   insn = p->bf_insns;
   if (option > 2) {
      printf("%d\n", n);
      for (i = 0; i < n; ++insn, ++i) {
         printf("%lu %lu %lu %lu\n", (long)insn->code,
                (long)insn->jt, (long)insn->jf, (long)insn->k);
      }
      return ;
   }
   if (option > 1) {
      for (i = 0; i < n; ++insn, ++i)
         printf("{ 0x%x, %d, %d, 0x%08x },\n",
                insn->code, insn->jt, insn->jf, insn->k);
      return;
   }
   for (i = 0; i < n; ++insn, ++i) {
#ifdef BDEBUG
      extern int bids[];
      printf(bids[i] > 0 ? "[%02d]" : " -- ", bids[i] - 1);
#endif
      puts(bpf_image(insn, i));
   }
}

/*
 * Copyright (c) 2000 QoSient, LLC
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * QOSIENT, LLC DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS, IN NO EVENT SHALL CARTER BULLARD BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

/*
 * Copyright (c) 1990, 1991, 1992, 1993, 1994
 *   The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code distributions
 * retain the above copyright notice and this paragraph in its entirety, (2)
 * distributions including binary code include the above copyright notice and
 * this paragraph in its entirety in the documentation or other materials
 * provided with the distribution, and (3) all advertising materials mentioning
 * features or use of this software display the following acknowledgement:
 * ``This product includes software developed by the University of California,
 * Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
 * the University nor the names of its contributors may be used to endorse
 * or promote products derived from this software without specific prior
 * written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 *  Internet, ethernet, port, and protocol string to address
 *  and address to string conversion routines
 */


#ifndef ArgusAddrtoName
#define ArgusAddrtoName
#endif

#include <sys/types.h>
#include <sys/socket.h>
#include <string.h>
#include <signal.h>
#include <netdb.h>

#include <compat.h>
#include <pcap.h>

#include <argus-namedb.h>


#include <ethernames.h>


static SIGRET nohostname(int);

#if defined(HAVE_ETHER_NTOHOST)
struct ether_addr;
extern int ether_ntohost(char *, struct ether_addr *);
#endif

/*
 * hash tables for whatever-to-name translations
 */

#define HASHNAMESIZE 4096

struct hnamemem {
   unsigned int addr;
   char *name;
   struct hnamemem *nxt;
};

struct hnamemem hnametable[HASHNAMESIZE];
struct hnamemem tporttable[HASHNAMESIZE];
struct hnamemem uporttable[HASHNAMESIZE];
struct hnamemem eprototable[HASHNAMESIZE];
struct hnamemem nnametable[HASHNAMESIZE];
struct hnamemem llcsaptable[HASHNAMESIZE];

struct enamemem {
   u_short e_addr0;
   u_short e_addr1;
   u_short e_addr2;
   char *e_name;
   u_char *e_nsap;         /* used only for nsaptable[] */
   struct enamemem *e_nxt;
};

struct enamemem enametable[HASHNAMESIZE];
struct enamemem nsaptable[HASHNAMESIZE];

struct protoidmem {
   unsigned int p_oui;
   arg_uint16 p_proto;
   char *p_name;
   struct protoidmem *p_nxt;
};

struct protoidmem protoidtable[HASHNAMESIZE];

/*
 * A faster replacement for inet_ntoa().
 */
char *
intoa(unsigned int addr)
{
   char *cp;
   u_int byte;
   int n;
   static char buf[sizeof(".xxx.xxx.xxx.xxx")];

        addr = ntohl(addr);
   cp = &buf[sizeof buf];
   *--cp = '\0';

   n = 4;
   do {
      byte = addr & 0xff;
      *--cp = byte % 10 + '0';
      byte /= 10;
      if (byte > 0) {
         *--cp = byte % 10 + '0';
         byte /= 10;
         if (byte > 0)
            *--cp = byte + '0';
      }
      *--cp = '.';
      addr >>= 8;
   } while (--n > 0);

   return cp + 1;
}

static unsigned int f_netmask;
static unsigned int f_localnet;
static unsigned int netmask;

/*
 * "getname" is written in this atrocious way to make sure we don't
 * wait forever while trying to get hostnames from yp.
 */
#include <setjmp.h>

jmp_buf getname_env;

static SIGRET
nohostname(int signo)
{
   longjmp(getname_env, 1);
}

/*
 * Return a name for the IP address pointed to by ap.  This address
 * is assumed to be in network byte order.
 */
char *
getname(u_char *ap)
{
   struct hostent *hp;
   char *cp;
   unsigned int addr;
   static struct hnamemem *p;      /* static for longjmp() */

#ifndef TCPDUMP_ALIGN
   addr = *(const unsigned int *)ap;
#else
   /*
    * Deal with alignment.
    */
   switch ((int)ap & 3) {

   case 0:
      addr = *(unsigned int *)ap;
      break;

   case 2:
#if BYTES_BIG_ENDIAN == FALSE
      addr = ((unsigned int)*(u_short *)(ap + 2) << 16) |
         (unsigned int)*(u_short *)ap;
#else
      addr = ((unsigned int)*(u_short *)ap << 16) |
         (unsigned int)*(u_short *)(ap + 2);
#endif
      break;

   default:
#if BYTES_BIG_ENDIAN == FALSE
      addr = ((unsigned int)ap[0] << 24) |
         ((unsigned int)ap[1] << 16) |
         ((unsigned int)ap[2] << 8) |
         (unsigned int)ap[3];
#else
      addr = ((unsigned int)ap[3] << 24) |
         ((unsigned int)ap[2] << 16) |
         ((unsigned int)ap[1] << 8) |
         (unsigned int)ap[0];
#endif
      break;
   }
#endif
   p = &hnametable[addr & (HASHNAMESIZE-1)];
   for (; p->nxt; p = p->nxt) {
      if (p->addr == addr)
         return (p->name);
   }
   p->addr = addr;
   p->nxt = (struct hnamemem *)calloc(1, sizeof (*p));

   /*
    * Only print names when:
    *   (1) -n was not given.
    *   (2) Address is foreign and -f was given.  If -f was not
    *       present, f_netmask and f_local are 0 and the second
    *       test will succeed.
    *   (3) The host portion is not 0 (i.e., a network address).
    *   (4) The host portion is not broadcast.
    */
   if (!(nflag)) 
      if ((addr & f_netmask) == f_localnet)
         if ((addr &~ netmask) != 0)
            if ((addr | netmask) != 0xffffffff)
               if (!setjmp(getname_env)) {
                  (void)signal(SIGALRM, nohostname);
                  (void)alarm(20);
                  addr = ntohl(addr);
                  hp = gethostbyaddr((char *)&addr, 4, AF_INET);
                  addr = htonl(addr);
                  (void)alarm(0);
                  if (hp) {
                     char *dotp;

                     p->name = savestr(hp->h_name);
                     if (Dflag) {
                        /* Remove domain qualifications */
                        dotp = strchr(p->name, '.');
                        if (dotp)
                           *dotp = 0;
                     }
                     return (p->name);
                  }
               }

#if BYTES_BIG_ENDIAN == FALSE
   addr = ((unsigned int)ap[0] << 24)  | ((unsigned int)ap[1] << 16) |
           ((unsigned int)ap[2] <<  8) | (unsigned int)ap[3];
#endif
   cp = intoa(addr);
   p->name = savestr(cp);
   return (p->name);
}

static char hex[] = "0123456789abcdef";


/* Find the hash node that corresponds the ether address 'ep'. */

static inline struct enamemem *
lookup_emem(const u_char *ep)
{
   u_int i, j, k;
   struct enamemem *tp;

   k = (ep[0] << 8) | ep[1];
   j = (ep[2] << 8) | ep[3];
   i = (ep[4] << 8) | ep[5];

   tp = &enametable[(i ^ j) & (HASHNAMESIZE-1)];
   while (tp->e_nxt)
      if (tp->e_addr0 == i &&
          tp->e_addr1 == j &&
          tp->e_addr2 == k)
         return tp;
      else
         tp = tp->e_nxt;
   tp->e_addr0 = i;
   tp->e_addr1 = j;
   tp->e_addr2 = k;
   tp->e_nxt = (struct enamemem *)calloc(1, sizeof(*tp));

   return tp;
}

/* Find the hash node that corresponds the NSAP 'nsap'. */

static inline struct enamemem *
lookup_nsap(const u_char *nsap)
{
   u_int i, j, k;
   int nlen = *nsap;
   struct enamemem *tp;
   const u_char *ensap = nsap + nlen - 6;

   if (nlen > 6) {
      k = (ensap[0] << 8) | ensap[1];
      j = (ensap[2] << 8) | ensap[3];
      i = (ensap[4] << 8) | ensap[5];
   }
   else
      i = j = k = 0;

   tp = &nsaptable[(i ^ j) & (HASHNAMESIZE-1)];
   while (tp->e_nxt)
      if (tp->e_addr0 == i &&
          tp->e_addr1 == j &&
          tp->e_addr2 == k &&
          tp->e_nsap[0] == nlen &&
          bcmp((char *)&(nsap[1]),
         (char *)&(tp->e_nsap[1]), nlen) == 0)
         return tp;
      else
         tp = tp->e_nxt;
   tp->e_addr0 = i;
   tp->e_addr1 = j;
   tp->e_addr2 = k;
   tp->e_nsap = (u_char *) calloc(1, nlen + 1);
   bcopy(nsap, tp->e_nsap, nlen + 1);
   tp->e_nxt = (struct enamemem *)calloc(1, sizeof(*tp));

   return tp;
}

/* Find the hash node that corresponds the protoid 'pi'. */

static inline struct protoidmem *
lookup_protoid(const u_char *pi)
{
   u_int i, j;
   struct protoidmem *tp;

   /* 5 octets won't be aligned */
   i = (((pi[0] << 8) + pi[1]) << 8) + pi[2];
   j =   (pi[3] << 8) + pi[4];
   /* XXX should be endian-insensitive, but do big-endian testing  XXX */

   tp = &protoidtable[(i ^ j) & (HASHNAMESIZE-1)];
   while (tp->p_nxt)
      if (tp->p_oui == i && tp->p_proto == j)
         return tp;
      else
         tp = tp->p_nxt;
   tp->p_oui = i;
   tp->p_proto = j;
   tp->p_nxt = (struct protoidmem *)calloc(1, sizeof(*tp));

   return tp;
}

char *
etheraddr_string(u_char *ep)
{
   u_int i, j;
   char *cp;
   struct enamemem *tp;

   tp = lookup_emem(ep);
   if (tp->e_name)
      return (tp->e_name);
#if defined(HAVE_ETHER_NTOHOST)
   if (!nflag) {
      char buf[128];
      if (ether_ntohost(buf, (struct ether_addr *)ep) == 0) {
         tp->e_name = savestr(buf);
         return (tp->e_name);
      }
   }
#endif
   tp->e_name = cp = (char *)ArgusMalloc(sizeof("00:00:00:00:00:00"));

   if ((j = *ep >> 4) != 0)
      *cp++ = hex[j];
   *cp++ = hex[*ep++ & 0xf];
   for (i = 5; (int)--i >= 0;) {
      *cp++ = ':';
      if ((j = *ep >> 4) != 0)
         *cp++ = hex[j];
      *cp++ = hex[*ep++ & 0xf];
   }
   *cp = '\0';
   return (tp->e_name);
}


#define ARGUS_MAXEPROTODB   0x10000

struct ArgusEtherTypeStruct *argus_eproto_db[ARGUS_MAXEPROTODB];

char *
etherproto_string(u_short port)
{
   struct ArgusEtherTypeStruct *p;
   char *retn = NULL, *cp = NULL;

   if ((p = argus_eproto_db[port]) != NULL) {
      retn =  p->tag;
   } else {
      if ((p = (struct ArgusEtherTypeStruct *) calloc (1, sizeof(*p))) != NULL) {
         if (nflag < 2) 
            p->tag = "unknown";
         else {
           p->tag = cp = (char *)ArgusMalloc(sizeof("000000"));
           sprintf (cp, "%d", port);
         }
      
         p->range = cp;
   
         argus_eproto_db[port] = p;
         retn = p->tag;
      }
   }

   return (retn);
}

char *
protoid_string(const u_char *pi)
{
   u_int i, j;
   char *cp;
   struct protoidmem *tp;

   tp = lookup_protoid(pi);
   if (tp->p_name)
      return tp->p_name;

   tp->p_name = cp = (char *)ArgusMalloc(sizeof("00:00:00:00:00"));

   if ((j = *pi >> 4) != 0)
      *cp++ = hex[j];
   *cp++ = hex[*pi++ & 0xf];
   for (i = 4; (int)--i >= 0;) {
      *cp++ = ':';
      if ((j = *pi >> 4) != 0)
         *cp++ = hex[j];
      *cp++ = hex[*pi++ & 0xf];
   }
   *cp = '\0';
   return (tp->p_name);
}

char *
llcsap_string(u_char sap)
{
   char *cp;
   struct hnamemem *tp;
   unsigned int i = sap;

   for (tp = &llcsaptable[i & (HASHNAMESIZE-1)]; tp->nxt; tp = tp->nxt)
      if (tp->addr == i)
         return (tp->name);

   tp->name = cp = (char *)ArgusMalloc(sizeof("00000"));
   tp->addr = i;
   tp->nxt = (struct hnamemem *)calloc(1, sizeof (*tp));

   *cp++ = '0';
   *cp++ = 'x';
   *cp++ = hex[sap >> 4 & 0xf];
   *cp++ = hex[sap & 0xf];
   *cp++ = '\0';
   return (tp->name);
}

char *
isonsap_string(const u_char *nsap)
{
   u_int i, nlen = nsap[0];
   char *cp;
   struct enamemem *tp;

   tp = lookup_nsap(nsap);
   if (tp->e_name)
      return tp->e_name;

   tp->e_name = cp = (char *)ArgusMalloc(nlen * 2 + 2);

   nsap++;
   *cp++ = '/';
   for (i = nlen; (int)--i >= 0;) {
      *cp++ = hex[*nsap >> 4];
      *cp++ = hex[*nsap++ & 0xf];
   }
   *cp = '\0';
   return (tp->e_name);
}

char *
tcpport_string(arg_uint16 port)
{
   struct hnamemem *tp;
   unsigned int i = port;

   for (tp = &tporttable[i & (HASHNAMESIZE-1)]; tp->nxt; tp = tp->nxt)
      if (tp->addr == i)
         return (tp->name);

   tp->name = (char *)ArgusMalloc(sizeof("00000"));
   tp->addr = i;
   tp->nxt = (struct hnamemem *)calloc(1, sizeof (*tp));

   (void)sprintf(tp->name, "%d", i);
   return (tp->name);
}

char *
udpport_string(u_short port)
{
   struct hnamemem *tp;
   unsigned int i = port;

        if (port) {
      for (tp = &uporttable[i & (HASHNAMESIZE-1)]; tp->nxt; tp = tp->nxt)
         if (tp->addr == i)
            return (tp->name);

      tp->name = (char *)ArgusMalloc(sizeof("00000"));
      tp->addr = i;
      tp->nxt = (struct hnamemem *)calloc(1, sizeof(*tp));

      (void)sprintf(tp->name, "%d", i);

      return (tp->name);
   } else
   return ("*");
}

static void
init_servarray(void)
{
   struct servent *sv;
   struct hnamemem *table;
   int i;

   while ((sv = getservent()) != NULL) {
      int port = ntohs(sv->s_port);
      i = port & (HASHNAMESIZE-1);
      if (strcmp(sv->s_proto, "tcp") == 0)
         table = &tporttable[i];
      else if (strcmp(sv->s_proto, "udp") == 0)
         table = &uporttable[i];
      else
         continue;

      while (table->name)
         table = table->nxt;

      if (nflag > 1) {
         char buf[32];

         (void)sprintf(buf, "%d", port);
         table->name = savestr(buf);
      } else
         table->name = savestr(sv->s_name);
      table->addr = port;
      table->nxt = (struct hnamemem *)calloc(1, sizeof(*table));
   }
   endservent();
}


static void
init_eprotoarray(void)
{
   struct ArgusEtherTypeStruct *p = argus_ethertype_names;

   bzero ((char *)argus_eproto_db, sizeof (argus_eproto_db));

   while (p->range != NULL) {
      int i, start, end;
      char *ptr;
      
      start = atoi(p->range);

      if ((ptr = strchr(p->range, '-')) != NULL)
         end = atoi(ptr + 1);
      else
         end = start;

      for (i = start; i < (end + 1); i++)
         argus_eproto_db[i] = p;

      p++;
   }
}


/*
 * SNAP proto IDs with org code 0:0:0 are actually encapsulated Ethernet
 * types.
 */
static void
init_protoidarray(void)
{
   struct ArgusEtherTypeStruct *p;
   int i;
   struct protoidmem *tp;
   u_char protoid[5];

   protoid[0] = 0;
   protoid[1] = 0;
   protoid[2] = 0;

   for (i = 0; i < ARGUS_MAXEPROTODB; i++) {
      if ((p = argus_eproto_db[i]) != NULL) {
         protoid[3] = i;
         tp = lookup_protoid(protoid);
         tp->p_name = p->tag;
      }
   }
}

static struct etherlist {
   u_char addr[6];
   char *name;
} etherlist[] = {
   {{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff }, "Broadcast" },
   {{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, NULL }
};

/*
 * Initialize the ethers hash table.  We take two different approaches
 * depending on whether or not the system provides the ethers name
 * service.  If it does, we just wire in a few names at startup,
 * and etheraddr_string() fills in the table on demand.  If it doesn't,
 * then we suck in the entire /etc/ethers file at startup.  The idea
 * is that parsing the local file will be fast, but spinning through
 * all the ethers entries via NIS & next_etherent might be very slow.
 *
 * XXX argus_next_etherent doesn't belong in the pcap interface, but
 * since the pcap module already does name-to-address translation,
 * it's already does most of the work for the ethernet address-to-name
 * translation, so we just argus_next_etherent as a convenience.
 */
static void
init_etherarray(void)
{
   struct etherlist *el;
   struct enamemem *tp;
#if defined(HAVE_ETHER_NTOHOST)
   char name[256];
#else
   struct argus_etherent *ep;

   FILE *fp;

   /* Suck in entire ethers file */
   fp = fopen(PCAP_ETHERS_FILE, "r");
   if (fp != NULL) {
      while ((ep = argus_next_etherent(fp)) != NULL) {
         tp = lookup_emem(ep->addr);
         tp->e_name = savestr(ep->name);
      }
      (void)fclose(fp);
   }
#endif

   /* Hardwire some ethernet names */
   for (el = etherlist; el->name != NULL; ++el) {
      tp = lookup_emem(el->addr);
      /* Don't override existing name */
      if (tp->e_name != NULL)
         continue;

#if defined(HAVE_ETHER_NTOHOST)
   /* Use yp/nis version of name if available */
   if (ether_ntohost(name, (struct ether_addr *)el->addr) == 0) {
      tp->e_name = savestr(name);
      continue;
   }
#endif
   tp->e_name = el->name;
   }
}

static struct ArgusTokenStruct llcsap_db[] = {
   { LLCSAP_NULL,	"null" },
   { LLCSAP_8021B_I,	"gsap" },
   { LLCSAP_8021B_G,	"isap" },
   { LLCSAP_SNAPATH,	"snapath" },
   { LLCSAP_IP,		"ipsap" },
   { LLCSAP_SNA1,	"sna1" },
   { LLCSAP_SNA2,	"sna2" },
   { LLCSAP_PROWAYNM,	"p-nm" },
   { LLCSAP_TI,		"ti" },
   { LLCSAP_BPDU,	"stp" },
   { LLCSAP_RS511,	"eia" },
   { LLCSAP_ISO8208,	"x25" },
   { LLCSAP_XNS,	"xns" },
   { LLCSAP_NESTAR,	"nestar" },
   { LLCSAP_PROWAYASLM,	"p-aslm" },
   { LLCSAP_ARP,	"arp" },
   { LLCSAP_SNAP,	"snap" },
   { LLCSAP_VINES1,	"vine1" },
   { LLCSAP_VINES2,	"vine2" },
   { LLCSAP_NETWARE,	"netware" },
   { LLCSAP_NETBIOS,	"netbios" },
   { LLCSAP_IBMNM,	"ibmnm" },
   { LLCSAP_RPL1,	"rpl1" },
   { LLCSAP_UB,		"ub" },
   { LLCSAP_RPL2,	"rpl2" },
   { LLCSAP_ISONS,	"clns" },
   { LLCSAP_GLOBAL,	"gbl" },
   { 0,         	 NULL }
};

static void
init_llcsaparray(void)
{
   int i;
   struct hnamemem *table;

   for (i = 0; llcsap_db[i].s != NULL; i++) {
      table = &llcsaptable[llcsap_db[i].v];
      while (table->name)
         table = table->nxt;
      table->name = llcsap_db[i].s;
      table->addr = llcsap_db[i].v;
      table->nxt = (struct hnamemem *)calloc(1, sizeof(*table));
   }
}

/*
 * Initialize the address to name translation machinery.  We map all
 * non-local IP addresses to numeric addresses if fflag is true (i.e.,
 * to prevent blocking on the nameserver).  localnet is the IP address
 * of the local network.  mask is its subnet mask.
 */
void
init_addrtoname(int fflag, unsigned int localnet, unsigned int mask)
{
   netmask = mask;
   if (fflag) {
      f_localnet = localnet;
      f_netmask = mask;
   }

   if (nflag > 1)
      /*
       * Simplest way to suppress names.
       */
      return;

   init_etherarray();
   init_servarray();
   init_eprotoarray();
   init_llcsaparray();
   init_protoidarray();
}



/*
 * Copyright (c) 2000 QoSient, LLC
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * QOSIENT, LLC DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS, IN NO EVENT SHALL CARTER BULLARD BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

/*
 * Copyright (c) 1990, 1991, 1992, 1993, 1994
 *   The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code distributions
 * retain the above copyright notice and this paragraph in its entirety, (2)
 * distributions including binary code include the above copyright notice and
 * this paragraph in its entirety in the documentation or other materials
 * provided with the distribution, and (3) all advertising materials mentioning
 * features or use of this software display the following acknowledgement:
 * ``This product includes software developed by the University of California,
 * Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
 * the University nor the names of its contributors may be used to endorse
 * or promote products derived from this software without specific prior
 * written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 * Name to id translation routines used by the scanner.
 * These functions are not time critical.
 */


#include <netinet/in.h>
#include <netdb.h>

#include <compat.h>

#include <pcap.h>
#include <argus-namedb.h>



#ifndef __GNUC__
#define inline
#endif

#ifndef NTOHL
#define NTOHL(x) (x) = ntohl(x)
#define NTOHS(x) (x) = ntohs(x)
#endif

/*
 *  Convert host name to internet address.
 *  Return 0 upon failure.
 */

unsigned int **
argus_nametoaddr(char *name)
{
#ifndef h_addr
   static unsigned int *hlist[2];
#endif
   unsigned int **p;
   struct hostent *hp;

   if ((hp = gethostbyname(name)) != NULL) {
#ifndef h_addr
      hlist[0] = (unsigned int *)hp->h_addr;
      NTOHL(hp->h_addr);
      return hlist;
#else
      for (p = (unsigned int **)hp->h_addr_list; *p; ++p)
         NTOHL(**p);
      return (unsigned int **)hp->h_addr_list;
#endif
   }
   else
      return 0;
}

/*
 *  Convert net name to internet address.
 *  Return 0 upon failure.
 */
unsigned int
argus_nametonetaddr(char *name)
{
   struct netent *np;

   if ((np = getnetbyname(name)) != NULL)
      return np->n_net;
   else
      return 0;
}

/*
 * Convert a port name to its port and protocol numbers.
 * We assume only TCP or UDP.
 * Return 0 upon failure.
 */
int
argus_nametoport(char *name, int *port, int *proto)
{
   struct servent *sp;
   char *other;

   sp = getservbyname(name, (char *)0);
   if (sp != NULL) {
      NTOHS(sp->s_port);
      *port = sp->s_port;
      *proto = argus_nametoproto(sp->s_proto);
      /*
       * We need to check /etc/services for ambiguous entries.
       * If we find the ambiguous entry, and it has the
       * same port number, change the proto to PROTO_UNDEF
       * so both TCP and UDP will be checked.
       */
      if (*proto == IPPROTO_TCP)
         other = "udp";
      else
         other = "tcp";

      sp = getservbyname(name, other);
      if (sp != 0) {
         NTOHS(sp->s_port);
         if (*port != sp->s_port)
            /* Can't handle ambiguous names that refer
               to different port numbers. */
            error("ambiguous port %s in /etc/services", name);
         *proto = PROTO_UNDEF;
      }
      return 1;
   }
#if defined(ultrix) || defined(__osf__)
   /* Special hack in case NFS isn't in /etc/services */
   if (strcmp(name, "nfs") == 0) {
      *port = 2049;
      *proto = PROTO_UNDEF;
      return 1;
   }
#endif
   return 0;
}

int
argus_nametoproto(char *str)
{
   struct protoent *p;

   p = getprotobyname(str);
   if (p != 0)
      return p->p_proto;
   else
      return PROTO_UNDEF;
}


int
argus_nametoeproto(char *s)
{
   struct ArgusEtherTypeStruct *p = argus_ethertype_names;

   while (p->tag != 0) {
      if (strcmp(p->tag, s) == 0) {
         return atoi(p->range);
      }
      p += 1;
   }

   return PROTO_UNDEF;
}

unsigned int
__argus_atoin(char *s, unsigned int *addr)
{
   u_int n;
   int len;

   *addr = 0;
   len = 0;
   while (1) {
      n = 0;
      while (*s && *s != '.')
         n = n * 10 + *s++ - '0';
      *addr <<= 8;
      *addr |= n & 0xff;
      len += 8;
      if (*s == '\0')
         return len;
      ++s;
   }
   /* NOTREACHED */
}

unsigned int
__argus_atodn(char *s)
{
#define AREASHIFT 10
#define AREAMASK 0176000
#define NODEMASK 01777

   unsigned int addr = 0;
   u_int node, area;

   if (sscanf((char *)s, "%d.%d", (int *) &area, (int *) &node) != 2)
      Argus_error("malformed decnet address '%s'", s);

   addr = (area << AREASHIFT) & AREAMASK;
   addr |= (node & NODEMASK);

   return(addr);
}

/*
 * Convert 's' which has the form "xx:xx:xx:xx:xx:xx" into a new
 * ethernet address.  Assumes 's' is well formed.
 */
u_char *
argus_ether_aton(char *s)
{
   register u_char *ep, *e;
   register u_int d;

   e = ep = (u_char *)ArgusMalloc(6);

   while (*s) {
      if (*s == ':')
         s += 1;
      d = xdtoi(*s++);
      if (isxdigit((int)*s)) {
         d <<= 4;
         d |= xdtoi(*s++);
      }
      *ep++ = d;
   }

   return (e);
}

#if !defined(HAVE_ETHER_HOSTTON)
u_char *
argus_ether_hostton(char *name)
{
   register struct argus_etherent *ep;
   register u_char *ap;
   static FILE *fp = NULL;
   static int init = 0;

   if (!init) {
      fp = fopen(PCAP_ETHERS_FILE, "r");
      ++init;
      if (fp == NULL)
         return (NULL);
   } else if (fp == NULL)
      return (NULL);
   else
      rewind(fp);
   
   while ((ep = argus_next_etherent(fp)) != NULL) {
      if (strcmp(ep->name, name) == 0) {
         ap = (u_char *)ArgusMalloc(6);
         if (ap != NULL) {
            memcpy(ap, ep->addr, 6);
            return (ap);
         }
         break;
      }
   }
   return (NULL);
}
#else
/* Use the os supplied routines */

#if !defined(sgi) && !defined(__NetBSD__) && !defined(__FreeBSD__)
extern int ether_hostton(char *, struct ether_addr *);
#endif

u_char *
argus_ether_hostton(char *name)
{
   register u_char *ap;
   u_char a[6];

   ap = NULL;
   if (ether_hostton((char*)name, (struct ether_addr *)a) == 0) {
      ap = (u_char *)ArgusMalloc(6);
      if (ap != NULL)
         memcpy(ap, a, 6);
   }
   return (ap);
}
#endif

u_short
__argus_nametodnaddr(char *name)
{
#ifndef   DECNETLIB
   Argus_error("decnet name support not included, '%s' cannot be translated\n", name);
   return(0);
#else
   struct nodeent *getnodebyname();
   struct nodeent *nep;
   unsigned short res;

   nep = getnodebyname(name);
   if (nep == ((struct nodeent *)0))
      Argus_error("unknown decnet host name '%s'\n", name);

   memcpy((char *)&res, (char *)nep->n_addr, sizeof(unsigned short));

   return(res);
#endif
}

#include <stdarg.h>

extern char *ArgusProgramName;

char ArgusPrintTimeBuf[64];

char *
print_time(struct timeval *tvp)
{
   char timeZoneBuf[32];
   char *retn = ArgusPrintTimeBuf, *ptr;
   struct tm *tm;

   bzero (timeZoneBuf, sizeof(timeZoneBuf));
   bzero (ArgusPrintTimeBuf, sizeof(ArgusPrintTimeBuf));

   if ((tm = localtime ((time_t *)&tvp->tv_sec)) != NULL) {
      if (uflag)
         sprintf (retn, "%9d", (int) tvp->tv_sec);
      else
         strftime ((char *) retn, 64, RaTimeFormat, tm);

      if (pflag) {
         ptr = &retn[strlen(retn)];

         if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0')) {
            if (uflag) {
               sprintf (ptr, ".%06d", (int) tvp->tv_usec);
               ptr[pflag + 1] = '\0';

            } else {
               sprintf (ptr, "%c0.%06d", RaFieldDelimiter, (int) tvp->tv_usec);
               ptr[pflag + 3] = '\0';
            }

         } else {
            sprintf (ptr, ".%06d", (int) tvp->tv_usec);
            ptr[pflag + 1] = '\0';
         }
      }
   } else
      retn = NULL;

   return (retn);
}

#ifdef ARGUSDEBUG
void
ArgusDebug (int d, char *fmt, ...)
{
   va_list ap;
   char buf[1024], *ptr;

   if (d <= Argusdflag) {
      (void) sprintf (buf, "%s[%d]: %s ", ArgusProgramName, (int)getpid(), print_time(&ArgusGlobalTime));
      ptr = &buf[strlen(buf)];

      va_start (ap, fmt);
      (void) vsprintf (ptr, fmt, ap);
      ptr = &buf[strlen(buf)];
      va_end (ap);

      if (*fmt) {
         fmt += (int) strlen (fmt);
         if (fmt[-1] != '\n')
            sprintf (ptr, "\n");
      }
      fprintf (stderr, "%s", buf);
   }
}
#endif /* ARGUSDEBUG */


#if defined(_LITTLE_ENDIAN)

#if !defined(ntohll)
long long ntohll (long long);

long long
ntohll (long long data)
{
   long long retn;
   unsigned char *ptr1, *ptr2; 
   int i, len;

   ptr1 = (unsigned char *) &data;
   ptr2 = (unsigned char *) &retn;

   for (i = 0, len = sizeof(retn); i < len; i++)
      ptr2[(len - 1) - i] = ptr1[i]; 

   return(retn);
}
#endif /* ntohll */

#if !defined(htonll)
long long htonll (long long);

long long
htonll (long long data)
{
   long long retn;
   unsigned char *ptr1, *ptr2;
   int i, len;

   ptr1 = (unsigned char *) &data;
   ptr2 = (unsigned char *) &retn;

   for (i = 0, len = sizeof(retn); i < len; i++) 
      ptr2[(len - 1) - i] = ptr1[i];

   return(retn);
}
#endif /* htonll */


void
ArgusNtoH (struct ArgusRecord *argus)
{
   int farlen = 0;

   argus->ahdr.length    = ntohs(argus->ahdr.length);
   argus->ahdr.argusid   = ntohl(argus->ahdr.argusid);
   argus->ahdr.seqNumber = ntohl(argus->ahdr.seqNumber);
   argus->ahdr.status    = ntohl(argus->ahdr.status);

   if (argus->ahdr.type & ARGUS_MAR) {

      argus->argus_mar.startime.tv_sec  = ntohl(argus->argus_mar.startime.tv_sec);
      argus->argus_mar.startime.tv_usec = ntohl(argus->argus_mar.startime.tv_usec);
      argus->argus_mar.now.tv_sec  = ntohl(argus->argus_mar.now.tv_sec);
      argus->argus_mar.now.tv_usec = ntohl(argus->argus_mar.now.tv_usec);
      argus->argus_mar.reportInterval = ntohs(argus->argus_mar.reportInterval);
      argus->argus_mar.argusMrInterval = ntohs(argus->argus_mar.argusMrInterval);
      argus->argus_mar.argusid = ntohl(argus->argus_mar.argusid);
      argus->argus_mar.localnet = ntohl(argus->argus_mar.localnet);
      argus->argus_mar.netmask = ntohl(argus->argus_mar.netmask);
      argus->argus_mar.nextMrSequenceNum = ntohl(argus->argus_mar.nextMrSequenceNum);

      argus->argus_mar.pktsRcvd  = ntohll(argus->argus_mar.pktsRcvd);
      argus->argus_mar.bytesRcvd = ntohll(argus->argus_mar.bytesRcvd);

      argus->argus_mar.pktsDrop = ntohl(argus->argus_mar.pktsDrop);
      argus->argus_mar.flows = ntohl(argus->argus_mar.flows);
      argus->argus_mar.flowsClosed = ntohl(argus->argus_mar.flowsClosed);

      argus->argus_mar.actIPcons = ntohl( argus->argus_mar.actIPcons);
      argus->argus_mar.cloIPcons = ntohl( argus->argus_mar.cloIPcons);
      argus->argus_mar.actICMPcons = ntohl( argus->argus_mar.actICMPcons);
      argus->argus_mar.cloICMPcons = ntohl( argus->argus_mar.cloICMPcons);
      argus->argus_mar.actIGMPcons = ntohl( argus->argus_mar.actIGMPcons);
      argus->argus_mar.cloIGMPcons = ntohl( argus->argus_mar.cloIGMPcons);
      argus->argus_mar.actFRAGcons = ntohl( argus->argus_mar.actFRAGcons);
      argus->argus_mar.cloFRAGcons = ntohl( argus->argus_mar.cloFRAGcons);
      argus->argus_mar.actSECcons = ntohl( argus->argus_mar.actSECcons);
      argus->argus_mar.cloSECcons = ntohl( argus->argus_mar.cloSECcons);

      argus->argus_mar.record_len = ntohl(argus->argus_mar.record_len);

   } else {
      unsigned int status, length = argus->ahdr.length - sizeof(argus->ahdr);
      struct ArgusFarHeaderStruct *farhdr = (struct ArgusFarHeaderStruct *) &argus->argus_far;

      farhdr->status = ntohs(farhdr->status);

      status = argus->ahdr.status;

      while (length > 0) {
         switch (farhdr->type) {
            case ARGUS_FAR: {
               struct ArgusFarStruct *far = (struct ArgusFarStruct *) farhdr;
               
               far->ArgusTransRefNum = ntohl(far->ArgusTransRefNum);

               switch (status & (ETHERTYPE_IP|ETHERTYPE_IPV6|ETHERTYPE_ARP)) {
                  case ETHERTYPE_IP: {
                     struct ArgusIPFlow *ipflow = &far->flow.ip_flow;

                     far->attr_ip.soptions = ntohs(far->attr_ip.soptions);
                     far->attr_ip.doptions = ntohs(far->attr_ip.doptions);

                     switch (ipflow->ip_p) {
                        case IPPROTO_UDP:
                        case IPPROTO_TCP:
                           ipflow->ip_src = ntohl(ipflow->ip_src);
                           ipflow->ip_dst = ntohl(ipflow->ip_dst);
                           ipflow->sport  = ntohs(ipflow->sport);
                           ipflow->dport  = ntohs(ipflow->dport);
                           ipflow->ip_id  = ntohs(ipflow->ip_id);
                           break;

                        case IPPROTO_ICMP: {
                           struct ArgusICMPFlow *icmpflow = &far->flow.icmp_flow;

                           icmpflow->ip_src = ntohl(icmpflow->ip_src);
                           icmpflow->ip_dst = ntohl(icmpflow->ip_dst);
                           icmpflow->id     = ntohs(icmpflow->id);
                           icmpflow->ip_id  = ntohs(icmpflow->ip_id);
                           break;
                        }

                        default: {
                           ipflow->ip_src = ntohl(ipflow->ip_src);
                           ipflow->ip_dst = ntohl(ipflow->ip_dst);
                           break;
                        }
                     }
                     break;
                  }
         
                  case ETHERTYPE_IPV6:
                     break;
         
                  case ETHERTYPE_ARP: {
                     struct ArgusArpFlow *arpflow = &far->flow.arp_flow;
         
                     arpflow->arp_tpa = ntohl(arpflow->arp_tpa);
                     arpflow->arp_spa = ntohl(arpflow->arp_spa);
                     break;
                  }
                  default:
                     break;
               }
         
               far->time.start.tv_sec  = ntohl(far->time.start.tv_sec);
               far->time.start.tv_usec = ntohl(far->time.start.tv_usec);
               far->time.last.tv_sec   = ntohl(far->time.last.tv_sec);
               far->time.last.tv_usec  = ntohl(far->time.last.tv_usec);
         
               far->src.count    = ntohl(far->src.count);
               far->src.bytes    = ntohl(far->src.bytes);
               far->src.appbytes = ntohl(far->src.appbytes);
         
               far->dst.count    = ntohl(far->dst.count);
               far->dst.bytes    = ntohl(far->dst.bytes);
               far->dst.appbytes = ntohl(far->dst.appbytes);
               break;
            }

            case ARGUS_MAC_DSR: {
               struct ArgusMacStruct *mac = (struct ArgusMacStruct *) farhdr;
               mac->status   = ntohs(mac->status);
               break;
            }


            case ARGUS_VLAN_DSR: {
               struct ArgusVlanStruct *vlan = (struct ArgusVlanStruct *) farhdr;
               vlan->status = ntohs(vlan->status);
               vlan->sid    = ntohs(vlan->sid);
               vlan->did    = ntohs(vlan->did);
               break;
            }

            case ARGUS_MPLS_DSR: {
               struct ArgusMplsStruct *mpls = (struct ArgusMplsStruct *) farhdr;
               mpls->status = ntohs(mpls->status);
               mpls->slabel = ntohl(mpls->slabel);
               mpls->dlabel = ntohl(mpls->dlabel);
               break;
            }

            case ARGUS_TCP_DSR: {
               struct ArgusTCPObject *tcp = (struct ArgusTCPObject *) farhdr; 

               tcp->status = ntohs(tcp->status); 
               tcp->state  = ntohl(tcp->state); 
               tcp->synAckuSecs  = ntohl(tcp->synAckuSecs); 
               tcp->ackDatauSecs = ntohl(tcp->ackDatauSecs); 
               tcp->options = ntohl(tcp->options); 
               tcp->src.seqbase  = ntohl(tcp->src.seqbase); 
               tcp->src.ackbytes = ntohl(tcp->src.ackbytes); 
               tcp->src.rpkts    = ntohl(tcp->src.rpkts); 
               tcp->src.win     = ntohs(tcp->src.win); 
               tcp->dst.seqbase  = ntohl(tcp->dst.seqbase); 
               tcp->dst.ackbytes = ntohl(tcp->dst.ackbytes); 
               tcp->dst.rpkts    = ntohl(tcp->dst.rpkts); 
               tcp->dst.win     = ntohs(tcp->dst.win); 
               break;
            }

            case ARGUS_ICMP_DSR: {
               struct ArgusICMPObject *icmp = (struct ArgusICMPObject *) farhdr;
 
               icmp->status   = ntohs(icmp->status);
               icmp->iseq     = ntohs(icmp->iseq);
               icmp->osrcaddr = ntohl(icmp->osrcaddr);
               icmp->odstaddr = ntohl(icmp->odstaddr);
               icmp->isrcaddr = ntohl(icmp->isrcaddr);
               icmp->idstaddr = ntohl(icmp->idstaddr);
               icmp->igwaddr  = ntohl(icmp->igwaddr);
               break;
            }

            case ARGUS_TIME_DSR: {
               struct ArgusTimeStruct *time = (void *) farhdr;

               time->status = ntohs(time->status);
               time->src.act.n       = ntohl(time->src.act.n);
               time->src.act.min     = ntohl(time->src.act.min);
               time->src.act.mean    = ntohl(time->src.act.mean);
               time->src.act.stdev   = ntohl(time->src.act.stdev);
               time->src.act.max     = ntohl(time->src.act.max);
               time->src.idle.n      = ntohl(time->src.idle.n);
               time->src.idle.min    = ntohl(time->src.idle.min);
               time->src.idle.mean   = ntohl(time->src.idle.mean);
               time->src.idle.stdev  = ntohl(time->src.idle.stdev);
               time->src.idle.max    = ntohl(time->src.idle.max);
               time->dst.act.n       = ntohl(time->dst.act.n);
               time->dst.act.min     = ntohl(time->dst.act.min);
               time->dst.act.mean    = ntohl(time->dst.act.mean);
               time->dst.act.stdev   = ntohl(time->dst.act.stdev);
               time->dst.act.max     = ntohl(time->dst.act.max);
               time->dst.idle.n      = ntohl(time->dst.idle.n);
               time->dst.idle.min    = ntohl(time->dst.idle.min);
               time->dst.idle.mean   = ntohl(time->dst.idle.mean);
               time->dst.idle.stdev  = ntohl(time->dst.idle.stdev);
               time->dst.idle.max    = ntohl(time->dst.idle.max);
               break;
            }

            case ARGUS_SRCUSRDATA_DSR: {
               struct ArgusUserStruct *user = (struct ArgusUserStruct *) farhdr;
               user->status   = ntohs(user->status);
               break;
            }

            case ARGUS_DSTUSRDATA_DSR: {
               struct ArgusUserStruct *user = (struct ArgusUserStruct *) farhdr;
               user->status   = ntohs(user->status);
               break;
            }

            case ARGUS_ESP_DSR: {
               struct ArgusESPStruct *esp = (struct ArgusESPStruct *) farhdr;
               esp->status      = ntohs(esp->status);
               esp->src.spi     = ntohl(esp->src.spi);
               esp->src.lastseq = ntohl(esp->src.lastseq);
               esp->src.lostseq = ntohl(esp->src.lostseq);
               esp->dst.spi     = ntohl(esp->dst.spi);
               esp->dst.lastseq = ntohl(esp->dst.lastseq);
               esp->dst.lostseq = ntohl(esp->dst.lostseq);
               break;
            }

            case ARGUS_AGR_DSR: {
               struct ArgusAGRStruct *agr = (struct ArgusAGRStruct *) farhdr;
 
               agr->status               = ntohs(agr->status);
               agr->count                = ntohl(agr->count);
               agr->laststartime.tv_sec  = ntohl(agr->laststartime.tv_sec);
               agr->laststartime.tv_usec = ntohl(agr->laststartime.tv_usec);
               agr->lasttime.tv_sec      = ntohl(agr->lasttime.tv_sec);
               agr->lasttime.tv_usec     = ntohl(agr->lasttime.tv_usec);
               agr->act.min              = ntohl(agr->act.min);
               agr->act.mean             = ntohl(agr->act.mean);
               agr->act.stdev            = ntohl(agr->act.stdev);
               agr->act.max              = ntohl(agr->act.max);
               agr->idle.min             = ntohl(agr->idle.min);
               agr->idle.mean            = ntohl(agr->idle.mean);
               agr->idle.stdev           = ntohl(agr->idle.stdev);
               agr->idle.max             = ntohl(agr->idle.max);
               break;
            }
         }
         if ((farlen = farhdr->length) == 0)
            break;

         if ((farhdr->type == ARGUS_SRCUSRDATA_DSR) ||
             (farhdr->type == ARGUS_DSTUSRDATA_DSR))
            farlen = farlen * 4;

         length -= farlen;
         farhdr = (struct ArgusFarHeaderStruct *)((char *)farhdr + farlen);
      }
   }
}


void
ArgusHtoN (struct ArgusRecord *argus)
{
   int farlen = 0;
   unsigned int length = argus->ahdr.length - sizeof(argus->ahdr);
   unsigned int status = argus->ahdr.status;

   argus->ahdr.length    = htons(argus->ahdr.length);
   argus->ahdr.argusid   = htonl(argus->ahdr.argusid);
   argus->ahdr.seqNumber = htonl(argus->ahdr.seqNumber);
   argus->ahdr.status = htonl(argus->ahdr.status);

   if (argus->ahdr.type & ARGUS_MAR) {

      argus->argus_mar.startime.tv_sec  = htonl(argus->argus_mar.startime.tv_sec);
      argus->argus_mar.startime.tv_usec = htonl(argus->argus_mar.startime.tv_usec);
      argus->argus_mar.now.tv_sec  = htonl(argus->argus_mar.now.tv_sec);
      argus->argus_mar.now.tv_usec = htonl(argus->argus_mar.now.tv_usec);
      argus->argus_mar.reportInterval = htons(argus->argus_mar.reportInterval);
      argus->argus_mar.argusMrInterval = htons(argus->argus_mar.argusMrInterval);
      argus->argus_mar.argusid = htonl(argus->argus_mar.argusid);
      argus->argus_mar.localnet = htonl(argus->argus_mar.localnet);
      argus->argus_mar.netmask = htonl(argus->argus_mar.netmask);
      argus->argus_mar.nextMrSequenceNum = htonl(argus->argus_mar.nextMrSequenceNum);

      argus->argus_mar.pktsRcvd  = htonll(argus->argus_mar.pktsRcvd);
      argus->argus_mar.bytesRcvd = htonll(argus->argus_mar.bytesRcvd);

      argus->argus_mar.pktsDrop = htonl(argus->argus_mar.pktsDrop);
      argus->argus_mar.flows = htonl(argus->argus_mar.flows);
      argus->argus_mar.flowsClosed = htonl(argus->argus_mar.flowsClosed);

      argus->argus_mar.actIPcons = htonl( argus->argus_mar.actIPcons);
      argus->argus_mar.cloIPcons = htonl( argus->argus_mar.cloIPcons);
      argus->argus_mar.actICMPcons = htonl( argus->argus_mar.actICMPcons);
      argus->argus_mar.cloICMPcons = htonl( argus->argus_mar.cloICMPcons);
      argus->argus_mar.actIGMPcons = htonl( argus->argus_mar.actIGMPcons);
      argus->argus_mar.cloIGMPcons = htonl( argus->argus_mar.cloIGMPcons);
      argus->argus_mar.actFRAGcons = htonl( argus->argus_mar.actFRAGcons);
      argus->argus_mar.cloFRAGcons = htonl( argus->argus_mar.cloFRAGcons);
      argus->argus_mar.actSECcons = htonl( argus->argus_mar.actSECcons);
      argus->argus_mar.cloSECcons = htonl( argus->argus_mar.cloSECcons);

      argus->argus_mar.record_len = htonl(argus->argus_mar.record_len);

   } else {
      struct ArgusFarHeaderStruct *farhdr =
                   (struct ArgusFarHeaderStruct *) &argus->argus_far;

      farhdr->status = htons(farhdr->status);

      while (length > 0) {
         switch (farhdr->type) {
            case ARGUS_FAR: {
               struct ArgusFarStruct *far = (struct ArgusFarStruct *) farhdr;

               far->ArgusTransRefNum = htonl(far->ArgusTransRefNum);

               switch (status & (ETHERTYPE_IP|ETHERTYPE_IPV6|ETHERTYPE_ARP)) {
                  case ETHERTYPE_IP: {
                     struct ArgusIPFlow *ipflow = &far->flow.ip_flow;

                     far->attr_ip.soptions = htons(far->attr_ip.soptions);
                     far->attr_ip.doptions = htons(far->attr_ip.doptions);

                     switch (ipflow->ip_p) {
                        case IPPROTO_UDP:
                        case IPPROTO_TCP:
                           ipflow->ip_src = htonl(ipflow->ip_src);
                           ipflow->ip_dst = htonl(ipflow->ip_dst);
                           ipflow->sport  = htons(ipflow->sport);
                           ipflow->dport  = htons(ipflow->dport);
                           ipflow->ip_id  = htons(ipflow->ip_id);
                           break;

                        case IPPROTO_ICMP: {
                           struct ArgusICMPFlow *icmpflow = &far->flow.icmp_flow;

                           icmpflow->ip_src = htonl(icmpflow->ip_src);
                           icmpflow->ip_dst = htonl(icmpflow->ip_dst);
                           icmpflow->id     = htons(icmpflow->id);
                           icmpflow->ip_id  = htons(icmpflow->ip_id);
                           break;
                        }

                        default: {
                           ipflow->ip_src = htonl(ipflow->ip_src);
                           ipflow->ip_dst = htonl(ipflow->ip_dst);
                           break;
                        }
                     }
                     break;
                  }
         
                  case ETHERTYPE_IPV6:
                     break;
         
                  case ETHERTYPE_ARP: {
                     struct ArgusArpFlow *arpflow = &far->flow.arp_flow;
         
                     arpflow->arp_tpa = htonl(arpflow->arp_tpa);
                     arpflow->arp_spa = htonl(arpflow->arp_spa);
                     break;
                  }
                  default:
                     break;
               }
         
               far->time.start.tv_sec  = htonl(far->time.start.tv_sec);
               far->time.start.tv_usec = htonl(far->time.start.tv_usec);
               far->time.last.tv_sec   = htonl(far->time.last.tv_sec);
               far->time.last.tv_usec  = htonl(far->time.last.tv_usec);
         
               far->src.count    = htonl(far->src.count);
               far->src.bytes    = htonl(far->src.bytes);
               far->src.appbytes = htonl(far->src.appbytes);
         
               far->dst.count    = htonl(far->dst.count);
               far->dst.bytes    = htonl(far->dst.bytes);
               far->dst.appbytes = htonl(far->dst.appbytes);
               break;
            }

            case ARGUS_MAC_DSR: {
               struct ArgusMacStruct *mac = (struct ArgusMacStruct *) farhdr;
               mac->status   = htons(mac->status);
               break;
            }


            case ARGUS_VLAN_DSR: {
               struct ArgusVlanStruct *vlan = (struct ArgusVlanStruct *) farhdr;
               vlan->status = ntohs(vlan->status);
               vlan->sid    = ntohs(vlan->sid);
               vlan->did    = ntohs(vlan->did);
               break;
            }

            case ARGUS_MPLS_DSR: {
               struct ArgusMplsStruct *mpls = (struct ArgusMplsStruct *) farhdr;
               mpls->status = ntohs(mpls->status);
               mpls->slabel = ntohl(mpls->slabel);
               mpls->dlabel = ntohl(mpls->dlabel);
               break;
            }

            case ARGUS_TCP_DSR: {
               struct ArgusTCPObject *tcp = (struct ArgusTCPObject *) farhdr; 

               tcp->status = htons(tcp->status); 
               tcp->state  = htonl(tcp->state); 
               tcp->synAckuSecs  = htonl(tcp->synAckuSecs); 
               tcp->ackDatauSecs = htonl(tcp->ackDatauSecs); 
               tcp->options = htonl(tcp->options); 
               tcp->src.seqbase  = htonl(tcp->src.seqbase); 
               tcp->src.ackbytes = htonl(tcp->src.ackbytes); 
               tcp->src.rpkts    = htonl(tcp->src.rpkts); 
               tcp->src.win     = htons(tcp->src.win); 
               tcp->dst.seqbase  = htonl(tcp->dst.seqbase); 
               tcp->dst.ackbytes = htonl(tcp->dst.ackbytes); 
               tcp->dst.rpkts    = htonl(tcp->dst.rpkts); 
               tcp->dst.win     = htons(tcp->dst.win); 
               break;
            }

            case ARGUS_ICMP_DSR: {
               struct ArgusICMPObject *icmp = (struct ArgusICMPObject *) farhdr; 

               icmp->status   = htons(icmp->status);
               icmp->iseq     = htons(icmp->iseq);
               icmp->osrcaddr = htonl(icmp->osrcaddr);
               icmp->odstaddr = htonl(icmp->odstaddr);
               icmp->isrcaddr = htonl(icmp->isrcaddr);
               icmp->idstaddr = htonl(icmp->idstaddr);
               icmp->igwaddr  = htonl(icmp->igwaddr);
               break;
            }

            case ARGUS_TIME_DSR: {
               struct ArgusTimeStruct *time = (void *) farhdr;

               time->status = htons(time->status);
               time->src.act.n       = htonl(time->src.act.n);
               time->src.act.min     = htonl(time->src.act.min);
               time->src.act.mean    = htonl(time->src.act.mean);
               time->src.act.stdev   = htonl(time->src.act.stdev);
               time->src.act.max     = htonl(time->src.act.max);
               time->src.idle.n      = htonl(time->src.idle.n);
               time->src.idle.min    = htonl(time->src.idle.min);
               time->src.idle.mean   = htonl(time->src.idle.mean);
               time->src.idle.stdev  = htonl(time->src.idle.stdev);
               time->src.idle.max    = htonl(time->src.idle.max);
               time->dst.act.n       = htonl(time->dst.act.n);
               time->dst.act.min     = htonl(time->dst.act.min);
               time->dst.act.mean    = htonl(time->dst.act.mean);
               time->dst.act.stdev   = htonl(time->dst.act.stdev);
               time->dst.act.max     = htonl(time->dst.act.max);
               time->dst.idle.n      = htonl(time->dst.idle.n);
               time->dst.idle.min    = htonl(time->dst.idle.min);
               time->dst.idle.mean   = htonl(time->dst.idle.mean);
               time->dst.idle.stdev  = htonl(time->dst.idle.stdev);
               time->dst.idle.max    = htonl(time->dst.idle.max);
               break;
            }

            case ARGUS_SRCUSRDATA_DSR: {
               struct ArgusUserStruct *user = (struct ArgusUserStruct *) farhdr;
               user->status   = htons(user->status);
               break;
            }

            case ARGUS_DSTUSRDATA_DSR: {
               struct ArgusUserStruct *user = (struct ArgusUserStruct *) farhdr;
               user->status   = htons(user->status);
               break;
            }

            case ARGUS_ESP_DSR: {
               struct ArgusESPStruct *esp = (struct ArgusESPStruct *) farhdr;
               esp->status      = htons(esp->status);
               esp->src.spi     = htonl(esp->src.spi);
               esp->src.lastseq = htonl(esp->src.lastseq);
               esp->src.lostseq = htonl(esp->src.lostseq);
               esp->dst.spi     = htonl(esp->dst.spi);
               esp->dst.lastseq = htonl(esp->dst.lastseq);
               esp->dst.lostseq = htonl(esp->dst.lostseq);
               break;
            }

            case ARGUS_AGR_DSR: {
               struct ArgusAGRStruct *agr = (struct ArgusAGRStruct *) farhdr;

               agr->status               = htons(agr->status);
               agr->count                = htonl(agr->count);
               agr->laststartime.tv_sec  = htonl(agr->laststartime.tv_sec);
               agr->laststartime.tv_usec = htonl(agr->laststartime.tv_usec);
               agr->lasttime.tv_sec      = htonl(agr->lasttime.tv_sec);
               agr->lasttime.tv_usec     = htonl(agr->lasttime.tv_usec);
               agr->act.min              = htonl(agr->act.min);
               agr->act.mean             = htonl(agr->act.mean);
               agr->act.stdev            = htonl(agr->act.stdev);
               agr->act.max              = htonl(agr->act.max);
               agr->idle.min             = htonl(agr->idle.min);
               agr->idle.mean            = htonl(agr->idle.mean);
               agr->idle.stdev           = htonl(agr->idle.stdev);
               agr->idle.max             = htonl(agr->idle.max);
               break;
            }
         }
         if ((farlen = farhdr->length) == 0)
            break;

         if ((farhdr->type == ARGUS_SRCUSRDATA_DSR) ||
             (farhdr->type == ARGUS_DSTUSRDATA_DSR))
            farlen = farlen * 4;

         length -= farlen;
         farhdr = (struct ArgusFarHeaderStruct *)((char *)farhdr + farlen);
      }
   }
}

#endif /* _LITTLE_ENDIAN */


void
ArgusPrintHex (const u_char *bp, u_int length)
{
   const u_short *sp;
   u_int i;
   int nshorts;

   sp = (u_short *)bp;
   nshorts = (u_int) length / sizeof(u_short);
   i = 0;
   while (--nshorts >= 0) {
      if ((i++ % 8) == 0) {
         (void)printf("\n\t");
      }
      (void)printf(" %04x", ntohs(*sp++));
   }

   if (length & 1) {
      if ((i % 8) == 0)
         (void)printf("\n\t");

      (void)printf(" %02x", *(u_char *)sp);
   }
   (void)printf("\n");
   fflush(stdout);
}



int ArgusMallocTotal = 0;
int ArgusCallocTotal = 0;
int ArgusFreeTotal = 0;

#define ARGUS_ALLOC   0x3210123

void *
ArgusMalloc (int size)
{
   void *retn = NULL;

   if (size) {
      ArgusMallocTotal++;
      retn = malloc (size);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (7, "ArgusMalloc (%d) returning 0x%x\n", size, retn);
#endif
   return (retn);
}


void *
ArgusCalloc (int nitems, int size)
{
   void *retn = NULL;

   if (size) {
      ArgusCallocTotal++;
      retn = calloc (nitems, size);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (7, "ArgusCalloc (%d, %d) returning 0x%x\n", nitems, size, retn);
#endif
   return (retn);
}


void
ArgusFree (void *buf)
{
   if (buf) {
      ArgusFreeTotal++;
      free (buf);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (7, "ArgusFree (0x%x) returning\n", buf);
#endif
}

extern char *print_time(struct timeval *);
#include <syslog.h>

struct ArgusLogPriorityStruct {
   int priority;
   char *label;
};

#define ARGUSPRIORITYSTR	8
struct ArgusLogPriorityStruct ArgusPriorityStr[ARGUSPRIORITYSTR] =
{
   { LOG_EMERG,   "ArgusEmergency" },
   { LOG_ALERT,   "    ArgusAlert" },
   { LOG_CRIT,    " ArgusCritical" },
   { LOG_ERR,     "    ArgusError" },
   { LOG_WARNING, "  ArgusWarning" },
   { LOG_NOTICE,  "   ArgusNotice" },
   { LOG_INFO,    "     ArgusInfo" },
   { LOG_DEBUG,   "    ArgusDebug" },
};


char ArgusErrorLog[MAXSTRLEN];

char * ArgusGetError(void);

char *
ArgusGetError()
{
   return ArgusErrorLog;
};

void
ArgusLog (int priority, char *fmt, ...)
{
   va_list ap;
   char *ptr = ArgusErrorLog;

#ifdef ARGUS_SYSLOG
#ifndef LOG_PERROR
#define LOG_PERROR	LOG_CONS
#endif

   openlog (ArgusProgramName, LOG_PID, LOG_DAEMON);
#else
   int i;
   char *label;

   if (priority == LOG_NOTICE)
      return;

   sprintf (ArgusErrorLog, "%s[%d]: %s ", ArgusProgramName, (int)getpid(), print_time(&ArgusGlobalTime));
   ptr = &ArgusErrorLog[strlen(ArgusErrorLog)];

#endif /* ARGUS_SYSLOG */

   va_start (ap, fmt);
   (void) vsprintf (ptr, fmt, ap);
   ptr = &ArgusErrorLog[strlen(ArgusErrorLog)];
   va_end (ap);

   if (*fmt) {
      fmt += (int) strlen (fmt);
      if (fmt[-1] != '\n')
         sprintf (ptr, "\n");
   }

#ifdef ARGUS_SYSLOG
   syslog (priority, ArgusErrorLog, strlen(ArgusErrorLog));
   closelog ();
#else

   for (i = 0; i < ARGUSPRIORITYSTR; i++)
      if (ArgusPriorityStr[i].priority == priority) {
         label = ArgusPriorityStr[i].label;
         break;
      }

   if (priority == LOG_ERR)
      fprintf (stderr, "%s: %s", label, ArgusErrorLog);

#endif /* ARGUS_SYSLOG */

   switch (priority) {
      case LOG_ERR: ArgusShutDown(-1); break;
      default: break;
   }
}

