/* $Id: ArkImage.cpp,v 1.19 2003/03/16 02:24:23 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2002 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <Ark/ArkImage.h>
#include <Ark/ArkSystem.h>
#include <Ark/ArkConfig.h>
#include <Ark/ArkFileSys.h>

#include <math.h>

#include <sstream>
#include <string>


namespace Ark
{

const int Image::BytesPerPixel[FORMAT_COUNT] =
{
	0, // NONE_0,
	3, // RGB_888,
	4, // RGBA_8888,
	1, // I_8,
	1, // A_8,
	2, // IA_88,
};

static const char *FormatToString (int format)
{
   switch (format)
   {
      case Image::NONE_0:
	 return "No data";
      case Image::RGB_888:
	 return "RGB 24 bits";
      case Image::RGBA_8888:
	 return "RGBA 32 bits";
      case Image::I_8:
	 return "Intensity 8 bits";
      case Image::A_8:
	 return "Alpha 8 bits";
      case Image::IA_88:
	 return "Intensity Alpha 16 bits";
   }

   return "Unknown";
}

Image::Image (const String &name, int w, int h, Format format, uchar *data) :
  Object (name)
{
  m_Data = NULL;
  m_Type = V_IMAGE;

  SetFormat (w, h, format, data);
}

Image::Image () : Object ("Image")
{
  m_Data = NULL;
  m_Format = NONE_0;
  m_Width = m_Height = 0;
  m_BytesPerPixel = 0;
  m_Type = V_IMAGE;
}

Image::~Image ()
{
  if (m_Data)
    delete m_Data;
}

/**
 * Returns the color of a point.
 */
Color 
Image::GetColor(float u_, float v_) const
{
    // Clamp to edges
    const float u = (u_<0.f) ? 0.f : (1.f<u_) ? 1.f : u_;
    const float v = (v_<0.f) ? 0.f : (1.f<v_) ? 1.f : v_;

    const float w = float(m_Width-1);
    const float xf = u * w;
    const float xff = floorf(xf);

    const float lu = xf - xff;
    const int x0 = int(xff);
    const int x1 = ((int)m_Width <= (x0+1)) ? x0 : x0+1;

    const float h = float(m_Height-1);
    const float yf = v * h;
    const float yff = floorf(yf);

    const float lv = yf - yff;
    const int y0 = int(yff);
    const int y1 = ((int)m_Height <= (y0+1)) ? y0 : y0+1;

    // Bilinearly interpolate from surrounding points
    Color c00 = GetPixelColor(x0, y0);
    Color c01 = GetPixelColor(x0, y1);
    Color c10 = GetPixelColor(x1, y0);
    Color c11 = GetPixelColor(x1, y1);

    Color c;
    c.R = Math::BilinearInterpolation(c00.R, c10.R, c01.R, c11.R, lu, lv);
    c.G = Math::BilinearInterpolation(c00.G, c10.G, c01.G, c11.G, lu, lv);
    c.B = Math::BilinearInterpolation(c00.B, c10.B, c01.B, c11.B, lu, lv);
    c.A = Math::BilinearInterpolation(c00.A, c10.A, c01.A, c11.A, lu, lv);

    return c;
}

/**
 * Returns the color of a pixel.
 */
Color 
Image::GetPixelColor(int x, int y) const
{
	const int offset = GetPixelOffset(x, y);
	switch (m_Format)
	{
		case Image::RGB_888:
		{
			const float red   = float(m_Data[offset+0]) / 255.f;
			const float green = float(m_Data[offset+1]) / 255.f;
			const float blue  = float(m_Data[offset+2]) / 255.f;
			return Color(red, green, blue, 1.f);
		}
		case Image::RGBA_8888:
		{
			const float red   = float(m_Data[offset+0]) / 255.f;
			const float green = float(m_Data[offset+1]) / 255.f;
			const float blue  = float(m_Data[offset+2]) / 255.f;
			const float alpha = float(m_Data[offset+3]) / 255.f;
			return Color(red, green, blue, alpha);
		}
		case Image::A_8:
		{
			const float alpha = float(m_Data[offset]) / 255.f;
			return Color(1.f, 1.f, 1.f, alpha);
		}
		case Image::IA_88:
		{
			const float alpha = float(m_Data[offset+1]) / 255.f;
			return Color(1.f, 1.f, 1.f, alpha);
		}
	   default: break;
	}

	// white is default
	return Color(1.f, 1.f, 1.f, 1.f);
}

/**
 * Returns the intensity of a pixel.
 */
float 
Image::GetPixelIntensity(int x, int y) const
{
	const int offset = GetPixelOffset(x, y);
	switch (m_Format)
	{
		case Image::RGB_888:
		case Image::RGBA_8888:
		{
			const float red   = float(m_Data[offset+0]) / 255.f;
			const float green = float(m_Data[offset+1]) / 255.f;
			const float blue  = float(m_Data[offset+2]) / 255.f;
			const float y = Color::GetLuminance(red, green, blue);
			return y;
		}
		case Image::I_8:
		case Image::IA_88:
		{
			const float y = float(m_Data[offset+0]) / 255.f;
			return y;
		}
	   default: break;
	}

	// white is default
	return 1.f;
}

/**
 * Returns the alpha value of a pixel.
 */
float 
Image::GetPixelAlpha(int x, int y) const
{
	const int offset = GetPixelOffset(x, y);
	switch (m_Format)
	{
		case Image::RGBA_8888:
		{
			const float alpha = float(m_Data[offset+3]) / 255.f;
			return alpha;
		}
		case Image::A_8:
		{
			const float alpha = float(m_Data[offset+0]) / 255.f;
			return alpha;
		}
		case Image::IA_88:
		{
			const float alpha = float(m_Data[offset+1]) / 255.f;
			return alpha;
		}
	   default:
	      break;
	}

	// 'white' is default
	return 1.f;
}

/// Set the image format, and image data if \c data isn't null.
bool
Image::SetFormat (int w, int h, Format format, uchar *data)
{
  m_Width = w;
  m_Height = h;
  m_Format = format;
  m_BytesPerPixel = GetBytesPerPixel(format);

  if (m_Data != NULL)
  {
	  delete[] m_Data;
	  m_Data = 0;
  }

  const int dsize = m_BytesPerPixel * m_Width * m_Height;

  if (0 < dsize)
  {
	  m_Data = new uchar[dsize];

	  if (data)
		  memcpy (m_Data, data, dsize);
	  else
		  memset (m_Data, 0, dsize);
  }

  return true;
}

void WriteTGA_RGB (WriteStream &file, uchar *image, int width, int height);
void WriteTGA_Gray (WriteStream &file, uchar *image, int width, int height);

void
Image::SaveTGA (const Ark::String &name)
{
  if (m_Format == RGBA_8888)
    return;

  String fname = Sys()->FS()->GetFileName(name);
  FileWriteStream file(fname.c_str(), std::ios::out | std::ios::binary);

  if (file.bad())
    return;

  if (m_Format == RGB_888)
    WriteTGA_RGB (file, m_Data, m_Width, m_Height);
  else if (m_Format == I_8)
    WriteTGA_Gray (file, m_Data, m_Width, m_Height);
  else
	  Sys()->Warning ("Saving TGA image format \"%s\" is not supported.", FormatToString(m_Format));
}

/// Returns a string describing the object
String
Image::Dump (bool long_version)
{
    std::ostringstream os;

    os << "[img] " << m_Name;
    
    if (long_version)
    {
	os << std::endl << "\twidth : " << m_Width;
	os << std::endl << "\theight : " << m_Height;
	os << std::endl << "\tformat : " << FormatToString (m_Format);
    }

    return os.str();
}

/* namespace Ark */
}
