/* $Id: ArkMath.h,v 1.30 2003/03/20 17:23:25 zongo Exp $
** 
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2002 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef ARK_MATH_H
#define ARK_MATH_H

#include <Ark/Ark.h>
#include <Ark/ArkTypes.h>
#include <Ark/ArkString.h>
#include <Ark/ArkStream.h>

#ifndef FASTCALL
#define FASTCALL
#endif

#ifdef WIN32
// Avoid clash with those stupid windows macros... 
#undef min
#undef max
#endif

#include <iostream>

namespace Ark
{

#ifndef PI
#define PI 3.1415926535897932f
#endif

/// Mathematics functions
class ARK_DLL_API Math
{
  static void MakeInvSqrtTable ();

public:

  /// Initialise the mathematics global tables
  static void Init ();

  /// Return the inverse of the square root of value (fast method,
  /// MakeInvSqrtTable should be called before)
  static scalar FASTCALL InvSqrt (scalar value);

  /// Converts 'd' from degrees to radians
  inline static float DegToRad (float d) {return (d * PI)/180.0f;}
  /// Converts 'd' from radians to degrees
  inline static float RadToDeg (float d) {return (d * 180.0f)/PI;}
  
  /**
   * Bilinearly interpolate from values
   *
   * ll Lower-left value
   * lr Lower-right value
   * ul Upper-left value
   * ur Upper-rifgt value
   * xf Factor between left and right
   * yf Factor between lower and upper
   */
  inline static float BilinearInterpolation(
		  float ll, float lr, float ul, float ur, 
		  float xf, float yf)
  {
	  const scalar v0 =  ll + (lr - ll) * xf;
	  const scalar v1 =  ul + (ur - ul) * xf;
	  const scalar v = v0 + (v1 - v0) * yf;
	  return v;
  }
	
};

/*---------------------------------------- Color operations */

class ARK_DLL_API Color
{
public:
  scalar R, G, B, A;

public:
  Color () : R(0.0f), G(0.0f), B(0.0f), A(1.0f) {}

  Color (scalar r, scalar g, scalar b, scalar a = 1.0f)
    : R(r), G(g), B(b), A(a) {}

  Color (uchar r, uchar g, uchar b, uchar a = 255) :
    R( scalar(r) / 255.0f ),
    G( scalar(g) / 255.0f ),
    B( scalar(b) / 255.0f ),
    A( scalar(a) / 255.0f ) {}

	/// Gives the luminance for a color
	inline static float GetLuminance(float r, float g, float b)
	{
		// This is the CIE definition
		const float y = 0.212671f * r + 0.715160f * g + 0.072169f * b;
		return y;
	}

	/// Gives the luminance of current color
	inline float GetLuminance() const
	{ return GetLuminance(R, G, B); }
};

typedef Color Colour;

inline bool
operator ==(const Color &c1, const Color &c2)
{
   return (c1.R == c2.R) && (c1.G == c2.G) && (c1.B == c2.B);
}

inline bool
operator != (const Color &c1, const Color &c2)
{
   return (c1.R != c2.R) || (c1.G != c2.G) || (c1.B != c2.B);
}

/**
 *  Class to manipulate 3D vectors
 */
class ARK_DLL_API Vector3
{
public:
  scalar X, Y, Z;

public:
  inline Vector3() : X(0.0f), Y(0.0f), Z(0.0f) {}

  inline Vector3(scalar x, scalar y, scalar z)
    : X(x), Y(y), Z(z) {}

  Vector3 GetUnitVector () const;
  Vector3 &Normalize ();

  /**
   *  Compute magnitude
   */
  scalar GetMagnitude() const;

  /**
   *  Compute squared magnitude which is faster than computing
   *  magnitude since we don't need to calculate a sqrt
   */
  scalar GetMagnitude2() const;

  Vector3 &Scale (scalar x);
  Vector3 &Scale (const Vector3 &f);

  Vector3 &operator += (const Vector3 &B);
  Vector3 &operator -= (const Vector3 &B);
  Vector3 &operator ^= (const Vector3 &B);

  Vector3 operator + (const Vector3 &B) const;
  Vector3 operator - (const Vector3 &B) const;
  /** 
   *  Cross product of the current vector and B
   */
  Vector3 operator ^ (const Vector3 &B) const;

  /**
   *  Dot product 
   */
  scalar  operator * (const Vector3 &B) const;

  /**
   *  Compute a normalized normal for an (A,B,C) triangle 
   */
  static Vector3 ComputeNormal (const Vector3 &A,
                                const Vector3 &B,
                                const Vector3 &C);

  friend std::ostream& operator<< (std::ostream& output, const Vector3& vector);
};

inline std::ostream& operator<< (std::ostream& output, const Vector3& v)
{
  return output << "(" << v.X << "," << v.Y << "," << v.Z << ")";
}

inline Vector3
operator * (scalar factor, const Vector3 &B)
{
   return Vector3 (B.X * factor, B.Y * factor, B.Z * factor);
}

/// This is only needed for map/set insertion (else it has no meaning..)
inline bool operator < (const Vector3 &v1, const Vector3 &v2)
{
   if (v1.X != v2.X)
      return v1.X < v2.X;
   if (v1.Y != v2.Y)
      return v1.Y < v2.Y;
   return v1.Z < v2.Z;
}

inline bool operator == (const Vector3 &v1, const Vector3 &v2)
{
   return (v1.X == v2.X) && (v1.Y == v2.Y) && (v1.Z == v2.Z);
}

inline bool operator != (const Vector3 &v1, const Vector3 &v2)
{
   return !(v1 == v2);
}

/**
 *  Class to manipulate 2D Vectors
 */
class ARK_DLL_API Vector2
{
public:
  scalar X, Y;

public:
  inline Vector2() : X(0.0f), Y(0.0f) {}
      
  inline Vector2(scalar x, scalar y) : X(x), Y(y) {}

  inline Vector2 GetUnitVector () const
  {
    Vector2 R = *this;
    R.Normalize();
    return R;
  }

  void Normalize ();

  /** Compute magnitude */
  scalar GetMagnitude() const;
  
  /** Compute squared magnitude which is faster since there is no sqrt 
   *  to calculate
   */
  scalar GetMagnitude2() const;

  Vector2 &Scale (scalar x);
  Vector2 &Scale (const Vector2 &f);

  Vector2 &operator += (const Vector2 &B);
  Vector2 &operator -= (const Vector2 &B);

  Vector2 operator + (const Vector2 &B) const;
  Vector2 operator - (const Vector2 &B) const;
  /** 
   *  Dot product of the current vector and B
   */
  scalar  operator * (const Vector2 &B) const;
};

/// This is only needed for map/set insertion (else it has no meaning..)
inline bool operator < (const Vector2 &v1, const Vector2 &v2)
{
   if (v1.X != v2.X)
      return v1.X < v2.X;
   return v1.Y < v2.Y;
}

inline bool operator == (const Vector2 &v1, const Vector2 &v2)
{
   return (v1.X == v2.X) && (v1.Y == v2.Y);
}

inline bool operator != (const Vector2 &v1, const Vector2 &v2)
{
   return (v1.X != v2.X) || (v1.Y != v2.Y);
}


/** 
 *  The Matrix44 class stores a 4x4 homogenous matrix and implements
 *  classical operations on them
 */
class ARK_DLL_API Matrix44
{
public:
  /** Store the 16 coefficients of the matrix */ 
  scalar m_Elems [16];

public:
  /** Create a new matrix non-initialised! */
  inline Matrix44 () {}

  /** Create a new matrix initialised to the identity matrix */
  inline Matrix44(int unused) { MakeIdentity (); }

  /** Set the matrix to identity */
  void MakeIdentity ();

  /** 
   *  Compute the invert of the matrix. Returns true if successful, false
   *  otherwise. Don't forget that matrices corresponding to a combination 
   *  of rotations are orthogonal, so their invert is their 
   *  transpose, which is much less expensive to calculate than using 
   *  this function.
   */
  bool Invert ();

  /**
   *  Set the current matrix so that it corresponds to a rotation along 
   *  X axis of angle radians
   */
  void MakeRotationX(scalar angle);

  /**
   *  Set the current matrix so that it corresponds to a rotation along 
   *  Y axis of angle radians
   */
  void MakeRotationY(scalar angle);

  /**
   *  Set the current matrix so that it corresponds to a rotation along 
   *  Z axis of angle radians
   */
  void MakeRotationZ(scalar angle);

  /**
   *  Generate a matrix which scales vectors by factors
   */
  void MakeScale(const Vector3 &factors);

  /**
   *  Generate a matrix which translate vectors by pos
   */
  void MakeTranslation(const Vector3 &pos);
 
  /**
   *  Store the result of the multiplication of the current matrix by the 
   *  matrix by in the current matrix
   */
  void Multiply (const Matrix44 &by);

  /**
   *  Changes the matrix to be composed by a translation.
   */
  void PostTranslate(const Vector3& vector);

  /** 
   *  Apply the transformation currently stored in the matric to pos
   */  
  Vector3 Transform (const Vector3 &pos) const;

  /** 
   *  Apply only the rotation part (ie the upper 3x3 matrix) stored in
   *  the matrix to pos
   */
  Vector3 Rotate    (const Vector3 &pos) const;

  /** 
   *  Apply the invert of the rotation stored in the matrix to pos
   */
  Vector3 InvRotate (const Vector3 &pos) const;

  /**
   *  Retrieve the [x,y] element from the matrix
   */
  inline scalar &M (int x, int y)
  {
    return m_Elems[x * 4 + y];
  }

  /**
   *  Retrieve the [x,y] element from the matrix
   */
  inline scalar M (int x, int y) const
  {
    return m_Elems[x * 4 + y];
  }
};


/*---------------------------------------- Plane Computation */
class ARK_DLL_API Plane
{
public:
  Vector3 Normal;
  scalar D;

public:
  inline Plane () : Normal(), D(0.0f) {}

  inline Plane (const Vector3& n, scalar d) : Normal(n), D(d) {}

  inline Plane (scalar X, scalar Y, scalar Z, scalar d)
      : Normal(X, Y, Z), D(d) {}

  /* The matrix should be inversed. If you already have computed it, you'd
  ** better call TransformInv */
  void Transform (const Matrix44 &mat);
  void TransformInv (const Matrix44 &mat);

  static Plane GetTriPlane (const Vector3& A, const Vector3& B, const Vector3& C);
};

/*---------------------------------------- Quaternion stuff */
class ARK_DLL_API Quaternion
{
public:
  scalar W, X, Y, Z;

public:
  inline Quaternion() : W(1.f), X(0.f), Y(0.f), Z(0.f) {}
  inline Quaternion(scalar w, scalar x, scalar y, scalar z) 
	  : W(w), X(x), Y(y), Z(z) {}

  /* Head, pitch, rotation */
  Quaternion (scalar y, scalar p, scalar r);
  /* Angle, axis */
  Quaternion (scalar angle, const Vector3 &axis);

  void Normalize ();
  void Zero ();

  const Quaternion& operator*= (const Quaternion& q);

  /** 
   *  Apply the rotation currently stored in the quaternion to pos
   */  
  Vector3 Transform(const Vector3 &pos) const;

  Matrix44 GetInvMatrix () const;
  Matrix44 GetMatrix () const;
  Vector3  GetDirVector () const;
  void GetAxisAngle (Vector3 *axis, scalar *angle) const;

  void GetEuler (scalar *y, scalar *p, scalar *r);

  static Quaternion Slerp (const Quaternion &from, const Quaternion &to, scalar t);

  // Return a quaternion conjuggate
  inline static Quaternion Conjugate(const Quaternion &from)
  { return Quaternion(from.W, -from.X, -from.Y, -from.Z); }

};

inline Quaternion operator * (const Quaternion& q1, const Quaternion& q2)
{
	Quaternion result(q1);
	result *= q2;
	return result;
}

// Typedef to disappear
typedef Quaternion Quat;

/*---------------------------------------- Bounding box stuff */

/// Bounding box
class ARK_DLL_API BBox
{
public:
  /// Bottom-Left-Near corner
  Vector3 m_Min;
  /// Top-Right-Far corner
  Vector3 m_Max;

public:
  /// Create an empty bounding box.
  BBox ();

  /// Make this bounding box empty.
  void MakeEmpty();

  /// Test wether this bounding box is empty
  bool Empty() const;

  /// Makes this bounding box bigger so can contain \c box.
  inline void AddBBox  (const BBox &box)
  { AddPoint (box.m_Min); AddPoint (box.m_Max); }

  /// Makes this bounding box bigger so can contain \c A.
  void AddPoint (const Vector3 &A);

  /// Return true if \c A is in the bounding box.
  bool IsInside (const Vector3 &A) const;

  /**
   * Find the smaller box containing all the transformed points (8) of this
   * bounding box.
   */
  void Transform (const Ark::Matrix44 &m);

  /// Get the eight corners of this bounding box.
  void GetCorners (Ark::Vector3 *vertices) const;

  /**
   * This function returns true if the two bounding boxes
   * overlap (their intersection isn't empty)
   */
  bool Overlap (const BBox &b) const;
};

/*---------------------------------------- Sphere stuff */

/// Bounding sphere
class ARK_DLL_API Sphere
{
public:
  /// Center
  Vector3 m_Center;
  /// Radius
  scalar m_Radius;

public:
  /// Create a bounding sphere.
  Sphere(const Vector3& center, scalar radius)
      : m_Center( center ), m_Radius( radius ) {}

};

/*---------------------------------------- Frustum ops */
  enum Visibility
  {
      OUTSIDE = 0,
      INSIDE = 1,
      SOME_CLIP = 2
  };

class ARK_DLL_API Frustum
{
public:
  Plane m_Planes[6];
  uint8 m_VertexCode[6];

public:
  Frustum (scalar near_ = 0.0, scalar far_ = 0.0,
           scalar hfov = 0.0, scalar vfov = 0.0);
  Frustum (Plane *planes);

  void Dump ();
  void ComputeVCode();


  void Transform (const Matrix44 &m);
  int  Outcode  (const Vector3 &point) const;
  inline bool IsInside (const Vector3 &point) const
       {return Outcode (point) == 0;}

  Visibility GetVisibility(const BBox &box) const;
  inline bool IsInside (const BBox &box) const
        {return GetVisibility (box) != OUTSIDE;}
};

/* namespace Ark */
}

#endif
