/* $Id: ArkModelOptimize.cpp,v 1.5 2003/03/26 15:43:01 mrq Exp $
** 
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <vector>

#include <Ark/ArkModel.h>
#include "NvTriStrip.h"

namespace Ark
{
   /// Optimize this mesh by creating triangle strips / fans.
   void
   Mesh::Optimize ()
   {
      PrimitiveGroup *primGroups;
      unsigned short numGroups;
      std::vector<unsigned short> indices;

      // Collect indices.
      for (BlockList::iterator pb = m_Blocks.begin();
	   pb != m_Blocks.end(); ++pb)
      {
	 switch (pb->Type())
	 {
	    case PRIM_TRIANGLES:
	    {
	       for (size_t i = 2; i < pb->Size(); i += 3)
	       {
		  indices.push_back ((*pb)[i-2]);
		  indices.push_back ((*pb)[i-1]);
		  indices.push_back ((*pb)[i]);
	       }
	       break;
	    }

	    case PRIM_TRIANGLE_FAN:
	    {
	       for (size_t i = 2; i < pb->Size(); i += 2)
	       {
		  indices.push_back ((*pb)[0]);
		  indices.push_back ((*pb)[i-1]);
		  indices.push_back ((*pb)[i]);
	       }
	    }
	    break;

	    case PRIM_TRIANGLE_STRIP:
	    {
	       for (size_t i = 2; i < pb->Size(); i++)
	       {
		  // the order of vertices in triangles is not the same
		  // when i is a multiple of two and when it's not..
		  
		  if (i & 1) 
		  {
		     indices.push_back ((*pb)[i-2]);
		     indices.push_back ((*pb)[i-1]);
		     indices.push_back ((*pb)[i]);
		  }
		  else
		  {
		     indices.push_back ((*pb)[i-2]);
		     indices.push_back ((*pb)[i]);
		     indices.push_back ((*pb)[i-1]);
		  }
	       }
	       break;
	    }
	    
	    default:
	       break;
	 }
	 
      }

      // Create triangle strips & copy them
      GenerateStrips (&indices[0], indices.size(), &primGroups, &numGroups);
      m_Blocks.clear();
      m_Blocks.resize(numGroups);

      for (size_t g = 0; g < numGroups; g++)
      {
	 m_Blocks.push_back(PrimitiveBlock ());

	 PrimitiveBlock &pb = m_Blocks.back();
	 PrimitiveType tp = PRIM_TRIANGLE_FAN;

	 if (primGroups[g].type == PT_STRIP)
	    tp = PRIM_TRIANGLE_STRIP;
	 else if (primGroups[g].type == PT_LIST)
	    tp = PRIM_TRIANGLES;

	 pb.SetType (tp);
	 pb.Resize (primGroups[g].numIndices);

	 memcpy (&pb[0], primGroups[g].indices, 
		 primGroups[g].numIndices*sizeof(unsigned short));
      }

      delete[] primGroups;
   }

   /**
    * Optimize the model, by stripifying the meshes, and reorganizing
    * the vertex data in order to reduce cache loss.
    */
   void
   SubModel::Optimize()
   {
      for (MeshLI i = m_Meshes.begin(); i != m_Meshes.end(); ++i)
	 i->Optimize();

#if 0 //FIXME: doesn't work (maybe the vertex buffer Remap function)

      // Remap the vertex indices to improve spatial locality in the vertex
      // buffer.

      // Caches oldIndex --> newIndex conversion
      vector<size_t> indexCache;
      indexCache.resize(m_VB.Size());
      
#define INVALID ((size_t)(-1))
      for (size_t i = 0; i < indexCache.size(); i++)
	 indexCache[i] = INVALID;

      size_t indexCtr = 0;

      for (MeshLI m = m_Meshes.begin(); m != m_Meshes.end(); ++m)
      {
	 for(BlockLI b = m->m_Blocks.begin(); b != m->m_Blocks.end(); ++b)
	 {
	    for(size_t j = 0; j < b->Size(); j++)
	    {
	       size_t cachedIndex = indexCache[(*b)[j]];
	       if(cachedIndex == INVALID) //we haven't seen this index before
	       {
		  // Point to "last" vertex in VB
		  (*b)[j] = indexCtr;
		  
		  // Add to index cache, increment
		  indexCache[(*b)[j]] = indexCtr++;
	       }
	       else
	       {
		  // We've seen this index before
		  (*b)[j] = cachedIndex;
	       }
	    }
	 }
      }

      cout << INVALID;
      for (size_t i = 0; i < indexCache.size(); i++)
      {
	 cout << indexCache[i] << " ";
	 if (indexCache[i] == INVALID)
	    indexCache[i] = indexCtr++;
      }

      // Remap vertex buffer.
      m_VB.Remap (indexCache);

      // Remap bone bindings.
      BoneBindingList oldbonebindings = m_BoneBindings;
      for (size_t i = 0; i < m_BoneBindings.size(); i++)
	 m_BoneBindings[indexCache[i]] = oldbonebindings[i];
#endif
			
   }


   void
   Model::Optimize ()
   {
      if (m_Optimized)
	 return;

      for (SubModelLI i = m_SubModels.begin(); i != m_SubModels.end(); i++)
	 i->Optimize();

      m_Optimized = true;
   }

}
