/* $Id: ArkRenderer.h,v 1.24 2003/03/15 17:49:04 mrq Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2002 The Contributors of the ArkRPG Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef ARK_RENDERER_H
#define ARK_RENDERER_H

#include <Ark/ArkCache.h>
#include <Ark/ArkFont.h>
#include <Ark/ArkFactory.h>

namespace Ark
{
   class Light;
   class Material;
   class PrimitiveBlock;
   class VertexBuffer;

   // Note: no ARK_DLL_API because there are no member functions but a template one
   class Camera
   {
	public:
		Vector3 m_PointOfView;
		Vector3 m_LookAt;
		scalar  m_FOV;

		template<class TargetChange, class PositionChange>
		inline void Move(TargetChange& target, PositionChange& position)
		{
			target(m_LookAt);
			position(m_PointOfView);
		}
   };
   
   class ARK_DLL_API CameraInstantChange
   {
		const Vector3& m_Point;
	public:
		inline CameraInstantChange(const Vector3& point) : m_Point(point) {}
		inline void operator()(Vector3& position) { position = m_Point; }
   };
   
   // ========================================================================
   /**
    * The Renderer class is used to abstract access to the different rendering
    * acceleration systems (ie Glide, OpenGL or DirectX). Even if now only
    * OpenGL is supported, having abstraction in mind it should be quite easy
    * to add a new DirectX renderer.
    */
   // ========================================================================
   
   class ARK_DLL_API Renderer : public DRefCounter
   {
      public:
	 /**
	  * Compute the current view frustum (ie six planes to cull out what
	  * is not currently in the view)
	  */
	 virtual const Frustum &GetFrustum() const = 0;

	 /**
	  * Get the current projection matrix
	  */
	 virtual const Matrix44 &GetProjectionMatrix() = 0;

	 /**
	  * Get the current view matrix
	  */
	 virtual const Matrix44 &GetViewMatrix() = 0;

	 /**
	  * Push a new view (ie model transformation) matrix
	  */
	 virtual void PushModelMatrix(Matrix44 &matx) = 0;

	 /**
	  * Pop last view (ie model transformation) matrix
	  */
	 virtual void PopModelMatrix() = 0;

	 /// Get the renderer object cache
	 virtual Cache &GetCache() = 0;

	 // =================================================================
	 /**
	  * Set the active vertex buffer. The data defined in this buffer will
	  * be used in subsequent calls to RenderBlock(), as long as they are
	  * not overriden by OverrideVB().
	  */
	 virtual bool SetActiveVB (const VertexBuffer &VB) = 0;

	 /**
	  * This overrides one or more values of the active vertex buffer. This
	  * can be used for multitexturing : the VB is set in the first pass,
	  * then the texture coordinates are overriden for the second pass,
	  * but the transformed coordinates are still kept in the card's
	  * memory.
	  *
	  * \arg override_flags This is a composition of VB_* flags, so you
	  * can have a full vertex buffer and choose to use only some one of
	  * its data.
	  */
	 virtual bool OverrideVB (int override_flags,
				  const VertexBuffer &VB) = 0;

	 /**
	  * Between a LockVB() / UnlockVB() call you can't change the \c n
	  * values of the active vertex buffer from \c begin. These values
	  * may be sent to the video card (if it supports T&L), that's why you
	  * can't change them. Moreover, being sure that they won't be changed
	  * allows the renderer to flag vertex as transformed when they are
	  * effectively processed and to reuse the already transformed vector
	  * if the same vertex is accessed several times in a primitive block. 
	  *
	  * The corresponding OpenGL extension is compiled_vertex_arrays, or
	  * CVA.
	  *
	  * It's OK to call OverrideVB() while the active vertex buffer is
	  * locked, but overriden data will of course need to be processed
	  * another time.
	  */
	 virtual bool LockVB (int begin, size_t n) = 0;

	 /**
	  * Unlock the active vertex buffer, so you can change its values at
	  * last.
	  */
	 virtual bool UnlockVB () = 0;

	 /**
	  * This function does not have to be called before rendering a block,
	  * but if might sometime enhance the rendering speed. Eg you have to
	  * renderer several blocks using the same vertex buffer, and the same
	  * texture generation, so you might try to call SetupMaterial() before
	  * calling LockVB() so texture coordinates might be reused for all
	  * the blocks.
	  */
	 virtual bool SetupMaterial (const Material &mat) = 0;

	 /**
	  * Render a primitive block. 
	  */
	 virtual bool RenderBlock (const Material &material,
				   PrimitiveType type,
				   const ushort *indices,
				   size_t n) = 0;

	 virtual void RenderLight(const Light &, int) = 0;

	 /**
	  * Render a primitive block (ie a list of primitives all using the
	  * same material).
	  */
	 inline bool RenderBlock (const Material &material, const PrimitiveBlock &block)
	 {
	    return RenderBlock(material, block.Type(), &block[0], block.Size());
	 }

	 /**
	  * Render a primitive block. This version of the function doesn't
	  * use the locked vertex buffer, and doesn't use index. This is
	  * slightly less fast for big primitive blocks (ie more than 100
	  * triangles), but for little primitive blocks, such as those
	  * corresponding to the user-interface, it has a lot less overhead.
	  */
	 virtual bool RenderBlock (const Material &material,
				   PrimitiveType type,
				   const VertexBuffer &vb,
				   int numvertices) = 0;

	 // ==================================================================

	 /// Render a particle system
	 virtual bool RenderParticleSys (ParticleSys *psys) = 0;

	 // ==================================================================

	 /// Set the viewport used for rendering.
	 virtual bool SetViewport (int x, int y, int w, int h) = 0;
	 
	 /**
	  * Sets the desired camera.
	  */
	 virtual bool SetCamera (const Camera &cam) = 0;
	 
	 /**
	  * Sets the desired near and far planes distance.
	  */
	 virtual bool SetPlanes (scalar np, scalar fp) = 0;
	 
	 /**
	  * Set an orthogonal identity camera, used for 2D rendering (so the
	  * vector \c (x,y,0.0) maps to window coordinates \c (x,y)).
	  */
	 virtual bool SetIdentity () = 0;
   };

   typedef Ptr<Renderer> RendererPtr;

   class ARK_DLL_API RendererFactory : public Factory
   {
      public:
         RendererFactory();
	 virtual ~RendererFactory() {}
	 virtual Renderer *NewRenderer(Cache *cache=0) = 0;

	 static Renderer *CreateRenderer(const String &implname,
					 Cache *cache=0,
					 FactoryList *factlist = 0);
   };

/* namespace Ark */
}

#endif
