/* $Id: Entity.cpp,v 1.18 2003/02/09 12:13:56 mrq Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <config.h>

#include <Engine/Engine.h>
#include <Engine/Entity.h>

namespace Ark
{

// The server entity constructor.
// Send a network update about the new entity.
EngineEntity::EngineEntity (Ark::World *world)
  : Ark::Entity (world)
{
  SetModel ("");

  m_Client = NULL;
  m_Flags |= JUSTCREATED;
}

// The server entity destructor.
// Send a network update about the destroyed entity.
EngineEntity::~EngineEntity ()
{}

/// Set the entity members using a list of (name, value) pairs.
bool
EngineEntity::SetEntries (Engine *engine,
			  const Ark::EntryList &entries)
{
   EntryList::const_iterator i;
   int j;

   i = entries.find ("class");
   m_Class = *i->second.d_str;
   if (m_Class == "PlayerStart")
   {
      i = entries.find ("position");
      
      engine->m_PlayerStarts.push_back( *i->second.d_vector3 );
   }
   
   // Set position
   i = entries.find ("position");
   SetPosition (*i->second.d_vector3);
   
   // Set name
   i = entries.find ("name");
   m_Name = *i->second.d_str;
   
   // Set short name
   i = entries.find ("shortname");
   m_ShortName = *i->second.d_str;
   
   // Set model
   i = entries.find ("model");
   SetModel(*i->second.d_str);
   
   // Set rotation.
   i = entries.find ("rotation");
   Ark::Vector3 r;
   if (i != entries.end())
      r = *(i->second.d_vector3);
   m_MState.m_Rotation = Ark::Quat (r.X, r.Y, r.Z);
   
   // Set flags
   struct {const char *name; int value;} flags[] = 
      {
	 {"static", STATIC},
	 {"visible", VISIBLE},
	 {"aidriven", AIDRIVEN},
	 {"collider", COLLISION},
	 {"pathfinder", PATHFINDING},
	 {"gravity", GRAVITY},
	 {0,0}
      };
   
   for (j = 0; flags[j].name != NULL; j++)
   {
      i = entries.find (flags[j].name);
      if (i->second.d_integer)
	 m_Flags |= flags[j].value;
      else
	 m_Flags &= ~flags[j].value;
   }
  
   return true;
}

// This calls the "normal" update function ; it then sends events to the
// entity if needed (ie: the goal has been reached, a timer has expired,
// etc..)
void EngineEntity::Update (scalar dt)
{
  // Update the entity;
    Entity::Update(dt);

  // If the path was set and it is not any more, it means we have reached our
  // goal.. Sent an event.
  if (m_Path.m_GoalReached)
  {
     EvGoalReached();
     m_Path.m_GoalReached = false;
  }

  std::vector<int> timers;

  // Check for timers..
  for (std::vector<EntityTimer>::iterator i = m_Timers.begin(); i != m_Timers.end() ;)
  {
    if (i->m_Timer.GetDelta() > i->m_Timeout)
    {
      // Remove timer from list, use return value of erase() for sequential containers
      timers.push_back (i->m_ID);
      i = m_Timers.erase( i );
    }
    else
    {
	// increment iterator if we do not erase
	++i;
    }
  }

  // Send events
  for (std::vector<int>::iterator ii = timers.begin();
       ii != timers.end();
       ii++)
    EvTimer (*ii);

  // Everything's OK!
  return;
}

/// Add a timer. When timeout is reached, the EvTimer function is called,
/// with id as parameter. Remark : the granularity of the timer depends on
/// the number of frame per seconds, since the entity is Updated only once
/// a frame.
bool
EngineEntity::AddTimer (int id, scalar timeout)
{
   m_Timers.push_back (EntityTimer (id, timeout));
  return true;
}

/// Called when someone speaks to this entity. If the entity is
/// a player, send a message to the corresponding client.
bool
EngineEntity::EvTell ()
{
   return true;
}

}
