/* $Id: GLRenderer.cpp,v 1.17 2003/04/05 12:25:15 mrq Exp $
** 
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2003 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <Ark/ArkConfig.h>
#include <Ark/ArkLight.h>
#include <Ark/ArkSystem.h>

#include "GLRenderer.h"

// Due to some NVIDIA includes, I had to add this for linux.
#ifndef GLAPIENTRY
#define GLAPIENTRY
#define GLAPI
#endif
#include <GL/glu.h>

#include <math.h>

#ifndef WIN32
#include <GL/glx.h>
#define glGetProcAddress(proc) glXGetProcAddressARB((const GLubyte*)proc)
#else
#define glGetProcAddress(proc) wglGetProcAddress(proc)
#endif


bool g_WireFrame = false;


namespace Ark
{
   /** This environment variable is set to true if the texture size has to 
    * be a power of two. 
    */
   bool g_TexSize_Power2 = false;

   /** Specify the maximum texture size. Textures bigger than that will be
    * scaled down.
    */
   int g_TexSize_Maximum = 1024;

   GLRenderer::GLRenderer (GLCache *cache) :
      m_State ("OpenGL state")
   {
      if (cache == NULL)
      {
	 m_Cache = Ptr<GLCache>(new GLCache(), 0);
	 m_Cache->AddRenderer(this);
      }
      else
      {
	 m_Cache = Ptr<GLCache>(cache);
	 m_Cache->AddRenderer(this);
      }

      const char *extensions = (const char*) glGetString (GL_EXTENSIONS);

      if (strstr (extensions, "EXT_compiled_vertex_array") != NULL)
      {
	 EXT_compiled_vertex_array
	    =  Sys()->Cfg()->GetInt("glrenderer::CompiledVertexArray", 1) != 0;

#ifdef GL_GLEXT_DYNAMIC
	 glLockArraysEXT =
	    (PFNGLLOCKARRAYSEXTPROC) glGetProcAddress ("glLockArraysEXT");
	 glUnlockArraysEXT =
	    (PFNGLUNLOCKARRAYSEXTPROC) glGetProcAddress ("glUnlockArraysEXT");
#endif
      }
      else
      {
	 EXT_compiled_vertex_array = false;
#ifdef GL_GLEXT_DYNAMIC
	 glLockArraysEXT = NULL;
	 glUnlockArraysEXT = NULL;
#endif
      }

      int texunits = 0;
      if (strstr (extensions, "ARB_multitexture") != NULL)
      {
	 glGetIntegerv (GL_MAX_TEXTURE_UNITS_ARB, &texunits);

	 if (texunits > 1)
	 {
	    ARB_multitexture
	       =  Sys()->Cfg()->GetInt("glrenderer::Multitexture", 1) != 0;

#ifdef GL_GLEXT_DYNAMIC
	    glActiveTextureARB =
	       (PFNGLACTIVETEXTUREARBPROC)
	       glGetProcAddress ("glActiveTextureARB");
	    glClientActiveTextureARB =
	       (PFNGLCLIENTACTIVETEXTUREARBPROC)
	       glGetProcAddress ("glClientActiveTextureARB");
#endif
	 }
      }

      if (texunits <= 1)
      {
	 ARB_multitexture = false;

#ifdef GL_GLEXT_DYNAMIC
	 glActiveTextureARB = NULL;
	 glClientActiveTextureARB = NULL;
#endif
      }

      CVA_hack =  Sys()->Cfg()->GetInt("glrenderer::CVAHack", 1) != 0;
      m_DumpNormals = Sys()->Cfg()->GetInt("glrenderer::DumpNormals", 0) != 0;

      int maxtexsize = 256;
      glGetIntegerv (GL_MAX_TEXTURE_SIZE, &maxtexsize);

      g_TexSize_Maximum = Sys()->Cfg()->GetInt
	 ("glrenderer::TexSize::Max", 1024);
      if (g_TexSize_Maximum > maxtexsize)
	 g_TexSize_Maximum = maxtexsize;
      
      g_TexSize_Power2 = Sys()->Cfg()->GetInt
	 ("glrenderer::TexSize::Power2", 1) != 0;

      glDisable (GL_ALPHA_TEST);
      glDisable (GL_BLEND);

      glEnable (GL_DEPTH_TEST);
      glDepthFunc (GL_LEQUAL);
      glDepthMask (GL_TRUE);

      glEnable (GL_CULL_FACE);
      glCullFace (GL_BACK);
      glFrontFace (GL_CW);
      glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

      glEnable (GL_RESCALE_NORMAL);

      m_TexUnit = 0;
      m_State.m_Passes[m_TexUnit].m_AlphaFunc = DEPTHFUNC_LEQUAL;
      m_State.m_Passes[m_TexUnit].m_DepthTest = true;
      m_State.m_Passes[m_TexUnit].m_DepthFunc = DEPTHFUNC_LEQUAL;
      m_State.m_Passes[m_TexUnit].m_BlendSrc = BLEND_UNDEFINED;
      m_State.m_Passes[m_TexUnit].m_BlendDst = BLEND_UNDEFINED;
      m_State.m_Passes[m_TexUnit].m_TextureEnv = TEXTUREENV_UNDEFINED;

      m_Near = Sys()->Cfg()->GetScalar ("glrenderer::Near", 0.1f);
      m_Far = Sys()->Cfg()->GetScalar ("glrenderer::Far", 1000.0f);
      
      m_FrameTime = 0.0;
   }

   GLRenderer::~GLRenderer()
   {
      m_Cache->RemoveRenderer(this);
   }

   // setdefault should be set to true if defaults values have to be
   // set when there's no information in the pass itself.
   bool GLRenderer::SetupPass (const ShaderPass &pass,
			       bool setdefault)
   {
      if (pass.m_Flags & PASS_HAS_TEXTURE)
      {
	 if (pass.m_Flags & PASS_HAS_TEXTUREANIM)
	 {
	    int texframe = (int) (m_FrameTime * pass.m_TexPerSec);
	    int tex = texframe % pass.m_AnimTex.size();

	    SetTexture (pass.m_AnimTex[tex]);
	 }
	 else
	    SetTexture (pass.m_Texture);
      }
      else
	 SetTexture (TexturePtr());

      if (pass.m_Flags & PASS_HAS_DEPTHWRITE)
	 SetDepthWrite (pass.m_DepthWrite);
      else if (setdefault)
	 SetDepthWrite (true);

      if (pass.m_Flags & PASS_HAS_DEPTHFUNC)
      {
	 SetDepthTest(pass.m_DepthTest);
	 SetDepthFunc (pass.m_DepthFunc);
      }
      else if (setdefault)
      {
	 SetDepthTest(true);
	 SetDepthFunc (DEPTHFUNC_LEQUAL);
      }

      if (pass.m_Flags & PASS_HAS_ALPHATEST)
	 SetAlphaTest (true, pass.m_AlphaFunc,
		       pass.m_AlphaValue);
      else if (setdefault)
	 SetAlphaTest (false);

      if (pass.m_Flags & PASS_HAS_BLENDING)
	 SetBlend (true, pass.m_BlendSrc, pass.m_BlendDst);
      else if (setdefault)
	 SetBlend (false);

      if (pass.m_Flags & PASS_HAS_TEXTUREENV)
	 SetTexEnv (pass.m_TextureEnv);
      else if (pass.m_Flags & PASS_HAS_TEXTURE)
	 SetTexEnv (TEXTUREENV_MODULATE);

      if (pass.m_Flags & PASS_HAS_TEXTUREGEN)
      {
	 if (pass.m_TextureGen == TEXTUREGEN_LINEAR)
	 {
	    SetLinearTexGen (pass.m_TextureGenData.m_Linear.m_PlaneS,
			     pass.m_TextureGenData.m_Linear.m_PlaneT);
	 }
      }
      else
	 SetTexGen(false);

      return true;
   }

   void
   GLRenderer::SetTexture (const TexturePtr& tex)
   {
      if (!tex)
      {
	 if (m_State.m_Passes[m_TexUnit].m_Flags & PASS_HAS_TEXTURE)
	 {
	    glDisable (GL_TEXTURE_2D);
	    m_State.m_Passes[m_TexUnit].m_Flags &= ~PASS_HAS_TEXTURE;
	 }
      }
      else
      {
	 if ((m_State.m_Passes[m_TexUnit].m_Flags & PASS_HAS_TEXTURE) == 0)
	 {
	    glEnable (GL_TEXTURE_2D);
	    m_State.m_Passes[m_TexUnit].m_Flags |= PASS_HAS_TEXTURE;
	 }

	 if (m_State.m_Passes[m_TexUnit].m_Texture != tex)
	 {
	    const GLTexture* glTex = static_cast<const GLTexture*>(&*tex);
	    glBindTexture (GL_TEXTURE_2D, glTex->m_GLID);
	    m_State.m_Passes[m_TexUnit].m_Texture = tex;
	 }
      }
   }
   
   void
   GLRenderer::SetDepthTest(bool dt)
   {
      if (dt == m_State.m_Passes[m_TexUnit].m_DepthTest)
	 return;

      if (dt)
	  glEnable(GL_DEPTH_TEST);
      else
	  glDisable(GL_DEPTH_TEST);

      m_State.m_Passes[m_TexUnit].m_DepthTest = dt;
   }

   void
   GLRenderer::SetDepthFunc (DepthFunc df)
   {
      if (df == m_State.m_Passes[m_TexUnit].m_DepthFunc)
	 return;

      switch (df)
      {
	 case DEPTHFUNC_LEQUAL:
	    glDepthFunc (GL_LEQUAL);
	    break;
	 case DEPTHFUNC_EQUAL:
	    glDepthFunc (GL_EQUAL);
	    break;
	 case DEPTHFUNC_ALWAYS:
	    glDepthFunc (GL_ALWAYS);
	    break;
	 default: return;
      }

      m_State.m_Passes[m_TexUnit].m_DepthFunc = df;
   }

   void
   GLRenderer::SetCulling (bool cull)
   {
      if (cull == false && 
	  (m_State.m_Flags & MATERIAL_IS_DOUBLESIDED) == 0)
      {
	 m_State.m_Flags |= MATERIAL_IS_DOUBLESIDED;
	 glDisable (GL_CULL_FACE);
      }
      else if (cull == true &&
	       (m_State.m_Flags & MATERIAL_IS_DOUBLESIDED) != 0)
      {
	 m_State.m_Flags &= ~MATERIAL_IS_DOUBLESIDED;
	 glEnable (GL_CULL_FACE);
      }
   }

   void
   GLRenderer::SetAlphaTest (bool alphatest, DepthFunc af, scalar value)
   {
      if (alphatest == true)
      {
	 if ((m_State.m_Passes[m_TexUnit].m_Flags & PASS_HAS_ALPHATEST) == 0)
	 {
	    glEnable (GL_ALPHA_TEST);
	    m_State.m_Passes[m_TexUnit].m_Flags |= PASS_HAS_ALPHATEST;
	 }

	 if (af == m_State.m_Passes[m_TexUnit].m_AlphaFunc
	     && value == m_State.m_Passes[m_TexUnit].m_AlphaValue)
	    return;

      }
      else if (alphatest == false)
      { 
	 if ((m_State.m_Passes[m_TexUnit].m_Flags & PASS_HAS_ALPHATEST) != 0)
	 {
	    glDisable (GL_ALPHA_TEST);
	    m_State.m_Passes[m_TexUnit].m_Flags &= ~PASS_HAS_ALPHATEST;
	 }

	 return;
      }

      switch (af)
      {
	 case DEPTHFUNC_GEQUAL:
	    glAlphaFunc (GL_GEQUAL, value);
	    break;
	 case DEPTHFUNC_EQUAL:
	    glAlphaFunc (GL_EQUAL, value);
	    break;
	 case DEPTHFUNC_ALWAYS:
	    glAlphaFunc (GL_ALWAYS, value);
	    break;
	 default: return;
      }

      m_State.m_Passes[m_TexUnit].m_AlphaFunc = af;
      m_State.m_Passes[m_TexUnit].m_AlphaValue = value;
   }

   void
   GLRenderer::SetDepthWrite (bool write)
   {
      if (write == m_State.m_Passes[m_TexUnit].m_DepthWrite)
	 return;

      glDepthMask (write == true ? GL_TRUE : GL_FALSE);
      m_State.m_Passes[m_TexUnit].m_DepthWrite = write;
   }

   GLenum GLBlendFromBlendFunc (BlendFunc f)
   {
      static const GLenum bf_to_glbf [] = 
	 {
	    GL_ONE,                      // BLEND_UNDEFINED
	    GL_ONE,                      // BLEND_ONE
	    GL_ZERO,                     // BLEND_ZERO
	    GL_DST_COLOR,                // BLEND_DST_COLOR
	    GL_ONE_MINUS_DST_COLOR,      // BLEND_ONE_MINUS_DST_COLOR
	    GL_SRC_ALPHA,                // BLEND_SRC_ALPHA
	    GL_ONE_MINUS_SRC_ALPHA,      // BLEND_ONE_MINUS_SRC_ALPHA
	    GL_DST_ALPHA,                // BLEND_DST_ALPHA
	    GL_ONE_MINUS_DST_ALPHA,      // BLEND_ONE_MINUS_DST_ALPHA
	    GL_SRC_COLOR,                // BLEND_SRC_COLOR
	    GL_ONE_MINUS_SRC_COLOR,      // BLEND_ONE_MINUS_SRC_COLOR
	    GL_CONSTANT_ALPHA,           // BLEND_CONSTANT_ALPAH
	    GL_ONE_MINUS_CONSTANT_ALPHA  // BLEND_ONE_MINUS_CONSTANT_ALPHA
	 };

      if (f > BLEND_ONE_MINUS_CONSTANT_ALPHA)
	 return GL_ONE;

      return bf_to_glbf[f];
   }

  void
  GLRenderer::SetBlend (bool blend, BlendFunc src , BlendFunc dst )
  {
	if (blend == false)
	{

	if (m_State.m_Passes[m_TexUnit].m_Flags & PASS_HAS_BLENDING)
	{
	   glDisable (GL_BLEND);
	   m_State.m_Passes[m_TexUnit].m_Flags &= ~PASS_HAS_BLENDING;
	}

	return;
     }

     if ((m_State.m_Passes[m_TexUnit].m_Flags & PASS_HAS_BLENDING) == 0)
     {
	glEnable (GL_BLEND);
	m_State.m_Passes[m_TexUnit].m_Flags |= PASS_HAS_BLENDING;
     }
     else if ((m_State.m_Passes[m_TexUnit].m_BlendSrc == src) &&
	      (m_State.m_Passes[m_TexUnit].m_BlendDst == dst))
     {
	return;
     }
     
     GLenum gsrc = GLBlendFromBlendFunc (src);
     GLenum gdst = GLBlendFromBlendFunc (dst);
     
     glBlendFunc (gsrc, gdst);
     m_State.m_Passes[m_TexUnit].m_BlendSrc = src;
     m_State.m_Passes[m_TexUnit].m_BlendDst = dst;
  }
   
   void
   GLRenderer::DisableLighting()
   {
       if (m_State.m_Flags & MATERIAL_HAS_LIGHTING)
       {
	   glDisable (GL_LIGHTING);
	   m_State.m_Flags &= ~MATERIAL_HAS_LIGHTING;
       }

       glColor4f (1.0, 1.0, 1.0, 1.0);
   }
   
   void
   GLRenderer::SetLighting (bool lighting,
			    const Color &ambient,
			    const Color &diffuse,
			    const Color &specular)
   {
      if (lighting == false)
      {
	 if (m_State.m_Flags & MATERIAL_HAS_LIGHTING)
	 {
	    glDisable (GL_LIGHTING);
	    m_State.m_Flags &= ~MATERIAL_HAS_LIGHTING;
	 }

	 glColor4f (1.0, 1.0, 1.0, 1.0);
	 return;
      }
      
      // FIXME: dunno why, but if I try to avoid state changes when
      // a material color doesn't change, objects get odd colors...

      //if (m_State.m_Ambient != ambient)
      {
	 glMaterialfv (GL_FRONT_AND_BACK, GL_AMBIENT, &ambient.R);
	 m_State.m_Ambient = ambient;
      }

      //if (m_State.m_Diffuse != diffuse)
      {
	 glMaterialfv (GL_FRONT_AND_BACK, GL_AMBIENT, &diffuse.R);
	 m_State.m_Diffuse = diffuse;
      }

      //if (m_State.m_Specular != specular)
      {
	 glMaterialfv (GL_FRONT_AND_BACK, GL_SPECULAR, &specular.R);
	 m_State.m_Specular = specular;
      }


      //if ((m_State.m_Flags & MATERIAL_HAS_LIGHTING) == 0)
      {
	 glEnable (GL_LIGHTING);
	 m_State.m_Flags |= MATERIAL_HAS_LIGHTING;
      }
   }
   
   // Texgen
   void
   GLRenderer::SetLinearTexGen (const scalar *planeU,
				const scalar *planeV)
   {
      SetTexGen (true);

      if (m_State.m_Passes[m_TexUnit].m_TextureGen != TEXTUREGEN_LINEAR)
      {
	 glTexGeni (GL_S, GL_TEXTURE_GEN_MODE, GL_OBJECT_LINEAR);
	 glTexGeni (GL_T, GL_TEXTURE_GEN_MODE, GL_OBJECT_LINEAR);
	 m_State.m_Passes[m_TexUnit].m_TextureGen = TEXTUREGEN_LINEAR;
      }

      scalar *plane;
      plane = m_State.m_Passes[m_TexUnit].m_TextureGenData.m_Linear.m_PlaneS;
      if (memcmp(plane, planeU, 4 * sizeof(scalar)))
      {
	 memcpy(plane, planeU, 4 * sizeof(scalar));
	 glTexGenfv (GL_S, GL_OBJECT_PLANE, planeU);
      }

      plane = m_State.m_Passes[m_TexUnit].m_TextureGenData.m_Linear.m_PlaneT;
      if (memcmp(plane, planeV, 4 * sizeof(scalar)))
      {
	 memcpy(plane, planeV, 4 * sizeof(scalar));
	 glTexGenfv (GL_T, GL_OBJECT_PLANE, planeV);
      }
   }

   void
   GLRenderer::SetTexEnv (TextureEnv texenv)
   {
      GLenum mode = GL_MODULATE;
      switch (texenv)
      {
	 case TEXTUREENV_REPLACE:
	    mode = GL_REPLACE;
	    break;

	 case TEXTUREENV_DECAL:
	    mode = GL_DECAL;
	    break;

	 case TEXTUREENV_MODULATE:
	 case TEXTUREENV_UNDEFINED:
	    mode = GL_MODULATE;
      }

      glTexEnvi (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, mode);
   }

   void
   GLRenderer::SetTexGen (bool enable)
   {
      if (enable == false
	  && m_State.m_Passes[m_TexUnit].m_Flags & PASS_HAS_TEXTUREGEN)
      {
	 if ((m_State.m_Passes[m_TexUnit].m_Flags & PASS_HAS_TEXTURE) != 0)
	 {
	    glDisable (GL_TEXTURE_GEN_S);
	    glDisable (GL_TEXTURE_GEN_T);
	    m_State.m_Passes[m_TexUnit].m_Flags &= ~PASS_HAS_TEXTUREGEN;
	 }
	 else
	 {
	    /// if there's no texture, why disable texture coord generation ??
	    /// it would invalidate vertex buffer texture coords...
	 }
      }
      else if (enable == true
	       && !(m_State.m_Passes[m_TexUnit].m_Flags & PASS_HAS_TEXTUREGEN))
      {
	 glEnable (GL_TEXTURE_GEN_S);
	 glEnable (GL_TEXTURE_GEN_T);
	 m_State.m_Passes[m_TexUnit].m_Flags |= PASS_HAS_TEXTUREGEN;
      }
   }

   bool GLRenderer::SetupMaterial (const Material &mat)
   {
      if (mat.m_Flags & MATERIAL_HAS_LIGHTING)
      {
	 SetLighting (true, mat.m_Ambient,
		      mat.m_Diffuse,
		      mat.m_Specular);
      }
      else
      {
	 Color col;
	 SetLighting (false, col, col, col);
      }

      if ((mat.m_Flags & MATERIAL_IS_DOUBLESIDED) != 0)
	 SetCulling (false);
      else
	 SetCulling (true);

      return true;
   }

   bool
   GLRenderer::SetVertexBuffer
      (int format, const VertexBuffer &VB, bool disable)
   {
      /// COORDINATE
      if (format & VertexBuffer::VB_HAS_COORD)
      {
	 glEnable (GL_VERTEX_ARRAY);
	 glVertexPointer (3, GL_FLOAT, VB.VertexSize(), VB.CoordP());

	 m_CoordPointer = (uchar*) VB.CoordP();
	 m_CoordStride = VB.VertexSize();
      }
      else if (disable)
      {
	 glDisable (GL_VERTEX_ARRAY);
	 m_CoordPointer = NULL;
      }

      /// NORMAL
      if (format & VertexBuffer::VB_HAS_NORMAL)
      {
	 glEnable (GL_NORMAL_ARRAY);
	 glNormalPointer (GL_FLOAT, VB.VertexSize(), VB.NormalP());

	 m_NormalPointer = (uchar*) VB.NormalP();
	 m_NormalStride = VB.VertexSize();
      }
      else if (disable)
      {
	 glDisable (GL_NORMAL_ARRAY);
	 m_NormalPointer = NULL;
      }

      /// TEXTURE COORD
	  const int offset = VB.GetTextureCoordinateOffset();
      if ((format & VertexBuffer::VB_HAS_UV0) && (offset == 0))
      {
	 glEnable (GL_TEXTURE_COORD_ARRAY);
	 glTexCoordPointer (2, GL_FLOAT, VB.VertexSize(), VB.UV0P ());
      }
	  else if ((format & VertexBuffer::VB_HAS_UV1) && (offset == 1))
      {
	 glEnable (GL_TEXTURE_COORD_ARRAY);
	 glTexCoordPointer (2, GL_FLOAT, VB.VertexSize(), VB.UV1P ());
      }
      else if (disable)
	 glDisable (GL_TEXTURE_COORD_ARRAY);

      /// COLOR
      if (format & VertexBuffer::VB_HAS_COLOR)
      {
	 glEnable (GL_COLOR_ARRAY);
	 glColorPointer (4, GL_UNSIGNED_BYTE, VB.VertexSize(), VB.Color4P());
      }
      else if (disable)
	 glDisable (GL_COLOR_ARRAY);

      return true;
   }

   bool GLRenderer::SetActiveVB (const VertexBuffer &VB)
   {
      return SetVertexBuffer (VB.Format(), VB, true);
   }

   bool GLRenderer::OverrideVB (int override_flags,
				const VertexBuffer &VB)
   {
      return SetVertexBuffer (override_flags, VB, false);
   }
      
   /**
    * Those two functions do nothing if compiled vertex arrays are not 
    * supported or are disabled.
    */
   bool GLRenderer::LockVB (int begin, size_t n)
   {
#ifndef NDEBUG
      if (m_DumpNormals && m_NormalPointer)
      {
	 /// Dump normals...
	 glColor3f (0.0, 0.0, 1.0);
	 glDisable (GL_LIGHTING);
	 
	 glBegin (GL_LINES);
	 for (size_t i = begin; i < n; i++)
	 {
	    Vector3 coord = * (Vector3*) &m_CoordPointer[i * m_CoordStride];
	    glVertex3fv (&coord.X);
	    
	    Vector3 normal = * (Vector3*) &m_NormalPointer[i * m_NormalStride];
	    normal.Scale (0.2f);
	    normal += coord;
	    glVertex3fv (&normal.X);
	 }
	 glEnd ();
      }
#endif

      if (EXT_compiled_vertex_array)
      {
	 if (CVA_hack && begin == 0)
	    glLockArraysEXT (1, n-1);
	 else
	    glLockArraysEXT (begin, n);
      }

      return true;
   }

   bool GLRenderer::UnlockVB ()
   {
      if (EXT_compiled_vertex_array)
	 glUnlockArraysEXT ();

      return true;
   }

   bool GLRenderer::RenderBlock (const Material &mat,
				 PrimitiveType type,
				 const ushort *indices,
				 size_t n)
   {
      if (n == 0)
	 return true;

      // FIXME: enhance multitexturing, so we can do two double texture
      // FIXME: passes instead of 4 single texture passes when hardware only
      // FIXME: has two texture units.
      bool multitex = ARB_multitexture && (mat.NumPasses() <= 2);

      if (!SetupMaterial (mat))
	 return false;

      int ca_enabled = glIsEnabled (GL_COLOR_ARRAY);
	 
      for (int i = 0; i < 4; i++)
      {
	 if (multitex)
	 {
	    m_TexUnit = i;

	    glActiveTextureARB (GLenum(GL_TEXTURE0_ARB + i));
	    glClientActiveTextureARB (GLenum(GL_TEXTURE0_ARB + i));
	 }

	 if (i > 0 && (mat.m_Flags & (MATERIAL_HAS_PASS1 << i)) == 0)
	 {
	    if (multitex)
	    {
	       glDisable (GL_TEXTURE_2D);
	       continue;
	    }

	    break;
	 }

	 if (i > 0 || ca_enabled == false)
	 {
	    if (ca_enabled)
	       glDisable (GL_COLOR_ARRAY);

	    glColor4fv (&mat.m_Passes[i].m_BlendColor.R);
	 }

	 if ((mat.m_Flags & (MATERIAL_HAS_PASS1 << i)))
	 {
	    // Set default values for depth test, alpha test, blending, etc
	    // only for the first pass..
	    bool setdefault = (i == 0);
	    SetupPass (mat.m_Passes[i], setdefault);
	 }
	 else if (i == 0)
	 {
	    SetAlphaTest (false);
	    SetDepthTest (true);
	    SetDepthFunc (DEPTHFUNC_LEQUAL);
	    SetDepthWrite ((mat.m_Passes[0].m_Flags & PASS_HAS_DEPTHWRITE)!=false);
	    SetTexture (TexturePtr());
	    SetBlend (false);
	 }
	 
	 /// If there's no multitexturing, we need to draw all the triangles
	 /// for each pass.
	 if (!multitex)
	 {
	    //Sys()->Log
	    //   ("No multitexture, pushing block (%d vertices, type %d)\n",
		//    n, type);
	    PushBlock (type, indices, n);
	 }
      }
      
      // XXX test
      //m_TexUnit = 0;

      if (ca_enabled)
	 glEnable (GL_COLOR_ARRAY);

      if (multitex)
	 PushBlock (type, indices, n);
      return true;
   }

   /** Send a list of polygons to OpenGL, WITHOUT caring about
    * the current state, and without setting the material.
    */
   bool
   GLRenderer::PushBlock (PrimitiveType type,
			  const ushort *indices,
			  size_t n)
   {
      //glPolygonMode (GL_FRONT_AND_BACK, GL_LINE);
      GLenum mode;
      switch (type)
      {
	 case PRIM_TRIANGLES:      mode = GL_TRIANGLES; break;
	 case PRIM_TRIANGLE_FAN:   mode = GL_TRIANGLE_FAN; break;
	 case PRIM_TRIANGLE_STRIP: mode = GL_TRIANGLE_STRIP; break;
	 default: return 0;
      }
      
      glDrawElements (mode, n, GL_UNSIGNED_SHORT, indices);
      return true;
   }
   
   ///////////////////////////////////////////////////////////////////////////
   // Direct vertex rendering (ie no vertex indices);
   ///////////////////////////////////////////////////////////////////////////

   bool
   GLRenderer::RenderBlock (const Material &mat,
			    PrimitiveType type,
			    const VertexBuffer &vb,
			    int numvertices)
   {
      // FIXME: I'd like to avoid code copy...
      bool multitex = ARB_multitexture && (mat.NumPasses() <= 2);

      if (!SetupMaterial (mat))
	 return false;

      int ca_enabled = glIsEnabled (GL_COLOR_ARRAY);
      SetActiveVB (vb);
	 
      for (int i = 0; i < 4; i++)
      {
	 if (multitex)
	 {
	    m_TexUnit = i;

	    glActiveTextureARB (GLenum(GL_TEXTURE0_ARB + i));
	    glClientActiveTextureARB (GLenum(GL_TEXTURE0_ARB + i));
	 }

	 if (i > 0 && (mat.m_Flags & (MATERIAL_HAS_PASS1 << i)) == 0)
	 {
	    if (multitex)
	    {
	       glDisable (GL_TEXTURE_2D);
	       continue;
	    }

	    break;
	 }

	 if (i > 0 || ca_enabled == false)
	 {
	    if (ca_enabled)
	       glDisable (GL_COLOR_ARRAY);

	    glColor4fv (&mat.m_Passes[i].m_BlendColor.R);
	 }

	 if ((mat.m_Flags & (MATERIAL_HAS_PASS1 << i)))
	 {
	    // Set default values for depth test, alpha test, blending, etc
	    // only for the first pass..
	    bool setdefault = (i == 0);
	    SetupPass (mat.m_Passes[i], setdefault);
	 }
	 else if (i == 0)
	 {
	    SetAlphaTest (false);
	    SetDepthTest (true);
	    SetDepthFunc (DEPTHFUNC_LEQUAL);
	    SetDepthWrite (true);
	    SetTexture (TexturePtr());
	    SetBlend (false);
	 }
	 
	 /// If there's no multitexturing, we need to draw all the triangles
	 /// for each pass.
	 if (!multitex)
	    PushBlock (type, vb, numvertices);
      }

      if (ca_enabled)
	 glEnable (GL_COLOR_ARRAY);

      if (multitex)
	 PushBlock (type, vb, numvertices);
      return true;

   }

   /** Send a list of polygons to OpenGL, WITHOUT caring about
    * the current state, and without setting the material.
    */
   bool
   GLRenderer::PushBlock (PrimitiveType type,
			  const VertexBuffer &vb,
			  size_t n)
   {
      //glPolygonMode (GL_FRONT_AND_BACK, GL_LINE);
      GLenum mode;
      switch (type)
      {
	 case PRIM_TRIANGLES:      mode = GL_TRIANGLES; break;
	 case PRIM_TRIANGLE_FAN:   mode = GL_TRIANGLE_FAN; break;
	 case PRIM_TRIANGLE_STRIP: mode = GL_TRIANGLE_STRIP; break;
	 default: return 0;
      }
      
      glDrawArrays (mode, 0, n);
      return true;
   }
   
   Frustum
   ComputeFrustum(const Matrix44& viewMatrix, const Matrix44& projectionMatrix)
   {
      Matrix44 viewmat(viewMatrix);
      viewmat.Multiply(projectionMatrix);
      
      Plane planes[6];

      /* Calculate the six OC plane equations. */
      
      /*=========*/
      planes[0] = Plane (viewmat.M(0,3) - viewmat.M(0,0),
			 viewmat.M(1,3) - viewmat.M(1,0),
			 viewmat.M(2,3) - viewmat.M(2,0),
			 viewmat.M(3,3) - viewmat.M(3,0));
      
      planes[1] = Plane (viewmat.M(0,3) + viewmat.M(0,0),
			 viewmat.M(1,3) + viewmat.M(1,0),
			 viewmat.M(2,3) + viewmat.M(2,0),
			 viewmat.M(3,3) + viewmat.M(3,0));
      
      /*=========*/
      planes[2] = Plane (viewmat.M(0,3) + viewmat.M(0,1),
			 viewmat.M(1,3) + viewmat.M(1,1),
			 viewmat.M(2,3) + viewmat.M(2,1),
			 viewmat.M(3,3) + viewmat.M(3,1));
      
      planes[3] = Plane (viewmat.M(0,3) - viewmat.M(0,1),
			 viewmat.M(1,3) - viewmat.M(1,1),
			 viewmat.M(2,3) - viewmat.M(2,1),
			 viewmat.M(3,3) - viewmat.M(3,1));
      
      /*=========*/
      planes[4] = Plane (viewmat.M(0,3) + viewmat.M(0,2),
			 viewmat.M(1,3) + viewmat.M(1,2),
			 viewmat.M(2,3) + viewmat.M(2,2),
			 viewmat.M(3,3) + viewmat.M(3,2));
      
      planes[5] = Plane (viewmat.M(0,3) - viewmat.M(0,2),
			 viewmat.M(1,3) - viewmat.M(1,2),
			 viewmat.M(2,3) - viewmat.M(2,2),
			 viewmat.M(3,3) - viewmat.M(3,2));
      
      return Frustum (planes);
   }

   void
   GLRenderer::MatrixChanged ()
   {
      glGetFloatv (GL_MODELVIEW_MATRIX, &m_ViewMatrix.M(0,0));
      glGetFloatv (GL_PROJECTION_MATRIX, &m_ProjectionMatrix.M(0,0));
      
      m_Frustum = ComputeFrustum(m_ViewMatrix, m_ProjectionMatrix);
      m_Frustum.ComputeVCode();
   }

   const Matrix44&
   GLRenderer::GetViewMatrix()
   {
      return m_ViewMatrix;
   }

   const Matrix44&
   GLRenderer::GetProjectionMatrix()
   {
      return m_ProjectionMatrix;
   }

   const Frustum&
   GLRenderer::GetFrustum() const
   {
      return m_Frustum;
   }

#if 0 // XXX 
   void
   GLRenderer::SetViewMatrix(Matrix44 &matx)
   {
      glMatrixMode (GL_MODELVIEW);
      glLoadMatrixf (&matx.M(0,0));

      MatrixChanged();
   }
#endif
   
   void
   GLRenderer::PushModelMatrix(Matrix44 &matx)
   {
	  glMatrixMode(GL_MODELVIEW);
	  glPushMatrix();
	  glMultMatrixf(&matx.M(0,0));
   }
   
   void
   GLRenderer::PopModelMatrix()
   {
	  glMatrixMode(GL_MODELVIEW);
	  glPopMatrix();
   }
   
   bool
   GLRenderer::SetPlanes (scalar np, scalar fp)
   {
       m_Near = np;
       m_Far = fp;
       return true;
   }
   
   bool
   GLRenderer::SetCamera (const Camera &cam)
   {
      m_FrameTime = m_Timer.GetDelta();

	  if (g_WireFrame)
	    glPolygonMode (GL_FRONT_AND_BACK, GL_LINE);
	  else
	    glPolygonMode (GL_FRONT_AND_BACK, GL_FILL);

      // FIXME: timer resolution...
      // if ((m_FrameTime - floor (m_FrameTime)) < 0.01)
      //   m_Timer.Update();

      Vector3 pov = cam.m_PointOfView, la = cam.m_LookAt;
      scalar aspect = scalar(m_Viewport[2])/scalar(m_Viewport[3]);
      
      m_Camera = cam;
      glMatrixMode (GL_PROJECTION);
      glLoadIdentity();
      gluPerspective (cam.m_FOV, aspect, m_Near, m_Far);
      
      glMatrixMode (GL_MODELVIEW);
      glLoadIdentity();
      gluLookAt (pov.X, pov.Y, pov.Z,
		 la.X, la.Y, la.Z,
		 0.0, 1.0, 0.0);

      MatrixChanged();
      return true;
   }
   
   bool
   GLRenderer::SetViewport (int x, int y, int w, int h)
   {
      glViewport (x, y, w, h);
      
      m_Viewport[0] = x;
      m_Viewport[1] = y;
      m_Viewport[2] = w;
      m_Viewport[3] = h;
      
      glMatrixMode (GL_PROJECTION);
      glLoadIdentity();
      glOrtho (x, w, h, y, -1, 1);
      
      glMatrixMode (GL_MODELVIEW);

      MatrixChanged();
      return true;
   }
   
   bool
   GLRenderer::SetIdentity ()
   {
      glMatrixMode (GL_PROJECTION);
      glLoadIdentity();
      glOrtho (m_Viewport[0], m_Viewport[2],
	       m_Viewport[3], m_Viewport[1], -1, 1);
      
      glMatrixMode (GL_MODELVIEW);
      glLoadIdentity();

      MatrixChanged();
      return true;
   }
  
   void 
   GLRenderer::RenderLight (const Light &light, int num)
   {     
      Color black;
      
      glEnable (GL_LIGHT0+num);

      glLightfv (GL_LIGHT0+num, GL_POSITION, &light.GetPosition().X);
      /* FIXME: this is lame, ambient light should be global
       * see GL_LIGHT_MODEL_AMBIENT
       */
      if (light.GetType () == LIGHT_AMBIENT)
      {
	 glLightfv (GL_LIGHT0+num, GL_AMBIENT, &light.GetColor().R);
	 glLightfv (GL_LIGHT0+num, GL_DIFFUSE, &(black.R));
      }
      else
      {
	 glLightfv (GL_LIGHT0+num, GL_DIFFUSE, &light.GetColor().R);
	 glLightfv (GL_LIGHT0+num, GL_AMBIENT, &(black.R));
      }
      
      glLightfv (GL_LIGHT0+num, GL_SPECULAR, &(black.R));
      glLightf (GL_LIGHT0+num, GL_CONSTANT_ATTENUATION, 0.0f);
      glLightf (GL_LIGHT0+num, GL_LINEAR_ATTENUATION, 0.0f);
      glLightf (GL_LIGHT0+num, GL_QUADRATIC_ATTENUATION, 
		light.GetAttenuation());
   }
}

#define ARK_MODULE
#include <Ark/ArkFactoryReg.h>

class GLRendererFactory : public Ark::RendererFactory
{
   public:
      virtual ~GLRendererFactory() {}
      virtual Ark::Renderer *NewRenderer(Ark::Cache *cache)
      {
	 return new Ark::GLRenderer(static_cast<Ark::GLCache*>(cache));
      }
};

ARK_REGISTER("ark::Renderer::OpenGL", GLRendererFactory);

