/*

*************************************************************************

ArmageTron -- Just another Tron Lightcycle Game in 3D.
Copyright (C) 2000  Manuel Moos (manuel@moosnet.de)

**************************************************************************

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
  
***************************************************************************

*/

#include "tMemManager.h"
#include "eGrid.h"
#include "eTess2.h"

#include "rTexture.h"
#include "tEventQueue.h"
#include "eTimer.h"
#include "eWall.h"
#include "eGameObject.h"
#include "eCamera.h"

#include <math.h>

#ifdef _MSC_VER
#include <float.h>
#define finite _finite
#endif

int se_debugExt=0;

eCoord    eTempEdge::Vec()         const{return halfEdges[0]->Vec();}
eCoord&   eTempEdge::Coord(int i)  const{return *halfEdges[i]->Point();}
ePoint    *eTempEdge::Point(int i) const{return halfEdges[i]->Point();}
eFace     *eTempEdge::Face(int i)  const{return halfEdges[i]->Face();}
eHalfEdge *eTempEdge::Edge(int i)  const{return halfEdges[i];}
eWall     *eTempEdge::Wall()       const{return halfEdges[0]->Wall();}



inline void eHalfEdge::SetWall(eWall *w){
  if (wall)
    {
      if (other && !other->wall)
	{
	  // simply flip the wall and attach it to the other edge :-)
	  w->Flip();
	  other->SetWall(w);
	  return;
	}

      st_Breakpoint();
      
      wall->edge = NULL;
      delete wall;
    }

  wall = w;
  if (w)
    {
      if (wall->edge)
	wall->edge->wall = NULL;
      
      wall->edge = this;
      wall->CalcLen();
    }
}

            // destructor unlinking all pointers
inline eHalfEdge::~eHalfEdge(){
  Unlink();
  tDESTROY(wall);
  refCtr = -100;
} 

inline void eHalfEdge::Unlink()
{
  if (point && this == point->edge)
    {
      if (prev && prev->other && prev->other)
	point->edge = prev->other;
      else if (other && other->next)
	point->edge = other->next;
      else
	point->edge = 0;
    }
  if (face && this == face->edge)
    {
      if (next)
	face->edge = next;
      if (prev)
	face->edge = prev;
    }

  if (next)
    next->prev = NULL;
  if (prev)
    prev->next = NULL;
  if (other)
    other->other = NULL;

  point   = NULL;
  face    = NULL;
  prev    = NULL;
  next    = NULL;
  other   = NULL;
}



// ********************************************

// consistency checks:

void eGrid::Check() const{
#ifdef DEBUG
  if (!doCheck)
    return;
#endif

  int i;
  for (i = points.Len()-1; i>=0; i--)
    points(i)->Check();

  for (i = faces.Len()-1; i>=0; i--)
    faces(i)->Check();

  for (i = edges.Len()-1; i>=0; i--)
    edges(i)->Check();
}


// **************************
//          eDual
// **************************

eDual::~eDual()  // destructor
{
  assert(refCtr <= 1);
  assert(!edge);
  refCtr = -100;
}

bool eDual::Check(int type)
{
  if (type)
    {
      assert((!edge ||  this == static_cast<eDual*>(edge->face)));
    }
  else
    {
      assert((!edge ||  this == static_cast<eDual*>(edge->point)));
    }

  if (ID>=0) 
    assert(!edge || edge->ID>=0);

  return true;
}
  // basic consistency check; type is 0 for points and 1 for faces.
inline void eDual::Unlink(int type)
{
  
}
  // removes itself from the edges' pointers. type is 0 for points and 1 for edges.




// **************************
//          ePoint
// **************************

void ePoint::Print(ostream &s) const
{
  s << "(" << x << ", " << y << ")";
}

// replaces all known pointers to *this with pointers to *P.
void ePoint::Replace(ePoint *P)
{
  AddRef();

  //  if (!(*this==*P))
  //    tERR_WARN("Unequal points are going to get exchaned!");

  eHalfEdge *run = edge;
  const eHalfEdge *stop = run;

  *static_cast<eCoord *>(P) = *this;
  
  do
    {
      assert(run && run->Point() == this);
      run->SetPoint(P);

      run = run->other->next;

    }
  while (stop != run);

  Release();

  edge = NULL;
}

void ePoint::Unlink()
{
  if (!edge)
    return;

  eHalfEdge *run = edge;
  const eHalfEdge *stop = run;
  
  do
    {
      assert(run && run->Point() == this && 
	     (!run->next || !run->next->next || run == run->next->next->next)) ;
      run = run->next;
      if (run)
	run= run->next;
      if (run)
	run = run->other;
    }
  while (run && stop != run);

  edge = NULL;
}

// **************************
//          eFace
// **************************

eFace::eFace (eHalfEdge *e1,eHalfEdge *e2,eHalfEdge *e3){
#ifdef DEBUG  
  assert(e1->other);
  assert(e2->other);
  assert(e3->other);

  assert(e1->Point() == e3->other->Point());
  assert(e2->Point() == e1->other->Point());
  assert(e3->Point() == e2->other->Point());

  //  assert(*e1->Point() != *e2->Point());
  //  assert(*e3->Point() != *e2->Point());
  //  assert(*e1->Point() != *e3->Point());

  assert((*e1->Point() - *e2->Point()).Norm_squared() < 1E+15);
  assert((*e3->Point() - *e2->Point()).Norm_squared() < 1E+15);
  assert((*e1->Point() - *e3->Point()).Norm_squared() < 1E+15);
#endif

  e1->SetFace(this);
  e2->SetFace(this);
  e3->SetFace(this);

  e1->next = e2;
  e2->next = e3;
  e3->next = e1;
  e1->prev = e3;
  e2->prev = e1;
  e3->prev = e2;

  e1->Point()->edge = e1;
  e2->Point()->edge = e2;
  e3->Point()->edge = e3;

  edge = e1;

#ifdef DEBUG
  REAL area = -(*e1->Point() - *e2->Point()) * (*e1->Point() - *e3->Point());
  if(area <= 0)
    {
      con << "Face " << *this << " has wrong orientation (area: " << area << ").\n";
      if (area < 0)
	{
	  static bool reported = false;
	  if (!reported)
	    st_Breakpoint();
	  reported = true;
	}
    }
#endif
}


void eFace::Unlink(){
  if (edge)
    {
      eHalfEdge *run = edge;
      const eHalfEdge *stop = run;
      
      do
	{
	  assert(run->Face() == this);
	  run->SetFace(NULL);
	  
	  run = run->next;
	}
      while (stop != run);
      
      edge = NULL;
    }
}

// ckeck if the given point lies inside this face (edges included)
bool eFace::IsInside(const eCoord &c)
{
  eHalfEdge *run = edge;
  if (!run)
    return false;

  eCoord *last_p = run->Point();
  run = run->next;
  const eHalfEdge *stop = run;
  
  do
    {
      assert(run && run->Face() == this);
      eCoord *p = run->Point();
      eCoord ed=*last_p - *p;
      eCoord cc= c - *p;
      
      if (ed*cc<-EPS)
	return 0;
      
      last_p = p;
      run = run->next;
    }
  while (run && stop != run);
  
  return 1;
}




void eCoord::Print(ostream &s)const{ s << "(" << x << "," << y << ")"; }
void eHalfEdge::Print(ostream &s)const{ s << "[" << *Point() << "->" << *other->Point() << "]"; }
void eFace::Print(ostream &s)const{ s << "[" << *edge->Point() << "->" << *edge->next->Point() << "->" << *edge->next->next->Point() << "]"; }


// **************************
//          eGrid
// **************************

static eHalfEdge *leakEdge =NULL;
static ePoint    *leakPoint=NULL;


ePoint * eGrid::DrawLine(ePoint *start, const eCoord &end, eWall *wal, bool change_grid){
  Range(end.Norm_squared());
  int restart=1;
  //  eEdge *todo=tNEW(eEdge) (start,end,wal);


  // now the loop: go from source to destination, 
  // "rotate" the edges we cross out of our way (if possible)
  // or cut them in half.

  tJUST_CONTROLLED_PTR<eHalfEdge> currentEdge  = NULL;
  // the edge we are currently considering.
  // the line we are to draw starts at currentEdge->Point()
  // and goes into currentEdge->Face().


  REAL left_test,right_test;
  int exactly_onLeft=0;

  assert(start->edge);

#ifdef DEBUG
  bool foundCurrentEdge;
#endif

  eCoord dir=eCoord(1,1); // the direction we are going

  int timeout = 10000;

  bool goon=1;
  while (goon && dir.Norm_squared()>0){
#ifdef DEBUG
    static int count = 0;
    count++;
    //    if (count == 40)
    //      st_Breakpoint();
#endif


    dir=end-*start; // the direction we are going

#ifdef DEBUG
    ePoint *laststarts[10];

    for (int i = 9; i>=1; i--)
      laststarts[i] = laststarts[i-1];

    laststarts[0] = start;

    if (timeout < 10)
	st_Breakpoint();
#endif

    if (timeout-- < 0)
      {
	requestCleanup = true;
	return start;  // give up.
      }

#ifdef DEBUG
    Check();
#endif

    // first, we need to find the first eFace:
    //    int i;

#ifdef DEBUG
    //    if (doCheck && start->id == 4792)
    //      st_Breakpoint();
#endif

    assert(restart || currentEdge);
    
    if (restart){
      exactly_onLeft = -1;

      eHalfEdge *run  = start->edge;
      assert(run);
      eHalfEdge *stop = run;

      eCoord lvec = run->Vec();
      eCoord rvec;
      left_test = lvec * dir;

      eHalfEdge *best = NULL;
      REAL best_score = -100;

      currentEdge = NULL;
      
      while (run)
	{
	  eHalfEdge *next = run->next->next;
	  assert(next->next = run);
	  next = next->other;

	  assert(next->Point() == start);

	  right_test = left_test;
	  rvec = lvec;
	  lvec = next->Vec();
	  left_test = lvec * dir;

	  if (right_test <= 0 && left_test >= 0)
	    {
	      bool good = rvec * lvec < 0 && right_test < 0;
	      if (good)
		next = NULL;
	      
	      if (good || !currentEdge)
		{
		  currentEdge = run;

		  // are dir and lvec exactly on one line?
		  exactly_onLeft=(left_test<=EPS*se_EstimatedRangeOfMult(lvec,dir)); 
		}
	    }
	  
	  REAL score = right_test * left_test;
	  if (right_test < 0 && left_test < 0)
	    score *= -1;     // here, - * - still is -.....

	  if (!best || score > best_score  && rvec * lvec < 0
)
	    best = run;
	  
	  run = next;
	  if (run == stop)
	    run = NULL;
	}

#ifdef DEBUG
      foundCurrentEdge = currentEdge;
      //      assert(run || currentEdge);
#endif

      
      if (!currentEdge && best)
	{
	  if (restart >= 3)
	    {
	      requestCleanup = true;
	      // give up.
	      st_Breakpoint();
	      return start;
	    }
	  
	  start = Insert(*start, best->Face());
	  restart ++;
	  continue;

	  currentEdge = best;
	}

      if (!currentEdge)
	tERR_ERROR_INT("Point " << *start << " does not have a eFace in direction " 
		       << dir << "!");
      // I know this bug happens a lot more often than"
      //  " I'd like to. Don't send a bug report!");
      
      restart=0;
    }

    if (start != currentEdge->Point())
      tERR_ERROR_INT("currentEdge not correct!");

    eHalfEdge *rightFromCurrent = currentEdge->next;
    eHalfEdge *leftFromCurrent  = rightFromCurrent->next;
    assert (currentEdge == leftFromCurrent->next);
    // are we finished?
    
    if (exactly_onLeft<0)
      {
	eCoord lvec = leftFromCurrent->Vec();
	exactly_onLeft =  dir * lvec < EPS * se_EstimatedRangeOfMult(lvec, dir);
      }

    eCoord left_point_to_dest = end - *(leftFromCurrent->Point());
    eCoord lvec               = rightFromCurrent->Vec();
    
    REAL test = lvec * left_point_to_dest;
    if(test<0){
      // endpoint lies within current eFace; cut it into three and be finnished.
      goon=0;
      
      /*
	
        C        a        B
	................
	.  .*cc     *** .
       .  . *  D    bb.
	   .  |      .
    e1	   .  |     .
	    . |aa  .
   .      b . |   .  c
 ..   e3     .|  .
E         G  .| .
    .	      ..
      e2      .

	      A

       */
      
      ePoint *A=currentEdge->Point();
      
      if ((*A) == end){
#ifdef DEBUG
	Check();
#endif
	return A;
      }
      else{
	eHalfEdge* a=rightFromCurrent;
	eHalfEdge* b=leftFromCurrent;
	eHalfEdge* c=currentEdge;

	ePoint*    B=a->Point();
	ePoint*    C=b->Point();

	if ((*B) == end)
	  return B;

	if ((*C) == end)
	  return C;

	ePoint*    D=tNEW(ePoint(end));

#ifdef DEBUG
	  Check();
#endif

	KillFace(a->Face());

	if(!exactly_onLeft)
	  {
	    // no problem: D really lies within current eFace.
	    // just split it in three parts:
	    
	    eHalfEdge *aa = tNEW(eHalfEdge) (A, D);
	    eHalfEdge *bb = tNEW(eHalfEdge) (B, D);
	    eHalfEdge *cc = tNEW(eHalfEdge) (C, D);
	    
	    //	  aa->SetOther(   tNEW(eHalfEdge) (D));
	    //      bb->SetOther(   tNEW(eHalfEdge) (D));
	    //	  cc->SetOther(   tNEW(eHalfEdge) (D));
	    
	    AddFaceAll(tNEW(eFace) (b,aa,cc->other));
	    AddFaceAll(tNEW(eFace) (bb,aa->other,c));
	    AddFaceAll(tNEW(eFace) (bb->other,a,cc));
	    
	    aa->SetWall(wal);
	    
#ifdef DEBUG
	    Check();
#endif
	    return D;
	  }
	else // exactly_onLeft
	  {
	    //	    st_Breakpoint();
	    
	    // bad luck: D lies on eEdge b. We are going to need the
	    // other triangle touching b:
	    assert(b->other);
	    
	    eFace *G=b->other->face;
	    
	    assert(G);

	    eHalfEdge *e1=b->other->next;
	    eHalfEdge *e2=e1->next;
	    assert(e2);
	    ePoint *E=e2->point;
	    assert(E);
	    
	    if (*E == *D)
	      {
		e2->other->SetWall(wal);
		return E;
	      }
	    
	    // delete faces inside quad ABCE and connect A...E with new
	    // ePoint D.
	    
	    KillFace(G);
	    KillEdge(b);
	    
	    eHalfEdge *DE=tNEW(eHalfEdge) (D,E);
	    eHalfEdge *DB=tNEW(eHalfEdge) (D,B);
	    eHalfEdge *DC=tNEW(eHalfEdge) (D,C);
	    eHalfEdge *DA=tNEW(eHalfEdge) (D,A);
	    DA->other->SetWall(wal);
	    
	    AddFaceAll(tNEW(eFace) (DE,e2,DA->other));
	    AddFaceAll(tNEW(eFace) (DC,e1,DE->other));
	    AddFaceAll(tNEW(eFace) (DB,a ,DC->other));
	    AddFaceAll(tNEW(eFace) (DA,c ,DB->other));
	    
#ifdef DEBUG
	    Check();
#endif
	    return D;
	  }
	}
      // finnished!
    }


    else{
      // some definitions and trivial tests:

      eHalfEdge* c = currentEdge;
      eHalfEdge* e = rightFromCurrent;
      eHalfEdge* d = leftFromCurrent;

      eFace  *F = c->Face();
      ePoint *C = c->Point();

      if (*C == end){
#ifdef DEBUG
	Check();
#endif
	return C;
      }
      //      eHalfEdge *dummy=NULL;


      ePoint *D = d->Point();

      if (*D == end){
	d->other->SetWall(wal);

#ifdef DEBUG
	Check();
#endif
	return D;
      }

      ePoint *B= e->Point();

      if (*B == end){
	c->SetWall(wal);

#ifdef DEBUG
	Check();
#endif
	return B;
      }

      // if dir is parallel with eEdge c:

      /*
      if(exactly_onLeft){
	todo->Split(d,dummy);
	delete todo;
	todo=dummy;
	start=todo->p[0];
	restart=1;
      }
      
      else
      */
      {
	if (!e->other || !e->other->Face()){
	  tERR_ERROR_INT ("Left Grid!");
	}
	else{
	  eFace  *G=e->other->Face();
	  
	  eHalfEdge* b = e->other->next;
	  eHalfEdge* a = b->next;
	  assert(a->next == e->other);
	  
	  ePoint *A = a->Point();
	  
	/* we have the following Situation:
	   
	   
                A      
	        *
               * **
	      *    ** b
	    a*       **
	    *   G |    **
	   *      |   e  ** 
	  D********|********* B
	    *      |(new eWall)
	     *   F |      *
	      *    |     *
	       *   |    *
	     d  *  |   * c 
	         * |  *
	          *| *
	           **
	           C
             
      */
      
      /*

	if the angles CDA and ABC are less than 180deg, we can just 
	"turn" eEdge e:


           A      
            *
	   ****
          * *  ** b
        a*  *    **
        *   * |    **
       *    e*|  G   ** 
     D*   F  *|       ** B
       *     *|(new eWall)
        *    *|      *
         *   *|     *
          *  *|    *
        d  *  *   * c 
            * *  *
	     ****
              **
	      C


      */

	  eCoord dd=*C-*D;
	  eCoord aa=*A-*D;
	  eCoord bb=*A-*B;
	  eCoord cc=*C-*B;
	  
	  REAL scale = aa.Norm_squared() + bb.Norm_squared() +
	    bb.Norm_squared() + cc.Norm_squared();

	  assert(scale > 0);
	  assert(aa.Norm_squared() > scale *1E-40f);
	  assert(bb.Norm_squared() > scale *1E-40f);
	  assert(dd.Norm_squared() > scale *1E-40f);
	  assert(cc.Norm_squared() > scale *1E-40f);

#ifdef DEBUG
	  Check();
#endif
			    
	  if (change_grid && e->Movable() && aa*dd>scale*.0001 && cc*bb >scale*.0001){
	    if (*A == *C)
	      {
		start = A;
		restart = 1;
		continue;
	      }


	    KillFace(F);
	    KillFace(G);
	    KillEdge(e);
	    
	    e=tNEW(eHalfEdge) (C,A);
	    F=tNEW(eFace)     (a,d,e);
	    G=tNEW(eFace)     (b,e->other,c);
	    
	    AddFaceAll(F);
	    AddFaceAll(G);

	    eCoord ee=*A-*C;
	    REAL x_test=ee*dir;
	    if (x_test>=0){
	      exactly_onLeft=(x_test<=EPS*se_EstimatedRangeOfMult(ee,dir));
	      currentEdge = c;
	      assert(currentEdge->ID >= 0);
	    }
	    else{
	      exactly_onLeft=-1;
	      currentEdge = e;
	      assert(currentEdge->ID >= 0);
	    }
	    
#ifdef DEBUG
	    Check();
#endif

	  }
	  
	  /* if not, we hace to introduce a new ePoint E: the cut between our
	     new eWall and eEdge e.
	     
	     A      
	     *
	   * **
          *  * ** b
        a*   a'  **
        *  G *| G' **
       *      *E     ** 
     D***d`*******b`**** B
       *      *(new eWall)
        *  F  *      *
         *    * F'  *
          *   *    *
        d  *  c'  * c 
            * *  *
	     ****
              **
	      C


     */

	  else{
	    // calculate the coordinates of the intersection ePoint E:
	    ePoint *E;
	    eHalfEdge *bb,*cc,*dd;

	    if (*C == *A)
	      {
		start = A;
		restart = 1;
		continue;
	      }

	    REAL ar = .5f;
	    
	    {
	      eHalfEdge *newtodo;
	      ePoint *CC = tNEW(ePoint(end));
	      eTempEdge todo (C, CC);
	      E=tNEW(ePoint) (e->IntersectWithCareless(todo.Edge(0)));

	      ar = e->Ratio(*E);

	      //	      REAL br = todo.Edge(0)->Ratio(*E);
	      //	      assert(-.1f < ar && ar < 1.1f);
	      //	      assert(-.1f < br && br < 1.1f);
	      

	      assert(E->Norm_squared() < 1E+15);
	      todo.Edge(0)->Split(cc,newtodo,E);
	      KillEdge(newtodo);  // not needed. Make sure it gets deleted.
	    }

	    if (wal)
	      {
#ifdef DEBUG
		assert(wal->Splittable());
		Check();
#endif

		eWall *wa, *wb;
		wal->Split(wa, wb, eCoord::V(*start, *E, end));
		delete wal;
		wal = wb;
		cc->SetWall(wa);
	      }

	    REAL arreal = ar;
	    if (ar > .99999f)
	      ar = .99999f;
	    if (ar < .00001f)
	      ar = .00001f;
	    *E = *B + (*D - *B) * ar;

	    if (*E == *D || (arreal >= 1.0f && !d->other->wall))
	      {
		d->other->SetWall(cc->wall);
		KillEdge(cc);
		start = D;
		restart = 1;
		continue;
	      }

	    if (*E == *B || (arreal <= 0.0f && !c->wall))
	      {
		c->SetWall(cc->wall);

		KillEdge(cc);
		start = B;
		restart = 1;
		continue;
	      }

		
	    
	    e->other->Split(dd,bb,E);

	    assert(!F->CorrectArea());
	    assert(!G->CorrectArea());
	    KillFace(F);
	    KillFace(G);
	    KillEdge(e);


	    start = E;

	    bool aeq = false, ceq = false;

	    if (*A == *E)
	      {
		if (A==leakPoint)
		  st_Breakpoint();

		A->Replace(E);
		KillPoint(A);

		a->other->SetWall(dd->wall);
		a->SetWall(dd->other->wall);
		b->SetWall(bb->other->wall);
		b->other->SetWall(bb->wall);
		KillEdge(dd);
		KillEdge(bb);
		dd = a->other;
		bb = b->other;
		aeq = true;
	      }


	    if (*C == *E)
	      {
		assert(!aeq);
		
		C->Replace(E);
		KillPoint(C);
		
		d->other->SetWall(dd->other->wall);
		d->SetWall(dd->wall);
		c->other->SetWall(bb->other->wall);
		c->SetWall(bb->wall);
		
		KillEdge(dd);
		KillEdge(bb);
		KillEdge(cc);
		dd = d;
		bb = c;
		ceq = true;
	      }
	    
	    if (!ceq)
	      {
		F=tNEW(eFace) (d,cc,dd->other);
		eFace *FF=tNEW(eFace) (c,bb->other,cc->other);
		
		AddFaceAll(F);
		AddFaceAll(FF);
	      }
	    
	    eHalfEdge *aa=NULL;
	    
	    
	    if (!aeq)
	      {
		aa = tNEW(eHalfEdge) (E,A);
		
		eFace *GG=tNEW(eFace) (b,aa->other,bb);
		G=tNEW(eFace) (a,dd,aa);
		AddFaceAll(G);
		AddFaceAll(GG);
	      }
	    else
	      {
		start = E;
		restart = true;
		continue;
	      }
	    
#ifdef DEBUG
	    Check();
#endif
	    if (end == *E)
	      return E;

	    //	    start = E;
	    eCoord ea=*A-*E;
	    dir   = end - *E;
	    REAL x_test=ea*dir;
	    
	    assert (aa);
	    if (x_test>=0){
	      currentEdge=bb;
	      assert(currentEdge->ID >= 0);
	      exactly_onLeft=(x_test<=EPS*se_EstimatedRangeOfMult(ea,dir));
	      // if (exactly_onLeft) cerr << "Achtung 3!\n";
	    }
	    
	    else{
	      currentEdge=aa;
	      assert(currentEdge->ID >= 0);
	      exactly_onLeft=-1;
	    }
	  }
	}
      }
    }
  }
  
  tERR_ERROR_INT("We shound never get here!");
  return NULL;
}


// **************************
//          eHalfEdge
// **************************




// splitting
bool eHalfEdge::Splittable() const
{
  eWall *w = wall;
  if (!w && other)
    w = other->wall;
  return !w || w->Splittable();
}

// split eEdge at s;
void eHalfEdge::Split(eHalfEdge *& e1,eHalfEdge *& e2,ePoint *s)
{
  assert(other);

  e1=tNEW(eHalfEdge) (Point(),s);
  e2=tNEW(eHalfEdge) (s,other->Point());

  if (wall)
    {
      if (!wall->Splittable())
	tERR_ERROR_INT("I told you not to split that!");
      // first, split the eWall structure:
      eWall *w1,*w2;    
      wall->Split(w1,w2,Ratio(*s));
      
      e1->SetWall(w1);
      e2->SetWall(w2);
    }
  if (other->wall)
    {
      if (!other->wall->Splittable())
	tERR_ERROR_INT("I told you not to split that!");
      // first, split the eWall structure:
      eWall *w1,*w2;    
      other->wall->Split(w2,w1,Ratio(*s));
      
      e1->other->SetWall(w1);
      e2->other->SetWall(w2);
    }
}


#define SIMPLIFY
#ifdef SIMPLIFY

bool eFace::CorrectArea()
{
  REAL area = edge->next->Vec() * edge->Vec();
  if (area > 0)
    return false;

  eHalfEdge *run = edge;

  // find the longest edge
  eHalfEdge *longest = NULL;
  REAL       length  = -1;

  for (int i=2; i>=0; i--)
    {
      REAL l = run->Vec().Norm_squared();
      if (l > length)
	{
	  length  = l;
	  longest = run;
	}
      run = run->next;
    }

  run = longest;
  eCoord &A = *run->Point();
  run       =  run->next;
  eCoord &B = *run->Point();
  run       =  run->next;
  eCoord &C = *run->Point();

  // now, C is the point opposite to the longest edge.
  eCoord MoveDir = (B-A);
  MoveDir = MoveDir.Turn(0, 1.1/MoveDir.Norm_squared() + 0.001f);
  C = C + MoveDir;

#ifdef DEBUG
  REAL newarea = edge->next->Vec() * edge->Vec();
  assert(newarea >= area);
#endif

  return true;
}


bool eHalfEdge::Simplify(eGrid *grid)
{
  if (wall && wall->Deletable() && face)
    {
      tDESTROY(wall);
    }
  
  if (wall || !other || other->wall || !face || !other->face)
    return false;

  ePoint  *to_be_removed=point; // we'll remove this ePoint
  ePoint  *stay=other->point;   // and redirect all his edges to this ePoint

  // first, we find the faces that are going to be modified:
  // they are neighbours of the ePoint we'll remove.

  eHalfEdge *run = this->other->next;
  do{
    //    eFace *F = run->face; // eFace to be modified
    assert(run->other);




    /*




                 C  this
   A ------------ --------------- stays
         Next    |
         	 | 
	         |r 
 	         |u   
	         |n    
	         |     
	         |
                 B
 



     */


    // if any of the to-be-modified edges is a eWall, abort the operation.
    if (run->wall || run->other->wall || !run->face)
      return false;
    
    eHalfEdge *Next = run->other->next;
    eCoord &A = *Next->other->point;
    eCoord &B = *run->other->point;
    eCoord &C = *stay;
#ifdef DEBUG
    //    eCoord &D = *to_be_removed;
    //    eCoord leftvec_b  = B-D;
    //    eCoord rightvec_b = A-D;
    //    REAL orientation_b=leftvec_b*rightvec_b;
#endif
    

    // if the eFace is to be modified (and not deleted), check
    // if it will flip orientation; if yes, abort.
    
    eCoord leftvec = B-C;
    eCoord rightvec= A-C;
    
    REAL orientation=leftvec*rightvec;
    
    if (orientation <= 0)
      return false;

    //    assert(orientation_b > 0);

    run = Next;
  }
  while (run && run->other &&  !(this == run->other->next));
  
  // dangerously close to the rim....
  if (!run || !run->other)
    return false;
  
  // Allright! everything is safe.
#ifdef DEBUG
  grid->Check();
#endif

  AddRef();

  eHalfEdge *a  = next;
  eHalfEdge *aa = a->next;
  ePoint *A     = aa->Point();

  eHalfEdge *b  = other->next;
  eHalfEdge *bb = b->next;
  ePoint *B     = bb->Point();

  if (a->wall || a->other->wall)
    return false;

  if (aa->wall || aa->other->wall)
    return false;

  if (b->wall || b->other->wall)
    return false;

  if (bb->wall || bb->other->wall)
    return false;

  stay->Replace(to_be_removed);
  



  // kill everything that goes away:
  grid->KillFace(face);
  grid->KillFace(other->face);
  grid->KillPoint(stay);

  grid->KillEdge(this);

  to_be_removed->edge = aa->other;
  A            ->edge = a->other;
  B            ->edge = b->other;

  // sew the edges of the faces that are going to disappear together:
  a->other->SetOther(aa->other);
  b->other->SetOther(bb->other);

  grid->KillEdge(a);
  grid->KillEdge(aa);
  grid->KillEdge(b);
  grid->KillEdge(bb);

  Release();

  return true;
}

void eGrid::SimplifyNum(int e){
#ifdef DEBUG
  Check();
#endif
  
  if (e>=0 && e < edges.Len()){
    eHalfEdge *E=edges(e);
#ifdef DEBUG
    eHalfEdge *Esave=E;
#endif
    E->Simplify(this);

#ifdef DEBUG
    assert(Esave);
    
    Check();
#endif
  }
}

#endif // SIMPLIFY

void eGrid::SimplifyAll(int count){
  if (requestCleanup)
    {
      requestCleanup = false;
      for (int i=faces.Len()-1; i>=0; i--)
	requestCleanup |= faces(i)->CorrectArea();
    }


  static int counter=0;
  
  for(;count>0;count--){
    counter--;
    
    if (counter<0 || counter>=edges.Len()) 
      counter=edges.Len()-1;
    
    SimplifyNum(counter);
  }
}



eGrid *currentGrid = NULL;

eGrid* eGrid::CurrentGrid(){return currentGrid;}

void eGrid::Create(){
  currentGrid = this;

  if (!A)
    Clear();

  base=eCoord(20,100);

#ifdef DEBUG
  doCheck = true;
#endif
  requestCleanup = false;


  A=tNEW(ePoint) (base.Turn(1,0));
  B=tNEW(ePoint) (base.Turn(-.5,.7));
  C=tNEW(ePoint) (base.Turn(-.5,-.7));
  
  a=tNEW(eHalfEdge) (B,C,tNEW(eWall(this)));
  b=tNEW(eHalfEdge) (C,A,tNEW(eWall(this)));
  c=tNEW(eHalfEdge) (A,B,tNEW(eWall(this)));

  a->other->next = c->other;
  a->other->prev = b->other;
  b->other->next = a->other;
  b->other->prev = c->other;
  c->other->next = b->other;
  c->other->prev = a->other;

  AddFaceAll(tNEW(eFace) (a,b,c));

  max_Norm_squared=base.Norm_squared();

#ifdef DEBUG
  Check();
#endif
  
}


void eGrid::Clear(){
  base=eCoord(0,100);
  max_Norm_squared=0;

  int i;
  for(i=faces.Len()-1; i>=0; i--)
    {
      faces(i)->Unlink();
      RemoveFace(faces(i));
    }
  for(i=points.Len()-1; i>=0; i--)
    {
      points(i)->Unlink();
      RemovePoint(points(i));
    }
  for(i=edges.Len()-1; i>=0; i--)
    {
      edges(i)->Unlink();
      RemoveEdge(edges(i));
    }

  
  A=B=C=NULL;
  a=b=c=NULL;
}

void eGrid::Range(REAL Norm_squared){
  if (A==NULL)
	Create();
//    tERR_ERROR_INT("Wanted to insert something into the nonexistant grid!");

  while (Norm_squared > max_Norm_squared *.1){
    base = base.Turn(-4,0);
    max_Norm_squared=base.Norm_squared();
    
    delete a->Wall();
    delete b->Wall();
    delete c->Wall();

    ePoint *newA=tNEW(ePoint) (base.Turn(1,0));
    ePoint *newB=tNEW(ePoint) (base.Turn(-.5,.7));
    ePoint *newC=tNEW(ePoint) (base.Turn(-.5,-.7));
  
    eHalfEdge *newa=tNEW(eHalfEdge) (newB,newC,tNEW(eWall(this)));
    eHalfEdge *newb=tNEW(eHalfEdge) (newC,newA,tNEW(eWall(this)));
    eHalfEdge *newc=tNEW(eHalfEdge) (newA,newB,tNEW(eWall(this)));
    
    eHalfEdge *AnewB=tNEW(eHalfEdge) (A,newB,NULL);
    eHalfEdge *AnewC=tNEW(eHalfEdge) (A,newC,NULL);
    eHalfEdge *BnewA=tNEW(eHalfEdge) (B,newA,NULL);
    eHalfEdge *BnewC=tNEW(eHalfEdge) (B,newC,NULL);
    eHalfEdge *CnewA=tNEW(eHalfEdge) (C,newA,NULL);
    eHalfEdge *CnewB=tNEW(eHalfEdge) (C,newB,NULL);
    
    AddFaceAll(tNEW(eFace) (a->other,BnewA,CnewA->other));
    AddFaceAll(tNEW(eFace) (b->other,CnewB,AnewB->other));
    AddFaceAll(tNEW(eFace) (c->other,AnewC,BnewC->other));

    AddFaceAll(tNEW(eFace) (newa,AnewC->other,AnewB));
    AddFaceAll(tNEW(eFace) (newb,BnewA->other,BnewC));
    AddFaceAll(tNEW(eFace) (newc,CnewB->other,CnewA));

    a=newa;
    b=newb;
    c=newc;
    
    A=newA;
    B=newB;
    C=newC;

    a->other->next = c->other;
    a->other->prev = b->other;
    b->other->next = a->other;
    b->other->prev = c->other;
    c->other->next = b->other;
    c->other->prev = a->other;
    
#ifdef DEBUG
    Check();
#endif

  }
}


void eTempEdge::CopyIntoGrid(eGrid *grid)
{
  if (Point(0) != Point(1)){
    ePoint *newp = grid->Insert(*Point(0));
    eWall *w = Edge(0)->Wall();
    w->Insert();
    assert(w->Splittable());
    //    Edge(0)->wall = NULL;
    //    w->edge       = NULL;
    //    Edge(0)->SetWall(NULL);
    grid->DrawLine(newp, *Point(1), w);
  }
}

eTempEdge::eTempEdge(const eCoord& a, const eCoord &b, eWall *w){
  halfEdges[0] = tNEW(eHalfEdge(tNEW(ePoint)(a),tNEW(ePoint)(b),w));
  halfEdges[1] = halfEdges[0]->Other();
  halfEdges[0]->AddRef();
} 

eTempEdge::eTempEdge(ePoint *a, ePoint *b, eWall *w){
  halfEdges[0] = tNEW(eHalfEdge(a, b ,w));
  halfEdges[1] = halfEdges[0]->Other();
  halfEdges[0]->AddRef();
} 



eTempEdge::~eTempEdge()
{
  halfEdges[0]->Unlink();
  halfEdges[0]->Release();
}






eHalfEdge::eHalfEdge(ePoint *a, ePoint *b, eWall *w)
  :ID(-1),next(NULL), prev(NULL), other(NULL), wall(0), refCtr(0)
{
  assert( a !=  b);

  SetPoint(a);
  SetOther(new eHalfEdge(b));
  SetWall(w);
}

// get the intersection point of two edges: 
// stupid intersection without checks; returned point
// does not allways lie within the two edges
eCoord eHalfEdge::IntersectWithCareless(const eHalfEdge *e2) const
{
  eCoord vec  = *other->Point()     -     *Point();
  eCoord e2vec= *e2->other->Point() - *e2->Point();
  eCoord e2_t = *Point()     - *e2->Point();

  REAL inv = (vec*e2vec);
  if (inv!=0)
    return (*Point()-vec*((e2_t*e2vec)/inv));
  else
    return (*Point()-vec*((e2_t*e2vec)));
}

// the same, but with checks: if the two edges don't intersect,
// NULL is returned.
ePoint * eHalfEdge::IntersectWith(const eHalfEdge *e2) const
{
  eCoord vec  = *other->Point()     -     *Point();
  eCoord e2vec= *e2->other->Point() - *e2->Point();
  eCoord e2_t = *Point()     - *e2->Point();
  
  REAL div=(vec*e2vec);

  if (div==0) return NULL;

  eCoord ret=*Point()-vec*((e2_t*e2vec)/ div);


  REAL v=Ratio(ret);
  if (v<0 || v>1) return NULL;
  v=e2->Ratio(ret);
  if (v<0 || v>1) return NULL;

  return tNEW(ePoint) (ret);

}

// inserts an eEdge into the grid
void eHalfEdge::CopyIntoGrid(eGrid *grid, bool change_grid){
  assert(other);
  ePoint *start = grid->Insert(*Point());
  grid->DrawLine(start, *other->Point(), wall, change_grid);
}

//bool eHalfEdge::Simplify(int dir);
// try to get rid of this eEdge; return value: can we delete it?


bool eHalfEdge::Check() const{
  if (other && other->next)
    assert(other->next->Point() == Point());
  if (next)
    assert(next->Face() == Face());
  if (next)
    assert(this == next->prev);
  if (prev)
    assert(this == prev->next);

  if (ID>=0)
    {
      assert(!Point() || Point()->ID >=0);
      assert(!Face()  || Face()->ID  >=0);
      assert(!next    || next->ID    >=0);
      assert(!prev    || prev->ID    >=0);
      assert(!other   || other->ID   >=0);
    }

  return true;
}



void eGrid::AddFace    (eFace     *f)
{
  if (f->ID >= 0)
    return;

  f->AddRef();
  faces.Add(f, f->ID);

  if (f->CorrectArea())
    requestCleanup = true;
}

void eGrid::RemoveFace (eFace     *f)
{
  if (f->ID < 0)
    return;

  faces.Remove(f, f->ID);
  f->Release();
}

void eGrid::AddEdge    (eHalfEdge *e)
{
  if (e->ID >= 0)
    return;

  assert(e->Other() && *e->Point() != *e->Other()->Point());

  e->AddRef();
  edges.Add(e, e->ID);
}

void eGrid::RemoveEdge (eHalfEdge *e)
{
  if (e == leakEdge)
    st_Breakpoint();

  if (e->ID < 0)
    return;

  edges.Remove(e, e->ID);
  e->Release();
}

void eGrid::AddPoint   (ePoint    *p)
{
  if (p->ID >= 0)
    return;

  p->AddRef();
  points.Add(p, p->ID);
}

void eGrid::RemovePoint(ePoint    *p)
{
  if (p == leakPoint)
    st_Breakpoint();

  if (p->ID < 0)
    return;

  points.Remove(p, p->ID);
  p->Release();
}

void eGrid::KillFace (eFace*      f)
{
  f->AddRef();
  RemoveFace(f);
  f->Unlink();
  f->Release();
}

void eGrid::KillEdge (eHalfEdge*  e)
{
  e->AddRef();
  RemoveEdge(e);
  eHalfEdge *o = e->other;

  if (o)
    o->AddRef();

  e->Unlink();
  e->Release();

  if (o)
    {
      KillEdge(o);
      o->Release();
    }
}

void eGrid::KillPoint(ePoint*     p)
{
  p->AddRef();
  RemovePoint(p);
  p->Unlink();
  p->Release();
}

void eGrid::AddFaceAll (eFace     *f)
{
  eHalfEdge *run = f->edge;
  for(int j=0;j<=2;j++){
    AddEdge(run);
    AddEdge(run->other);
    AddPoint(run->Point());
    run = run->next;
  }
  assert (run == f->edge);

  AddFace(f);
}



ePoint * eGrid::Insert(const eCoord& x, eFace *start){
#ifdef DEBUG
  Check();
#endif
  Range(x.Norm_squared());

  eFace *f = FindSurroundingFace(x, start);
  if (!f)
    return NULL;

  /*
  for(int i=faces.Len()-1;i>=0;i--)
    {
      eFace *f = faces(i);
      if (f->IsInside(x)){

  */

  eHalfEdge *run = f->edge;
  for(int j=0;j<=2;j++){
    if (*run->Point() == x)
      return run->Point();
    run = run->next;
  }
  assert (run == f->edge);

  return DrawLine(f->edge->Point(), x);
  /*
	break;
      }
    }
  return NULL; 
  */
}

const eCoord se_zeroCoord(0,0);


eFace * eGrid::FindSurroundingFace(const eCoord &pos, eFace *currentFace) const{
  if (faces.Len()<1)
    return NULL;

  if (!currentFace || currentFace->ID < 0 || currentFace->ID >= faces.Len() || faces(currentFace->ID) != currentFace)
    currentFace=faces(0);

  int timeout=faces.Len()+2;

  while (currentFace && timeout >0 && !currentFace->IsInside(pos)){
    timeout--;

    eHalfEdge *run   = currentFace->Edge(); // runs through all edges of the face
    eHalfEdge *best  = NULL;                // the best face to leave
    eHalfEdge *first = NULL;
    eHalfEdge *in    = NULL;
    REAL bestScore   = 0;


    // look for the best way out
    while(run != first)
      {
	if (run == in)
	  {
	    run = run->Next();
	    continue;
	  }

	eCoord vec=pos - (*run->Point()); // the vector to our destination

	REAL score = run->Vec() * vec;

	if (!first)
	  first = run;

	if (score > bestScore)
	  {
	    best      = run;
	    bestScore = score;
	  }

	run = run->Next();
      }
    
    if (best)
      {
	in = best->Other();
	assert(in);
	
	currentFace = in->Face();
	timeout--;
      }
    else
      {
	timeout = 0;
	st_Breakpoint();
      }
  }

  if (timeout<=0){ // normal way failed
#ifdef DEBUG
    con << "WARNING! FindSurroundingFace failed.\n";
#endif
    // do it the hard way:
    for(int i=faces.Len()-1;i>=0;i--)
      if(faces(i)->IsInside(pos)){
	currentFace=faces(i);
	i=-1;
      }
  }

  return currentFace;
}

void eGrid::AddGameObjectInteresting    (eGameObject *o){
  gameObjectsInteresting.Add(o, o->interestingID);
}

void eGrid::RemoveGameObjectInteresting (eGameObject *o){
  gameObjectsInteresting.Remove(o, o->interestingID);
}

void eGrid::AddGameObjectInactive       (eGameObject *o){
  gameObjectsInactive.Add(o, o->inactiveID);
}

void eGrid::RemoveGameObjectInactive    (eGameObject *o){
  gameObjectsInactive.Remove(o, o->inactiveID);
}


/*


void eEdge::calc_Len() const{
  REAL len2=(*p[0] - *p[1]).Norm_squared();
  *reinterpret_cast<REAL *>(reinterpret_cast<void *>(&len))=sqrt(len2);
}


void ePoint::InsertIntoGrid(){
  if (f.Len()==0){
    for(int i=eFace::faces.Len()-1;i>=0;i--)
      
      if (eFace::faces(i)->IsInside(*this)){
	for(int j=0;j<=2;j++){
	  if (*eFace::faces(i)->p[j]==*this){
	    ePoint *same=eFace::faces(i)->p[j];
	    same->Replace(this);
	    delete same;
	    return;
	  }
	}
	eFace::faces(i)->p[0]->DrawLineTo(this);
	return;
      }
  }
}



      eHalfEdge *run  = start->edge;
      assert(run);
      const eHalfEdge *stop = run;

      do
	{
	  
	  run = run->other->next;
	}
      while (stop != start);

*/


eGrid::eGrid()
  :refCtr(0),
   requestCleanup(false),
   A(NULL), B(NULL), C(NULL),
   a(NULL), b(NULL), c(NULL),
   max_Norm_squared(-1),
   base(100,100)
{
  currentGrid = this;
}


eGrid::~eGrid()
{
  if (currentGrid == this)
    currentGrid = NULL;
}






void eHalfEdge::AddRef(){
#ifdef DEBUG
  if (this == leakEdge)
    st_Breakpoint();
#endif

  assert (refCtr>=0);
  refCtr++;
}

void eHalfEdge::Release(){
#ifdef DEBUG
  if (this == leakEdge)
    st_Breakpoint();
#endif

  assert(refCtr > 0);
  
  refCtr--; 
  if (refCtr<=0)
    delete this;
}

void ePoint::AddRef(){
#ifdef DEBUG
  if (this == leakPoint)
    st_Breakpoint();
#endif

  assert (refCtr>=0);
  refCtr++;
}

void ePoint::Release(){
#ifdef DEBUG
  if (this == leakPoint)
    st_Breakpoint();
#endif

  assert(refCtr > 0);
  
  refCtr--; 
  if (refCtr<=0)
    delete this;
}

void eFace::AddRef(){
  assert (refCtr>=0);
  refCtr++;
}

void eFace::Release(){
  assert(refCtr > 0);
  
  refCtr--; 
  if (refCtr<=0)
    delete this;
}
