/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	CD information class
 *
 *	by Tony Sideris	(05:01PM Feb 11, 2002)
 *================================================*/
#include "arson.h"

#include <qfile.h>

#include "cdinfoeditor.h"
#include "cdinfo.h"
#include "konfig.h"

#ifndef ARSONDBG
#	define dump()
#endif

/*========================================================*/
/*	Format a track filename (basename) with this track
 *	information using the given format.
 *
 *	Example, "%a - %t" becomes:
 *	"Black Sabbath - Snowblind"
 *
 *	%a - Artist
 *	%v - Album artist or "Various Artists"
 *	%c - CD Title
 *	%t - Track Title
 *	%n - Track Number
 *	%N - Track Number (2 characters wide, '0' filled)
 *	%z - Zero-based Track Number
 *	%Z - Zero-based Track Number (2 characters wide, '0' filled)
 *========================================================*/

QString arson_STRIP (const QString &repl)
{
	QString copy;

	for (int index = 0; index < repl.length(); ++index)
		if (!ACONFIG.ripper().isStrippedChar(repl[index].latin1())
			&& repl[index].latin1() != '/')
			copy += repl[index];

	return copy;
}

bool ArsonCdInfo::Track::formatTrackName (const ArsonCdInfo *parent,
	const QString &fmt, QString &result) const
{
	struct {
		const char *find;
		QString with;
	}	repls[] = {
		{ "%a", arson_STRIP(parent->variousArtistDisk() ? m_artist : parent->artist()) },
		{ "%v", arson_STRIP(parent->variousArtistDisk() ? i18n("Various Artists") : parent->artist()) },
		{ "%c", arson_STRIP(parent->title()) },
		{ "%t", arson_STRIP(m_title) },
		{ "%n", QString::number(trackNo()) },
		{ "%z", QString::number(trackNo() - 1) },
		{ "%N", QString().sprintf("%02d", trackNo()) },
		{ "%Z", QString().sprintf("%02d", trackNo() - 1) },
//		{ "%x", QString().sprintf("0x02x", trackNo()) }
	};

	result = fmt;

	for (int index = 0; index < ARRSIZE(repls); ++index)
		arsonReplace(repls[index].find, repls[index].with, result);

	return true;
}

/*========================================================*/

ArsonCdInfo::ArsonCdInfo (QObject *parent, const char *name)
	: QObject(parent, name), m_leadout(0)
{
	ARSON_INSTANCE_INCR("ArsonCdInfo");
}

ArsonCdInfo::~ArsonCdInfo (void)
{
	ARSON_INSTANCE_DECR("ArsonCdInfo");
}

/*========================================================*/

ArsonCdInfo::Track::Track (void)
	: m_offset(0),
	m_trackNo(0)
{
	//	Nothing...
}

/*========================================================*/

bool ArsonCdInfo::formatTrackName (int track,
	const QString &fmt, QString &result) const
{
	if (track == -1)
		return Track().formatTrackName(
			this, fmt, result);

	/*	Do not assume that m_tracks[1] == TRACK_1 ...
	 *	it PROBABLY does, but do not assume it, the
	 *	track number is given in the cdindex data,
	 *	use that.
	 */
	for (int index = 0; index < m_tracks.count(); index++)
		if (track == m_tracks[index].trackNo())
			return m_tracks[index].formatTrackName(this, fmt, result);

	return false;
}

/*========================================================*/

ArsonCdInfo::Track &ArsonCdInfo::addTrack (const ArsonCdInfo::Track &track)
{
	return *(m_tracks.append(track));
}

/*========================================================*/

void ArsonCdInfo::clear (void)
{
	m_tracks.clear();

	m_title = m_artist =
		m_year = m_genre = m_categ =
		m_cddbID = m_cdIndexID =
		m_ext = m_playorder =
		QString::null;

	m_leadout = 0;
}

/*========================================================*/

uint ArsonCdInfo::totalSeconds (void) const
{
	uint total, index;

	for (total = index = 0; index < trackCount(); ++index)
		total += track(index).time().totalSeconds();

	return total;
}

/*========================================================*/

void ArsonCdInfo::edit (QWidget *parent, const char *name)
{
	ArsonCdInfoEditor dlg (kapp->mainWidget(), *this);

	if (dlg.exec() == QDialog::Accepted)
		merge(dlg.info());
}

void ArsonCdInfo::edit (void)
{
	edit(kapp->mainWidget(), NULL);
}

/*========================================================*/
#ifdef ARSONDBG
void ArsonCdInfo::dump (void) const
{
#define STR(s)		(s).latin1()
	
	Trace(
		"ArsonCdInfo: %s / %s\n"
		"IDs: %s / %s\n"
		"Year/Genre/Category: %s/%s/%s\n"
		"Extra: %s\n"
		"Playorder: %s\n"
		"Leadout: %u\n\nTracks:\n",
		STR(m_title), STR(m_artist),
		STR(m_cddbID), STR(m_cdIndexID),
		STR(m_year), STR(m_genre), STR(m_categ),
		STR(m_ext),
		STR(m_playorder),
		m_leadout);

	for (int index = 0; index < m_tracks.count(); ++index)
	{
		const Track &track = m_tracks[index];

		Trace(
			" %d: %s / %s\n"
			"   Extra: %s\n"
			"   Offset: %u\n"
			"   Time: %s\n\n",
			track.trackNo(), STR(track.title()), STR(track.artist(this)),
			STR(track.comment()),
			track.offset(),
			STR(track.time().display()));
	}
}
#endif	//	ARSONDBG
/*========================================================*/

void ArsonCdInfo::merge (const ArsonCdInfo &info)
{
/*
	const QString cddb (info.cddbID());
	const QString cdi (info.cdIndexID());
	ArsonCdInfo *that = this;

	(*that) = info;

	setCdIndexID(cdi);
	setCddbID(cddb);
*/
	setTitle(info.title());
	setArtist(info.artist());
	setCddbID(info.cddbID());
	setCdIndexID(info.cdIndexID());
	setYear(info.year());
	setGenre(info.genre());
	setCateg(info.categ());
	setComment(info.comment());
	setPlayOrder(info.playOrder());
	setLeadout(info.leadout());
	setTracks(info.tracks());

	emit changed();
}

/*========================================================*/
