/*
 * Copyright (c) 2001 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	Document widget impl
 *
 *	by Tony Sideris	(05:47PM Aug 24, 2002)
 *================================================*/
#include "arson.h"

#include <qdragobject.h>
#include <qpopupmenu.h>
#include <qstatusbar.h>
#include <qfileinfo.h>
#include <qlabel.h>
#include <qxml.h>
#include <qfile.h>
#include <qmultilineedit.h>

#include <kglobalsettings.h>
#include <kiconloader.h>
#include <kglobal.h>
#include <kfiledialog.h>
#include <klocale.h>
#include <krun.h>
#include <kopenwith.h>
#include <kaction.h>
#include <kfileitem.h>

#include "progressdlg.h"
#include "progressbar.h"
#include "filewriter.h"
#include "docwidget.h"
//#include "mainwnd.h"
#include "listwnd.h"
#include "konfig.h"

const QString ArsonDocWidget::ARSON_FILTER(
	I18N_NOOP("*.arson|arson Files")
	);

/*========================================================*/
/*	Simple wrapper over an action collection
 *	for enabling and disabling
 *========================================================*/

void ArsonActionEnabler::enable (KStdAction::StdAction act, bool en)
{
	if (KAction *pa = m_actions.action(KStdAction::stdName(act)))
		enable(pa, en);

#ifdef ARSONDBG
	else { Trace("no action found, std: %d\n", act); }
#endif
}

void ArsonActionEnabler::enable (const char *name, bool en)
{
	if (KAction *pa = m_actions.action(name))
		enable(pa, en);

#ifdef ARSONDBG
	else { Trace("no action found %s\n", name); }
#endif
}

void ArsonActionEnabler::enable (KAction *pa, bool en)
{
	pa->setEnabled(en);
}

/*========================================================*/
/*	Action Container - Passed to ArsonObject::connectTo
 *	so that docs/views can connect/disconnect the actions
 *	they need.
 *========================================================*/

ArsonActionConnector::ArsonActionConnector(
	KActionCollection &col, ArsonDocWidget *pw, bool connect)
	: ArsonActionEnabler(col), m_pWidget(pw), m_connect(connect)
{
	//	Nothing...
}

/*========================================================*/

void ArsonActionConnector::connect (KStdAction::StdAction act,
	const char *slot, QObject *recv, bool enable)
{
	if (KAction *pa = m_actions.action(KStdAction::stdName(act)))
		connect(pa, slot, recv, enable);

#ifdef ARSONDBG
	else { Trace("Failed to find action %d\n", act); }
#endif
}

void ArsonActionConnector::connect (const char *name,
	const char *slot, QObject *recv, bool enable)
{
	if (KAction *pa = m_actions.action(name))
		connect(pa, slot, recv, enable);

#ifdef ARSONDBG
	else { Trace("Failed to find action %s\n", name); }
#endif
}

void ArsonActionConnector::connect (KAction *pa,
	const char *slot, QObject *recv, bool enable)
{
	if (!recv)
		recv = m_pWidget;

	if (m_connect)
		QObject::connect(pa, SIGNAL(activated()), recv, slot);
	else
		pa->disconnect(recv, slot);

	pa->setEnabled(m_connect && enable);
}

/*========================================================*/

void ArsonActionConnector::disconnect (KStdAction::StdAction act)
{
	if (KAction *pa = m_actions.action(KStdAction::stdName(act)))
		disconnect(pa);
}

void ArsonActionConnector::disconnect (const char *name)
{
	if (KAction *pa = m_actions.action(name))
		disconnect(pa);
}

void ArsonActionConnector::disconnect (KAction *pa)
{
	pa->disconnect(m_pWidget, NULL);
}

/*========================================================*/
/*	Base document widget class implementation
 *========================================================*/

ArsonDocWidget::ArsonDocWidget (QWidget *parent, const char *name)
	: QVBox(parent, name), m_pActions(NULL),
	m_flags(stdDocActions | sizeCanChange),
	m_pStatus(NULL)
{
	QObject::connect(this, SIGNAL(updateUI()), this, SLOT(slotUpdateUI()));
}

ArsonDocWidget::~ArsonDocWidget (void)
{
	//	Nothing...
}

/*========================================================*/

void ArsonDocWidget::setModified (bool mod)
{
	boolf(modified, mod);
	emit updateUI();
}

/*========================================================*/

void ArsonDocWidget::refresh (void) { }

/*========================================================*/

ArsonActionConnector ArsonDocWidget::connector (bool active)
{
	Assert(m_pActions != NULL);

	return ArsonActionConnector(
		*m_pActions, this, active);
}

ArsonActionEnabler ArsonDocWidget::enabler (void)
{
	Assert(m_pActions != NULL);

	return ArsonActionEnabler(*m_pActions);
}

/*========================================================*/

void ArsonDocWidget::setSizeCanChange (bool ch)
{
	boolf(sizeCanChange, ch);

	if (ch)
		emitSizeChanged();
}

/*========================================================*/

void ArsonDocWidget::emitSizeChanged (void)
{
	if (canSizeChange())
	{
		emit sizeChanged();
	}
}

/*========================================================*/

void ArsonDocWidget::newDocument (void)
{
	try {
		deleteContents();
		m_url = KURL();

		setModified(false);
	}
	catch (ArsonAbortAction &err) {
		//	Cancelled
	}
}

/*========================================================*/

bool ArsonDocWidget::saveDocument (const KURL &url)
{
	bool success = false;
	
	if (ArsonBaseWriter *pw = createFileWriter(url))
	{
		if ((success = writeDocumentFile(*pw)))
			setModified(false);
		
		delete pw;
	}

	return success;
}

/*========================================================*/

void ArsonDocWidget::setDocumentName (const KURL &url)
{
	m_url = url;
}

/*========================================================*/

void ArsonDocWidget::deleteContents (void)
{
	if (isModified() && is(stdDocActions))
	{
		const int res = KMessageBox::warningYesNoCancel(this,
			i18n("This document has been modified, do you want to save first?"),
			i18n("Document Modified"));

		if (res == KMessageBox::Cancel)
			throw ArsonAbortAction();

		else if (res == KMessageBox::Yes)
			slotSave();
	}
}

/*========================================================*/

void ArsonDocWidget::setActive (bool active)
{
	ArsonActionConnector ac (connector(active));
	
	if (m_pStatus)
		m_pStatus->show(active);

	connectTo(ac);

	if (active)
	{
		emit updateUI();
	}
}

/*========================================================*/

void ArsonDocWidget::connectTo (ArsonActionConnector &ac)
{
	if (const bool en = is(stdDocActions))
	{
		ac.connect(KStdAction::New, SLOT(slotNew()), NULL, en);
		ac.connect(KStdAction::Save, SLOT(slotSave()), NULL, en);
		ac.connect(KStdAction::SaveAs, SLOT(slotSaveAs()), NULL, en);
		ac.connect("file_export_text", SLOT(slotExportText()), NULL, en);
	}
}

void ArsonDocWidget::slotUpdateUI (void)
{
	if (m_pActions)
	{
		ArsonActionEnabler en (enabler());
		uiUpdated(en);
	}
}

void ArsonDocWidget::uiUpdated (ArsonActionEnabler &en)
{
	en.enable(KStdAction::Save, isModified());
}

/*========================================================*/

bool ArsonDocWidget::writeDocumentFile (ArsonBaseWriter &writer)
{
	bool success = false;

	if (writer.valid())
	{
		writer.beginDocument();

		if ((success = writeDocument(writer)))
			writer.endDocument();
	}

	return success;
}

/*========================================================*/

void ArsonDocWidget::setStatusBar (QStatusBar *psb)
{
	delete m_pStatus;
	m_pStatus = initStatus(psb);
}

ArsonDocWidget::Status *ArsonDocWidget::initStatus (QStatusBar *psb)
{
	return new Status(psb);
}

/*========================================================*/

void ArsonDocWidget::setPaneText (const char *name, const QString &text)
{
	const QString pad ("  ");
	QLabel *pl = (QLabel *) pane(name);

	if (pl)
		pl->setText(pad + text + pad);
}

QWidget *ArsonDocWidget::pane (const char *name)
{
	return m_pStatus ? m_pStatus->pane(name) : NULL;
}

void ArsonDocWidget::status (const QString &str)
{
	if (m_pStatus)
		m_pStatus->message(str);
}

/*========================================================*/

void ArsonDocWidget::Status::addPane (const char *name, const QString &text)
{
	QLabel *ptr = new QLabel(m_pStatus);

	addPane(name, ptr);
	ptr->setText(text);
}

void ArsonDocWidget::Status::addPane (const char *name, QWidget *pw)
{
	if (pw->parent() != m_pStatus)
		pw->reparent(m_pStatus, QPoint());

	m_pStatus->addWidget(pw, false, true);
	m_panes.insert(name, pw);
	pw->hide();
}

QWidget *ArsonDocWidget::Status::pane (const char *name)
{
	//	Hehe... try and say this three times quickly...
	if (m_panes.contains(name))
		return m_panes[name];

	return NULL;
}

void ArsonDocWidget::Status::message (const QString &text)
{
	m_pStatus->message(text);
}

void ArsonDocWidget::Status::show (bool sh)
{
	for (STATUSPANES::Iterator it = m_panes.begin();
		 it != m_panes.end(); it++)
	{
		if (sh)
			it.data()->show();
		else
			it.data()->hide();
	}
}

/*========================================================*/

void ArsonDocWidget::slotNew (void)
{
	newDocument();
}

/*========================================================*/

namespace arson {
	KURL appendExtension(const KURL &url)
	{
		KURL saveurl(url);
		QString savename = saveurl.fileName();

		if (savename.right(6) != ".arson")
			savename += ".arson";

		saveurl.setFileName(savename);
		return saveurl;
	}

	KURL getSaveFile (void)
	{
		KURL url = KFileDialog::getSaveURL(NULL, i18n(ArsonDocWidget::ARSON_FILTER));

		if (url.isEmpty())
			return KURL();

		url = appendExtension(url);

		if (url.isLocalFile() && QFileInfo(url.path()).exists())
		{
			if (KMessageBox::questionYesNo(kapp->mainWidget(),
					i18n("The document %1 exists, overwrite?").arg(url.path()),
					i18n("Overwrite Prompt")) == KMessageBox::No)
				return KURL();
		}

		return url;
	}
};

/*========================================================*/

void ArsonDocWidget::slotSave (void)
{
	if (m_url.isEmpty())
	{
		const KURL url = arson::getSaveFile();

		if (url.isEmpty())
			return;

		setDocumentName(url);
	}

	saveDocument(m_url);
}

/*========================================================*/

void ArsonDocWidget::slotSaveAs (void)
{
	const KURL url = arson::getSaveFile();

	if (!url.isEmpty())
	{
		setDocumentName(url);
		saveDocument(url);
	}
}

/*========================================================*/

void ArsonDocWidget::slotModified (void)
{
	setModified(true);
}

/*========================================================*/

void ArsonDocWidget::slotExportText (void)
{
	const KURL url = KFileDialog::getSaveURL(QString::null,
		ArsonFileFilter(i18n("*.txt|Text Files")).toString());

	if (!url.isEmpty())
	{
		ArsonTextFileWriter writer (this, url);
		writeDocumentFile(writer);
	}
}

/*========================================================*/
/*	A list item
 *========================================================*/

ArsonListItem::ArsonListItem (void)
{
	ARSON_INSTANCE_INCR("ArsonListItem");
}

ArsonListItem::~ArsonListItem (void)
{
	ARSON_INSTANCE_DECR("ArsonListItem");
}

/*========================================================*/

QListViewItem *ArsonListItem::createItem (ArsonListWnd *parentWnd,
		QListViewItem *parentItem, QListViewItem *pAfter)
{
	QListViewItem *pi = arson::newListViewItem<QListViewItem>(parentWnd, parentItem, pAfter);

	return pi;
}

void ArsonListItem::refresh (QListViewItem *pi, ArsonDocWidget *pd)
{
	pi->setText(0, display());
}

/*========================================================*/

ArsonLvPos::ArsonLvPos (ArsonListDoc *pd)
	: result(NULL)
{
	pd->defaultLvPos(*this);
}

/*========================================================*/
/*	Root item info
 *========================================================*/

void ArsonRootItemInfo::createItem (QListView *pl)
{
	pl->setRootIsDecorated(true);
	
	m_pItem = new QListViewItem(pl, defaultText());
	m_pItem->setPixmap(0, ArsonFileListItem::loadIcon(icon()));
	m_pItem->setExpandable(true);
	m_pItem->setOpen(true);
}

void ArsonRootItemInfo::resetItem (const QString &text)
{
	if (m_pItem)
		m_pItem->setText(0, text.isNull() ? defaultText() : text);
}

QString ArsonRootItemInfo::itemText (void) const
{
	return m_pItem ? m_pItem->text(0) : QString::null;
}

/*========================================================*/
/*	The list document widget
 *========================================================*/

ArsonListDoc::ArsonListDoc (QWidget *parent, const char *name)
	: ArsonDocWidget(parent, name),
	m_renameCol(-1),
	m_editableList(true),
	m_pList(NULL),
	m_pRoot(NULL)
{
}

ArsonListDoc::~ArsonListDoc (void)
{
	delete m_pRoot;
}

/*========================================================*/

void ArsonListDoc::create (void)
{
	ArsonDocWidget::create();

	m_pList = createListWnd();

	QObject::connect(m_pList, SIGNAL(moved()), this, SLOT(slotModified()));
	QObject::connect(m_pList, SIGNAL(selectionChanged()), SLOT(slotUpdateUI()));

	if (m_pRoot)
		m_pRoot->createItem(m_pList);
}

ArsonListWnd *ArsonListDoc::createListWnd (void)
{
	return new ArsonListWnd(this);
}

/*========================================================*/

void ArsonListDoc::setRootItemInfo (ArsonRootItemInfo *pi)
{
	delete m_pRoot;
	m_pRoot = pi;
}

/*========================================================*/

QListViewItem *ArsonListDoc::rootItem (void) const
{
	return m_pRoot ? m_pRoot->item() : NULL;
}

/*========================================================*/

void ArsonListDoc::addItem (ArsonListItem *ptr, ArsonLvPos *pp)
{
	ArsonLvPos pos;

	if (!pp)
		defaultLvPos(pos);
	else
		pos = *pp;

//	Trace("parent=%p after=%p\n", pos.parent, pos.after);

	if (QListViewItem *pi = ptr->createItem(m_pList, pos.parent, pos.after))
	{
		ptr->refresh(pi, this);

		m_items.insert(pi, ptr);

		emitSizeChanged();
		setModified();

		if (pp)
			pp->result = pi;
	}
}

void ArsonListDoc::delItem (QListViewItem *ptr)
{
	if (ArsonListItem *pi = item(ptr))
	{
		m_items.remove(ptr);

		delete pi;
		delete ptr;

		emitSizeChanged();
		setModified();
	}
}

/*========================================================*/

void ArsonListDoc::delTreeListItem (QListViewItem *pi)
{
	while (QListViewItem *pc = pi->firstChild())
		delTreeListItem(pc);

	delItem(pi);
}

/*========================================================*/

void ArsonListDoc::deleteContents (void)
{
	QListViewItem *pr = rootItem();
	ArsonDocWidget::deleteContents();

	ArsonDocSizeLock dsl (this);

	//	Recursively remove all listview items (and arson items)
	while (QListViewItem *pi = pr ? pr->firstChild() : m_pList->firstChild())
		delTreeListItem(pi);

	if (m_pRoot)
		m_pRoot->resetItem();
}

/*========================================================*/

ArsonListItem *ArsonListDoc::item (QListViewItem *pi) const
{
	return m_items.contains(pi)
		? m_items[pi]
		: NULL;
}

/*========================================================*/

ArsonListDoc::ItemIterator::ItemIterator (const ArsonListDoc *pd, QListViewItem *pi)
	: m_pd(pd), m_pListItem(pi)
{
	ArsonListWnd *pw = pd ? pd->listWnd() : NULL;

	if (!pi && pw)
	{
		if (!(m_pListItem = pd->rootItem()))
			m_pListItem = pw->firstChild();
	}
}

ArsonListDoc::ItemIterator ArsonListDoc::ItemIterator::childIterator (void) const
{
	if (m_pListItem)
	{
		QListViewItem *pi = m_pListItem->firstChild();

		return ItemIterator(m_pd, pi);
	}

	return ItemIterator(NULL);
}

ArsonListDoc::ItemIterator &ArsonListDoc::ItemIterator::operator++ (void)
{
	m_pListItem = m_pListItem->nextSibling();
	return *this; // FIXME
}

ArsonListDoc::ItemIterator &ArsonListDoc::ItemIterator::operator++ (int)
{
	m_pListItem = m_pListItem->nextSibling();
	return *this;
}

/*========================================================*/

namespace arson {
	QListViewItem *lastChildItem (QListViewItem *pi)
	{
		for (QListViewItem *next = pi ? pi->nextSibling() : NULL;
			 next; pi = next, next = next->nextSibling());

		return pi;
	}
};

void ArsonListDoc::defaultLvPos (ArsonLvPos &pos)
{
	pos = ArsonLvPos();

	if (QListViewItem *pi = m_pList->currentItem())
	{
		QListViewItem *child = pi->firstChild();

		pos.parent = (child || pi->isExpandable())
			? pi : pi->parent();
	}

	if (!pos.parent)
		pos.parent = rootItem();

	if (pos.parent)
		pos.after = arson::lastChildItem(pos.parent->firstChild());
	else
		pos.after = arson::lastChildItem(m_pList->firstChild());
}

/*========================================================*/

void ArsonListDoc::connectTo (ArsonActionConnector &ac)
{
	ac.connect(KStdAction::SelectAll, SLOT(slotListSelAll()));
	ac.connect("list_invert", SLOT(slotListInvertSel()));

	if (editableList())
	{
		ac.connect("list_del", SLOT(slotListDel()));
		ac.connect("list_up", SLOT(slotListUp()));
		ac.connect("list_down", SLOT(slotListDn()));
		ac.connect("list_shuffle", SLOT(slotShuffle()));
	}

	if (renameColumn() != -1)
		ac.connect("list_rename", SLOT(slotRename()));
	
	ArsonDocWidget::connectTo(ac);
}

/*========================================================*/

void ArsonListDoc::uiUpdated (ArsonActionEnabler &en)
{
	const bool empty = this->empty();
	QListViewItem *pc = (m_pList ? m_pList->currentItem() : NULL);

	en.enable(KStdAction::SelectAll, !empty);
	en.enable("list_invert", !empty);
	en.enable("list_del", pc != NULL);
	en.enable("list_up", pc && pc->itemAbove());
	en.enable("list_down", pc && pc->itemBelow());
	en.enable("list_shuffle", count() > 1);
	en.enable("list_rename",
		(pc && pc == rootItem()) || (!empty && renameColumn() != -1));

	ArsonDocWidget::uiUpdated(en);
}

/*========================================================*/

bool ArsonListDoc::buildContextMenu (QPopupMenu *pup)
{
	KActionCollection *pa = actionCollection();//ArsonFrame::getFrame()->actionCollection();

	if (editableList())
	{
		const char *items[] = {
			"list_add",
			"list_add_dir",
			"list_del",
			NULL,
			"list_up",
			"list_down",
			NULL,
			"list_shuffle",
			NULL,
			"list_write_cd",
		};

		for (int index = 0; index < ARRSIZE(items); ++index)
		{
			if (!items[index])
			{
				pup->insertSeparator();
				continue;
			}

			pa->action(items[index])->plug(pup);
		}
	}

	if (renameColumn() != -1)
	{
		pup->insertSeparator();
		pa->action("list_rename")->plug(pup);
	}

	return true;
}

/*========================================================*/
/*	List item visit (for each)
 *========================================================*/

bool ArsonListDoc::visit (ArsonListVisiter &obj, QListViewItem *pi)
{
	do
	{
		if (ArsonListItem *ptr = item(pi))
			if (!obj.visit(pi, ptr))
				return false;

		if (QListViewItem *pc = pi->firstChild())
			if (!visit(obj, pc))
				return false;
	}
	while ((pi = pi->nextSibling()));

	return true;
}

bool ArsonListDoc::visit (ArsonTreeVisiter &obj, QListViewItem *pi)
{
	QListViewItem *pc;

	do
	{
		if ((pc = pi->firstChild()) || pi->isExpandable())
		{
			ArsonListItem *pl = item(pi);

			if (pl && !obj.beginBranch(pi, pl))
				return false;

			if (pc && !visit(obj, pc))
				return false;

			if (pl)
				obj.endBranch();
		}

		else if (ArsonListItem *pl = item(pi))
			if (!obj.visit(pi, pl))
				return false;
	}
	while ((pi = pi->nextSibling()));	

	return true;
}

/*========================================================*/

bool ArsonListDoc::visitAll (ArsonListVisiter &obj)
{
	if (QListViewItem *pi = m_pList->firstChild())
		return visit(obj, pi);

	return false;
}

bool ArsonListDoc::visitAll (ArsonTreeVisiter &obj)
{
	if (QListViewItem *pi = m_pList->firstChild())
		return visit(obj, pi);

	return false;
}

bool ArsonListDoc::visitAll (ArsonListVisiter &obj) const
{
	ArsonListDoc *that = (ArsonListDoc *) this;
	QListViewItem *pi = m_pList ? m_pList->firstChild() : NULL;

	if (pi)
		return that->visit(obj, pi);

	return false;
}

bool ArsonListDoc::visitAll (ArsonTreeVisiter &obj) const
{
	ArsonListDoc *that = (ArsonListDoc *) this;
	QListViewItem *pi = m_pList ? m_pList->firstChild() : NULL;

	if (pi)
		return that->visit(obj, pi);

	return false;
}

/*========================================================*/

class refreshVisiter : public ArsonListVisiter
{
public:
	refreshVisiter (ArsonListDoc *pd) : m_pd(pd) { }

	virtual bool visit (QListViewItem *pi, ArsonListItem *pl)
	{
		pl->refresh(pi, m_pd);
		return true;
	}

private:
	ArsonListDoc *m_pd;
};

void ArsonListDoc::refresh (void)
{
	refreshVisiter rv (this);
	visitAll(rv);
}

/*========================================================*/

bool ArsonListDoc::onStartElement (const QString &name,
	const QXmlAttributes &attr, ArsonListInserter &pos)
{
	if (name == QString("arson-") + propDocType() && m_pRoot)
	{
		const QString label = attr.value(m_pRoot->xmlAttr());

		if (!label.isEmpty())
			m_pRoot->resetItem(label);
	}

	return ArsonDocWidget::onStartElement(name, attr, pos);
}

void ArsonListDoc::editRootElement (ArsonXmlWriter &xml)
{
	ArsonDocWidget::editRootElement(xml);

	if (m_pRoot)
		xml.addAttribute("label", m_pRoot->itemText());
}

/*========================================================*/

void ArsonListDoc::slotContextMenu (KListView *ptr, QListViewItem *item, const QPoint &pnt)
{
	QPopupMenu menu;

	if (buildContextMenu(&menu))
		menu.exec(pnt);
}

/*========================================================*/
/*	Write the contents to file
 *========================================================*/

bool ArsonListDoc::writeDocument (ArsonBaseWriter &writer)
{
	return visitAll(writer);
}

/*========================================================*/

void ArsonListDoc::slotListDel (void)
{
	ArsonDocSizeLock dsl (this);
	QList<QListViewItem> il = m_pList->selectedItems();
	
	while (!il.isEmpty())
	{
		delTreeListItem(il.first());

		il = m_pList->selectedItems();

		for (il.first(); il.current(); )
		{
			if (!item(il.current()))
				il.remove();
			else
				il.next();
		}
	}
}

/*========================================================*/

void ArsonListDoc::slotListUp (void)
{
	QListViewItem *pi, *pre;

	if ((pi = m_pList->currentItem()) &&
		(pre = pi->itemAbove()) &&
		(pi->parent() == pre->parent()))
	{
		m_pList->moveItem(pre, NULL, pi);
		setModified();
	}
}

void ArsonListDoc::slotListDn (void)
{
	QListViewItem *pi, *next;

	if ((pi = m_pList->currentItem()) &&
		(next = pi->nextSibling()))
	{
		pi->moveItem(next);
		setModified();
	}
}

/*========================================================*/

class invertSelVisiter : public ArsonListVisiter
{
	virtual bool visit (QListViewItem *pi, ArsonListItem *pl)
	{
		pi->listView()->setSelected(pi, !pi->isSelected());
		return true;
	}
};

void ArsonListDoc::slotListInvertSel (void)
{
	invertSelVisiter isv;
	visitAll(isv);
}

/*========================================================*/

void ArsonListDoc::slotShuffle (void)
{
	typedef QValueList<QListViewItem*> LVILIST;

	QListViewItem *pf;
	LVILIST items;

	//	Gets the first item in the currently selected branch
	if (QListViewItem *pi = m_pList->currentItem())
		pf = ((pi = pi->parent())) ? pi->firstChild() : m_pList->firstChild();
	else
		pf = m_pList->firstChild();

	//	Make a list fo the currently selected branch
	while (pf)
	{
		items.append(pf);
		pf = pf->nextSibling();
	}

	if (items.count() > 0)
	{
		int index;
		const int count = items.count();
		
		//	Shuffle the list
		for (index = 0; index < (count << 1); ++index)
		{
			ts::swap(
				items[rand() % count],
				items[rand() % count]);
		}

		for (index = 0; index < count; ++index)
		{
			QListView *pl = items[0]->listView();

			pl->takeItem(items[index]);
			pl->insertItem(items[index]);
		}

		slotModified();
	}
}

/*========================================================*/

void ArsonListDoc::slotListSelAll (void)
{
	m_pList->selectAll(true);
}

/*========================================================*/

void ArsonListDoc::slotRename (void)
{
	if (QListViewItem *pi = listWnd()->currentItem())
	{
		const int col = renameColumn();

		if (col != -1)
			listWnd()->rename(pi, col);
	}
}

/*========================================================*/
/*	File list document
 *========================================================*/

ArsonFileListDoc::ArsonFileListDoc (QWidget *parent, const char *name)
	: ArsonListDoc(parent, name),
	m_pProgress(NULL),
	m_pd(NULL),
	m_maxProgress(0),
	m_totalLength(0)
{
	QObject::connect(this, SIGNAL(sizeChanged()),
		this, SLOT(slotSizeChanged()));

	m_pd = new ArsonProgressDisplay(i18n("MB"), (1024 * 1024));
}

ArsonFileListDoc::~ArsonFileListDoc (void)
{
	delete m_pd;
}

/*========================================================*/

void ArsonFileListDoc::create (void)
{
	ArsonListDoc::create();

	m_pProgress = new ArsonDiskUsageBar(m_pd ? m_pd->maxProgress() : 0, 0, this);

	if (m_pd)
	{
		m_pProgress->setSuffix(m_pd->suffix());
		m_pProgress->setMod(m_pd->mod());
	}

	if (ArsonListWnd *pl = listWnd())
	{
		QObject::connect(
			pl, SIGNAL(dropped(QDropEvent*,QListViewItem*,QListViewItem*)),
			this, SLOT(slotDropped(QDropEvent*,QListViewItem*,QListViewItem*))
			);
	}
}

/*========================================================*/

ArsonListWnd *ArsonFileListDoc::createListWnd (void)
{
	return new ArsonFileListWnd(this);
}

/*========================================================*/

void ArsonFileListDoc::slotDropped (QDropEvent *pd,
	QListViewItem *parent, QListViewItem *after)
{
	QStrList sl;

	if (QUriDrag::decode(pd, sl))
	{
		ArsonDocSizeLock dsl (this);
		ArsonListInserter pos (parent, after);
		
		for (char *ps = sl.first(); ps; ps = sl.next())
		{
			KURL url (ps);
         url.adjustPath(1);
         KFileItem kfi(KFileItem::Unknown, KFileItem::Unknown, url);

			Trace("dropped: %s\n", ps);

         if (kfi.isDir())
         {
            url.adjustPath(1);
            addDirectory(url, pos);
         }
			else if (kfi.isFile())
			{
            ArsonLvPos lvp (pos.getParent(), pos.getAfter());
				if (addFileItem(ps, &lvp))
					pos.setAfter(lvp.result);
			} else
         {
            Trace("Invalid/Unknown file dropped!\n");
         }
		}
	}
}

/*========================================================*/

ArsonBaseWriter *ArsonFileListDoc::createFileWriter (const KURL &url)
{
	return new ArsonXmlFileListWriter(this, url);
}

ArsonFileListItem *ArsonFileListDoc::createFileItem (const KURL &url) const
{
	return new ArsonFileListFileItem(url);
}

ArsonFileListItem *ArsonFileListDoc::createDirItem (const QString &dn) const
{
	return new ArsonFileListDirItem(dn);
}

/*========================================================*/

void ArsonFileListDoc::addItem (ArsonListItem *ptr, ArsonLvPos *pos)
{
	ArsonFileListItem *pi = (ArsonFileListItem *) ptr;

	m_totalLength += pi->length();

	ArsonListDoc::addItem(ptr, pos);
}

void ArsonFileListDoc::delItem (QListViewItem *ptr)
{
	if (ArsonFileListItem *pi = (ArsonFileListItem *) item(ptr))
		m_totalLength -= pi->length();

	ArsonListDoc::delItem(ptr);
}

/*========================================================*/

void ArsonFileListDoc::deleteContents (void)
{
	ArsonListDoc::deleteContents();

	setProgress(0);
}

/*========================================================*/

void ArsonListInserter::setParent (QListViewItem *pi)
{
	parent = pi;
	after = NULL;
}

void ArsonListInserter::clearParent (void)
{
	after = parent;
	parent = parent ? parent->parent() : NULL;
}

void ArsonListInserter::setAfter (QListViewItem *pi)
{
	after = pi;
}

/*========================================================*/

bool ArsonFileListDoc::onStartElement (const QString &name,
	const QXmlAttributes &attr, ArsonListInserter &pos)
{
	int index;
	ArsonLvPos lvp (pos.getParent(), pos.getAfter());

	if (!ArsonListDoc::onStartElement(name, attr, pos))
		return false;

	if (name == propFolder() && (index = attr.index("name")) != -1)
	{
		addDirItem(KURL::decode_string(attr.value(index)), &lvp);

		pos.setParent(lvp.result);
		return true;
	}

	else if (name == propItem() && (index = attr.index("url")) != -1)
	{
		const QString url = KURL::decode_string(attr.value(index));

		if (url != QString::null)
		{
			const KURL u (url);

			//	Warn if the file does now exist
			if (u.isLocalFile() && !QFileInfo(QFile::encodeName(u.path())).exists())
			{
				if (!arsonWarning(i18n("%1 doesn't exist!\nContinue?")
						.arg(url), QString::null))
					return false;

				return true;
			}

			if (addFileItem(u, &lvp))
				pos.setAfter(lvp.result);
		}

		return true;
	}

	return false;
}

bool ArsonFileListDoc::onEndElement (const QString &name, ArsonListInserter &pos)
{
	if (!ArsonListDoc::onEndElement(name, pos))
		return false;

	if (name == propFolder())
		pos.clearParent();

	return true;
}

/*========================================================*/

bool ArsonFileListDoc::addFileItem (const KURL &url, ArsonLvPos *pp)
{
	if (ArsonListItem *ptr = createFileItem(url))
	{
		addItem(ptr, pp);

		return true;
	}

	return false;
}

bool ArsonFileListDoc::addDirItem (const QString &dirname, ArsonLvPos *pp)
{
	if (ArsonListItem *ptr = createDirItem(dirname))
	{
		addItem(ptr, pp);
		return true;
	}

	return false;
}

/*========================================================*/

void ArsonFileListDoc::slotListAdd (void)
{
	ArsonDocSizeLock dsl (this);
	ArsonFileFilter filter (this);
	KURL::List::ConstIterator it;

	const KURL::List url = KFileDialog::getOpenURLs(
		NULL, filter.toString());

	try {
		ArsonLvPos pos;

		defaultLvPos(pos);

		for (it = url.begin(); it != url.end(); ++it)
		{
			if ((*it).isValid())
			{
				if (addFileItem(*it, &pos))
					pos.swapCurrent();
			}
		}
	}
	catch (ArsonError &err) {
		err.report();
	}
/*
	catch (...) {
		arsonErrorMsg(
			i18n("Caught unknown exception adding file"));
	}
*/
}

void ArsonFileListDoc::slotListAddDir (void)
{
	ArsonLvPos pos;
	const QString dir = KFileDialog::getExistingDirectory();

	defaultLvPos(pos);

	//	Iterate each mp3 file in the directory...
	if (dir != QString::null)
	{
		ArsonDocSizeLock dsl (this);
		ArsonListInserter li (pos.parent, pos.after);

      KURL url(dir);
      url.adjustPath(1);
      
		addDirectory(url, li);
	}
}

void ArsonFileListDoc::slotWriteCD (void)
{
	ArsonProgressDlg dlg (kapp->mainWidget(), NULL);

	if (!count())
	{
		arsonErrorMsg(
			i18n("Document empty, add somthing to burn first."));
	}

	else if (ArsonProgress *pDlg = createProgress(&dlg))
	{
		dlg.setProgressWidget (pDlg);
		dlg.exec();
	}
}

/*========================================================*/

void ArsonFileListDoc::slotExportM3u (void)
{
	const KURL url = KFileDialog::getSaveURL(QString::null,
		ArsonFileFilter(i18n("*.m3u|M3U Files")).toString());

	if (!url.isEmpty())
	{
		ArsonM3uFileWriter writer (this, url);
		writeDocumentFile(writer);
	}
}

/*========================================================*/

void ArsonFileListDoc::connectTo (ArsonActionConnector &ac)
{
	if (editableList())
	{
		ac.connect("list_add", SLOT(slotListAdd()));
		ac.connect("list_add_dir", SLOT(slotListAddDir()));
//		ac.connect("list_shuffle", SLOT(slotShuffle()));
		ac.connect("list_write_cd", SLOT(slotWriteCD()));
		ac.connect("file_export_m3u", SLOT(slotExportM3u()));

		ac.connect("list_open", SLOT(slotFileOpen()));
		ac.connect("list_open_with", SLOT(slotFileOpenWith()));
		ac.connect("list_file_view", SLOT(slotFileView()));
	}

	ArsonListDoc::connectTo(ac);
}

/*========================================================*/

void ArsonFileListDoc::uiUpdated (ArsonActionEnabler &en)
{
	const bool empty = this->empty();
	QListViewItem *pc = (!empty ? listWnd()->currentItem() : NULL);

	en.enable("list_write_cd", !empty);
	en.enable("file_export_m3u", !empty);
	en.enable("list_open", pc != NULL);
	en.enable("list_open_with", pc != NULL);
	en.enable("list_file_view", pc != NULL);
	
	ArsonListDoc::uiUpdated(en);
}

/*========================================================*/

bool ArsonFileListDoc::buildContextMenu (QPopupMenu *pm)
{
	KActionCollection *pa = actionCollection();//ArsonFrame::getFrame()->actionCollection();

	pa->action("list_open")->plug(pm);
	pa->action("list_open_with")->plug(pm);
	pa->action("list_file_view")->plug(pm);

	pm->insertSeparator();
	return ArsonListDoc::buildContextMenu(pm);
}

/*========================================================*/
/*	Read a line from a file regardless of the EOL
 *	used in the file.
 *========================================================*/

namespace arson {
	QString readFileLine (QFile &file)
	{
		QString result = QString::null;
		int ch;

		while ((ch = file.getch()) != -1)
		{
			if (ch == '\r' || ch == '\n')
				break;

			result.append(char(ch));
		}

		while ((ch = file.getch()) != -1)
			if (ch != '\r' && ch != '\n')
			{
				file.ungetch(ch);
				break;
			}

		return result;
	}
};

bool arsonMd5Check (const char *filename);	//	in md5check.cpp

void ArsonFileListDoc::openMd5File (const KURL &url)
{
	ArsonNetFile nf (url);

	//	Check the md5 check sum if the option is set.
	if (!ACONFIG.is(ArsonConfig::flagMd5Verify) || arsonMd5Check(nf.path()))
	{
		QFile in (nf.path());

		//	Open the input md5 file
		if (in.open(IO_ReadOnly))
		{
			QString line;
			const QDir dir = QFileInfo(nf.path()).dir();

			//	Reset the contents of the document (new doc)
			if (ACONFIG.is(ArsonConfig::flagMd5Reset))
			{
				try { deleteContents(); }
				catch (ArsonAbortAction&) {
					//	Cancelled
					return;
				}
			}

			ArsonDocSizeLock dsl (this);
			
			//	Read contents line-by-line
			while ((line = arson::readFileLine(in)) != QString::null)
			{
				int pos = line.find(' ');	//	Format is: CHECKSUM   FILENAME

				if (pos != -1)
				{
					/*	Skip all spaces top find the start of the filename,
					 *	also skip the first '*', md5 files created with the
					 *	"-b" switch have one '*' preceding the filename.
					 */
					while (pos < line.length() && line[++pos].isSpace());
					if (line[pos] == '*') ++pos;

					const bool res = addFileItem(QString("file:")
						+ dir.absFilePath((line.latin1() + pos)));

					//	Allow the user to bail if there was an error.
					if (!res && !arsonWarning(
							i18n("Failed to add %1. Continue?")
							.arg(line.latin1() + pos), QString::null))
						break;
				}
			}
		}
	}
}

/*========================================================*/

void ArsonFileListDoc::openMd5Dir (const QString &dirName)
{
	const QDir dir (dirName);
	const QStringList files = dir.entryList("*.md5");

	if (files.count() == 1)
		openMd5File(KURL(QString("file:") + files[0]));

	else if (files.count() > 1)
	{
		ArsonRadioDlg dlg;
		QStringList::ConstIterator it, end;

		for (it = files.begin(), end = files.end(); it != end; ++it)
			dlg.addItem(*it);

		if (dlg.exec() == QDialog::Accepted)
		{
			openMd5File(QString("file:") +
				dir.absFilePath(dlg.selection()));
		}
	}
	else
		arsonErrorMsg(
			i18n("No md5 files found in %1")
			.arg(dirName));
}

/*========================================================*/

void ArsonFileListDoc::openM3uFile (const KURL &url)
{
	ArsonNetFile nf (url);
	QFile file (nf.path());

	//	Open the m3u file
	if (file.open(IO_ReadOnly))
	{
		QString line;
		bool ext = false;
		const QDir dir = QFileInfo(nf.path()).dir();

		//	Reset the document
		try { deleteContents(); }
		catch (ArsonAbortAction&) {
			//	Cancelled
			return;
		}

		ArsonDocSizeLock dsl (this);
		
		//	Iterate the file line by line
		for (int index = 0; (line = arson::readFileLine(file)) != QString::null; ++index)
		{
			//	If the first line is "#EXTM3U" then enable comment skipping
			if (!index && line.left(7) == "#EXTM3U")
				ext = true;

			//	Skip comments is this is an EXTM3U file
			if (ext && line[0] == '#')
				continue;

			//	Add the current file
			const bool res = addFileItem(
				QString("file:") + dir.absFilePath(line));

			if (!res && !arsonWarning(
					i18n("Failed to add %1. Continue?").arg(line), QString::null))
				break;
		}
	}
	else
		arsonErrorMsg(
			i18n("Failed to open m3u file: %1")
			.arg(url.path()));
}

/*========================================================*/

void ArsonFileListDoc::addDirectory (const KURL &url, ArsonListInserter &pos)
{
	ArsonFileFilter filter (this);
	QDir qd (QFile::encodeName(url.directory(false,false)));
   
	QStringList::ConstIterator it, end;
	const QStringList sl = qd.entryList(
		filter.filters(),
		QDir::Files, QDir::Name);

	for (it = sl.begin(), end = sl.end(); it != end; it++)
	{
		ArsonLvPos lvp (pos.getParent(), pos.getAfter());
		const QString absPath = qd.absFilePath(*it);

		if (addFileItem(KURL(absPath), &lvp))
			pos.setAfter(lvp.result);
	}
}

/*========================================================*/

void ArsonFileListDoc::setMaxProgress (int maxl)
{
	m_pProgress->setMax((m_maxProgress = maxl));
}

void ArsonFileListDoc::setProgress (int length)
{
	if (m_pd)
		setPaneText("time", m_pd->fullText(length));
	
	m_pProgress->setPos(length);
}

QString ArsonProgressDisplay::fullText (int length) const
{
	return i18n("%1 / %2")
		.arg(arsonByteSize(length))
		.arg(arsonByteSize(maxProgress()));
}

int ArsonProgressDisplay::maxProgress (void) const
{
	return (ACONFIG.cdlenMB() * mod());
}

/*========================================================*/

void ArsonFileListDoc::setProgressInfo (ArsonProgressDisplay *pd)
{
	delete m_pd;
	m_pd = pd;
}

/*========================================================*/

void ArsonFileListDoc::slotSizeChanged (void)
{
	setProgress(m_totalLength);
}

/*========================================================*/

ArsonFileListFileItem *ArsonFileListDoc::currentFileItem (void) const
{
	ArsonFileListFileItem *pi = (ArsonFileListFileItem *)
		item(listWnd()->currentItem());

	return (pi && !pi->isDir()) ? pi : NULL;
}

/*========================================================*/

void ArsonFileListDoc::slotFileOpen (void)
{
	if (ArsonFileListFileItem *pi = currentFileItem())
	{
		KRun *pr = new KRun(pi->local());
		pr->setAutoDelete(true);
	}
}

void ArsonFileListDoc::slotFileOpenWith (void)
{
	if (ArsonFileListFileItem *pi = currentFileItem())
	{
		KURL url (QString("file:") + pi->local());
		KURL::List kul (QStringList(url.path()));
		KOpenWithDlg dlg (kul, this);
		KService *ptr;

		if (dlg.exec() && (ptr = dlg.service()))
			KRun::run(*ptr, kul);
	}
}

#include "_textviewer.h"

class arsonFileViewer : public ArsonTextViewerBase
{
public:
	arsonFileViewer (const KURL &url, QWidget *parent)
		: ArsonTextViewerBase(parent, NULL, TRUE)
	{
		try {
			ArsonNetFile nf (url);
			QFile file (nf.path());
			QString str;

			if (file.open(IO_ReadOnly))
			{
				QString line;

				while ((line = arson::readFileLine(file)) != QString::null)
					str.append(line + "\n");

				text->setFont(KGlobalSettings::fixedFont());
				text->setWordWrap(QMultiLineEdit::NoWrap);
				text->setText(str);
			}
		}
		catch (ArsonError &err) {
			err.report();
		}
	}
};

void ArsonFileListDoc::slotFileView (void)
{
	if (ArsonFileListFileItem *pi = currentFileItem())
	{
		arsonFileViewer dlg (KURL(pi->local()), this);

		dlg.exec();
	}
}

/*========================================================*/

ArsonDocWidget::Status *ArsonFileListDoc::initStatus (QStatusBar *psb)
{
	Status *ps = ArsonListDoc::initStatus(psb);

	ps->addPane("time", m_pd ? m_pd->fullText(0) : QString());
	return ps;
}

/*========================================================*/
/*	File list item impl
 *========================================================*/

void ArsonFileListFileItem::refresh (QListViewItem *pi, ArsonDocWidget *pd)
{
	ArsonListItem::refresh(pi, pd);
	pi->setText(1, arsonByteSize(length()));
}

uint ArsonFileListFileItem::length (void) const
{
	return QFileInfo(local()).size();
}

/*========================================================*/
/*	Directory list item
 *========================================================*/

ArsonFileListDirItem::ArsonFileListDirItem (const QString &dn)
	: ArsonFileListItem()
{
	QStringList sl = QStringList::split(QString("/"), dn);

	if (!sl.isEmpty())
		m_name = sl[sl.count() - 1];
	else
		m_name = dn;
}

uint ArsonFileListDirItem::length (void) const
{
	return 0;	//	FIXME: Change to size of dir in iso9660?
}

/*========================================================*/
/*	For proper sorting prepend a '0' or a '1' before
 *	the dir/file based on ascending/descending.
 *========================================================*/

namespace arson {
	QString sortKey (const QListViewItem *pi, const char *chars, bool ascending)
	{
		return QString((ascending) ? QChar(chars[0]) : QChar(chars[1])) + pi->text(0);
	}

	class dirItem : public QListViewItem
	{
	public:
		dirItem (QListViewItem *pp, QListViewItem *pa)
			: QListViewItem(pp, pa) { }

		dirItem (QListView *pw, QListViewItem *pa)
			: QListViewItem(pw, pa) { }

		virtual QString key (int col, bool asc) const
		{
			return sortKey(this, "01", asc);
		}
	};

	class fileItem : public QListViewItem
	{
	public:
		fileItem (QListViewItem *pp, QListViewItem *pa)
			: QListViewItem(pp, pa) { }

		fileItem (QListView *pw, QListViewItem *pa)
			: QListViewItem(pw, pa) { }

		virtual QString key (int col, bool asc) const
		{
			return sortKey(this, "10", asc);
		}
	};
};

QListViewItem *ArsonFileListFileItem::createItem (ArsonListWnd *parentWnd,
	QListViewItem *parentItem, QListViewItem *pAfter)
{
	return arson::newListViewItem<arson::fileItem>(parentWnd, parentItem, pAfter);
}

QListViewItem *ArsonFileListDirItem::createItem (ArsonListWnd *parentWnd,
	QListViewItem *parentItem, QListViewItem *pAfter)
{
	QListViewItem *pi = arson::newListViewItem<arson::dirItem>(
		parentWnd, parentItem, pAfter);

	pi->setExpandable(true);
	return pi;
}

/*========================================================*/

QPixmap ArsonFileListItem::loadIcon (const char *name)
{
	return KGlobal::iconLoader()->loadIcon(name, KIcon::Small);
}

/*========================================================*/
/*	File item visiter
 *========================================================*/

bool ArsonFileListVisiter::visit (QListViewItem *pi, ArsonListItem *pl)
{
	ArsonFileListItem *li = (ArsonFileListFileItem *) pl;

	if (!li->isDir())
		return visitFile(pi, (ArsonFileListFileItem *) li);
	else
		return visitDir(pi, (ArsonFileListDirItem *) li);

	return true;
}

/*========================================================*/

bool ArsonFileTreeVisiter::beginBranch (QListViewItem *pi, ArsonListItem *pl)
{
	return visitDir(pi, (ArsonFileListDirItem *) pl);
}

bool ArsonFileTreeVisiter::visit (QListViewItem *pi, ArsonListItem *pl)
{
	return visitFile(pi, (ArsonFileListFileItem *) pl);
}

/*========================================================*/
