/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	Code to open an md5 file, and verify the
 *	contents using md5sum
 *
 *	by Tony Sideris	(07:56PM May 25, 2002)
 *================================================*/
#include "arson.h"

#include <klocale.h>

#include "processmgr.h"
#include "waitdlg.h"
#include "process.h"

/*========================================================*/

class ArsonMd5CheckMgr : public ArsonProcessMgr
{
public:
	ArsonMd5CheckMgr (ArsonProcessUI *pUI, const char *filename);

	virtual void begin (const ArsonProcessOpts &opts);
	virtual void taskComplete (ArsonProcess *ptr);

	bool successful (void) const { return m_success; }
	
private:
	QCString m_filename;
	bool m_success;
};

/*========================================================*/

class ArsonMd5WaitDlg : public ArsonWaitDlg
{
public:
	ArsonMd5WaitDlg (QWidget *parent, const char *filename);
	virtual ~ArsonMd5WaitDlg (void);

	bool successful (void) const;
	void dismiss (void);

private:
	virtual void reject (void);
	
	ArsonMd5CheckMgr *m_pMgr;

	class WaitDlgUI : public ArsonProcessUI
	{
	public:
		WaitDlgUI (ArsonMd5WaitDlg *ptr)
			: ArsonProcessUI(), m_dlg(ptr) { }

		virtual void output (const QString &str, bool error) { }
		virtual void setText (const QString &text);
		virtual void begin (void) { }
		virtual void end (void);

	private:
		ArsonMd5WaitDlg *m_dlg;
	};

	ArsonProcessUI *createUI (void) { return new WaitDlgUI(this); }
};

/*========================================================*/
/*	The waitdlg the user sees while the md5 file
 *	is checked.
 *========================================================*/

ArsonMd5WaitDlg::ArsonMd5WaitDlg (QWidget *parent, const char *filename)
	: ArsonWaitDlg(parent)
{
	setMessage(
		i18n("Verifying MD5 checksum..."));

	setCaption(
		i18n("MD5 Check"));

	m_pMgr = new ArsonMd5CheckMgr(createUI(), filename);
	m_pMgr->begin(ArsonProcessOpts());
}

ArsonMd5WaitDlg::~ArsonMd5WaitDlg (void)
{
	delete m_pMgr;
}

/*========================================================*/

void ArsonMd5WaitDlg::reject (void)
{
	m_pMgr->abort();
	ArsonWaitDlg::reject();
}

/*========================================================*/

bool ArsonMd5WaitDlg::successful (void) const
{
	return (m_pMgr && m_pMgr->successful());
}

/*========================================================*/

void ArsonMd5WaitDlg::dismiss (void)
{
	ArsonWaitDlg::dismiss(successful());
}

/*========================================================*/
/*	The processUI interface impl for the wait dlg.
 *========================================================*/

void ArsonMd5WaitDlg::WaitDlgUI::setText (const QString &text)
{
	m_dlg->setMessage(text);
}

void ArsonMd5WaitDlg::WaitDlgUI::end (void)
{
	m_dlg->dismiss();
}

/*========================================================*/
/*	The md5 check process manager
 *========================================================*/

ArsonMd5CheckMgr::ArsonMd5CheckMgr (ArsonProcessUI *pUI, const char *filename)
	: ArsonProcessMgr(pUI), m_filename(filename), m_success(false)
{
	//	Nothing...
}

/*========================================================*/

void ArsonMd5CheckMgr::begin (const ArsonProcessOpts &opts)
{
	ArsonProcessMgr::begin(opts);

	try {
		setProcess(
			new ArsonMd5sumProcess(this, m_filename));
	}
	catch (ArsonError &err) {
		err.report();
	}
}

/*========================================================*/

void ArsonMd5CheckMgr::taskComplete (ArsonProcess *ptr)
{
	if (!(m_success = ptr->successful()))
	{
		ArsonMd5sumProcess *pm = (ArsonMd5sumProcess *) ptr;
		const QString errs = pm->errors().join(" ");

		arsonErrorMsg(
			i18n("MD5 verification failed, errors:\n\n ") + errs);
	}
	
	ArsonProcessMgr::taskComplete(ptr);
}

/*========================================================*/
/*	Big fat interface function for it all.
 *========================================================*/

bool arsonMd5Check (const char *filename)
{
	ArsonMd5WaitDlg dlg (kapp->mainWidget(), filename);

	return (dlg.exec() == QDialog::Accepted);
}

/*========================================================*/
