/*
 * Copyright (c) 2001 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#ifndef __WIZARD_H__
#define __WIZARD_H__

#include <qstringlist.h>
#include <qdialog.h>
#include <qmap.h>

#include <klocale.h>

/*========================================================*/
/**
 *	A simple wizard page containing a label,
 *	path edit field, and a browse button.
 *	Override selectPath() to display a file
 *	dialog and return the selcted path.
 */

class QLineEdit;
class QLabel;

class ArsonWizardPathPage : public QWidget
{
	Q_OBJECT
	
public:
	ArsonWizardPathPage (const QString &label,
		QWidget *parent, const char *name);

	virtual QString selectPath (void) = 0;
	virtual bool isValid (const QString &text) const = 0;

	void setMessage (const QString &msg);

	QString path (void) const;

private:
	QLineEdit *m_pEdit;
	QLabel *m_pMsg;
	
private slots:
	void slotBrowse (void);
	void slotTextChange (const QString &text);

signals:
	void validate (uint btns);
};

/*========================================================*/

class ArsonFileFilter;

class ArsonWizardDirPage : public ArsonWizardPathPage
{
public:
	ArsonWizardDirPage (QWidget *parent, const char *name);

private:
	virtual QString selectPath (void);
	virtual bool isValid (const QString &text) const;
};

class ArsonWizardFilePage : public ArsonWizardPathPage
{
public:
	ArsonWizardFilePage (QWidget *parent, const char *name);

private:
	virtual QString selectPath (void);
	virtual bool isValid (const QString &text) const;
	virtual void fileNameFilter (ArsonFileFilter &filter) const;
};

/*========================================================*/
/**
 *	Adds pages to a ArsonWizard. For each program
 *	job (burn, copy, rip, etc.) a class should be
 *	derived from this class, and ONE global/static
 *	instance created. The constructor will add the
 *	instance to a static name->instance mapping
 *	which allows the wizard to be created by name,
 *	and to show up in the main wizard.
 */

class ArsonWizard;

class ArsonWizardFactory
{
	typedef QMap<QCString,ArsonWizardFactory*> MANAGERS;

public:
	enum ActionType {
		PrimaryAction,
		SecondaryAction,
		ToolAction,
	};

	ArsonWizardFactory (const char *name, const char *desc, ActionType type);

	/**
	 *	Override for each action (burn, copy, rip, etc.)
	 *	and add necessary pages to @p pw. This function
	 *	is also responsible for connect()ing the wizard's
	 *	pageChange() signal to the page's SLOT.
	 */
	virtual void addPages (ArsonWizard *pw) = 0;

	QString description (void) const { return i18n(m_desc); }
	ActionType type (void) const { return m_type; }

	/**
	 *	Retrives the global/static factory instance by name, for example:
	 *	ArsonWizardFactory::byName("cdcopy");
	 *	(in practice all names should be symbolic constants,
	 *	ARSON_WIZARD_CDCOPY, instead of "cdcopy" in this
	 *	example.
	 */
	static ArsonWizardFactory *byName (const char *name);
	static void freeAll (void) { delete allFactories; }

private:
	static MANAGERS *allFactories;

	const char *m_desc;
	ActionType m_type;
};

/*========================================================*/
/**
 *	The wizard dialog class. I don't like [KQ]Wizard
 *	i wrote this one instead. It's alot more
 *	customizable.
 */

class ArsonProgress;
class QWidgetStack;
class QVBoxLayout;
class QHBoxLayout;
class QPushButton;
class QLabel;

class ArsonWizard : public QDialog
{
	Q_OBJECT

public:
	ArsonWizard (QWidget *parent, const char *name = NULL, bool modal = true);

	void addPage (ArsonProgress *ptr, const QString &text);
	void addPage (QWidget *ptr, const QString &text);
	void addPage (ArsonWizardPathPage *ptr, const QString &text);

	/**
	 *	Sets the factory object for this wizard,
	 *	pages added by ArsonWizardFactory::addPages()
	 *	will be APPENDED to this wizard.
	 */
	void setFactory (ArsonWizardFactory *pMgr);

	/**
	 *	Show a wizard by name, @p name is the
	 *	name of a ArsonWizardFactory instance.
	 */
	static void showWizard (const char *name);

	/**
	 *	Set the decorative static widget displayed
	 *	in the right side of the dialog, can add more
	 *	than one, each addition will be inserted on
	 *	the right side.
	 */
	void setDecorationWidget (QWidget *widget);

	/**
	 *	Sets the title widget displayed at the
	 *	top of the dialog. Should handle the
	 *	titleChange() slot to properly change
	 *	the text as needed.
	 */
	void setTitleWidget (QWidget *title);

	int index (QWidget *page) const;
	QWidget *page (int page) const;

	void setCurrentPage (QWidget *page);
	void setCurrentPage (int page);

	int currentPage (void) const { return m_page; }
	int pageCount (void) const { return m_count; }

	enum Buttons {
		Finish = 0x01,
		Cancel = 0x02,
		Back = 0x04,
		Next = 0x08,
		All = (Finish | Cancel | Back | Next)
	};

private:
	QWidgetStack *m_pages;			///	The wizard pages
	QWidget *m_decor;				///	Decoration widget on left side
	QPushButton *m_finish;			///	Finish button
	QPushButton *m_cancel;			///	Cancel button
	QPushButton *m_back;			///	Back button
	QPushButton *m_next;			///	Next button
	QVBoxLayout *m_pRightLayout;	///	Layout for right side (content)
	QHBoxLayout *m_pCenterLayout;	///	Center layout
	QWidget *m_title;				///	Title label
	QStringList m_titles;			///	Page titles
	int m_count;					///	The number of pages
	int m_page;						///	The current page index

	void createButtons (void);
	
public slots:
	void enableButtons (uint buttons);
	void enableButtons (void);
	void next (void);
	void back (void);

signals:
	/**
	 *	Emitted when the page is changed, was will be
	 *	the previously active page, now will be the
	 *	currently active page.
	 */
	void pageChange (QWidget *was, QWidget *now);

	/**
	 *	Emitted whenever the page changes, and
	 *	hence the title.
	 */
	void titleChange (const QString &title);
};

/*========================================================*/
#endif	/*	__WIZARD_H__	*/
