#ifndef asdlibh
#define asdlibh

#include <glib.h>

#include <protocol-asd-spec.h>
#include <asderrno.h>

struct _AsdConnection;
typedef struct _AsdConnection AsdConnection;

typedef enum { ASD_CONNECTION_CONTROL, ASD_CONNECTION_STREAM, ASD_CONNECTION_DIRECT, ASD_CONNECTION_SAMPLE } AsdConnectionState;

// Creates a new connection to a speaker. speaker maybe NULL. The library than will try to connect to 
// the speaker specified in the ASDSPEAKER environment variable. If it does not exist, first an UNIX, than an
// INET connection will be tried with the default server adresses.
// Returns a pointer to an AsdConnection object or NULL on failure.
// The object will be in ASD_CONNECTION_CONTROL mode
AsdConnection* asd_connection_new(gchar *speaker);

// Frees an AsdConnection object and closes the connection
void asd_connection_free(AsdConnection *c);

// Returns the file handle of an AsdConnection object
// Use this to get a file handle to write audio data to after putting the connection in ASD_CONNECTION_STREAM mode
int asd_connection_fd(AsdConnection *c);

// Tells us whether the connection has already been authenticated
gboolean asd_connection_authenticated(AsdConnection *c);

// Tells us the mode of the given AsdConnection
AsdConnectionState asd_connection_state(AsdConnection *c);

// Tries to authenticate the connection
// Note: all commands below try to switch to authenticated mode automatically when needed
// So normally not needed
gboolean asd_authenticate(AsdConnection *c);

// Writes up to l bytes of the server version string to s
// c must be in ASD_CONNECTION_CONTROL mode before call, and will be after
gboolean asd_server_version(AsdConnection *c, gchar *s, guint l);

// Reads general server info from the server. 
// c must be in ASD_CONNECTION_CONTROL mode before call, and will be after
gboolean asd_server_info(AsdConnection *c, ProtocolAsdServerInfoResponse *info);

// The desired stream action: play to a given sink as source, capture from a given source as sink,
// monitor a given sink as sink, be a generic unlinked source, be a generic unlinked sink.
// You may use asd_link_source_sink() to create a conjunction between a source and a link thereafter.
// Or asd_link_sink_sink() to create a conjunction between a sink and another sink (monitoring).
// c must be in ASD_CONNECTION_CONTROL mode before call, and will be in ASD_CONNECTION_STREAM mode after
typedef enum { ASD_STREAM_PLAY, ASD_STREAM_CAPTURE, ASD_STREAM_MONITOR, ASD_STREAM_SOURCE, ASD_STREAM_SINK } AsdStreamAction;

// Tries to switch c into ASD_CONNECTION_STREAM mode
// action:         specifies the desired operation
// name:           client name (NULL for appname)
// device:         the device (source/sink) to connect to. Only needed for ASD_STREAM_PLAY,
//                 ASD_STREAM_CAPTURE and ASD_STREAM_MONITOR. NULL for default.
//                 ("play0" for PLAY and MONITOR, "capt0" for CAPTURE)
// sample_type:    the desired sample type, NULL for 44khz, 16b, Stereo, LE, Signed
// volume:         the desired initial volume. NULL for maximum.
// immediate_stop: TRUE, when asd shall not wait until all queued data is played, when a source dies
// q_len:          for CAPTURE, MONITOR, PLAY specifies queue length in blocks
//                 Maybe zero for default length
// q_hold:         specifies hold value of block queue. That is how many blocks to wait, before pop succeeds
//                 May be zero for default hold value
// shortname:      When not NULL, asdlib will write the shortname of the new created device to this buffer
// shortname_size: Buffer length for shortname
// block_size:     block size in bytes the client should use. Converted internal block size of asd
gboolean asd_stream(AsdConnection *c, 
                    AsdStreamAction action, 
                    gchar *name, 
                    gchar *device, 
                    SampleType *sample_type, 
                    Volume *volume, 
                    gboolean immediate_stop, 
                    guint8 q_len, 
                    guint8 q_hold, 
                    gchar* shortname, 
                    guint shortname_size, 
                    guint32 *block_size);


// Some convenience functions. Wrappers for asd_stream with default parameters
gboolean asd_stream_simple_play(AsdConnection *c, SampleType *sample_type);
gboolean asd_stream_simple_capture(AsdConnection *c, SampleType *sample_type);
gboolean asd_stream_simple_monitor(AsdConnection *c, SampleType *sample_type);

// Locks or unlocks the server
// c must be in CONTROL mode, and will be after
gboolean asd_lock(AsdConnection *c, gboolean b);

// Sets the volume of a sink or source called shortname
// c must be in CONTROL mode, and will be after
gboolean asd_volume_set(AsdConnection *c, gchar *shortname, Volume v);

// Gets the volume of a sink or source called shortname
// c must be in CONTROL mode, and will be after
gboolean asd_volume_get(AsdConnection *c, gchar *shortname, Volume *v);

// Gets info about a source called shortname
// c must be in CONTROL mode, and will be after
gboolean asd_info_source(AsdConnection *c, gchar *shortname, ProtocolAsdInfoSourceResponse *info);

// Gets info about a sink called shortname
// c must be in CONTROL mode, and will be after
gboolean asd_info_sink(AsdConnection *c, gchar *shortname, ProtocolAsdInfoSinkResponse *info);

// Function prototype for following callback-functions
// For every sink/source enumerated, a function of this type will be called with a terse info (names, type) 
// of the device. Look into protocol-asd-spec.h for a description of this structure
typedef void (*AsdListProc) (ProtocolAsdListResponse* response, gpointer userdata);

// Enumerates all sources or sinks. Calls a user defined procedure with each entry.
// You may call asd_info_XXX() from these procedures and the same connection object.
// The caller may specify a userdata pointer, which is used on every callback-proc call
// c must be in CONTROL mode, and will be after.
gboolean asd_list(AsdConnection *c, AsdListProc proc, gpointer userdata, gboolean source);

// Two convenience macros calling the above function
#define asd_list_sources(c,proc,userdata) asd_list(c,proc,userdata,TRUE)
#define asd_list_sinks(c,proc,userdata) asd_list(c,proc,userdata,FALSE)

/* // Link functions */
/* gboolean asd_link_source_sink(AsdConnection *c, gchar* source, gchar *sink, guint q_len, guint q_hold); */
/* gboolean asd_link_sink_sink(AsdConnection *c, gchar* sink1, gchar *sink2, guint q_len, guint q_hold); */

/* gboolean asd_unlink_source_sink(AsdConnection *c, gchar* source, gchar *sink); */
/* gboolean asd_unlink_sink_sink(AsdConnection *c, gchar* sink1, gchar *sink2); */

/* gboolean asd_unlink_source(AsdConnection *c, gchar* source); */
/* gboolean asd_unlink_sink(AsdConnection *c, gchar* sink); */

#endif
