#include <pthread.h>
#include <signal.h>
#include <sys/types.h>
#include <unistd.h>
#include <string.h>
#include <getopt.h>
#include <stdlib.h>

#include <asdutil.h>
#include <asdlib.h>

typedef struct 
{
  int from;
  int to;
  gboolean filter;
  guint block_size;
  pthread_t t;
} Stream; 

gboolean quit = FALSE;

gboolean play, capture, monitor, filter;

SampleType sample_type;

gboolean st_sign = TRUE;
gboolean st_be = FALSE;
guint st_rate = 44100;
guint st_channels = 2;
guint st_bits = 16;
guint queue_length = 0;
guint queue_hold = 0;
gboolean st_mono = FALSE;
gboolean st_stereo = FALSE;
gboolean immediate_stop = FALSE;

Volume volume1, volume2;

pid_t main_pid;

gchar in_dev[ASD_SHORTNAME_LENGTH] = "", out_dev[ASD_SHORTNAME_LENGTH] = "";

gchar speaker[120] = "";
gchar name[ASD_NAME_LENGTH] = "";

static void* _thread(void *p)
{
  sigset_t ss;
  Stream *s;
  guint8 *data;

  g_assert(s = (Stream*) p);

  pthread_sigmask(SIG_BLOCK, NULL, &ss);
  sigaddset(&ss, SIGINT);
  sigaddset(&ss, SIGQUIT);
  sigaddset(&ss, SIGHUP);
  pthread_sigmask(SIG_BLOCK, &ss, NULL);

  pthread_setcanceltype(PTHREAD_CANCEL_DEFERRED, NULL);
  pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, NULL);

  g_assert(data = g_malloc(s->block_size));
  pthread_cleanup_push(g_free, data);

  while (!quit)
    {
      ssize_t l;
      
      if ((l = read(s->from, data, s->block_size)) <= 0)
	break;

      if (atomic_write(s->to, data, l) != l)
	break;
      
      if (s->filter)
	if (atomic_write(1, data, l) != l)
	  break;
    }

  pthread_cleanup_pop(1);

  kill(main_pid, SIGINT);
  return NULL;
}

void help()
{
  g_print("Usage: %s ... [sink [source]]\n\n", g_get_prgname());
  g_print("Options:\n");
  g_print(" -h, --help                Show this help\n"
          " -S, --signed              Signed data (default)\n"
          " -U, --unsigned            Unsigned data\n"
          " -B, --be                  Big endian data\n"
          " -L, --le                  Little endian data (default)\n"
          " -2, --stereo              Stereo data (default)\n"
          " -1, --mono                Mono data\n"
          " -b, --bits=BITS           Set bit depth to BITS\n"
          " -8                        Set bit depth to eight\n"
          " -c, --channels=CHANNELS   Set channels to CHANNELS (Default: 2, stereo)\n"
          " -r, --rate=RATE           Set sampling rate to RATE (Default: 44100)\n"
          " -s, --server=SERVERSPEC   Set server to SERVERSPEC\n"
          " -f, --filter              Enable copy of STDIN to STDOUT\n"
          " -o, --capture[=SOURCE]    Capture data from SOURCE, capt0 if ommited\n"
          " -m, --monitor[=SINK]      Capture data from SINK, play0 if ommited\n"
          " -i, --play[=SINK]         Play data to SINK, play0 if ommited\n"
          " -u, --usage               Show terse usage info\n"
	  " -I, --immediate           Enable immediate stop\n"
          " -Q, --queue-length=LEN    Set queue length to LEN blocks\n"
          " -H, --queue-hold=LEN      Wait for LEN blocks in queue before starting playback\n"
          " -n, --name=NAME           Sets the stream name to NAME (Default: %s)\n"
          " -v, --volume=VOLSPEC      Set volume of playback to VOLSPEC\n"
          " -w, --volume2=VOLSPEC     Set volume of capture to VOLSPEC\n\n", g_get_prgname());
  
  
  g_print("VOLSPEC has the following format: V1,V2,V3,V4,...,Vn\n"
          "Vi is the volume for the i-th channel from 0 to 65535.\n"
          "The last specified volume Vn will be used for all unspecified channels.\n\n");
  
  g_print("Example:\n"
          " 0,32000,65535    will mute channel #1 and channel #2 will have half the\n"
          "                  volume of channel #3 and further.\n"
          " 0                will mute all channels.\n"
          " 65535            will maximize all channels.\n"
          " -1               will also maximize all channels using a dirty trick.\n\n");

  g_print("SERVERSPEC has the following format: PROTOCOL:SERVER\n"
          "PROTOCOL maybe 'unix:' for unix-socket, and 'inet:' for tcp-socket.\n"
          "SERVER specifies a unix-socket filename for unix-protocol, maybe ommitted for default.\n"
          "SERVER specifies a hostname for inet-protocol, maybe ommited for 'localhost'.\n"
          "You may add a port number to SERVER for inet-protocol by adding ':PORT' to it.\n\n");
  
  g_print("Example:\n"
          " unix:            connect via unix socket.\n"
          " unix:/tmp/gurke  connect via unix socket /tmp/gurke.\n"
          " inet:tomate:4711 connect via inet socket to host tomate on port 4711.\n"
          " inet:banane      connect via inet socket to host banane.\n"
          " inet:            connect via inet socket to localhost.\n\n");

  g_print("%s tries to evaluate $ASDSPEAKER, when no --server is specified.\n", g_get_prgname());

  exit(0);
}

void  usage()
{
  g_print("Usage: %s [-h] [-b BITS] [-c CHANNELS] [-r RATE] [-s SERVERSPEC] [-f] [-o SOURCE] [-m SINK] [-i SINK] "
          "[-u] [-I] [-Q] [-H] [-v VOLSPEC] [-w VOLSPEC] [-n NAME] [sink [source]]\n", g_get_prgname());
  exit(0);
}


void parse_args(int argc, char *argv[])
{
  static struct option long_options[] =
  {
    {"signed",   no_argument, &st_sign, TRUE},
    {"unsigned", no_argument, &st_sign, FALSE},
    {"be",       no_argument, &st_be,   TRUE},
    {"le",       no_argument, &st_be,   FALSE},
    {"mono",     no_argument, &st_mono, TRUE},
    {"stereo",   no_argument, &st_stereo, TRUE},
    {"bits",     required_argument, NULL, 'b'},
    {"channels", required_argument, NULL, 'c'},
    {"rate",     required_argument, NULL, 'r'},
    {"server",   required_argument, NULL, 's'},
    {"filter",   no_argument, &filter, TRUE },
    {"capture",  optional_argument, NULL, 'o'},
    {"monitor",  optional_argument, NULL, 'm'},
    {"play",     optional_argument, NULL, 'i'},
    {"help",     no_argument,       NULL, 'h'},
    {"usage",    no_argument,       NULL, 'u'},
    {"queue-length", required_argument, NULL, 'Q'},
    {"queue-hold", required_argument, NULL, 'H'},
    {"volume",   required_argument, NULL, 'v'},
    {"volume2",  required_argument, NULL, 'w'},
    {"immediate",no_argument,       NULL, 'I'},
    {"name",     required_argument, NULL, 'n'},
    {NULL, 0, NULL, 0}
  };

  gchar *b, c;
  int long_index;

  volume_max(&volume1);
  volume_max(&volume2);

  b = g_basename(g_get_prgname());

  if (strcmp(b, "asdmon") == 0)
    {
      filter = play = !(capture = monitor = TRUE);
      strncpy(out_dev, "play0", sizeof(out_dev));
    }
  else if (strcmp(b, "asdrec") == 0)
    {
      filter = monitor = play = !(capture = TRUE);
      strncpy(out_dev, "capt0", sizeof(out_dev));
    }
  else if (strcmp(b, "asdfilter") == 0)
    {
      monitor = capture = !(filter = play = TRUE);
      strncpy(in_dev, "play0", sizeof(in_dev));
    }
  else if (strcmp(b, "asdduplex") == 0)
    {
      filter = monitor = !(play = capture = TRUE);
      strncpy(in_dev, "play0", sizeof(in_dev));
      strncpy(out_dev, "capt0", sizeof(out_dev));
    }
  else if (strcmp(b, "asdcat") == 0)
    {
      filter = monitor = capture = !(play = TRUE);
      strncpy(in_dev, "play0", sizeof(in_dev));
    }
  else
    filter = monitor = capture = play = FALSE;

  while ((c = getopt_long(argc, argv, "12SUBLb:8c:r:s:fo::m::i::huv:w:n:", long_options, &long_index)) >= 0)
    {
      switch (c) 
        {
        case '1' : 
          st_mono = TRUE;
          break;
        case '2' :
          st_stereo = TRUE;
          break;
        case 'S' :
        case 'U' :
          st_sign = c == 'S';
          break;
        case 'B' :
        case 'L' :
          st_be = c == 'B';
          break;
        case 'b' :
          g_assert(optarg);
          st_bits = atoi(optarg);
          break;
        case '8' :
          st_bits = 8;
          break;
        case 'c' :
          g_assert(optarg);
          st_channels = atoi(optarg);
          break;
        case 'r' :
          g_assert(optarg);
          st_rate = atoi(optarg);
          break;
        case 's' :
          g_assert(optarg);
          strncpy(speaker, optarg, sizeof(speaker));
          break;
        case 'f' :
          filter = TRUE;
          break;
        case 'o' :
          monitor = !(capture = TRUE);
          if (optarg)
            strncpy(out_dev, optarg, sizeof(out_dev));
          break;
        case 'm' :
          capture = monitor = TRUE;
          if (optarg)
            strncpy(out_dev, optarg, sizeof(out_dev));
          break;
        case 'i' :
          play = TRUE;
          if (optarg)
            strncpy(in_dev, optarg, sizeof(in_dev));
          break;
        case 'h' :
          help();
        case '?' :
        case 'u' :
          usage();
	case 'I' :
	  immediate_stop = TRUE;
	  break;
        case 'Q' :
          g_assert(optarg);
          queue_length = atoi(optarg);
          break;
        case 'H' :
          g_assert(optarg);
          queue_hold = atoi(optarg);
          break;
        case 'v' :
          g_assert(optarg);
          volume_parse(&volume1, optarg);
          break;
        case 'w' :
          g_assert(optarg);
          volume_parse(&volume2, optarg);
          break;
        case 'n' :
          g_assert(optarg);
          strncpy(name, optarg, sizeof(name));
          break;
        default:
          g_error("Not yet implemented!");
        }
    }

  if (argc-optind > 2)
    {
      g_printerr("Too many arguments.\n");
      exit(1);
    }

  if (argc-optind >= 1)
    {
      strncpy(in_dev, argv[optind], sizeof(in_dev));
      strncpy(out_dev, argv[argc-optind >= 2 ? 2 : 1], sizeof(out_dev));
    }
  
  if (!capture && !play)
    {
      g_printerr("Need something to do.\n");
      exit(1);
    }

  if (capture && filter)
    {
      g_printerr("Cannot capture/monitor and filter at the same time.\n");
      exit(1);
    }

  if (st_stereo)
    st_channels = 2;
  if (st_mono)
    st_channels = 1;

  sample_type.channels = st_channels;
  sample_type.bits = st_bits;
  sample_type.sign = st_sign;
  sample_type.be = st_be;
  sample_type.rate = st_rate;
  
  if (!sample_type_valid(&sample_type))
    {
      g_printerr("Specified sample type not valid.\n");
      exit(1);
    }

}

void sighandler(int sig)
{
  quit = TRUE;
  signal(sig, sighandler);
}

int main(int argc , char *argv[])
{
  AsdConnection *play_connection = NULL, *capture_connection = NULL;
 
  g_set_prgname(g_basename(argv[0]));

  parse_args(argc, argv);
 
  signal(SIGPIPE, sighandler); siginterrupt(SIGPIPE, 1);
  signal(SIGHUP,  sighandler); siginterrupt(SIGHUP,  1);
  signal(SIGINT,  sighandler); siginterrupt(SIGINT,  1);
  signal(SIGQUIT, sighandler); siginterrupt(SIGQUIT, 1);
  signal(SIGTERM, sighandler); siginterrupt(SIGTERM, 1);

  g_thread_init(NULL);

  main_pid = getpid();

  if (play)
    play_connection = asd_connection_new(speaker[0] ? speaker : NULL);

  if (capture)
    capture_connection = asd_connection_new(speaker[0] ? speaker : NULL);

  if ((play && !play_connection) || (capture && !capture_connection))
    asd_perror("Could not create capturing connection");
  else
    {
      Stream play_stream, capture_stream;
      gboolean ok = TRUE;

      if (play)
	{
	  play_stream.from = 0;
	  play_stream.to = asd_connection_fd(play_connection);
	  play_stream.filter = filter;

	  if (!asd_stream(play_connection, ASD_STREAM_PLAY, name[0] ? name : NULL, in_dev[0] ? in_dev : NULL, &sample_type, &volume1, immediate_stop, queue_length, queue_hold, NULL, 0, &play_stream.block_size))
	    {
	      asd_perror("Could not switch to play stream mode");
	      ok = FALSE;
	    }
	}
      
      if (ok && capture)
	{
	  capture_stream.from = asd_connection_fd(capture_connection);
	  capture_stream.to = 1;
	  capture_stream.filter = FALSE;

	  if (!monitor)
	    {
              if (!asd_stream(capture_connection, ASD_STREAM_CAPTURE, name[0] ? name : NULL, out_dev[0] ? out_dev : NULL, &sample_type, &volume2, FALSE, queue_length, queue_hold, NULL, 0, &capture_stream.block_size))
		{
		  asd_perror("Could not switch to capture stream mode");
		  ok = FALSE;
		}
	    }
	  else
	    {
              if (!asd_stream(capture_connection, ASD_STREAM_MONITOR, name[0] ? name : NULL, out_dev[0] ? out_dev : NULL, &sample_type, &volume2, FALSE, queue_length, queue_hold, NULL, 0, &capture_stream.block_size))
		{
		  asd_perror("Could not switch to monitor stream mode");
		  ok = FALSE;
		}
	    }
	}

      if (ok)
	{
	  if (play)
	    pthread_create(&play_stream.t, NULL, _thread, (void*) &play_stream);

	  if (capture)
	    pthread_create(&capture_stream.t, NULL, _thread, (void*) &capture_stream);

	  select(FD_SETSIZE, NULL, NULL, NULL, NULL);

	  if (play)
	    {
	      pthread_cancel(play_stream.t);
	      pthread_join(play_stream.t, NULL);
	    }
	  
	  if (capture)
	    {
	      pthread_cancel(capture_stream.t);
	      pthread_join(capture_stream.t, NULL);
	    }
	}
    }

  if (play_connection)
    asd_connection_free(play_connection);
  
  if (capture_connection)
    asd_connection_free(capture_connection);

  return 0;
}
