------------------------------------------------------------------------------
--                                                                          --
--                      GNAT METRICS TOOLS COMPONENTS                       --
--                                                                          --
--                      M E T R I C S . O P T I O N S                       --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                       Copyright (C) 2008, AdaCore                        --
--                                                                          --
-- GNAT Metrics Toolset  is free software;  you can  redistribute it and/or --
-- modify it under terms of the  GNU General Public License as published by --
-- the Free Software Foundation;  either version 2, or (at your option) any --
-- later version.  GNAT Metrics Toolset is  distributed in the hope that it --
-- will be useful, but  WITHOUT ANY WARRANTY; without even the implied war- --
-- ranty of  MERCHANTABILITY or  FITNESS FOR A PARTICULAR PURPOSE.  See the --
-- GNU General Public License for more details.  You should have received a --
-- copy of the  GNU General Public License distributed with  GNAT; see file --
-- COPYING.  If not,  write to the  Free Software  Foundation,  51 Franklin --
-- Street, Fifth Floor, Boston, MA 02110-1301, USA.                         --
--                                                                          --
-- GNAT Metrics Toolset is maintained by AdaCore (http://www.adacore.com).  --
--                                                                          --
------------------------------------------------------------------------------

pragma Ada_05;

with Ada.Characters.Handling;          use Ada.Characters.Handling;
with Ada.Containers.Ordered_Sets;
with Ada.Containers.Vectors;

with GNAT.HTable;

with Asis.Compilation_Units;           use Asis.Compilation_Units;
with Asis.Compilation_Units.Relations; use Asis.Compilation_Units.Relations;
with Asis.Declarations;                use Asis.Declarations;
with Asis.Elements;                    use Asis.Elements;
with Asis.Extensions.Flat_Kinds;       use Asis.Extensions.Flat_Kinds;
with Asis.Iterator;                    use Asis.Iterator;
with Asis.Text;                        use Asis.Text;

with ASIS_UL.Metrics.Definitions;      use ASIS_UL.Metrics.Definitions;
with ASIS_UL.Misc;                     use ASIS_UL.Misc;
with ASIS_UL.Strings;                  use ASIS_UL.Strings;

with METRICS.Common;                   use METRICS.Common;
with METRICS.Options;                  use METRICS.Options;
with METRICS.Output;                   use METRICS.Output;

package body METRICS.Coupling is

   -----------------------------------------------
   -- Data structures for coupling depentencies --
   -----------------------------------------------

   type Unit_Id is new Integer range 0 .. Integer'Last;
   --  Index of a unit in the unit Table. We need IDs to organize dependency
   --  lists.

   No_Unit    : constant Unit_Id := Unit_Id'First;
   First_Unit : constant Unit_Id := No_Unit + 1;

   function Present (CU_Id : Unit_Id) return Boolean;
   function No      (CU_Id : Unit_Id) return Boolean;
   --  Check if CU_Id represents an existing entry in the unit table.

   subtype Existing_Unit_Id is Unit_Id range First_Unit .. Unit_Id'Last;

   package Dependency_Lists is new Ada.Containers.Ordered_Sets
     (Element_Type => Unit_Id);
   --  Lists to represent unit dependencies.

   type Unit_Record is record
      SF          : SF_Id;
      Unit_Name   : String_Loc;
      Unit_Span   : Span;
      Dependents  : Dependency_Lists.Set;
      Descendants : Dependency_Lists.Set;
      Supporters  : Dependency_Lists.Set;

      Hash_Link   : Unit_Id;
   end record;
   --  Comments are needed!!!!

   package Unit_Containers is new Ada.Containers.Vectors
      (Index_Type   => Existing_Unit_Id,
       Element_Type => Unit_Record);

   Unit_Table : Unit_Containers.Vector;

   ----------------
   -- Hash table --
   ----------------

   Hash_Num : constant Integer := 2**8;
   --  Number of headers in the hash table. There is no special reason in this
   --  choice.

   Hash_Max : constant Integer := Hash_Num - 1;
   --  Indexes in the hash header table run from 0 to Hash_Num - 1

   subtype Hash_Index_Type is Integer range 0 .. Hash_Max;
   --  Range of hash index values

   Hash_Table : array (Hash_Index_Type) of Unit_Id := (others => No_Unit);
   --  The hash table is used to locate existing entries in the unit table.
   --  The entries point to the first nodes table entry whose hash value
   --  matches the hash code. Then subsequent nodes table entries with the
   --  same hash code value are linked through the Hash_Link fields.

   function Hash is new GNAT.HTable.Hash (Header_Num => Hash_Index_Type);
   function Hash (CU : Asis.Compilation_Unit) return Hash_Index_Type;

   procedure Print_Unit (U : Unit_Id);
   --  Prints out the debug info of the unit.

   procedure Print_List (List : Dependency_Lists.Set);
   --  Prints the debug image of the argument node list

   -----------------------------------------------
   -- Access and update routines for unit table --
   -----------------------------------------------

   type Unit_Record_Access is access Unit_Record;

   function Table (CU_Id : Unit_Id) return Unit_Record_Access;
   --  Mimics the notation Instantce_Name.Table (N) in the instantiation of the
   --  GNAT Table package. Returns the (pointer to the) unit with the index
   --  CU_Id from Unit_Table (see the body of the package). Raises
   --  Constraint_Error if a node with this index does not exsist.

   function Unit_SF (CU_Id : Unit_Id) return SF_Id;
   --  Returns source file table entry for the argumeny unit. Returns No_SF_Id
   --  in case if No (CU_Id).

   function Unit_Name (CU_Id : Unit_Id) return String_Loc;
   --  Returns string table entry for the argumeny unit. Returns Nil_String_Loc
   --  in case if No (CU_Id).

   function Unit_Hash_Link (CU_Id : Unit_Id) return Unit_Id;
   --  Returns the hash link for the argumeny unit. Returns No_Unit in case
   --  if No (CU_Id).

   procedure Set_Hash_Link (CU_Id : Unit_Id; Val : Unit_Id);
   --  Sets the hash link of the unit CU_Id to Val. It is erroneous to call it
   --  for No (CU_Id).

   procedure Add_Descendant (To : Unit_Id; Descendant : Unit_Id);
   procedure Add_Dependent  (To : Unit_Id; Dependent  : Unit_Id);
   procedure Add_Supporter  (To : Unit_Id; Supporter  : Unit_Id);

   function Find_CU (CU : Asis.Compilation_Unit) return Unit_Id;
   --  Tries to locate in the unit table the entry corresponding to the
   --  argument unit. Returns No_Unit if there is no such entry.

   function Is_Equal
     (CU_Id : Unit_Id;
      CU    : Asis.Compilation_Unit)
      return  Boolean;
   --  Checks if CU_Id represents CU. CU_Id should not be No_Unit. This
   --  function assumes that Has_Coupling_Metrics (CU) is true.

   -----------------------
   -- Local subprograms --
   -----------------------

   procedure Check_For_OO_Type
     (Element :        Asis.Element;
      Control : in out Traverse_Control;
      State   : in out Boolean);
   --  Used as Pre_Operation. Checks if its argument is a declaration (or
   --  a definition) of a tagged or interface type. If it is, sets State to
   --  True and terminates the traversal. Assumes that the top element of the
   --  traversal is a (generic) package declaration.

   procedure Do_Nothing
     (Element :        Asis.Element;
      Control : in out Traverse_Control;
      State   : in out Boolean);
   --  Used as a Post_Operation, does nothing.

   procedure Look_For_OO_Type is new Traverse_Element
     (State_Information => Boolean,
      Pre_Operation     => Check_For_OO_Type,
      Post_Operation    => Do_Nothing);
   --  Checks if the argument Element contains a declaration of OO type.

   function Collect_Coupling_Dependencies
     (CU   : Asis.Compilation_Unit;
      SF   : SF_Id)
      return Unit_Id;
   --  Does the same as the procedure Collect_Coupling_Dependencies and returns
   --  the ID of the unit as the result.

   function Corresponding_Unit
     (CU   : Asis.Compilation_Unit)
      return Asis.Compilation_Unit;
   --  If the argument unit is a body or a subunit, returns the corresponding
   --  spec unit. Otherwise returns the argument Unit

   Ident_String : constant String := "   ";
   --  Used in the debug output of the unit table

   -----------------------
   -- Check_For_OO_Type --
   -----------------------

   procedure Check_For_OO_Type
     (Element :        Asis.Element;
      Control : in out Traverse_Control;
      State   : in out Boolean)
   is
   begin

      case Flat_Element_Kind (Element) is

         when An_Ordinary_Type_Declaration  |
              A_Private_Type_Declaration    |
              A_Package_Declaration         |
              A_Generic_Package_Declaration =>
            --  We have to look into the construct
            null;
         when A_Tagged_Incomplete_Type_Declaration              |
              A_Private_Extension_Declaration                   |
              A_Tagged_Record_Type_Definition                   |
              A_Tagged_Private_Type_Definition                  |
              A_Derived_Record_Extension_Definition             |
              An_Ordinary_Interface .. A_Synchronized_Interface =>
            --  We've found OO type, so:
            State := True;
            Control := Terminate_Immediately;

         when A_Package_Instantiation =>
            Look_For_OO_Type
              (Corresponding_Declaration (Element), Control, State);
         when others =>
            --  Definitely nothing interesting inside
            Control := Abandon_Children;
      end case;

   end Check_For_OO_Type;

   -----------------------------------
   -- Collect_Coupling_Dependencies --
   -----------------------------------

   function Collect_Coupling_Dependencies
     (CU   : Asis.Compilation_Unit;
      SF   : SF_Id)
      return Unit_Id
   is
      Package_CU : constant Asis.Compilation_Unit := Corresponding_Unit (CU);
      Result     :          Unit_Id               := Find_CU (Package_CU);
      New_Unit   :          Boolean               := False;

      Next_Parent    : Asis.Compilation_Unit;
      Next_Supporter : Unit_Id;

   begin

      if No (Result) then
         --  We have to create the corresponding entry in the unit table:
         declare
            New_Unit         : Unit_Record;
            Hash_Value       : constant Hash_Index_Type := Hash (Package_CU);
            Last_In_Chain    :          Unit_Id := Hash_Table (Hash_Value);

         begin
            New_Unit.SF   := SF;
            New_Unit.Unit_Span := Element_Span (Unit_Declaration (CU));
            New_Unit.Unit_Name :=
              Enter_String (To_String (Unit_Full_Name (CU)));
            New_Unit.Hash_Link := No_Unit;

            Unit_Containers.Append
              (Container => Unit_Table,
               New_Item  => New_Unit);

            Result := Unit_Containers.Last_Index (Unit_Table);

            if No (Last_In_Chain) then
               Hash_Table (Hash_Value) := Result;
            else

               while Present (Unit_Hash_Link (Last_In_Chain)) loop
                  Last_In_Chain := Unit_Hash_Link (Last_In_Chain);
               end loop;

               Set_Hash_Link
                 (CU_Id => Last_In_Chain,
                  Val   => Result);
            end if;

         end;

         New_Unit := True;

      end if;

      if New_Unit
       or else
         not Is_Equal (CU, Package_CU)
      then
         --  We have to create or to update the list of supporters and to
         --  update the dependents lists for new supporters accordingly

         declare

            Supporters : constant Asis.Compilation_Unit_List :=
               Semantic_Dependence_Order
                 (Compilation_Units => (1 => CU),
                  Dependent_Units   => Nil_Compilation_Unit_List,
                  The_Context       => Common.The_Context,
                  Relation          => Asis.Supporters).Consistent;

            Next_Supporter    : Unit_Id;
            Next_Supporter_SF : SF_Id;

         begin

            for J in Supporters'Range loop

               Next_Supporter := Find_CU (Supporters (J));

               if No (Next_Supporter) then
                  --  May be this is a unit of interest that has not been
                  --  processed yet.

                  if Has_Coupling_Metrics (Supporters (J)) then
                     Next_Supporter_SF :=
                       File_Find (To_String (Text_Name (Supporters (J))));
                  else
                     Next_Supporter_SF := No_SF_Id;
                  end if;

                  if Present (Next_Supporter_SF) then
                     Next_Supporter :=
                       Collect_Coupling_Dependencies
                         (Supporters (J), Next_Supporter_SF);
                  end if;

               end if;

               if Present (Next_Supporter)
                 and then
                  Next_Supporter /= Result
               then
                  Add_Dependent (To => Next_Supporter, Dependent => Result);
                  Add_Supporter (To => Result, Supporter => Next_Supporter);

               end if;

            end loop;

         end;

      end if;

      if New_Unit then
         --  we have to update  descendants relations

         Next_Parent := Corresponding_Parent_Declaration (CU);

         if not Is_Nil (Next_Parent) then

            while not Is_Nil
                        (Corresponding_Parent_Declaration (Next_Parent))
            loop
               --  Loop stops when Next_Parent is package Standard
               Next_Supporter := Find_CU (Next_Parent);

               if Present (Next_Supporter) then
                  Add_Descendant
                    (To => Next_Supporter, Descendant => Result);
               end if;

               Next_Parent :=
                 Corresponding_Parent_Declaration (Next_Parent);

            end loop;

         end if;

      end if;

      return Result;
   end Collect_Coupling_Dependencies;

   procedure Collect_Coupling_Dependencies
     (CU : Asis.Compilation_Unit;
      SF : SF_Id)
   is
      Result : Unit_Id := Collect_Coupling_Dependencies (CU, SF);
      pragma Unreferenced (Result);
   begin
      null;
   end Collect_Coupling_Dependencies;

   ------------------------------
   -- Compute_Coupling_Metrics --
   ------------------------------

   procedure Compute_Coupling_Metrics is
   begin
      --  Do we really have anything to compute?
      null;
   end Compute_Coupling_Metrics;

   ------------------------
   -- Corresponding_Unit --
   ------------------------

   function Corresponding_Unit
     (CU   : Asis.Compilation_Unit)
      return Asis.Compilation_Unit
   is
      Result : Asis.Compilation_Unit := CU;
   begin

      case Unit_Kind (CU) is
         when A_Package_Body =>
            Result := Corresponding_Declaration (CU);
         when A_Subunit =>
            Result :=
              Corresponding_Unit (Corresponding_Subunit_Parent_Body (Result));
         when others =>
            null;
      end case;

      return Result;
   end Corresponding_Unit;

   ----------------
   -- Do_Nothing --
   ----------------

   procedure Do_Nothing
     (Element :        Asis.Element;
      Control : in out Traverse_Control;
      State   : in out Boolean)
   is
   begin
      null;
   end Do_Nothing;

   -------------
   -- Find_CU --
   -------------

   function Find_CU (CU : Asis.Compilation_Unit) return Unit_Id is
      Result : Unit_Id;
   begin

      Result := Hash_Table (Hash (CU));

      while Present (Result) loop

         if Is_Equal (Result, CU) then
            exit;
         end if;

         Result := Table (Result).Hash_Link;
      end loop;

      return Result;
   end Find_CU;

   --------------------------
   -- Has_Coupling_Metrics --
   --------------------------

   function Has_Coupling_Metrics
     (CU : Asis.Compilation_Unit)
      return Boolean
   is
      Arg_Unit : constant Asis.Compilation_Unit := Corresponding_Unit (CU);

      Unit    : Asis.Element;
      Result  : Boolean := False;
      Control : Traverse_Control := Continue;
   begin

      case Unit_Kind (Arg_Unit) is
         when A_Package          |
              A_Generic_Package  |
              A_Package_Instance =>

            Unit := Unit_Declaration (Arg_Unit);

            if Unit_Kind (Arg_Unit) = A_Package_Instance then
               Unit := Corresponding_Declaration (Unit);
            end if;

            Look_For_OO_Type (Unit, Control, Result);

         when others =>
            null;
      end case;

      return Result;
   end Has_Coupling_Metrics;

   ----------
   -- Hash --
   ----------

   function Hash (CU : Asis.Compilation_Unit) return Hash_Index_Type is
   begin
      return Hash (To_String (Unit_Full_Name (CU)));
   end Hash;

   --------------
   -- Is_Equal --
   --------------

   function Is_Equal
     (CU_Id : Unit_Id;
      CU    : Asis.Compilation_Unit)
      return  Boolean
   is
      Result : Boolean := False;
   begin

      if Present (CU_Id)
        and then
         not Is_Nil (CU)
      then
         Result :=
           Get_String (Unit_Name (CU_Id)) = To_String (Unit_Full_Name (CU));
      end if;

      return Result;
   end Is_Equal;

   --------
   -- No --
   --------

   function No (CU_Id : Unit_Id) return Boolean is
   begin
      return CU_Id not in
        First_Unit .. Unit_Containers.Last_Index (Unit_Table);
   end No;

   -------------
   -- Present --
   -------------

   function Present (CU_Id : Unit_Id) return Boolean is
   begin
      return CU_Id in First_Unit .. Unit_Containers.Last_Index (Unit_Table);
   end Present;

   ----------------
   -- Print_List --
   ----------------

   procedure Print_List (List : Dependency_Lists.Set) is
      use Dependency_Lists;
      Next_El : Cursor := First (List);
   begin

      if Next_El = No_Element then
         Info_No_EOL (" ...nothing...");
      else

         while Next_El /= No_Element loop
            Info_No_EOL (Dependency_Lists.Element (Next_El)'Img);
            Next_El := Next (Next_El);
         end loop;

         Info_No_EOL (" (" & Length (List)'Img & ")");

      end if;

      Info ("");

   end Print_List;

   ----------------
   -- Print_Unit --
   ----------------

   procedure Print_Unit (U : Unit_Id) is
   begin
      Info ("Unit_Id =" & U'Img);

      Info_No_EOL (Ident_String);
      Info ("Source file - " &  Source_Name (Unit_SF (U)));

      Info_No_EOL (Ident_String);
      Info ("Unit name   - " &  Get_String (Unit_Name (U)));

      Info_No_EOL (Ident_String);
      Info ("Hash link   - " &  Unit_Hash_Link (U)'Img);

      Info_No_EOL (Ident_String & "Descendants :");
      Print_List (Table (U).Descendants);

      Info_No_EOL (Ident_String & "Supporters  :");
      Print_List (Table (U).Supporters);

      Info_No_EOL (Ident_String & "Dependents  :");
      Print_List (Table (U).Dependents);

   end Print_Unit;

   ----------------------
   -- Print_Unit_Table --
   ----------------------

   procedure Print_Unit_Table is
   begin

      for J in First_Unit .. Unit_Containers.Last_Index (Unit_Table) loop
         Print_Unit (J);
         Info ("");
      end loop;

   end Print_Unit_Table;

   -----------------------------
   -- Report_Coupling_Metrics --
   -----------------------------

   procedure Report_Coupling_Metrics is
      Crown : Dependency_Lists.Set;
      --  Contains terminal (leaf) units belonging to the category

      Category_Supporters : Dependency_Lists.Set;
      Category_Dependents : Dependency_Lists.Set;
      --  External dependencies of the category (a unit from Category set
      --  cannot belong to any of these lists.

      use Dependency_Lists;
      Next_El : Cursor;
   begin

      Report ("");
      Report ("Coupling metrics:");
      Report ("=================");

      Open_Tag ("coupling");

      for J in First_Unit .. Unit_Containers.Last_Index (Unit_Table) loop

         --  At the moment we use a rahther straightforward, but not very
         --  effective way of computing the category coupling.
         --
         --  For afferent coupling we collect all the units depending on
         --  a root unit and all its descendants, and then substract from
         --  the result the category itself (that is, the guven unit
         --  plus all its descendants).
         --
         --  For efferent coupling, we get a set of all the terminal
         --  units of the category. Because a descendant unit belongs on
         --  all the units all its ancestor units depends upon, we
         --  collect the dependencies of the terminal units only and then
         --  substract from the result the category itself.
         --
         --  The disadvantage of this approach is that the same things
         --  are computed many times...

         if Compute_Category_Efferent_Coupling then
            --  First, create the set of terminal nodes.

            Clear (Crown);
            Clear (Category_Supporters);

            if Is_Empty (Table (J).Descendants) then
               Insert (Crown, J);
            else
               Next_El := First (Table (J).Descendants);

               while Next_El /= No_Element loop

                  if Is_Empty
                    (Table (Dependency_Lists.Element (Next_El)).Descendants)
                  then
                     Insert (Crown, Dependency_Lists.Element (Next_El));
                  end if;

                  Next_El := Next (Next_El);
               end loop;

            end if;

            Next_El := First (Crown);

            while Next_El /= No_Element loop
               Union
                 (Category_Supporters,
                  Table (Dependency_Lists.Element (Next_El)).Supporters);

               Next_El := Next (Next_El);
            end loop;

            Difference (Category_Supporters, Table (J).Descendants);
            Exclude    (Category_Supporters, J);

         end if;

         if Compute_Category_Afferent_Coupling then
            Clear (Category_Dependents);
            Union (Category_Dependents, Table (J).Dependents);

            Next_El := First (Table (J).Descendants);

            while Next_El /= No_Element loop

               Union
                 (Category_Dependents,
                  Table (Dependency_Lists.Element (Next_El)).Dependents);

               Next_El := Next (Next_El);
            end loop;

            Difference (Category_Dependents, Table (J).Descendants);
            Exclude    (Category_Dependents, J);

         end if;

         Report ("Unit " & Get_String (Unit_Name (J)) &
                 " (" & Source_Name (Unit_SF (J)) & ")");

         Report_XML
           ("<file name=" & '"' &
            Source_Name_For_Output (Unit_SF (J))  & """>",
            Depth => 1);

         Report_XML
           ("<unit name=""" & Get_String (Unit_Name (J))                &
            """ line="""    & Image (Table (J).Unit_Span.First_Line)    &
            """ col="""     & Image (Table (J).Unit_Span.First_Column)  &
            """>",
            Depth => 1);

         if Compute_Package_Efferent_Coupling then
            Report ("   package efferent coupling  :" &
                    Length (Table (J).Supporters)'Img);

            Output_XML_Metric
              ("package_efferent_coupling",
               Metric_Count (Length (Table (J).Supporters)),
               Depth => 2);
         end if;

         if Compute_Category_Efferent_Coupling then
            Report ("   category efferent coupling :" &
                    Length (Category_Supporters)'Img);

            Output_XML_Metric
              ("category_efferent_coupling",
               Metric_Count (Dependency_Lists.Length (Category_Supporters)),
               Depth => 2);
         end if;

         if Compute_Package_Afferent_Coupling then
            Report ("   package afferent coupling  :" &
                    Length (Table (J).Dependents)'Img);

            Output_XML_Metric
              ("package_afferent_coupling",
               Metric_Count (Length (Table (J).Dependents)),
               Depth => 2);
         end if;

         if Compute_Category_Afferent_Coupling then
            Report ("   category afferent coupling :" &
                    Length (Category_Dependents)'Img);

            Output_XML_Metric
              ("category_afferent_coupling",
               Metric_Count (Length (Category_Dependents)),
               Depth => 2);
         end if;

         Report ("");

      end loop;

      Close_Tag ("coupling");

   end Report_Coupling_Metrics;

   -----------
   -- Table --
   -----------

   function Table (CU_Id : Unit_Id) return Unit_Record_Access is
      Result : Unit_Record_Access;

      procedure Process (E : in out Unit_Record);

      procedure Process (E : in out Unit_Record) is
      begin
         Result := E'Unrestricted_Access;
      end Process;
   begin

      Unit_Containers.Update_Element
        (Container => Unit_Table,
         Index     => CU_Id,
         Process   => Process'Access);

      return Result;

   end Table;

   --------------------
   -- Unit_Hash_Link --
   --------------------

   function Unit_Hash_Link (CU_Id : Unit_Id) return Unit_Id is
   begin

      if No (CU_Id) then
         return No_Unit;
      else
         return Table (CU_Id).Hash_Link;
      end if;

   end Unit_Hash_Link;

   ---------------
   -- Unit_Name --
   ---------------

   function Unit_Name (CU_Id : Unit_Id) return String_Loc is
   begin

      if No (CU_Id) then
         return Nil_String_Loc;
      else
         return Table (CU_Id).Unit_Name;
      end if;

   end Unit_Name;

   function Unit_SF (CU_Id : Unit_Id) return SF_Id is
   begin

      if No (CU_Id) then
         return No_SF_Id;
      else
         return Table (CU_Id).SF;
      end if;

   end Unit_SF;

   --------------------------------
   -- Unit table update routines --
   --------------------------------

   Unit_Id_Tmp : Unit_Id;

   procedure Add_Descendant (For_Unit_Rec : in out Unit_Record);
   procedure Add_Dependent  (For_Unit_Rec : in out Unit_Record);
   procedure Add_Supporter  (For_Unit_Rec : in out Unit_Record);
   procedure Set_Hash_Link  (For_Unit_Rec : in out Unit_Record);

   ----------------------------------------------------------

   procedure Add_Descendant (For_Unit_Rec : in out Unit_Record) is
      Tmp_Cursor  : Dependency_Lists.Cursor;
      Tmp_Boolean : Boolean;
      pragma Warnings (Off, Tmp_Cursor);
      pragma Warnings (Off, Tmp_Boolean);
   begin
      Dependency_Lists.Insert
          (Container => For_Unit_Rec.Descendants,
           New_Item  => Unit_Id_Tmp,
           Position  => Tmp_Cursor,
           Inserted  => Tmp_Boolean);
   end Add_Descendant;

   procedure Add_Dependent (For_Unit_Rec : in out Unit_Record) is
      Tmp_Cursor  : Dependency_Lists.Cursor;
      Tmp_Boolean : Boolean;
      pragma Warnings (Off, Tmp_Cursor);
      pragma Warnings (Off, Tmp_Boolean);
   begin
      Dependency_Lists.Insert
          (Container => For_Unit_Rec.Dependents,
           New_Item  => Unit_Id_Tmp,
           Position  => Tmp_Cursor,
           Inserted  => Tmp_Boolean);
   end Add_Dependent;

   procedure Add_Supporter (For_Unit_Rec : in out Unit_Record) is
      Tmp_Cursor  : Dependency_Lists.Cursor;
      Tmp_Boolean : Boolean;
      pragma Warnings (Off, Tmp_Cursor);
      pragma Warnings (Off, Tmp_Boolean);
   begin
      Dependency_Lists.Insert
          (Container => For_Unit_Rec.Supporters,
           New_Item  => Unit_Id_Tmp,
           Position  => Tmp_Cursor,
           Inserted  => Tmp_Boolean);
   end Add_Supporter;

   procedure Set_Hash_Link (For_Unit_Rec : in out Unit_Record) is
   begin
      For_Unit_Rec.Hash_Link := Unit_Id_Tmp;
   end Set_Hash_Link;

   ----------------------------------------------------------

   procedure Add_Descendant (To : Unit_Id; Descendant : Unit_Id) is
   begin
      Unit_Id_Tmp := Descendant;

      Unit_Containers.Update_Element
        (Container => Unit_Table,
         Index     => To,
         Process   => Add_Descendant'Access);
   end Add_Descendant;

   procedure Add_Dependent (To : Unit_Id; Dependent : Unit_Id) is
   begin
      Unit_Id_Tmp := Dependent;

      Unit_Containers.Update_Element
        (Container => Unit_Table,
         Index     => To,
         Process   => Add_Dependent'Access);
   end Add_Dependent;

   procedure Add_Supporter (To : Unit_Id; Supporter : Unit_Id) is
   begin
      Unit_Id_Tmp := Supporter;

      Unit_Containers.Update_Element
        (Container => Unit_Table,
         Index     => To,
         Process   => Add_Supporter'Access);
   end Add_Supporter;

   procedure Set_Hash_Link (CU_Id : Unit_Id; Val : Unit_Id) is
   begin
      Unit_Id_Tmp := Val;

      Unit_Containers.Update_Element
        (Container => Unit_Table,
         Index     => CU_Id,
         Process   => Set_Hash_Link'Access);
   end Set_Hash_Link;

end METRICS.Coupling;
