// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#include "Puma/CCInstantiation.h"
#include "Puma/CStructure.h"
#include "Puma/CClassInstance.h"
#include "Puma/CUnionInstance.h"
#include "Puma/CFctInstance.h"
#include "Puma/CNamespaceInfo.h"
#include "Puma/CTemplateInfo.h"
#include "Puma/CTemplateParamInfo.h"
#include "Puma/CTree.h"
#include "Puma/CCOverloading.h"
#include "Puma/CConstant.h"
#include "Puma/CCConversions.h"
#include "Puma/CUnit.h"
#include "Puma/CCParser.h"
#include "Puma/TokenStream.h"
#include "Puma/UnitManager.h"
#include "Puma/CSemDatabase.h"
#include "Puma/PreMacroManager.h"
#include "Puma/CTranslationUnit.h"
#include "Puma/PreprocessorParser.h"
#include "Puma/CFileInfo.h"
#include "Puma/CSourceInfo.h"
#include "Puma/CCConversions.h"

#include <stdio.h>      /* sprintf() */
#include <sstream>  /* ostringstream */
using namespace std;

namespace Puma {


/*DEBUG*/int TRACE_INSTANCE_CODE = 0;


#define SEM_ERROR(loc__,mesg__) \
  {if (report) {\
    err << sev_error << loc__->token ()->location () \
        << mesg__ << endMessage;}}


// maximal instantiation depth
const unsigned CCInstantiation::MAX_DEPTH = 50;

unsigned int CCInstantiation::m_ScopeNameCounter = 0;


CCInstantiation::CCInstantiation (ErrorSink &e, bool rep) : err (e) {
  instance      = 0;
  last_instance = 0;
  report        = rep;
  trans_unit    = 0;
  current_scope = 0;

  base_candidate = new InstantiationCandidate;
  candidates[0] = base_candidate;
  First ().initialize (rep ? &e : (ErrorSink*)0);
}


CCInstantiation::~CCInstantiation () {
  for (long i = 0; i < candidates.length (); i++) {
    InstantiationCandidate *candidate = candidates.lookup (i);
    if (candidate == base_candidate)
      base_candidate = 0;
    delete candidate;
  }
  if (base_candidate)
    delete base_candidate;
}


// start instantiation process
CObjectInfo *CCInstantiation::instantiate (CTree *node, 
 CObjectInfo *binfo, const CCOverloading &ovl) {
  // function call arguments
  unsigned args = ovl.Arguments ();
  for (unsigned i = 0; i < args; i++)
    First ().addArgument (ovl.Argument (i));

  return instantiate (node, binfo, true);
}


// start instantiation process
CObjectInfo *CCInstantiation::instantiate (CTree *node, 
 CObjectInfo *binfo, bool real_inst, CStructure *cs) {
  // don't call twice
  if (instance || ! binfo || ! node)
    return instance;

  First ().initialize (node, binfo);
  current_scope = cs;
  
// TEMPORARY HACK --->
  if (! ObjectInfo ()->SemDB ()->Project ()->config ().Option ("--real-instances")) 
    real_inst = false;
// TEMPORARY HACK <---

  // §14.8.2 deduce template arguments
  if (! First ().deduceArguments (real_inst))
    return instance;
  
  // choose the appropriate template specialization (if any)
  if (! chooseSpecialization ()) {
    return instance;
  }

  // instantiate template if not already done before
  if (! alreadyInstantiated (real_inst)) {
    // check maximal instantiation depth
    if (TemplateInfo ()->Depth () >= MAX_DEPTH) {
      err << sev_fatal << PointOfInstantiation ()->token ()->location () 
          << "maximal instantiation depth ("
          << MAX_DEPTH << ") reached" << endMessage;
      return instance;
    }

    // create an instance of the template
    if (real_inst)
      createInstance ();
    else
      createPseudoInstance ();
    
    // do not delete the deduced arguments
    if (instance) {
      First ().forgetDeducedArgs ();
      if (base_candidate != &First ())
        base_candidate->forgetDeducedArgs ();
    }
  }    
    
  return instance;
}


// §14.5.4.1 matching of template partial specializations
bool CCInstantiation::chooseSpecialization () {
  bool no_default;
  unsigned entries;
  CObjectInfo *info;
  CTemplateInfo *tinfo;
  CT_TemplateName *name;
  CT_TemplateArgList *args;

  // only class templates can be partially specialized
  if (ObjectInfo ()->FunctionInfo ()) 
    return true;

  // iterate specializations
  for (unsigned i = 0; i < TemplateInfo ()->Specializations (); i++) {
    tinfo = TemplateInfo ()->Specialization (i);
    if (! tinfo->ObjectInfo ()) // internal error
      continue;
    info = tinfo->ObjectInfo ()->DefObject ();
    name = tinfo->SpecializationName ();
    if (! name) // internal error
      continue;
    args = name->Arguments ();
    if (! args) // internal error
      continue;

    // create new instantiation candidate
    InstantiationCandidate *cand = new InstantiationCandidate;
    cand->initialize (PointOfInstantiation (), info, tinfo, report ? &err : (ErrorSink*)0);
    
    // add the template arguments of the partial specialization
    entries = args->Entries ();
    for (unsigned j = 0; j < entries; j++)
      cand->addArgument (args->Entry (j));
    // add default arguments of the base template if not overwritten
    no_default = false;
    for (unsigned j = entries; j < TemplateInfo ()->Parameters (); j++) {
      // no default argument? this is an error
      if (! TemplateInfo ()->Parameter (i)->DefaultArgument ()) {
        no_default = true;
        cand->reset ();
        delete cand;
        break;
      }
      cand->addArgument (TemplateInfo ()->Parameter (i)->DefaultArgument ());
    }
    if (no_default)
      continue;
    
    // §14.5.4.1.2 try to match the partial specialization 
    // against the actual template argument list
    if (! cand->match ()) {
      // does not match, not a candidate for instantiation
      cand->reset ();
      delete cand;
    } else {
      candidates[candidates.length ()] = cand;
    }
  }
  
  // if exactly one matching specialization is found, the 
  // instantiation is generated from that specialization
  if (candidates.length () == 2) {
    // remove the base template as candidate
    candidates.remove (0); 
  // if more than one specialization is matching, the
  // best specialization is chosen using the partial 
  // ordering rules (§14.5.4.2 and §14.5.5.2)
  } else if (candidates.length () > 2) {
    return chooseBestSpecialization ();
  }
  
  return true;
}


bool CCInstantiation::chooseBestSpecialization () {
  // run a tournament to choose the best specialization 
  // comparing two specializations using the partial 
  // ordering rules (§14.5.4.2 and §14.5.5.2)
  InstantiationCandidate *challenger, *champion;
  unsigned num;

  num = candidates.length ();
  champion = candidates.lookup (num-1);
  for (unsigned i = num-1; i > 1; i--) {
    // current challenger
    challenger = candidates.lookup (i-1); 
    
    // let the champion face the challenger 
    switch (champion->compare (*challenger)) {
      // champion has won the round 
      case 1: 
        // elliminate challenger
        delete candidates.lookup (i-1); 
        candidates.remove (i-1); 
        break;
      // challenger has won the round
      case -1:
        // elliminate old champion
        delete candidates.lookup (i); 
        candidates.remove (i); 
        // challenger is the new champion
        champion = challenger; 
        break;
      // challenger and champion are equal
      default:
        // case 0: ambiguous up till now
        champion = challenger; 
        break;
    }
  }
  
  // verify that the champion is better than all other remaining 
  // candidates (the champion did not yet faced)
  for (unsigned i = candidates.length (); i > 1; i--) {
    challenger = candidates.lookup (i-1); 
    if (challenger == champion)
      continue;
      
    if (champion->compare (*challenger) == 1) {
      delete candidates.lookup (i-1); 
      candidates.remove (i-1);
    }
  }
  
  // if exactly one specialization left over, the 
  // instantiation is generated from that specialization
  if (candidates.length () == 2) {
    // remove the base template as candidate
    candidates.remove (0); 
    return true;
  } 

  // more than one specialization left over,
  // so the instantiation is ambiguous 
  if (report) {
    std::ostringstream name;
    name << ObjectInfo ()->Name ();
    listArguments (*base_candidate, name);
    err << sev_error << PointOfInstantiation ()->token ()->location () 
        << "Instantiation of `" << name.str ().c_str () 
        << "' is ambiguous" << endMessage;
    for (int i = 1; i < candidates.length (); i++) {
      InstantiationCandidate *cand = candidates[i];
      if (cand->ObjectInfo ()->Tree () && cand->ObjectInfo ()->Tree ()->token ())
        err << cand->ObjectInfo ()->Tree ()->token ()->location ();
      if (i == 1)
        err << "candidates are: ";
      else
        err << "                ";
      err << name.str ().c_str () << endMessage;
    }
  }

  return false;
}


bool CCInstantiation::alreadyInstantiated (bool real_inst) {
  CObjectInfo *ti;
  bool cont;

  last_instance = 0;
  for (unsigned i = 0; i < TemplateInfo ()->Instances (); i++) {
    ti = TemplateInfo ()->Instance (i);
    cont = false;
// TEMPORARY HACK --->
    if ((TemplateInfo ()->isBaseTemplate () == 
         ti->TemplateInstance ()->Template ()->isBaseTemplate ()) && 
        (DeducedArgs () == ti->TemplateInstance ()->DeducedArgs ())) {
      for (unsigned j = 0; j < ti->TemplateInstance ()->DeducedArgs (); j++)
        if (! (*DeducedArg (j) == *ti->TemplateInstance ()->DeducedArg (j)))
          cont = true;
    } else
      cont = true;
// TEMPORARY HACK <---
    if (! cont) {
      instance = ti;
      if ((ti->FunctionInfo () && ti->FunctionInfo ()->isFctDef ()) ||
          (ti->Record () && ti->Record ()->isDefined ())) {
        last_instance = 0;
        return true;
      } else
        last_instance = ti;
    }
  }
  
  instance = 0;
  if (last_instance) { 
    if (real_inst &&
        ((ObjectInfo ()->FunctionInfo () && ObjectInfo ()->FunctionInfo ()->isFctDef ()) ||
         (ObjectInfo ()->Record () && ObjectInfo ()->Record ()->isDefined ())))
      return false;
    instance = last_instance;
    last_instance = 0;
    return true;    
  } 
  return false;
}


void CCInstantiation::createPseudoInstance () {
  CTemplateInstance *ti = 0;
  if (ObjectInfo ()->FunctionInfo ()) {
    instance = makeScope ()->newFunction (true);
    ti = instance->TemplateInstance ();
  } else if (ObjectInfo ()->UnionInfo ()) {
    instance = makeScope ()->newUnion (true);
    ti = instance->TemplateInstance ();
  } else if (ObjectInfo ()->ClassInfo ()) {
    instance = makeScope ()->newClass (true);
    ti = instance->TemplateInstance ();
  } else if (ObjectInfo ()->TemplateParamInfo ()) {
    instance = ObjectInfo ()->TemplateParamInfo ()->TemplateTemplate ()->newTemplateParam (false);
    instance->TemplateParamInfo ()->TemplateInstance (new CTemplateInstance);
    instance->TemplateParamInfo ()->isTypeParam (ObjectInfo ()->TemplateParamInfo ()->isTypeParam ());
    instance->TemplateParamInfo ()->ValueType (ObjectInfo ()->TemplateParamInfo ()->ValueType ()->Duplicate ());
    instance->TemplateParamInfo ()->TemplateTemplate (ObjectInfo ()->TemplateParamInfo ()->TemplateTemplate ());
    ObjectInfo ()->NextObject (instance);
    ti = instance->TemplateParamInfo ()->TemplateInstance ();
  } else {
    return; // unsupported object type
  }
  instance->Name (ObjectInfo ()->Name ());  
  instance->SemDB (ObjectInfo ()->SemDB ());
  instance->SourceInfo ()->FileInfo (ObjectInfo ()->SourceInfo ()->FileInfo ());
  instance->SourceInfo ()->StartToken (PointOfInstantiation ()->token_node ());
  instance->TypeInfo (ObjectInfo ()->TypeInfo ()->Duplicate ());
  ti->PointOfInstantiation (getSimpleName (), current_scope);
  ti->Template (TemplateInfo ());
  ti->isPseudoInstance (true);
  if (instance->FunctionInfo ())
    instance->TypeInfo ()->TypeFunction ()->FunctionInfo (instance->FunctionInfo ());
  else if (instance->TemplateParamInfo ())
    instance->TypeInfo ()->TypeTemplateParam ()->TemplateParamInfo (instance->TemplateParamInfo ());
  else
    instance->TypeInfo ()->TypeRecord ()->Record (instance->Record ());
  for (unsigned i = 0; i < DeducedArgs (); i++)
    ti->addDeducedArg (DeducedArg (i));
  for (unsigned i = 0; i < base_candidate->DeducedArgs (); i++)
    ti->addInstantiationArg (base_candidate->DeducedArg (i));
  if (! instance->TemplateParamInfo ()) {
    TemplateInfo ()->addPseudoInstance (instance);
  }
}


void CCInstantiation::createInstance () {
  CTranslationUnit *tu;
  CStructure *scope;
  CProject *project;
  bool fct_inst;
  CUnit *unit;

  // increase instantiation depth
  TemplateInfo ()->increaseDepth ();

  scope = makeScope ();
  fct_inst = (ObjectInfo ()->FunctionInfo ());
  project = ObjectInfo ()->SemDB ()->Project ();
  unit = new CUnit (err);
  unit->name (scope->Name ());

  // create code for the template instance
  fillUnit (*unit);

  // setup new parser for the instantiated code
  trans_unit = tu = new CTranslationUnit (*unit, *project);
  CCParser p;
  p.syntax ().configure (project->config ());
  p.semantic ().configure (project->config ());
  // setup the preprocessor
  TokenStream stream;
  stream.push (unit); 
  PreprocessorParser cpp (&project->err (), 
    &project->unitManager (), &tu->local_units ());
  cpp.macroManager ()->init (unit->name ());
  cpp.stream (&stream);
  cpp.configure (project->config (), false); // do not process --include option
  // initialize semantic analyzer
  p.semantic ().init (*ObjectInfo ()->SemDB (),
    *ObjectInfo ()->SourceInfo ()->FileInfo ()->Primary (), 
    scope, fct_inst, ! fct_inst, this);   
  ((ErrorCollector&)p.builder ().err ()).index (0);
  p.semantic ().error_sink (p.builder ().err ());
  TokenProvider provider (cpp);
  // start parsing 
  tu->tree (p.syntax ().run (provider)); 
  tu->cpp_tree (cpp.syntaxTree ());

  // decrease instantiation depth
  TemplateInfo ()->decreaseDepth ();

  // report errors and clean up
  if (((ErrorCollector&)p.builder ().err ()).severity () > sev_warning) {
    removeInstance ();
    if (report) {
      err << PointOfInstantiation ()->token ()->location () 
          << "In instantiation of `" << scope->Name ().c_str ()+1 
          << "':" << endMessage;
      p.builder ().errors (err);
    }
    //delete trans_unit;
    //trans_unit = 0;
    //delete unit;
  // no errors detected
  } else {
    if (instance) {
      //instance->TemplateInstance ()->canDelete ();
    } else {
      //delete trans_unit;
      //trans_unit = 0;
      //delete unit;
    }
  }
}


void CCInstantiation::insertInstance (CObjectInfo *info) {
  if (! info || ! info->TemplateInstance ())
    return;

  instance = info;
  for (unsigned i = 0; i < DeducedArgs (); i++)
    instance->TemplateInstance ()->addDeducedArg (DeducedArg (i));
  for (unsigned i = 0; i < base_candidate->DeducedArgs (); i++)
    instance->TemplateInstance ()->addInstantiationArg (base_candidate->DeducedArg (i));
  if (last_instance && last_instance != instance)
    instance->NextObject (last_instance);
  instance->TemplateInstance ()->PointOfInstantiation (getSimpleName (),
    current_scope);
  instance->TemplateInstance ()->Template (TemplateInfo ());
  instance->TemplateInstance ()->TranslationUnit (trans_unit);
  TemplateInfo ()->addInstance (instance);
}    


void CCInstantiation::removeInstance () {
  if (! instance)
    return;

  TemplateInfo ()->removeInstance (instance);
  instance->TemplateInstance ()->clearDeducedArgs ();
  instance->TemplateInstance ()->PointOfInstantiation (0, 0);
  instance->TemplateInstance ()->TranslationUnit (0);
  instance = 0;
}    


void CCInstantiation::fillUnit (CUnit &unit) {
  DeducedArgument *arg;
  CTemplateParamInfo *param;
  CTree *default_arg;
  CTypeInfo *type;
  std::ostringstream psname;
  
  psname << "__puma_" << m_ScopeNameCounter;
  m_ScopeNameCounter++;
  for (unsigned i = 0; i < DeducedArgs (); i++) {
    arg = DeducedArg (i);
    param = arg->TemplateParam ();
    // unnamed and template template parameters are special, 
    // nothing needs to be generated
    if (! param || param->isTemplate () || ! param->Name () || 
        *param->Name () == '%')
      continue;
      
    // deduced type argument
    if (param->isTypeParam ()) {
      // default template argument
      if (arg->isDefaultArg ()) {
        unit << "typedef ";
        default_arg = arg->TemplateArg ();
        genArgType (unit, default_arg, findPrivateName (default_arg), param->Name ());
        unit << ";\n" << endu;
      } else if (arg->Type ()) {
        unit << "namespace " << psname.str () << " {\n";
        unit << "typedef ";
        arg->Type ()->TypeText (unit, param->Name (), true);
        unit << ";\n}\n" << "using " << psname.str () << "::" 
             << param->Name () << ";\n" << endu;
      }
    // deduced non-type argument
    } else {
      unit << "static ";
      type = param->ValueType ();
      if (! type->isConst ()) {
        // turn it into a const type
        type = new CTypeQualified (type->Duplicate (), true, false, false);
        type->TypeText (unit, param->Name (), true);
        delete type;
      } else {
        type->TypeText (unit, param->Name (), true);
      }
      // default template argument
      if (arg->isDefaultArg ()) {
        unit << " = ";
        copyTree (unit, arg->TemplateArg ());
      } else if (arg->Value ()) {
        unit << " = " << *(arg->Value ());
      }
      unit << ";\n" << endu;
    }
  }
  
  // check for a template parameter in the class name -> desturbs
  Array<CTree*> desturbing;
  if (ObjectInfo ()->Tree ()->NodeName () == CT_ClassDef::NodeId ()) {
    CT_ClassDef *clsdef = (CT_ClassDef*)ObjectInfo ()->Tree ();
    if (clsdef->Name ()->NodeName () == CT_TemplateName::NodeId ())
      desturbing.append (((CT_TemplateName*)clsdef->Name ())->Arguments ());
    CT_MembList *members = clsdef->Members ();
    if (members) { // normally true, but in case of parse errors (?) ...
      for (int i = 0; i < members->Entries (); i++) {
	if (members->Entry (i)->NodeName () == CT_FctDef::NodeId ()) {
	  CT_FctDef *fctdef = (CT_FctDef*)members->Entry (i);
	  CT_SimpleName *nm =((CT_Declarator*)fctdef->Declarator ())->Name ();
	  if (nm->NodeName () == CT_TemplateName::NodeId ())
	    desturbing.append (((CT_TemplateName*)nm)->Arguments ());
	}
      }
    }
  }
  
  // copy the object declaration without the template header
  if (TemplateInfo ()->Tree ()) {
    copyTree (unit, TemplateInfo ()->Tree ()->Declaration (), false, &desturbing);
  }
  
  unit << "\n" << endu;

  if (TRACE_INSTANCE_CODE) {
    cout << "\ninstantiation unit: " << unit.name () << endl;
    cout << "-------------------" << endl;
    unit.print (cout);
    cout << "\n-------------------\n" << endl;
  }
}


void CCInstantiation::genArgType (CUnit &unit, CTree *tree, CTree *pname, const char *name) {
  // replace private name with name of the current template parameter
  if (tree == pname) {
    unit << name << " ";
  // replace template template parameter by its deduced value
  } else if (tree->NodeName () == CT_SimpleName::NodeId ()) {
    replaceName (unit, (CT_SimpleName*)tree, true);
    return;
  // copy the rest
  } else if (tree->NodeName () == CT_Token::NodeId ()) {
    if (tree->token ())
      unit << tree->token ()->text () << " ";
  } 
  
  for (unsigned i = 0; i < (unsigned)tree->Sons (); i++) {
    genArgType (unit, tree->Son (i), pname, name);
  }
}


void CCInstantiation::copyTree (CUnit &unit, CTree *tree, bool gen_spaces, 
 Array<CTree*> *desturbing) {
  // first check if this subtree is desturbing and should 
  // be omitted in the generated unit
  if (desturbing) 
    for (int i = 0; i < desturbing->length (); i++)
      if (desturbing->lookup (i) == tree)
        return; // prune here
      
  // replace template template parameter by its deduced value
  if (tree->NodeName () == CT_SimpleName::NodeId ()) {
    replaceName (unit, (CT_SimpleName*)tree, gen_spaces);
    return;
  // copy the rest
  } else if (tree->NodeName () == CT_Token::NodeId ()) {
    if (tree->token ()) {
      unit << tree->token ()->text () << " ";
    }
  }
  
  for (unsigned i = 0; i < (unsigned)tree->Sons (); i++)
    copyTree (unit, tree->Son (i), gen_spaces, desturbing);
    
  // function bodies might be error nodes, because they were skipped
  // ==> replace by ';'
  if (tree->NodeName () == CT_FctDef::NodeId ()) {
    if (((CT_FctDef*)tree)->Body ()->NodeName () == CT_Error::NodeId ()) {
      unit << ";";
    }
  }
}


void CCInstantiation::replaceName (CUnit &unit, CT_SimpleName *name, bool gen_spaces) {
  bool replaced = false;
  
  CObjectInfo *info = name->Object ();
  if (info) {
    // is template template parameter?
    CTemplateParamInfo *pinfo = info->TemplateParamInfo ();
    if (pinfo && pinfo->isTemplate ()) {
      // which template parameter is it?
      int pos = First ().getPosition (pinfo);
      if (pos != -1) {
        // get deduced argument for this parameter and insert it here
        DeducedArgument *darg = DeducedArg (pos);
        if (darg->TemplateArg ()) {
          copyTree (unit, darg->TemplateArg (), gen_spaces);
          replaced = true;
        } else {
          replaced = dumpDeducedArg (darg, unit);
        }
      }
    }
  }
  // not replaced, so copy it as it is
  if (! replaced) {
    copyTree (unit, name->token_node (), gen_spaces);
  }
}


CStructure *CCInstantiation::makeScope () {
  CStructure *scope;
  
  // create namespace name
  std::ostringstream sname;
  sname << "%" << ObjectInfo ()->Name ();
  listArguments (*base_candidate, sname);

//  if (current_scope)
//    scope = current_scope->newNamespace ();    
//  else
  scope = TemplateInfo ()->Parent ()->newNamespace ();    
  scope->NamespaceInfo ()->aroundInstantiation (true);
  scope->Name (sname.str ().c_str ());
  scope->TypeInfo (&CTYPE_UNDEFINED);
  scope->SemDB (ObjectInfo ()->SemDB ());
  scope->SourceInfo ()->FileInfo (ObjectInfo ()->SourceInfo ()->FileInfo ());
  scope->SourceInfo ()->StartToken (PointOfInstantiation ()->token_node ());
  return scope;
}


void CCInstantiation::listArguments (InstantiationCandidate &cand, std::ostringstream &out) {
  out << "<";
  for (unsigned i = 0; i < cand.DeducedArgs (); i++) {
    DeducedArgument *arg = cand.DeducedArg (i);
    // do not list default arguments
    if (arg->isDefaultArg ())
      break;

    if (i) out << ",";
    dumpDeducedArg (arg, out);
  }
  out << ">";
}


bool CCInstantiation::dumpDeducedArg (DeducedArgument *arg, std::ostringstream &out) {
  if (arg->Type ()) {
    arg->Type ()->TypeText (out);
  } else if (arg->Value ()) {
    out << *(arg->Value ());
  } else {
    return false;
  }
  return true;
}


CT_SimpleName *CCInstantiation::getSimpleName () {
  if (PointOfInstantiation ()->NodeName () == CT_SimpleName::NodeId () ||
      PointOfInstantiation ()->NodeName () == CT_QualName::NodeId () ||
      PointOfInstantiation ()->NodeName () == CT_RootQualName::NodeId () ||
      PointOfInstantiation ()->NodeName () == CT_TemplateName::NodeId () ||
      PointOfInstantiation ()->NodeName () == CT_OperatorName::NodeId () ||
      PointOfInstantiation ()->NodeName () == CT_SpecialName::NodeId () ||
      PointOfInstantiation ()->NodeName () == CT_PrivateName::NodeId () ||
      PointOfInstantiation ()->NodeName () == CT_DestructorName::NodeId () ||
      PointOfInstantiation ()->NodeName () == CT_ConversionName::NodeId ())
    return (CT_SimpleName*)PointOfInstantiation ();
  return (CT_SimpleName*)0;
}


CT_SimpleName *CCInstantiation::findPrivateName (CTree *node) const {
  const char *id = node->NodeName ();
  if (id == CT_PrivateName::NodeId ())
    return (CT_SimpleName*)node;
  else if (id == CT_NamedType::NodeId ())
    return findPrivateName (((CT_NamedType*)node)->Declarator ());
  else if (id == CT_FctDeclarator::NodeId ())
    return findPrivateName (((CT_FctDeclarator*)node)->Declarator ());
  else if (id == CT_ArrayDeclarator::NodeId ())
    return findPrivateName (((CT_ArrayDeclarator*)node)->Declarator ());
  else if (id == CT_PtrDeclarator::NodeId ())
    return findPrivateName (((CT_PtrDeclarator*)node)->Declarator ());
  else if (id == CT_MembPtrDeclarator::NodeId ())
    return findPrivateName (((CT_MembPtrDeclarator*)node)->Declarator ());
  else if (id == CT_BracedDeclarator::NodeId ())
    return findPrivateName (((CT_BracedDeclarator*)node)->Declarator ());
  else if (id == CT_BitFieldDeclarator::NodeId ())
    return findPrivateName (((CT_BitFieldDeclarator*)node)->Declarator ());
  else if (id == CT_RefDeclarator::NodeId ())
    return findPrivateName (((CT_RefDeclarator*)node)->Declarator ());
  else if (id == CT_InitDeclarator::NodeId ())
    return findPrivateName (((CT_InitDeclarator*)node)->Declarator ());
  return (CT_SimpleName*)0;
}


} // namespace Puma
