/****************************************************************
**
** Attal : Lords of Doom
**
** questConditionEditor.cpp
** Editor of conditions used in quests
**
** Version : $Id: questConditionEditor.cpp,v 1.6 2004/06/29 19:58:40 audoux Exp $
**
** Author(s) : Pascal Audoux
**
** Date : 14/06/2004
**
** Licence :
**	This program is free software; you can redistribute it and/or modify
**   	it under the terms of the GNU General Public License as published by
**     	the Free Software Foundation; either version 2, or (at your option)
**      any later version.
**
**	This program is distributed in the hope that it will be useful,
** 	but WITHOUT ANY WARRANTY; without even the implied warranty of
**	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**	GNU General Public License for more details.
**
****************************************************************/

#include "questConditionEditor.h"

// generic include files
// include files for QT
#include <qlabel.h>
#include <qlayout.h>
#include <qlistbox.h>
#include <qpopupmenu.h>
#include <qpushbutton.h>
// application specific include files
#include "libCommon/log.h"
#include "libCommon/quest.h"

#include "libClient/gui.h"

#include "scenarioEditor/questConditionProperties.h"

QuestConditionEditor::QuestConditionEditor( QWidget * parent, const char * name )
: QWidget( parent, name )
{
	_condition = 0;

	_title = new QLabel( this );
	_title->setText( tr( "Condition: " ) );
	FIXEDSIZE( _title );

	QPushButton * pbEdit = new QPushButton( this );
	pbEdit->setText( tr( "Edit" ) );
	FIXEDSIZE( pbEdit );

	QHBoxLayout * layout = new QHBoxLayout( this );
	layout->addWidget( _title );
	layout->addSpacing( 5 );
	layout->addWidget( pbEdit );
	layout->addStretch( 1 );
	layout->activate();

	connect( pbEdit, SIGNAL( clicked() ), SLOT( slot_edit() ) );
}

void QuestConditionEditor::setTitle( const QString & title )
{
	_title->setText( title );
	FIXEDSIZE( _title );
}

void QuestConditionEditor::slot_edit()
{
	if( ! _condition ) {
		_condition = new QuestConditionComposite();
	}

	QuestConditionDialog dialog( _condition );
	dialog.exec();
}


//
// ----- QuestConditionDialog -----
//

QuestConditionDialog::QuestConditionDialog( QuestCondition * condition, QWidget * parent, const char * name )
: QDialog( parent, name, true )
{
	_contextItem = 0;
	_condition = condition;
	_menu = new QPopupMenu();

	_list = new QListView( this );
	_list->addColumn( tr( "Condition" ) );
	_list->setRootIsDecorated( true );
	buildTree();

	QPushButton * pbOk = new QPushButton( this );
	pbOk->setText( tr( "Ok" ) );
	FIXEDSIZE( pbOk );

	QPushButton * pbCan = new QPushButton( this );
	pbCan->setText( tr( "Cancel" ) );
	FIXEDSIZE( pbCan );

	QHBoxLayout * layH1 = new QHBoxLayout();
	layH1->addStretch( 1 );
	layH1->addWidget( pbOk );
	layH1->addStretch( 1 );
	layH1->addWidget( pbCan );
	layH1->addStretch( 1 );

	QVBoxLayout * layout = new QVBoxLayout( this );
	layout->setMargin( 5 );
	layout->setSpacing( 5 );
	layout->addWidget( _list, 1 );
	layout->addLayout( layH1 );
	layout->activate();

	connect( pbOk, SIGNAL( clicked() ), SLOT( accept() ) );
	connect( pbCan, SIGNAL( clicked() ), SLOT( reject() ) );
	connect( _list, SIGNAL( contextMenuRequested( QListViewItem *, const QPoint &, int ) ), SLOT( slot_menu( QListViewItem *, const QPoint &, int ) ) );
}

void QuestConditionDialog::buildTree()
{
	_list->clear();
	if( _condition ) {
		QuestConditionItem * item = new QuestConditionItem( _list );
		_list->setOpen( item, true );
		item->setCondition( _condition );
		if( _condition->getType() == QuestCondition::COMPOSITE ) {
			QuestConditionComposite * composite = ( QuestConditionComposite * ) _condition;
			uint nb = composite->getConditionNumber();
			for( uint i = 0; i < nb; ++i ) {
				buildSubTree( item, composite->getCondition( i ) );
			}
		}
	}
}

void QuestConditionDialog::buildSubTree( QuestConditionItem * root, QuestCondition * condition )
{
	QuestConditionItem * item = new QuestConditionItem( root );
	_list->setOpen( item, true );
	item->setCondition( condition );
	if( condition->getType() == QuestCondition::COMPOSITE ) {
		QuestConditionComposite * composite = ( QuestConditionComposite * ) condition;
		uint nb = composite->getConditionNumber();
		for( uint i = 0; i < nb; ++i ) {
			buildSubTree( item, composite->getCondition( i ) );
		}
	}
}

void QuestConditionDialog::slot_menu( QListViewItem * item, const QPoint & point, int /* column */ )
{
	_menu->clear();
	_contextItem = (QuestConditionItem *) item;
	if( _contextItem ) {
		_menu->insertItem( tr( "&Properties" ), this, SLOT( slot_actionPropertiesCondition() ) );
		if( _contextItem->getCondition()->getType() == QuestCondition::COMPOSITE ) {
			_menu->insertItem( tr( "&Add condition" ), this, SLOT( slot_actionAddCondition() ) );
		}
		if( _contextItem->parent() ) {
			_menu->insertItem( tr( "&Delete condition" ), this, SLOT( slot_actionDelCondition() ) );
		}
		_menu->popup( point );
	} else {
		if( _list->childCount() == 0 ) {
			_menu->insertItem( tr( "&Add condition" ), this, SLOT( slot_actionAddCondition() ) );
			_menu->popup( point );
		}
	}
}

void QuestConditionDialog::slot_actionPropertiesCondition()
{
	QuestCondition * condition = _contextItem->getCondition();
	switch( condition->getType() ) {
	case QuestCondition::DATE: {
		QuestConditionDateProperties dialog( this );
		dialog.setCondition( (QuestConditionDate*)condition );
		dialog.exec();
		} break;
	case QuestCondition::LORD: {
		QuestConditionLordProperties dialog( this );
		dialog.setCondition( (QuestConditionLord*)condition );
		dialog.exec();
		} break;
	case QuestCondition::PLAYER: {
		QuestConditionPlayerProperties dialog( this );
		dialog.setCondition( (QuestConditionPlayer*)condition );
		dialog.exec();
		} break;
	default:
		break;
	}
}

void QuestConditionDialog::slot_actionAddCondition()
{
	NewQuestConditionDialog dialog;
	if( dialog.exec() ) {
		QuestConditionItem * item;
		QuestCondition * condition = dialog.getNewCondition();
		if( _contextItem ) {
			QuestCondition * parentCondition = _contextItem->getCondition();
			if( parentCondition->getType() == QuestCondition::COMPOSITE ) {
				( ( QuestConditionComposite * )parentCondition )->addCondition( condition );
			}
			buildTree();
		} else {
			item = new QuestConditionItem( _list );
			item->setCondition( condition );
		}
	}
}

void QuestConditionDialog::slot_actionDelCondition()
{
	if( _contextItem ) {
		QuestConditionItem * parentItem = dynamic_cast<QuestConditionItem*> ( _contextItem->parent() );
		QuestCondition * parentCondition = parentItem->getCondition();
		if( parentCondition->getType() == QuestCondition::COMPOSITE ) {
			( ( QuestConditionComposite * )parentCondition )->delCondition( _contextItem->getCondition() );
		}
		buildTree();
	}
}

//
// ----- QuestConditionItem -----
//

QuestConditionItem::QuestConditionItem( QListView * parent )
: QListViewItem( parent )
{
	_condition = 0;
}

QuestConditionItem::QuestConditionItem( QuestConditionItem * parent )
: QListViewItem( parent )
{
	_condition = 0;
}

void QuestConditionItem::setCondition( QuestCondition * condition )
{
	setText( 0, condition->getLabel() );
	_condition = condition;
}

//
// ----- NewQuestConditionDialog -----
//

NewQuestConditionDialog::NewQuestConditionDialog( QWidget * parent, const char * name )
: QDialog( parent, name, true )
{
	_type = TC_NONE;

	_list = new QListBox( this );
	_list->insertItem( tr( "AND" ), TC_AND );
	_list->insertItem( tr( "OR" ), TC_OR );
	_list->insertItem( tr( "Date" ), TC_DATE );
	_list->insertItem( tr( "Lord" ), TC_LORD );
	_list->insertItem( tr( "Player" ), TC_PLAYER );

	QVBoxLayout * layout = new QVBoxLayout( this );
	layout->addWidget( _list, 1 );
	layout->activate();

	connect( _list, SIGNAL( doubleClicked( QListBoxItem * ) ), this, SLOT( slot_select() ) );
}

QuestCondition * NewQuestConditionDialog::getNewCondition()
{
	QuestCondition * ret = 0;

	switch( _type ) {
	case TC_AND:
		ret = new QuestConditionComposite( QuestConditionComposite::AND );
		break;
	case TC_OR:
		ret = new QuestConditionComposite( QuestConditionComposite::OR );
		break;
	case TC_DATE:
		ret = new QuestConditionDate();
		break;
	case TC_LORD:
		ret = new QuestConditionLord();
		break;
	case TC_PLAYER:
		ret = new QuestConditionPlayer();
		break;
	default:
		break;
	}

	return ret;
}

void NewQuestConditionDialog::slot_select()
{
	_type = ( TypeCondition )_list->currentItem();
	accept();
}


