/*
  Main.C
  Algebraical Virtual Assembler
  Uros Platise, Feb. 1998
*/

#ifdef WIN32
#pragma warning( disable : 4786)
#endif
#include <sys/stat.h>
#ifdef WIN32
#include <io.h>
#else
#include <unistd.h>
#include <dirent.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "Global.h"
#include "Preproc.h"
#include "Lexer.h"
#include "Syntax.h"
#include "Segment.h"
#include "Symbol.h"
#include "Keywords.h"
#include "Reports.h"
#include "Object.h"
#include "Avr.h"

#ifdef WIN32
#define S_ISDIR(k) ((k)&_S_IFDIR)
#define S_ISREG(k) ((k)&_S_IFREG)
#endif
#define __AVA_MAJOR_VERSION	0
#define __AVA_MINOR_VERSION	3
#define __AVA_PATCH_VERSION	0

const char* version = "AVA Version 0.3b, Uros Platise (c) 1997-1999";
const char* arch_file = AVA_ARCH;
const char* help = 
"Syntax: ava [-pLv] [--version] [--multiple-output-files] [--motorola]\n"
"            [--intel] [-Adevice] [-lI<filename>] [-I<dirname>] [-T{filename}]"
"\n"
"            [-Dmacro{=val}] [-fmacro{=val}] [-o outfile] files libraries\n"
"Files:\n"
"  file.s     Assembler source.\n"
"  file.o     Object; if not else specifed, this is the assembler default.\n"
"  a.out      Linker default output file name.\n"
"  a.out.name Linker extended file name if --multiple-output-files is given.\n"
"             The extension 'name' is replaced with segment name.\n"
"  library    Library is a directory of object files linked in alphabetical\n"
"             order.\n"
"\n"
"Switches:\n"
"  -o         Set output file name (to redirect the output to stdout: -o"
" stdout)\n"
"  -p         Use stdin for input and stdout for output if -o is not set.\n"
"  -A         Declare target device; same as: -Ddevice -Tarch.inc\n"
"  -D         Define macro with value 'val'. If 'val' is not given 1 is"
" assumed.\n"
"  -f         Define public macro of the form __macro{=val}\n"
"  -I         Add directory to the search list or include a file.\n"
"  -T         Auto-include the 'target.inc' file or the 'filename' if given.\n"
"             File specifed by the -T option is always first in the queue.\n"
"\n"
"  -L         Generate listing report (assembler only)\n"
"  -v         Verbose (enable info)\n"
"  -l         All reports, infos, errors and warnings are printed on the\n"
"             stderr. If log file is specified info is redirected.\n"
"             Errors and warnings are reported to the stderr and log file.\n"
"\n"
"  --version                    Output version information.\n"
"  --multiple-output-files      Force linker to split segments in files.\n"
"  --intel                      Set Intel standard 16 bit output format.\n"
"                               (sets --multiple-output-files by default)\n"
"  --motorola                   Set Motorola S-record S1/S2 (16/24) bit\n"
"                               output format.\n"
"  --uasm                       Set Micro Assembler output format (default)\n"
"\n"
"AVR specific options:\n"
"  -favr_noskipbug              Do not check bad return address.\n"
"  -favr_noendianbug            Do not swap bytes in the flash segment.\n"
"                               (this will be the default in future releases)\n"
"\n"
"Report bugs to: uros.platise@ijs.si (http://medo.fov.uni-mb.si/mapp)"
"";


TAVAStatus avasta;
TPreproc preproc;
TLexer lexer;
TGAS gas;
TSyntax syntax;
TSegment segment;
TSymbol symbol;
TKeywords keywords;
TReports reports;
TObject object;
PArch archp;


/* Auxilirary Functions */

#ifdef WIN_DOS
#define alphasort 1

void scandir(char *dirname, dirent ***dirlist, 
            int (*select)(const struct dirent *dir_entry), int sort) {              
    int i;
    struct dirent *help;
    DIR *d = opendir(dirname);
    
    i = 0;
    while ( (help = readdir(d)) ) 
        if (select(help)) 
            i++;

    dirlist = (dirent***)malloc(i * sizeof(dirent));

    i = 0;
    while ( (help = readdir(d)) ) 
        if (select(help)) 
            memcpy(dirlist[i++], help, sizeof(dirent)) ;

    closedir(d);
}
#endif

#ifndef WIN32
int select_object_file(const struct dirent *dir_entry){
  int eos = strlen(dir_entry->d_name);
  if (dir_entry->d_name[eos-2]=='.' && dir_entry->d_name[eos-1]=='o'){
    return 1;
  }
  return 0;
}
#endif

/* AVA Start-Up Procedure */

int main(int argc, char* argv[]){
  char ftype;
  bool linker=false, assembler=false;
  const char* include_targetfile=NULL;	/* always pushed last! */
  char* outfile=NULL;
  char* asmfile=NULL;
  int ai;
  TMicroStack<const char *> files;
  TMicroStack<string> object_lib;
  try{
    for (ai=1; ai<argc; ai++){
      ftype = argv[ai][strlen(argv[ai])-1];
      
      if (argv[ai][0]=='-'){
        if (argv[ai][2]==0){
          switch(argv[ai][1]){
  	  case 'h': printf("%s\n%s\n", version, help); exit(1);
  	  case 'v': reports.IncVerboseLevel(); break;
	  case 'L': reports.listing.Enable(); break;
          case 'T':
	    if (include_targetfile!=NULL){
	      throw generic_error("Target file already defined.");
	    }
	    include_targetfile=&argv[ai][2]; 
	    break;
          case 'p': 
            files.push("stdin"); outfile="stdout"; asmfile="stdin";
            assembler=true; 
            break;
	  case 'e': break;
          case 'o': 
	    ai++; 
	    if (ai >= argc){throw generic_error("-o: File name is missing.");}
            if (strstr(argv[ai],".s")!=NULL){
              throw generic_error("Output file name has source extension"
                " `*.s'");
            }
            outfile=argv[ai];
	    break;	  
	  default: throw generic_error("Invalid switch:", argv[ai]);
	  }
	}
	else if (argv[ai][0]=='-' && argv[ai][1]=='-'){
	  if (strcmp(&argv[ai][2], "help")==0){
	    printf("%s\n%s\n", version, help); exit(1);
	  }
	  else if (strcmp(&argv[ai][2], "version")==0){
	    printf("%s\n", version); exit(1);
	  }
	  else if (strcmp(&argv[ai][2], "multiple-output-files")==0){
	    object.SetMultipleFiles();
	  }
	  else if (strcmp(&argv[ai][2], "motorola")==0){
	    object.SetOutputFormat(TObject::OF_MOTOROLA);
	  }	  
	  else if (strcmp(&argv[ai][2], "intel")==0){
	    object.SetOutputFormat(TObject::OF_INTEL);
	  }	  	  
	  else if (strcmp(&argv[ai][2], "uasm")==0){
	    object.SetOutputFormat(TObject::OF_UASM);
	  }	  	  	  
	  else{throw generic_error("Invalid switch:", argv[ai]);}
	}
        else{
          switch(argv[ai][1]){      
	  case 'l': reports.Config(&argv[ai][2]); break;
	  case 'D': {
              char *val = strchr(&argv[ai][2],'=');
              if (val==NULL){symbol.addMacro(&argv[ai][2],"1");}
              else{
                char buf[LX_STRLEN]; int i=0; 
                while(val!=&argv[ai][i+2]){buf[i]=argv[ai][i+2];i++;} buf[i]=0;
                symbol.addMacro(buf,val+1);
              }
            } break;
          case 'f': {
              char *val = strchr(&argv[ai][2],'=');
              char buf[LX_STRLEN]; strcpy(buf, "__");
              if (val==NULL){
                strcat(buf, &argv[ai][2]); 
		symbol.addMacro(buf, "1", TSymbolRec::Public);
              } else{
                int i=2; 
                while(val!=&argv[ai][i]){buf[i]=argv[ai][i];i++;} buf[i]=0;
                symbol.addMacro(buf, val+1, TSymbolRec::Public);
              }
            } break;
	  case 'I': {
	      struct stat file_info;
	      if (stat(&argv[ai][2], &file_info)<0){
	        throw file_error(&argv[ai][2]);
	      }
	      if (S_ISDIR(file_info.st_mode)){preproc.AddDir(&argv[ai][2]);}
	      else if (S_ISREG(file_info.st_mode)){files.push(&argv[ai][2]);}
	      else{
	        throw generic_error("Invalid file type for -I switch.");
	      }
	    } break;
	  case 'T':
	    if (include_targetfile!=NULL){
	      throw generic_error("Target file already defined.");
	    } 
	    include_targetfile=&argv[ai][2]; 
	    break;
	  case 'A':
	    if (include_targetfile!=NULL){
	      throw generic_error("Target file already defined.");
	    }
	    symbol.addMacro(&argv[ai][2],"1"); 
	    include_targetfile = arch_file;
	    break;
	  default: throw generic_error("Invalid switch:", argv[ai]);
          }
	}
      }else{      
        switch(ftype){
        case 's': 
          if (assembler){
	    throw generic_error("Only one assembler source can"
	                        " be assembled at a time.");
          }
	  assembler=true;
          asmfile=argv[ai];
	  break;
        case 'o': 
          linker=true; 
          files.push(argv[ai]); 
          break;
        default: {
	    /* add object files from directory to the list */
 	    struct stat file_info;
	    if (stat(argv[ai], &file_info)<0){throw file_error(argv[ai]);}
	    if (S_ISDIR(file_info.st_mode)){	    
	      /* find .o files */
	      reports.Info(TReports::VL_LINK4,"Scanning directory: %s\n"
	                   "Adding: ", argv[ai]);
#ifdef WIN32    
              struct _finddata_t c_file;    
              long hFile;
              char full_path[PPC_MAXFILELEN];
              strcpy(full_path, argv[ai]);
              int edge = strlen(full_path);
              if (full_path[edge - 1] != '\\' && full_path[edge - 1] != '/'){
                full_path[edge] = '/';
                ++edge;
              }
              strcpy(&full_path[edge], "*.o");
              if ((hFile = _findfirst(full_path, &c_file )) != -1L){
        	do{
                  strcpy(&full_path[edge], c_file.name);
                  object_lib.push(full_path);
                  files.push(object_lib.top().c_str());
                  linker=true;
                  reports.Info(TReports::VL_LINK4,"%s ", c_file.name);
        	}while (_findnext(hFile, &c_file)==0);
              }			   
#else			   
	      struct dirent **dir_list;
	      char full_path [PPC_MAXFILELEN];
	      int cat_pos;
	      strcpy(full_path, argv[ai]);
	      cat_pos = strlen(full_path);
	      scandir(argv[ai], &dir_list, select_object_file, alphasort);
	      for(int i=0; dir_list[i] != NULL; i++){
		strcpy(&full_path[cat_pos], dir_list[i]->d_name);
		object_lib.push(full_path);
		files.push(object_lib.top().c_str());
		linker=true;
		reports.Info(TReports::VL_LINK4,"%s ", dir_list[i]->d_name);
		free(dir_list[i]);
              }
	      free(dir_list);
#endif	      
	      reports.Info(TReports::VL_LINK4,"\n");
	    } else{
	      throw generic_error("Unknown file type.");
	    }
	  }	  
        }
      }
    }
    
    if (linker && assembler){ 
      throw generic_error("Only assembler or linker can be invoked at a time.");
    }
    /* if assembler source was given, put it on the list as last */
    if (assembler){files.push(asmfile);}
    
    /* resort files */
    if (files.empty()){
      fprintf(stderr,"%s: No input files.\n",argv[0]);exit(1);
    }
    while(!files.empty()){preproc.insert(files.pop());}

    /* Include AVA dedicated macros */
    symbol.addMacro("__AVA_MAJOR_VERSION","0");
    symbol.addMacro("__AVA_MINOR_VERSION","3");
    symbol.addMacro("__AVA_PATCH_VERSION","0");
    
    /* Copy Command Line Parameters */
    reports.Info(TReports::VL_LINK1, "Running:\n  ");
    for (int i=0; i<argc; i++){
      reports.Info(TReports::VL_LINK1, "%s ", argv[i]);
    }
    reports.Info(TReports::VL_LINK1, "\n\n");
    
    /* Invoke processor */
    if (assembler){
      preproc.AddDir(AVA_LIB);
      if (include_targetfile){
        char buf[LX_STRLEN];
        if (*include_targetfile==0){strcpy(buf,AVA_TARGET);}
        else{strcpy(buf,include_targetfile);}
        preproc.insert(preproc.FindFullPathName(buf));
      }
      object.assemble(outfile, asmfile);
    }
    if (linker){object.link(outfile);}
  }
  catch (global_error &x){reports.Error(x);}  
  
  return reports.ErrorCount();
}

