#ifndef AVM_ARGS_H
#define AVM_ARGS_H

#include "avm_default.h"

AVM_BEGIN_NAMESPACE;

class AVMEXPORT Args
{
public:
    class Option {
    public:
	enum Type {
	    NONE =	   '#',
	    HELP =	   'h',	// use default help text and options
	    CODEC =	   'c',	// show available codec options (char*)
	    OPTIONS =	   'O',	// @value is pointer to another Options array
	    SUBOPTIONS =   'o',	// @value is pointer to SubOptions array
				// these options are separated by ':'
	    BOOL =	   'b',	// true flag when option is present (default is false)
	    DOUBLE =	   'd',	// double from given range (max == min -> unlimited)
	    INT =	   'i',	// integer from given range (max == min -> unlimited)
	    STRING =	   's',	// any string could be given
	    SELECTSTRING = 'x',	// string from the list

	    REGBOOL =	   'B',
	    REGDOUBLE =    'D',
	    REGINT =	   'I',
	    REGSTRING =    'S'
	};
	Option(const char* o, const void* v = 0, int32_t mi = 0, int32_t ma = 0)
	    : opts(o), value(v), min(mi), max(ma) {}

	Option(const char* o, const bool* b)
	    : opts(o), value(b) {}

	const char* getLongOption() const { return opts + 1; }
	const char* getShortOption() const { return findString(1); }
	const char* getHelp() const { return findString(2); }

	int32_t getInt() const { return (value) ? *(const int32_t*)value : 0; }
	int32_t getMin() const { return min; }
	int32_t getMax() const { return max; }
	bool getBool() const { return (value) ? *(const bool*)value : false; }
	double getDouble() const { return (value) ? *(const double*)value : 0.; }
	const char* getString() const { return (const char*)value; }
	const Option* getOptions() const { return (const Option*)value; }
	Type getType() const { return (Type) opts[0]; }

	bool is(Type t) const { return (t == getType()); }
	bool isInRange(int32_t v) const { return (getMin() == getMax())
	    || (getMin() <= v && v <= getMax()); }

	int setBool(bool b) const { return (value) ? *(bool*)value = b, 0 : -1; }
	int setDouble(double d) const { return (value) ? *(double*)value = d, 0 : -1; }
	int setInt(int32_t i) const { return (value) ? *(int32_t*)value = i, 0 : -1; }
	int setString(char* s) const { return (value) ? *(char**)value = s, 0 :  -1; }

    private:
	// go through adjanced zero ended strings
	const char* findString(int cnt) const {
	    const char* r = opts;
	    while (cnt--)
		while (*r++)
		    ;
	    return r;
	}

	const char* opts;	// 'Type' long-option-str \0  short-option-str \0 help-str
	const void* value;	// storage pointer for value
				// for SUB/OPTIONS contains subarray with Options
				// for REG+  contains pointer to default value
	union {
	    struct {
		int32_t min;	// min integer value
		int32_t max;	// max integer value
	    };
	    const char* defstr;	// list of string options
	};
    };

    Args(const Option* options, int* argc, char** argv,
	 const char* help = 0, const char* regname = 0);
    ~Args();
    // could be used to parse CodecInfo arguments
    static void ParseCodecInfo(const char* str);

protected:
    int findOpt(int olong = 0);

    const Option* opts;
    int* argc;
    char** argv;
    const char* help;
    const char* regname;
    int idx;
};

#ifdef __GNUC__
#define ARGSOPTION(a, b, c, d, args...) \
    Args::Option(a b "\0" c "\0" d, ## args)
#else
#define ARGSOPTION(a, b, c, d, ...) \
    Args::Option(a b "\0" c "\0" d, __VA_ARGS__)
#endif

AVM_END_NAMESPACE;

#endif /* AVM_ARGS_H */
