# Arch Perl library, Copyright (C) 2004 Mikhael Goikhman
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.005;
use strict;

package AXP::Command::tree::history;
use base 'AXP::Command::tree';

use Arch::Tree;

sub infoline {
	"show descriptions of ancestry revisions from logs"
}

sub optusage {
	"[options] [filepath]"
}

sub options {
	(
		$_[0]->tree_options,
		reverse  => { sh => 'r', desc => "sort from oldest to newest" },
		summary  => { sh => 's', desc => "show summary of each revision" },
		creator  => { sh => 'c', desc => "show creator of each revision" },
		date     => { sh => 'D', desc => "show date of each revision" },
		desc     => { desc => "equivalent to -Dcs plus -n for renamed files" },
		no_full  => { sh => 'F', desc => "show short revision names" },
		filename => { sh => 'n', desc => "show orig file/dir name if applicable" },
		no_email => { sh => 'E', desc => "don't show email of creator" },
		one_version => { sh => 'o', desc => "don't follow tags from other versions" },
	)
}   
 
sub helptext {
	q{
		Without filepath parameter, show all ancestry revision names
		and optionally their summary, date and creator.  The format is
		the same as in 'tla logs' and 'tla revisions'.

		With filepath parameter (that is existing file or directory
		managed by arch), only show the revisions that modified it.
		In this case, additional field may be shown too (--filename)
		and all renames are reported properly.
	}
}

sub execute {
	my $self = shift;
	my %opt = %{$self->{options}};

	$opt{date} = $opt{creator} = $opt{summary} = 1 if $opt{desc};

	my $tree = $self->tree;
	my $filepath = shift(@ARGV);
	warn "Post file-name parameters (@ARGV) are ignored\n" if @ARGV;

	my %args = ();
	$args{one_version} = 1 if $opt{one_version};

	my $revision_descs = $tree->get_history_revision_descs($filepath, %args);

	if ($filepath && !$opt{filename} && $opt{desc}) {
		$opt{filename} ||= $_->{is_filepath_renamed} foreach @$revision_descs;
	}

	@$revision_descs = reverse @$revision_descs if $opt{reverse};

	foreach my $revision_desc (@$revision_descs) {
		my $creator_line = "";
		$creator_line .= $revision_desc->{date} if $opt{date};
		if ($opt{creator}) {
			$creator_line .= "      " if $opt{date};
			$creator_line .= $revision_desc->{creator};
			$creator_line .= " <" . $revision_desc->{email} . ">" unless $opt{no_email};
		}

		print "$revision_desc->{version}--" unless $opt{no_full};
		print $revision_desc->{name}, "\n";
		print "    ", $revision_desc->{filepath}, "\n" if $filepath && $opt{filename};
		print "    $creator_line\n" if $creator_line;
		print "    ", $revision_desc->{summary}, "\n" if $opt{summary};
	}
}

1;
