//
// Licensed under same license as Banshee
// Copyright (C) 2006 Novell, Inc.
// Written by Aaron Bockover <abock@gnome.org>
//
// This tool extracts translator information from .po files
// to generate information for display in the about dialog
//

using System;
using System.IO;
using System.Collections.Generic;
using System.Text.RegularExpressions;

public class TranslatorExtractor
{
    public static void Main(string [] args)
    {
        Console.WriteLine("<!-- AUTOGENERATED - DO NOT EDIT -->");
        Console.WriteLine("<translators>");
        foreach(string filename in Directory.GetFiles(args.Length == 0 ? "." : args[0], "*.po")) {
            using(FileStream stream = new FileStream(filename, FileMode.Open, FileAccess.Read)) {
                using(StreamReader reader = new StreamReader(stream)) {
                    ParseFile(filename, reader);
                }
            }
        }
        Console.WriteLine("</translators>");
    }

    private static void ParseFile(string file, TextReader reader)
    {
        Dictionary<string, string> names = new Dictionary<string, string>();
        string language_code = Path.GetFileNameWithoutExtension(file);
        string language_name = null;
    
        bool past_plural_forms = false;
    
        while(true) {
            string line = reader.ReadLine();
            if(line == null || (line.StartsWith("#:") && past_plural_forms)) {
                break;
            } else if(line.StartsWith("\"Plural-Forms:")) {
                past_plural_forms = true;
            }
            
            try {
                Match match = Regex.Match(line, ".*#(.*)<(.*)>(.*,.*|[ \n]*$)");
                if(!match.Success) {
                    match = Regex.Match(line, ".*\\\"Last-Translator:(.*)<(.*)>.*");
                    if(!match.Success) {
                        match = Regex.Match(line, ".*\\\"Language-Team:(.*)<.*>.*");
                        if(!match.Success) {
                            continue;
                        }

                        language_name = match.Groups[1].Value.Trim();
                        int pos = language_name.IndexOf('(');
                        if(pos > 0) {
                            language_name = language_name.Substring(0, pos).Trim();
                        }
                    }
                }
                
                string name = match.Groups[1].Value.Trim();
                string email = match.Groups[2].Value.Trim();
            
                if(name == String.Empty || email == String.Empty) {
                    continue;
                } else if(name.StartsWith("Maintainer:")) {
                    name = name.Substring(11).Trim();
                }

                if(!names.ContainsKey(email)) {
                    names.Add(email, name);
                }
            } catch {
                continue;
            }
        }

        // hack for banshee, remove if used elsewhere
        if(language_code == "sr") {
            return;
        } else if(language_code == "sr@Latn") {
            language_code = "sr";
        }
        // end special

        // override language names from extracted
        switch(language_code) {
            case "ca": language_name = "Catalan"; break;
            case "zh_CN": language_name = "Simplified Chinese"; break;
            case "es": language_name = "Spanish"; break;
            case "fr": language_name = "French"; break;
            default: break;
        }
        
        Console.WriteLine("  <language code=\"{0}\" name=\"{1}\">", language_code, language_name);
        foreach(KeyValuePair<string, string> person in names) {
            Console.WriteLine("    <person>{0}</person>", person.Value);
        }
        Console.WriteLine("  </language>");
    }
}

