/*  BMP - Cross-platform multimedia player
 *  Copyright (C) 2003-2004  BMP development team.
 *
 *  Based on XMMS:
 *  Copyright (C) 1998-2003  XMMS development team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "widget.h"

#include <glib.h>
#include <gdk/gdk.h>

#include "debug.h"


void
widget_init(Widget * widget, GdkPixmap * parent, GdkGC * gc,
            gint x, gint y, gint width, gint height, gint visible)
{
    widget->parent = parent;
    widget->gc = gc;
    widget_set_position(widget, x, y);
    widget_set_size(widget, width, height);
    widget->visible = visible;
    widget->redraw = TRUE;
    widget->mutex = g_mutex_new();
}

void
widget_set_position(Widget * widget, gint x, gint y)
{
    widget->x = x;
    widget->y = y;
    widget_queue_redraw(widget);
}

void
widget_move(Widget * widget, gint dx, gint dy)
{
    widget->x += dx;
    widget->y += dy;
    widget_queue_redraw(widget);
}

void
widget_set_size(Widget * widget, gint width, gint height)
{
    widget->width = width;
    widget->height = height;
    widget_queue_redraw(widget);
}

void
widget_queue_redraw(Widget * widget)
{
    widget->redraw = TRUE;
}

gboolean
inside_widget(gint x, gint y, gpointer p)
{
    Widget *w = WIDGET(p);

    if (x >= w->x && x < w->x + w->width && y >= w->y
        && y < w->y + w->height && w->visible)
        return TRUE;
    return FALSE;
}

void
show_widget(gpointer widget)
{
    WIDGET(widget)->visible = TRUE;
    draw_widget(widget);
}

void
hide_widget(gpointer widget)
{
    WIDGET(widget)->visible = FALSE;
}

void
resize_widget(gpointer widget, gint width, gint height)
{
    WIDGET(widget)->width = width;
    WIDGET(widget)->height = height;
    draw_widget(widget);
}

void
move_widget(gpointer widget, gint x, gint y)
{
    lock_widget(widget);
    widget_set_position(widget, x, y);
    unlock_widget(widget);
}

void
draw_widget(gpointer widget)
{
    lock_widget(widget);
    WIDGET(widget)->redraw = TRUE;
    unlock_widget(widget);
}

void
add_widget(GList ** list, gpointer widget)
{
    (*list) = g_list_append(*list, widget);
}

void
handle_press_cb(GList * widget_list, GtkWidget * widget,
                GdkEventButton * event)
{
    GList *wl;

    for (wl = widget_list; wl; wl = g_list_next(wl)) {
        if (WIDGET(wl->data)->button_press_cb)
            WIDGET(wl->data)->button_press_cb(widget, event, wl->data);
    }
}

void
handle_release_cb(GList * widget_list, GtkWidget * widget,
                  GdkEventButton * event)
{
    GList *wl;

    for (wl = widget_list; wl; wl = g_list_next(wl)) {
        if (WIDGET(wl->data)->button_release_cb)
            WIDGET(wl->data)->button_release_cb(widget, event, wl->data);
    }
}

void
handle_motion_cb(GList * widget_list, GtkWidget * widget,
                 GdkEventMotion * event)
{
    GList *wl;

    for (wl = widget_list; wl; wl = g_list_next(wl)) {
        if (WIDGET(wl->data)->motion_cb)
            WIDGET(wl->data)->motion_cb(widget, event, wl->data);
    }
}

void
handle_scroll_cb(GList * wlist, GtkWidget * widget, GdkEventScroll * event)
{
    GList *wl;

    for (wl = wlist; wl; wl = g_list_next(wl)) {
        if (WIDGET(wl->data)->mouse_scroll_cb)
            WIDGET(wl->data)->mouse_scroll_cb(widget, event, wl->data);
    }
}

void
draw_widget_list(GList * widget_list, gboolean * redraw, gboolean force)
{
    GList *wl;
    Widget *w;

    *redraw = FALSE;
    wl = widget_list;

    for (wl = widget_list; wl; wl = g_list_next(wl)) {
        w = WIDGET(wl->data);

        REQUIRE_LOCK(w->mutex);

        if (!w->draw)
            continue;

        if (!w->visible)
            continue;

        if (w->redraw || force) {
            w->draw(w);
/*             w->redraw = FALSE; */
            *redraw = TRUE;
        }
    }
}

void
widget_list_change_pixmap(GList * widget_list, GdkPixmap * pixmap)
{
    GList *wl;

    for (wl = widget_list; wl; wl = g_list_next(wl)) {
        Widget *widget = wl->data;
        widget->parent = pixmap;
        widget_queue_redraw(widget);
    }
}

void
clear_widget_list_redraw(GList * widget_list)
{
    GList *wl;

    for (wl = widget_list; wl; wl = g_list_next(wl)) {
        REQUIRE_LOCK(WIDGET(wl->data)->mutex);
        WIDGET(wl->data)->redraw = FALSE;
    }
}

void
lock_widget(gpointer widget)
{
    g_mutex_lock(WIDGET(widget)->mutex);
}

void
unlock_widget(gpointer widget)
{
    g_mutex_unlock(WIDGET(widget)->mutex);
}

void
lock_widget_list(GList * widget_list)
{
    g_list_foreach(widget_list, (GFunc) lock_widget, NULL);
}

void
unlock_widget_list(GList * widget_list)
{
    g_list_foreach(widget_list, (GFunc) unlock_widget, NULL);
}
