/* Bezerk
 * Copyright (C) 1998 Tony Gale.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <string.h>
#include <pwd.h>
#include <unistd.h>
#include <sys/stat.h>

#include <gtk/gtk.h>
#include <glib.h>

#include "bezerk.h"
#include "util.h"
#include "persona.h"
#include "debug.h"

GSList *personae=NULL;
char *persona_dir;

void persona_save()
{
  FILE *persona_fd;
  char buff[BUFFLEN];
  GSList *list_entry;
  Persona *persona;

  bs_function_enter();

  if (!personae) {
      bs_function_leave();
      return;
  }

  g_snprintf(buff, BUFFLEN, "%s/%s", persona_dir, PERSONA_FILE);
  if ( (persona_fd = fopen(buff, "w")) == NULL) {
    bs_printf(3, "Error saving personae to %s", buff);
    bs_function_leave();
    return;
  }
  
  fputs("# Persona Settings\n", persona_fd);
  fputs("# DO NOT EDIT - Automatically Generated\n", persona_fd);

  list_entry = personae;
  while(list_entry) {
    persona = (Persona *) list_entry->data;
    fputs(":", persona_fd);
    fputs(persona->nick, persona_fd);
    fputs("\n", persona_fd);
    fputs(persona->real_name, persona_fd);
    fputs("\n", persona_fd);
    fputs(persona->away_nick, persona_fd);
    fputs("\n", persona_fd);
    list_entry = g_slist_next(list_entry);
  }

  fclose(persona_fd);
  bs_function_leave();
  return;
}

void persona_clear()
{
  Persona *persona;

  bs_function_enter();

  while (personae) {
    persona = (Persona *) personae->data;
    personae = g_slist_remove(personae, persona);
    g_free(persona->nick);
    g_free(persona->real_name);
    g_free(persona->away_nick);
    g_free(persona);
  }

  bs_function_leave();
  return;
}

Persona *persona_get(gint persona_num)
{
  GSList *list_entry = g_slist_nth(personae, persona_num);
  
  if (list_entry) {
    return((Persona *)list_entry->data);
  } else {
    return(NULL);
  }
}

void persona_add(char *nick, char *real_name, char *away_nick)
{
  Persona *persona;

  if (!nick || !real_name) {
    bs_function_leave();
    return;
  }

  persona = g_malloc(sizeof(Persona));
  persona->nick = nick;
  persona->real_name = real_name;
  if (away_nick) {
    persona->away_nick = away_nick;
  } else {
    persona->away_nick = NULL;
  }
  personae = g_slist_append(personae, persona);

  bs_function_leave();
  return;
}

char *persona_read_element(FILE *persona_fd, gint is_start)
{
  char buff[BUFFLEN];

  bs_function_enter();

  do {
    if ( !fgets(buff, BUFFLEN, persona_fd) ) {
      bs_function_leave();
      return(NULL);
    }
  } while (*buff == '#' || (is_start && (*buff != ':')) || (!is_start && (*buff == ':')) );

  if ( buff[strlen(buff)-1] == '\n' ) {
    buff[strlen(buff)-1] = '\0';
  }

  if ( !strlen(buff) ) {
    bs_function_leave();
    return(NULL);
  } else {
    bs_function_leave();
    return( (is_start ? g_strdup(buff+1) : g_strdup(buff)) );
  }
}
  
gint persona_init()
{
  int uid;
  struct stat stat_buf;
  struct passwd *pwdent;
  FILE *persona_fd;
  char buff[BUFFLEN];
  char *nick, *away_nick, *real_name;

  bs_function_enter();

  /* Get the UID of the current user */
  uid = getuid();

 /* Get the password entry using the UID */
  if ((pwdent = getpwuid(uid)) == NULL) {
    g_error("Can't find password file entry");
  }

  g_snprintf(buff, BUFFLEN, "%s/.bezerk", pwdent->pw_dir);
  persona_dir = g_strdup(buff);

  if (stat(persona_dir, &stat_buf) == -1) {
    if (mkdir(persona_dir, S_IRUSR | S_IWUSR | S_IXUSR) == -1) {
      g_error("Creating bezerk persona directory");
    }
  } else if ( !S_ISDIR(stat_buf.st_mode) ) {
    g_error("~/.bezerk is not a directory");
  }

  g_snprintf(buff, BUFFLEN, "%s/%s", persona_dir, PERSONA_FILE);
  if ( (persona_fd = fopen(buff, "r")) == NULL) {
    bs_function_leave();
    return(FALSE);
  } 
 
  while ( !feof(persona_fd) ) {
    nick = persona_read_element(persona_fd, TRUE);
    real_name = persona_read_element(persona_fd, FALSE);
    away_nick = persona_read_element(persona_fd, FALSE);
    if (!nick || !real_name) {
      break;
    }
    persona_add(nick, real_name, away_nick);
  }
  fclose(persona_fd);

  bs_function_leave();
  return(TRUE);
}

/* ------------------------------------------------------------------- */
/* Persona Preferences Functions                                       */
/* ------------------------------------------------------------------- */

gchar **persona_to_strs(Persona *persona, PersonaPrefsData *persona_prefs)
{
  gchar **persona_strs;

  persona_strs = g_malloc(sizeof(gchar *) *3);
  persona_strs[0] = g_strdup(persona->nick);
  persona_strs[1] = g_strdup(persona->real_name);
  if (!persona->away_nick) {
    persona_strs[2] = g_strdup("");
  } else {
    persona_strs[2] = g_strdup(persona->away_nick);
  }

  return(persona_strs);
}

void persona_to_strs_free(gchar **persona_strs)
{
  g_free(persona_strs[0]);
  g_free(persona_strs[1]);
  g_free(persona_strs[2]);
  g_free(persona_strs);
}

void persona_prefs_up(GtkWidget *widget, PersonaPrefsData *persona_prefs)
{
  gchar *row_strings[3];
  gpointer row_data;
  gint selection;

  if ( persona_prefs->selected_row == 0 ) {
    bs_function_enter();
    return;
  }

  persona_prefs->changed = TRUE;

  selection = persona_prefs->selected_row;
  gtk_clist_freeze(  GTK_CLIST(persona_prefs->clist) );
  gtk_clist_get_text( GTK_CLIST(persona_prefs->clist), persona_prefs->selected_row, 0, &row_strings[0] );
  gtk_clist_get_text( GTK_CLIST(persona_prefs->clist), persona_prefs->selected_row, 1, &row_strings[1] );
  gtk_clist_get_text( GTK_CLIST(persona_prefs->clist), persona_prefs->selected_row, 2, &row_strings[2] );
  row_data = gtk_clist_get_row_data( GTK_CLIST(persona_prefs->clist), persona_prefs->selected_row );

  /* Need to do it in this order, otherwise the strings are destroyed by the remove */
  gtk_clist_insert( GTK_CLIST(persona_prefs->clist), persona_prefs->selected_row-1, row_strings);
  gtk_clist_set_row_data( GTK_CLIST(persona_prefs->clist), persona_prefs->selected_row-1, row_data);
  gtk_clist_remove( GTK_CLIST(persona_prefs->clist), persona_prefs->selected_row+1 );
  gtk_clist_select_row( GTK_CLIST(persona_prefs->clist), selection-1, -1);
  gtk_clist_thaw(  GTK_CLIST(persona_prefs->clist) );

  bs_function_enter();

  return;
}

void persona_prefs_down(GtkWidget *widget, PersonaPrefsData *persona_prefs)
{
  gchar *row_strings[3];
  gpointer row_data;

  if ( persona_prefs->selected_row == (GTK_CLIST(persona_prefs->clist)->rows-1) ) {
    bs_function_enter();
    return;
  }

  persona_prefs->changed = TRUE;

  gtk_clist_freeze(  GTK_CLIST(persona_prefs->clist) );
  gtk_clist_get_text( GTK_CLIST(persona_prefs->clist), persona_prefs->selected_row, 0, &row_strings[0] );
  gtk_clist_get_text( GTK_CLIST(persona_prefs->clist), persona_prefs->selected_row, 1, &row_strings[1] );
  gtk_clist_get_text( GTK_CLIST(persona_prefs->clist), persona_prefs->selected_row, 2, &row_strings[2] );
  row_data = gtk_clist_get_row_data( GTK_CLIST(persona_prefs->clist), persona_prefs->selected_row );

  /* Need to do it in this order, otherwise the strings are destroyed by the remove */
  gtk_clist_insert( GTK_CLIST(persona_prefs->clist), persona_prefs->selected_row+2, row_strings);
  gtk_clist_set_row_data( GTK_CLIST(persona_prefs->clist), persona_prefs->selected_row+2, row_data);
  gtk_clist_remove( GTK_CLIST(persona_prefs->clist), persona_prefs->selected_row );
  gtk_clist_select_row( GTK_CLIST(persona_prefs->clist), persona_prefs->selected_row+1, -1);
  gtk_clist_thaw(  GTK_CLIST(persona_prefs->clist) );

  bs_function_enter();

  return;
}

void persona_prefs_clist_set_column_widths( PersonaPrefsData *persona_prefs )
{
  gtk_clist_set_column_width(GTK_CLIST(persona_prefs->clist), 0, persona_prefs->widths[0]);
  gtk_clist_set_column_width(GTK_CLIST(persona_prefs->clist), 1, persona_prefs->widths[1]);
  gtk_clist_set_column_width(GTK_CLIST(persona_prefs->clist), 2, persona_prefs->widths[2]);
}

gint persona_prefs_clist_add_entry( PersonaPrefsData  *persona_prefs,
				    gchar            **persona_strs,
				    gint               position)
{
  persona_prefs->widths[0] = MAX(persona_prefs->widths[0],
				     gdk_string_width(persona_prefs->clist->style->font, persona_strs[0]));
  persona_prefs->widths[1] = MAX(persona_prefs->widths[1],
				     gdk_string_width(persona_prefs->clist->style->font, persona_strs[1]));
  persona_prefs->widths[2] = MAX(persona_prefs->widths[2],
				     gdk_string_width(persona_prefs->clist->style->font, persona_strs[2]));

  if (position < 0) {
    position = gtk_clist_append( GTK_CLIST(persona_prefs->clist) , persona_strs);
  } else {
    gtk_clist_insert( GTK_CLIST(persona_prefs->clist), position, persona_strs);
  }

  bs_function_leave();
  return(position);
}

gint persona_edit_destroy(GtkWidget *widget, GdkEvent *event, PersonaEditData *persona_data)
{
  bs_function_enter();

  if (!persona_data) {
    bs_function_leave();
    return(FALSE);
  }

  if (persona_data->positive_func && persona_data->negative_func) {
    persona_data->negative_func(persona_data->user_data);
    g_free(persona_data);
    return(FALSE);
  }
  
  if (persona_data->positive_func && !persona_data->negative_func) {
    return(TRUE);
  }

  g_free(persona_data);
  bs_function_leave();
  return(FALSE);
}

void persona_edit_ok(GtkWidget *widget, PersonaEditData *persona_data)
{
  gchar **persona_strs;

  bs_function_enter();

  if (persona_data->positive_func) {
    persona_strs = g_malloc(sizeof(gchar *) *3);
    persona_strs[0] = g_strdup(gtk_entry_get_text( GTK_ENTRY(persona_data->nick)) );
    persona_strs[1] = g_strdup(gtk_entry_get_text( GTK_ENTRY(persona_data->real_name) ));
    persona_strs[2] = g_strdup(gtk_entry_get_text( GTK_ENTRY(persona_data->away_nick) ));
    if (!strlen(persona_strs[0]) || !strlen(persona_strs[1])) {
      gdk_beep();
      bs_function_leave();
      return;
    }
    persona_data->positive_func(persona_strs, persona_data->user_data);
    persona_to_strs_free(persona_strs);
  }

  persona_data->positive_func = NULL;
  gtk_widget_destroy(persona_data->dialog);

  bs_function_leave();
  return;
}

void persona_edit_cancel(GtkWidget *widget, PersonaEditData *persona_data)
{

  bs_function_enter();

  if (persona_data->negative_func) {
    persona_data->negative_func(persona_data->user_data);
    persona_data->positive_func = NULL;
  }

  gtk_widget_destroy(persona_data->dialog);
  
  bs_function_leave();
  return;
}

void persona_edit_dialog(void   *user_data,
			 gchar **persona_strs,
			 void  (*positive_func)(gchar **, void *),
			 void  (*negative_func)(void *))
{
  PersonaEditData *persona_data;
  GtkWidget *hbox;
  GtkWidget *label;
  GtkWidget *button;

  bs_function_enter();

  persona_data = g_malloc(sizeof(PersonaEditData));
  persona_data->dialog = gtk_dialog_new();
  persona_data->user_data = user_data;
  persona_data->positive_func = positive_func;
  persona_data->negative_func = negative_func;

  gtk_window_set_title (GTK_WINDOW (persona_data->dialog), "Edit Persona");
  gtk_window_position (GTK_WINDOW (persona_data->dialog), GTK_WIN_POS_CENTER);
  gtk_signal_connect (GTK_OBJECT (persona_data->dialog), "delete_event",
		      GTK_SIGNAL_FUNC (persona_edit_destroy), persona_data);

  persona_data->nick = gtk_entry_new();
  hbox = create_labelled_widget("Nick:*", 70, persona_data->nick, 0);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(persona_data->dialog)->vbox), hbox, FALSE, FALSE, 10);
  gtk_widget_show (persona_data->nick);

  persona_data->real_name = gtk_entry_new();
  hbox = create_labelled_widget("Real Name:*", 70, persona_data->real_name, 0);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(persona_data->dialog)->vbox), hbox, FALSE, FALSE, 10);
  gtk_widget_show (persona_data->real_name);

  persona_data->away_nick = gtk_entry_new();
  hbox = create_labelled_widget("Away Nick:", 70, persona_data->away_nick, 0);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(persona_data->dialog)->vbox), hbox, FALSE, FALSE, 10);
  gtk_widget_show (persona_data->away_nick);
  
  label = gtk_label_new("* - required field");
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(persona_data->dialog)->vbox), label, FALSE, FALSE, 10);
  gtk_widget_show (label);
  
  button = gtk_button_new_with_label ("Ok");
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(persona_data->dialog)->action_area), button, TRUE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (persona_edit_ok),
		      persona_data);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (button);
  gtk_widget_show (button);

  if (negative_func) {
    button = gtk_button_new_with_label ("Cancel");
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG(persona_data->dialog)->action_area), button, TRUE, FALSE, 0);
    gtk_signal_connect(GTK_OBJECT (button), "clicked",
		       GTK_SIGNAL_FUNC (persona_edit_cancel),
		       persona_data);
    gtk_widget_show (button);
  }

  if (persona_strs) {
    gtk_entry_set_text( GTK_ENTRY(persona_data->nick), persona_strs[0] );
    gtk_entry_set_text( GTK_ENTRY(persona_data->real_name), persona_strs[1] );
    gtk_entry_set_text( GTK_ENTRY(persona_data->away_nick), persona_strs[2] );
  }

  gtk_grab_add (persona_data->dialog);
  gtk_widget_show (persona_data->dialog);

  bs_function_leave();
  return;
}

void persona_prefs_add_ok(gchar **persona_strs, void *data)
{
  PersonaPrefsData *persona_prefs = data;

  persona_prefs_clist_add_entry(persona_prefs, persona_strs, -1);
  persona_prefs_clist_set_column_widths(persona_prefs);
  persona_prefs->changed = TRUE;
  bs_function_leave();
  return;
}

void persona_prefs_add(GtkWidget *widget, PersonaPrefsData *persona_prefs)
{

  bs_function_enter();

  persona_edit_dialog(persona_prefs, NULL, persona_prefs_add_ok, NULL);

  bs_function_leave();
  return;
}

void persona_prefs_edit_cancel(void *data)
{
  /* Place holder so we get a cancel button */
  return;
}

void persona_prefs_edit_ok(gchar **persona_strs, void *data)
{
  PersonaPrefsData *persona_prefs = data;
  gint position;

  position = persona_prefs->selected_row;
  gtk_clist_freeze(GTK_CLIST(persona_prefs->clist));
  gtk_clist_remove(GTK_CLIST(persona_prefs->clist), persona_prefs->selected_row);
  persona_prefs_clist_add_entry(persona_prefs, persona_strs, position);
  persona_prefs_clist_set_column_widths(persona_prefs);
  gtk_clist_select_row( GTK_CLIST(persona_prefs->clist), position, -1);
  gtk_clist_thaw(GTK_CLIST(persona_prefs->clist));
  persona_prefs->changed = TRUE;

  bs_function_leave();
  return;
}

void persona_prefs_edit(GtkWidget *widget, PersonaPrefsData *persona_prefs)
{
  gchar *persona_strs[3];

  bs_function_enter();

  gtk_clist_get_text( GTK_CLIST(persona_prefs->clist), persona_prefs->selected_row, 0, &persona_strs[0]);
  gtk_clist_get_text( GTK_CLIST(persona_prefs->clist), persona_prefs->selected_row, 1, &persona_strs[1]);
  gtk_clist_get_text( GTK_CLIST(persona_prefs->clist), persona_prefs->selected_row, 2, &persona_strs[2]);
  
  persona_edit_dialog(persona_prefs, persona_strs, persona_prefs_edit_ok, persona_prefs_edit_cancel);

  bs_function_leave();
  return;
}

void persona_prefs_delete(GtkWidget *widget, PersonaPrefsData *persona_prefs)
{
  bs_function_enter();

  gtk_clist_remove(GTK_CLIST(persona_prefs->clist), persona_prefs->selected_row);
  persona_prefs->changed = TRUE;

  bs_function_leave();
  return;
}

void persona_prefs_select_callback (GtkWidget       *widget,
				    gint              row,
				    gint              column, 
				    GdkEventButton   *bevent,
				    PersonaPrefsData *persona_prefs)
{
  bs_function_enter();

  persona_prefs->selected_row = row;

  bs_function_leave();
  return;
}

GtkWidget *persona_prefs_create(void **pref_data)
{
  PersonaPrefsData *persona_prefs;
  GtkWidget *vbox;
  GtkWidget *hbox;
  GtkWidget *sub_vbox;
  GtkWidget *button;
  GtkWidget *arrow;

  static char *column_titles[] = { "Nick", "Real Name", "Away Nick" };

  persona_prefs =  g_malloc(sizeof(PersonaPrefsData));
  persona_prefs->changed = FALSE;

  vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER (vbox), 4);
  gtk_widget_show (vbox);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER (hbox), 4);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 3);
  gtk_widget_show (hbox);

  persona_prefs->clist = gtk_clist_new_with_titles(3, column_titles);
  gtk_clist_column_titles_passive ( GTK_CLIST(persona_prefs->clist) );
  gtk_clist_set_selection_mode (GTK_CLIST (persona_prefs->clist), GTK_SELECTION_BROWSE);
  gtk_clist_set_policy (GTK_CLIST (persona_prefs->clist),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_signal_connect (GTK_OBJECT (persona_prefs->clist),
		      "select_row",
		      (GtkSignalFunc) persona_prefs_select_callback,
		      persona_prefs);
  gtk_box_pack_start(GTK_BOX(hbox), persona_prefs->clist, TRUE, TRUE, 0);
  gtk_widget_show (persona_prefs->clist);

  persona_prefs->widths[0] = gdk_string_width(persona_prefs->clist->style->font, column_titles[0]);
  persona_prefs->widths[1] = gdk_string_width(persona_prefs->clist->style->font, column_titles[1]);
  persona_prefs->widths[2] = gdk_string_width(persona_prefs->clist->style->font, column_titles[2]);

  sub_vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER (sub_vbox), 4);
  gtk_box_pack_start(GTK_BOX(hbox), sub_vbox, FALSE, TRUE, 0);
  gtk_widget_show (sub_vbox);

  button = gtk_button_new ();
  gtk_box_pack_start(GTK_BOX(sub_vbox), button, FALSE, TRUE, 0);
  arrow = gtk_arrow_new (GTK_ARROW_UP, GTK_SHADOW_OUT);
  gtk_container_add (GTK_CONTAINER (button), arrow);
  gtk_widget_show (arrow);
  gtk_signal_connect( GTK_OBJECT (button), "clicked",
                      GTK_SIGNAL_FUNC (persona_prefs_up),
		      persona_prefs );
  gtk_widget_show (button);

  button = gtk_button_new ();
  gtk_box_pack_start(GTK_BOX(sub_vbox), button, FALSE, TRUE, 0);
  arrow = gtk_arrow_new (GTK_ARROW_DOWN, GTK_SHADOW_OUT);
  gtk_container_add (GTK_CONTAINER (button), arrow);
  gtk_widget_show (arrow);
  gtk_signal_connect( GTK_OBJECT (button), "clicked",
                      GTK_SIGNAL_FUNC (persona_prefs_down),
		      persona_prefs );
  gtk_widget_show (button);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER (hbox), 4);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, TRUE, 3);
  gtk_widget_show (hbox);

  button = gtk_button_new_with_label("Add");
  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, TRUE, 3);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (persona_prefs_add),
		      persona_prefs);
  gtk_widget_show (button);

  button = gtk_button_new_with_label("Edit");
  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, TRUE, 3);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (persona_prefs_edit),
		      persona_prefs);
  gtk_widget_show (button);

  button = gtk_button_new_with_label("Delete");
  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, TRUE, 3);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (persona_prefs_delete),
		      persona_prefs);
  gtk_widget_show (button);

  *pref_data = persona_prefs;

  bs_function_leave();
  return(vbox);
}

gint persona_prefs_save(void *pref_data)
{
  PersonaPrefsData *persona_prefs = pref_data;
  char *nick, *real_name, *away_nick;
  guint row;

  bs_function_enter();

  if (persona_prefs->changed == TRUE) {
    persona_clear();
    for (row = 0 ; row < GTK_CLIST(persona_prefs->clist)->rows ; row++) {
      gtk_clist_get_text( GTK_CLIST(persona_prefs->clist), row, 0, &nick);
      gtk_clist_get_text( GTK_CLIST(persona_prefs->clist), row, 1, &real_name);
      gtk_clist_get_text( GTK_CLIST(persona_prefs->clist), row, 2, &away_nick);
      persona_add(g_strdup(nick), g_strdup(real_name), g_strdup(away_nick));
    }
    persona_save();
  }

  bs_function_leave();
  return(TRUE);
}

void persona_prefs_reset(void *pref_data)
{
  PersonaPrefsData *persona_prefs = pref_data;
  GSList *list_entry;
  Persona *persona;
  gchar **persona_strs;
  gint row;

  bs_function_enter();

  gtk_clist_clear( GTK_CLIST(persona_prefs->clist) );

  list_entry = personae;
  while (list_entry) {
    persona = (Persona *) list_entry->data;
    persona_strs = persona_to_strs(persona, persona_prefs);
    row = persona_prefs_clist_add_entry(persona_prefs, persona_strs, -1);
    persona_to_strs_free(persona_strs);
    gtk_clist_set_row_data ( GTK_CLIST(persona_prefs->clist), row, persona );
    list_entry = g_slist_next(list_entry);
  }

  persona_prefs_clist_set_column_widths(persona_prefs);
  
  bs_function_leave();
  return;
}
