/* Bezerk
 * Copyright (C) 1998 Tony Gale.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <string.h>
#include <pwd.h>
#include <unistd.h>
#include <sys/stat.h>

#include <gtk/gtk.h>
#include <glib.h>

#include "servers.h"
#include "persona.h"
#include "prefs.h"
#include "bezerk.h"
#include "dialogs.h"
#include "ch_utils.h"
#include "irc.h"
#include "util.h"
#include "debug.h"

GSList *connections=NULL;
GSList *networks=NULL;
GSList *servers=NULL;
char *servers_dir;

GSList *fav_servers = NULL;

/* void toolbars_set_favs(GList *favs_list) */
/* { */
/*   Connection *connection; */
/*   GSList *list_entry; */

/*   list_entry = connections; */
/*   while (list_entry) { */
/*     connection = (Connection *) list_entry->data; */
/*     if (connection->console) { */
/*       gtk_combo_set_popdown_strings (GTK_COMBO(BEZ_CHANNEL_WINDOW(connection->console)->toolbar->combo), */
/* 				     favs_list); */
/*     } */
/*     list_entry = g_slist_next(list_entry); */
/*   } */
/* } */

Network *network_find(char *name)
{
  Network *network;
  GSList *list_entry;

  bs_function_enter();

  list_entry = networks;
  while (list_entry) {
    network = (Network *) list_entry->data;
    if ( !strcmp(network->name, name)) {
      bs_function_leave();
      return(network);
    }
    list_entry = g_slist_next(list_entry);
  }

  bs_function_leave();
  return(NULL);
}

Network *network_add(char *name)
{
  Network *new_network;

  bs_function_enter();

  new_network = g_malloc(sizeof(Network));
  new_network->name = g_strdup(name);
  networks = g_slist_append(networks, new_network);

  bs_function_leave();
  return(new_network);
}

GList *networks_get_list()
{
  static GList *networks_list = NULL;
  GList *list_entry=NULL;
  GSList *networks_entry;
  Network *network;

  bs_function_enter();

  if ( networks_list) {
    list_entry = networks_list;
    while (list_entry) {
      g_free(list_entry->data);
      list_entry = g_list_next(list_entry);
    }
    g_list_free(networks_list);
    networks_list = NULL;
  }

  networks_entry = networks;
  while (networks_entry) {
    network = (Network *) networks_entry->data;
    list_entry = g_list_append(list_entry, g_strdup(network->name));
    networks_entry = g_slist_next(networks_entry);
  }

  bs_function_leave();
  return(list_entry);
} 

void networks_clear()
{
  Network *network;

  bs_function_enter();

  while (networks) {
    network = (Network *) networks->data;
    networks = g_slist_remove(networks, network);
    g_free(network->name);
    g_free(network);
  }

  bs_function_leave();
  return;
}

gint servers_compare(Server *a, Server *b)
{
  int retval;

  if ( (retval = strcmp(a->network->name, b->network->name)) != 0) {
    return(retval);
  } else {
    return(strcasecmp(a->name, b->name));
  }
}

gint servers_find_insert_position(char *net_name, char *name)
{
  Network *network;
  Server *server;
  GSList *list_entry;
  gint position = 0;

  network = g_malloc(sizeof(Network));
  network->name = g_strdup(net_name);
  
  server = g_malloc(sizeof(Server));
  server->network = network;
  server->name = name;

  list_entry = servers;
  while (list_entry) {
    if (servers_compare(server, (Server *) list_entry->data) <= 0) {
      return(position);
    }
    list_entry = g_slist_next(list_entry);
    position++;
  }

  bs_function_leave();
  return(position);
}
  
gint favs_compare(Server *a, Server *b)
{
  if ( !a ) return(-1);
  if ( !b ) return(1);

  return (a->favourite - b->favourite);
}

gint servers_add(Network *network,
		 char    *server,
		 char    *low_port,
		 char    *high_port,
		 char    *fav,
		 char    *descr)
{
  Server *new_server;
  gint position;

  bs_function_enter();

  new_server = g_malloc(sizeof(Server));
  new_server->network = network;
  new_server->name = g_strdup(server);
  new_server->low_port = atoi(low_port);
  new_server->high_port = atoi(high_port);
  new_server->favourite = atoi(fav);
  new_server->descr = g_strdup(descr);

  if (new_server->favourite >= 0) {
    fav_servers = g_slist_insert_sorted(fav_servers, new_server, (GCompareFunc) favs_compare);
  }

  servers = g_slist_insert_sorted(servers, new_server, 
				  (GCompareFunc) servers_compare);

  position = g_slist_position(servers, g_slist_find(servers, new_server));

  bs_function_leave();
  return(position);
}

/* Server *servers_get_fav(int fav_num) */
/* { */
/*   if ( (fav_num >= 0) && (fav_num <= 4) ) { */
/*     return(fav_servers[fav_num]); */
/*   } */
/*   return(NULL); */
/* } */

GList *servers_get_fav_list()
{
  static GList *fav_list = NULL;
  GList *list_entry=NULL;
  GSList *fav_entry;
  Server *server;

  bs_function_enter();

  if ( fav_list) {
    list_entry = fav_list;
    while (list_entry) {
      g_free(list_entry->data);
      list_entry = g_list_next(list_entry);
    }
    g_list_free(fav_list);
    fav_list = NULL;
  }

  fav_entry = fav_servers;
  while (fav_entry) {
    server = (Server *) fav_entry->data;
    /* TODO: use of description should depend upon user pref (see below) */
    list_entry = g_list_append(list_entry, g_strdup(server->descr));
    fav_entry = g_slist_next(fav_entry);
  }

  bs_function_leave();
  return(list_entry);
} 

Server *servers_find(char *object)
{
  GSList *list_entry;
  Server *candidate;

  bs_function_enter();

  list_entry = servers;
  while (list_entry) {
    candidate = (Server *) list_entry->data;
    /* TODO: use of description should depend upon user pref (see above) */
    if ( !strcmp(candidate->descr, object) ) {
        bs_function_leave();
	return(candidate);
    }
    list_entry = g_slist_next(list_entry);
  }

  bs_function_leave();
  return(NULL);
}

void servers_clear()
{
  Server *server;

  bs_function_enter();

  g_slist_free(fav_servers);
  fav_servers = NULL;

  while (servers) {
    server = (Server *) servers->data;
    servers = g_slist_remove(servers, server);
    g_free(server->name);
    g_free(server->descr);
    g_free(server);
  }

  bs_function_leave();
  return;
} 

void servers_save()
{
  FILE *servers_fd;
  char buff[BUFFLEN];
  GSList *list_entry;
  Server *server;
  char *network;

  bs_function_enter();

  if (!servers) {
      bs_function_leave();
      return;
  }

  g_snprintf(buff, BUFFLEN, "%s/%s", servers_dir, SERVERS_FILE);
  if ( (servers_fd = fopen(buff, "w")) == NULL) {
    bs_printf(3, "Error saving servers to %s", buff);
    bs_function_leave();
    return;
  }
  
  fputs("# Server Settings\n", servers_fd);
  fputs("# DO NOT EDIT - Automatically Generated\n", servers_fd);
  
  list_entry = servers;
  server = (Server *) list_entry->data;
  network = server->network->name;
  fputs(":", servers_fd);
  fputs(network, servers_fd);
  fputs("\n", servers_fd);
  while(list_entry) {
    server = (Server *) list_entry->data;
    if ( strcmp(network, server->network->name) ) {
      network = server->network->name;
      fputs(":", servers_fd);
      fputs(network, servers_fd);
      fputs("\n", servers_fd);
    }      
    g_snprintf(buff, BUFFLEN, "%s %d %d %d %s\n", server->name, server->low_port,
	       server->high_port, server->favourite, server->descr);
    fputs(buff, servers_fd);
    list_entry = g_slist_next(list_entry);
  }

  fclose(servers_fd);
  bs_function_leave();
  return;
}

void servers_write_default(char *file_name)
{
  Network *current_network;

  bs_function_enter();

  current_network = network_add("EFnet");
  servers_add(current_network, "efnet.demon.co.uk", "6667", "-1", "-1", "EFnet: demon.uk");
  servers_add(current_network, "irc.emory.net", "6667", "-1", "0", "EFnet: emory.net");
  servers_add(current_network, "irc.colorado.edu", "6667", "-1", "1", "EFnet: colorado");
  current_network = network_add("Undernet");
  servers_add(current_network, "london.uk.eu.undernet.org", "6667", "-1", "4", "Undernet: London");
  servers_add(current_network, "undernet.mindspring.com", "6667", "-1", "5", "Undernet: mindspring.com");
  current_network = network_add("IRCnet");
  servers_add(current_network, "chat.btinternet.com", "6667", "-1", "-1", "IRCnet: bt.uk");
  servers_add(current_network, "irc.stealth.net", "6667", "-1", "2", "IRCnet: stealth.net");
  servers_add(current_network, "irc.anet.com", "6667", "-1", "3", "IRCnet: anet.com");

  servers_save();

  bs_function_leave();
  return;
}

gint servers_init()
{
  int uid;
  struct stat stat_buf;
  struct passwd *pwdent;
  FILE *servers_fd;
  char buff[BUFFLEN];
  char *server, *low_port, *high_port, *fav, *descr;
  Network *current_network = NULL;

  bs_function_enter();

  /* Get the UID of the current user */
  uid = getuid();

 /* Get the password entry using the UID */
  if ((pwdent = getpwuid(uid)) == NULL) {
    g_error("Can't find password file entry");
  }

  g_snprintf(buff, BUFFLEN, "%s/.bezerk", pwdent->pw_dir);
  servers_dir = g_strdup(buff);

  if (stat(servers_dir, &stat_buf) == -1) {
    if (mkdir(servers_dir, S_IRUSR | S_IWUSR | S_IXUSR) == -1) {
      g_error("Creating bezerk servers directory");
    }
  } else if ( !S_ISDIR(stat_buf.st_mode) ) {
    g_error("~/.bezerk is not a directory");
  }

  g_snprintf(buff, BUFFLEN, "%s/%s", servers_dir, SERVERS_FILE);
  if ( (servers_fd = fopen(buff, "r")) == NULL) {
    if (stat(buff, &stat_buf) == -1) {
      servers_write_default(buff);
      bs_function_leave();
      return(TRUE);
    }
    bs_function_leave();
    return(FALSE);
  }  

  while ( !feof(servers_fd) ) {
    if ( !fgets(buff, BUFFLEN, servers_fd) ) {
      continue;
    }
    if ( *buff == '#' ) {
      continue;
    }
    if ( buff[strlen(buff)-1] == '\n' ) {
      buff[strlen(buff)-1] = '\0';
    }
    if ( *buff == ':' ) {
      current_network = network_add(buff+1);
      continue;
    }
    server = bs_strtok(buff, " ");
    low_port = bs_strtok(NULL, " ");
    high_port = bs_strtok(NULL, " ");
    fav = bs_strtok(NULL, " ");
    descr = bs_strtok(NULL, "\0");
    if ( !server || !low_port ) {
      continue;
    }
    if (!current_network) {
      current_network = network_add("General");
    }
    servers_add(current_network, server, low_port, high_port, fav, descr);
  }
  fclose(servers_fd);

  bs_function_leave();
  return(TRUE);
}

gint connection_destroy(Connection *connection)
{
  GSList *list_entry;
  ChannelInfo *channel_info;

  bs_function_enter();

  if (!connection) {
    bs_function_leave();
    return( connections ? TRUE : FALSE );
  }

  if (connection->nick) {
    g_free(connection->nick);
  }
  if (connection->oldnick) {
    g_free(connection->oldnick);
  }
  if (connection->name) {
    g_free(connection->name);
  }
  if (connection->server) {
    g_free(connection->server);
  }

/*   list_entry = connection->channels; */
  while(connection->channels) {
    channel_info = (ChannelInfo *) connection->channels->data;
    if ( channel_info->window != BEZ_CHANNEL_WINDOW(connection->console)) {
      /* This will free the appropriate memory */
      gtk_widget_destroy( GTK_WIDGET(channel_info->window->window) );
      connection->channels = g_slist_remove(connection->channels, channel_info);
    } else {
      connection->channels = remove_channel(connection->channels, channel_info->name);
    }
/*     list_entry = g_slist_next(list_entry); */
  }
/*   gtk_widget_destroy(BEZ_CHANNEL_WINDOW(connection->console)->window); */
  g_slist_free(connection->channels);

  list_entry = connection->messages;
  while(list_entry) {
    gtk_widget_destroy( BEZ_MESSAGE_WINDOW(list_entry->data)->window );
    list_entry = g_slist_next(list_entry);
  }
  g_slist_free(connection->messages);
  connections = g_slist_remove(connections, connection);
  g_free(connection);

  bs_function_leave();
  return( connections ? TRUE : FALSE );
}

Connection *connection_create(Server *server_ptr, int sd, char *server, int port, BezWindow *console)
{
  Connection *connection;
  Persona *persona;

  bs_function_enter();
  
  if ( (sd < 0) || !server || (port < 0) || !IS_BEZ_WINDOW(console) ) {
      bs_function_leave();
      return(NULL);
  }

  connection = g_malloc(sizeof(Connection));
  connection->server_ptr = server_ptr;
  connection->sd = sd;
  /* TODO: This will need to change later */
  persona = persona_get(0);
  connection->nick = g_strdup( persona->nick );
  connection->name = g_strdup( persona->real_name );
  connection->server = g_strdup(server);
  connection->port = port;
  connection->mode[0] = '\0';
  connection->lag[0] = '\0';
  connection->status = NOTREG;
  connection->oldnick = NULL;
  connection->console = console;
  connection->channels = NULL;
  connection->messages = NULL;
  connections = g_slist_append(connections, connection);

  bs_function_leave();
  return(connection);
}

Connection *connection_find_by_sd(int sd)
{
  Connection *target;
  static Connection *last_found=NULL;
  GSList *list_entry;

  bs_function_enter();

  if (last_found && (last_found->sd == sd) ) {
    bs_function_leave();
    return(last_found);
  }

  list_entry = connections;
  while ( list_entry ) {
    target = (Connection *) list_entry->data;
    if (target->sd == sd) {
      last_found = target;
      bs_function_leave();
      return(target);
    }
    list_entry = g_slist_next(list_entry);
  }

  bs_function_leave();
  return(NULL);
}

Connection *connection_find_by_name(char *server_name)
{
  Connection *target;
  GSList *list_entry;

  list_entry = connections;
  while ( list_entry ) {
    target = (Connection *) list_entry->data;
    /* TODO: use of description should depend upon user pref (see above) */
    if ( target->server_ptr && !strcmp(server_name, target->server_ptr->descr) ) {
      bs_function_leave();
      return(target);
    }
    list_entry = g_slist_next(list_entry);
  }

  bs_function_leave();
  return(NULL);
}

/* ------------------------------------------------------------------- */
/* Servers Preferences Functions                                       */
/* ------------------------------------------------------------------- */


static char *column_title[] = { "Network", "Server", "Low Port", "High Port", "Favourite", "Description" };

gchar **server_to_strs(Server *server, ServersPrefsData *servers_prefs)
{
  gchar **server_strs;
  gchar buff[BUFFLEN];

  server_strs = g_malloc(sizeof(gchar *) *6);
  server_strs[0] = g_strdup(server->network->name);
  server_strs[1] = g_strdup(server->name);
  g_snprintf(buff, BUFFLEN, "%d", server->low_port);
  server_strs[2] = g_strdup(buff);
  if (server->high_port >= 0) {
    g_snprintf(buff, BUFFLEN, "%d", server->high_port);
  } else {
    buff[0]='\0';
  }
  server_strs[3] = g_strdup(buff);
  if (server->favourite >= 0) {
    g_snprintf(buff, BUFFLEN, "%d", server->favourite);
  } else {
    buff[0]='\0';
  }
  server_strs[4] = g_strdup(buff);
  server_strs[5] = g_strdup(server->descr);

  return(server_strs);
}

void server_to_strs_free(gchar **server_strs)
{
  g_free(server_strs[0]);
  g_free(server_strs[1]);
  g_free(server_strs[2]);
  g_free(server_strs[3]);
  g_free(server_strs[4]);
  g_free(server_strs);
}

void server_prefs_clist_set_column_widths( ServersPrefsData  *servers_prefs )
{
  gtk_clist_set_column_width(GTK_CLIST(servers_prefs->clist), 0, servers_prefs->network_width);
  gtk_clist_set_column_width(GTK_CLIST(servers_prefs->clist), 1, servers_prefs->server_width);
  gtk_clist_set_column_width(GTK_CLIST(servers_prefs->clist), 2, servers_prefs->hp_width);
  gtk_clist_set_column_width(GTK_CLIST(servers_prefs->clist), 3, servers_prefs->lp_width);
  gtk_clist_set_column_width(GTK_CLIST(servers_prefs->clist), 4, servers_prefs->fav_width);
  gtk_clist_set_column_width(GTK_CLIST(servers_prefs->clist), 5, servers_prefs->descr_width);
}
  
void server_prefs_clist_add_entry(ServersPrefsData  *servers_prefs,
				  gchar            **server_strs,
				  gint               position)
{

  servers_prefs->network_width = MAX(servers_prefs->network_width,
				     gdk_string_width(servers_prefs->clist->style->font, server_strs[0]));
  servers_prefs->server_width = MAX(servers_prefs->server_width,
				     gdk_string_width(servers_prefs->clist->style->font, server_strs[1]));
  servers_prefs->lp_width = MAX(servers_prefs->lp_width,
				     gdk_string_width(servers_prefs->clist->style->font, server_strs[2]));
  servers_prefs->hp_width = MAX(servers_prefs->hp_width,
				     gdk_string_width(servers_prefs->clist->style->font, server_strs[3]));
  servers_prefs->fav_width = MAX(servers_prefs->fav_width,
				     gdk_string_width(servers_prefs->clist->style->font, server_strs[4]));
  servers_prefs->descr_width = MAX(servers_prefs->descr_width,
				     gdk_string_width(servers_prefs->clist->style->font, server_strs[5]));

  if (position < 0) {
    gtk_clist_append( GTK_CLIST(servers_prefs->clist) , server_strs);
  } else {
    gtk_clist_insert( GTK_CLIST(servers_prefs->clist), position, server_strs);
  }

  bs_function_leave();
  return;
}

void server_edit_favs_up(GtkWidget *widget, ServerFavsData *favs_data)
{
  gchar *row_strings[3];
  gpointer row_data;
  gint selection;

  if ( favs_data->selected_row == 0 ) {
    bs_function_enter();
    return;
  }

  favs_data->changed = TRUE;

  selection = favs_data->selected_row;
  gtk_clist_freeze(  GTK_CLIST(favs_data->clist) );
  gtk_clist_get_text( GTK_CLIST(favs_data->clist), favs_data->selected_row, 0, &row_strings[0] );
  gtk_clist_get_text( GTK_CLIST(favs_data->clist), favs_data->selected_row, 1, &row_strings[1] );
  gtk_clist_get_text( GTK_CLIST(favs_data->clist), favs_data->selected_row, 2, &row_strings[2] );
  row_data = gtk_clist_get_row_data( GTK_CLIST(favs_data->clist), favs_data->selected_row );

  /* Need to do it in this order, otherwise the strings are destroyed by the remove */
  gtk_clist_insert( GTK_CLIST(favs_data->clist), favs_data->selected_row-1, row_strings);
  gtk_clist_set_row_data( GTK_CLIST(favs_data->clist), favs_data->selected_row-1, row_data);
  gtk_clist_remove( GTK_CLIST(favs_data->clist), favs_data->selected_row+1 );
  gtk_clist_select_row( GTK_CLIST(favs_data->clist), selection-1, -1);
  gtk_clist_thaw(  GTK_CLIST(favs_data->clist) );

  bs_function_enter();

  return;
}

void server_edit_favs_down(GtkWidget *widget, ServerFavsData *favs_data)
{
  gchar *row_strings[3];
  gpointer row_data;

  if ( favs_data->selected_row == (GTK_CLIST(favs_data->clist)->rows-1) ) {
    bs_function_enter();
    return;
  }

  favs_data->changed = TRUE;

  gtk_clist_freeze(  GTK_CLIST(favs_data->clist) );
  gtk_clist_get_text( GTK_CLIST(favs_data->clist), favs_data->selected_row, 0, &row_strings[0] );
  gtk_clist_get_text( GTK_CLIST(favs_data->clist), favs_data->selected_row, 1, &row_strings[1] );
  gtk_clist_get_text( GTK_CLIST(favs_data->clist), favs_data->selected_row, 2, &row_strings[2] );
  row_data = gtk_clist_get_row_data( GTK_CLIST(favs_data->clist), favs_data->selected_row );

  /* Need to do it in this order, otherwise the strings are destroyed by the remove */
  gtk_clist_insert( GTK_CLIST(favs_data->clist), favs_data->selected_row+2, row_strings);
  gtk_clist_set_row_data( GTK_CLIST(favs_data->clist), favs_data->selected_row+2, row_data);
  gtk_clist_remove( GTK_CLIST(favs_data->clist), favs_data->selected_row );
  gtk_clist_select_row( GTK_CLIST(favs_data->clist), favs_data->selected_row+1, -1);
  gtk_clist_thaw(  GTK_CLIST(favs_data->clist) );

  bs_function_enter();

  return;
}

void server_edit_favs_ok(GtkWidget *widget, ServerFavsData *favs_data)
{
  Server *server;
  gint row;

  bs_function_enter();

  if (favs_data->changed) {
    g_slist_free(fav_servers);
    fav_servers = NULL;

    for (row = 0 ; row < GTK_CLIST(favs_data->clist)->rows ; row++) {
      server = (Server *) gtk_clist_get_row_data( GTK_CLIST(favs_data->clist), row );
      server->favourite = row;
      fav_servers = g_slist_append(fav_servers, server);
    }

    if (favs_data->servers_prefs) {
      favs_data->servers_prefs->changed = TRUE;
      servers_prefs_reset(favs_data->servers_prefs);
    }
/*     toolbars_set_favs(servers_get_fav_list()); */
  }

  gtk_widget_destroy(favs_data->dialog);
  
  bs_function_leave();
  return;
}

void server_edit_favs_cancel(GtkWidget *widget, ServerFavsData *favs_data)
{

  bs_function_enter();

  gtk_widget_destroy(favs_data->dialog);
  
  bs_function_leave();
  return;
}

void servers_edit_favs_select_callback (GtkWidget       *widget,
					gint              row,
					gint              column, 
					GdkEventButton   *bevent,
					ServerFavsData *favs_data)
{
  bs_function_enter();

  favs_data->selected_row = row;

  bs_function_leave();
  return;
}

void servers_edit_favs_destroy(GtkWidget *widget, ServerFavsData *favs_data)
{
  bs_function_enter();

  gtk_grab_remove(favs_data->dialog);

  bs_function_leave();
  return;
}

void servers_edit_favourites(GtkWidget *widget, ServersPrefsData *servers_prefs)
{
  ServerFavsData *favs_data;
  GtkWidget *hbox;
  GtkWidget *vbox;
  GtkWidget *arrow;
  GtkWidget *button;
  GSList *fav_list;
  gchar *row_data[3];
  gint row_widths[3];
  gint row;
  Server *server;

  static char *column_titles[] = { "Network", "Server", "Description"};

  favs_data = g_malloc(sizeof(ServerFavsData));
  favs_data->servers_prefs = servers_prefs;
  favs_data->changed = FALSE;

  favs_data->dialog = gtk_dialog_new();
  gtk_widget_set_usize(favs_data->dialog, 400, 350);
  gtk_window_set_title (GTK_WINDOW (favs_data->dialog), "Favourites");
  gtk_window_position (GTK_WINDOW (favs_data->dialog), GTK_WIN_POS_CENTER);
  gtk_signal_connect (GTK_OBJECT (favs_data->dialog), "destroy",
		      GTK_SIGNAL_FUNC (servers_edit_favs_destroy), favs_data);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER (hbox), 4);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(favs_data->dialog)->vbox), hbox, TRUE, TRUE, 3);
  gtk_widget_show (hbox);

  favs_data->clist = gtk_clist_new_with_titles(3, column_titles);
  gtk_clist_column_titles_passive ( GTK_CLIST(favs_data->clist) );
  gtk_clist_set_selection_mode (GTK_CLIST (favs_data->clist), GTK_SELECTION_BROWSE);
  gtk_clist_set_policy (GTK_CLIST (favs_data->clist),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_signal_connect (GTK_OBJECT (favs_data->clist),
		      "select_row",
		      (GtkSignalFunc) servers_edit_favs_select_callback,
		      favs_data);
  gtk_box_pack_start(GTK_BOX(hbox), favs_data->clist, TRUE, TRUE, 0);
/*   gtk_widget_realize(favs_data->clist); */
  gtk_widget_show (favs_data->clist);

  row_widths[0] = gdk_string_width(favs_data->clist->style->font, column_titles[0]);
  row_widths[1] = gdk_string_width(favs_data->clist->style->font, column_titles[1]);
  row_widths[2] = gdk_string_width(favs_data->clist->style->font, column_titles[2]);

  fav_list = fav_servers;
  while(fav_list) {
    server = (Server *) fav_list->data;
    row_data[0] = server->network->name;
    row_data[1] = server->name;
    row_data[2] = server->descr;
    row_widths[0] = MAX(row_widths[0], gdk_string_width(favs_data->clist->style->font, row_data[0]) );
    row_widths[1] = MAX(row_widths[1], gdk_string_width(favs_data->clist->style->font, row_data[1]) );
    row_widths[2] = MAX(row_widths[2], gdk_string_width(favs_data->clist->style->font, row_data[2]) );
    row = gtk_clist_append( GTK_CLIST(favs_data->clist) , row_data);
    gtk_clist_set_row_data ( GTK_CLIST(favs_data->clist), row, server );
    fav_list = g_slist_next(fav_list);
  }

  gtk_clist_set_column_width(GTK_CLIST(favs_data->clist), 0, row_widths[0]);
  gtk_clist_set_column_width(GTK_CLIST(favs_data->clist), 1, row_widths[1]);
  gtk_clist_set_column_width(GTK_CLIST(favs_data->clist), 2, row_widths[2]);

  vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER (vbox), 4);
  gtk_box_pack_start (GTK_BOX (hbox), vbox, FALSE, TRUE, 3);
  gtk_widget_show (vbox);

  button = gtk_button_new ();
  gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, TRUE, 3);
  arrow = gtk_arrow_new (GTK_ARROW_UP, GTK_SHADOW_OUT);
  gtk_container_add (GTK_CONTAINER (button), arrow);
  gtk_widget_show (arrow);
  gtk_signal_connect( GTK_OBJECT (button), "clicked",
                      GTK_SIGNAL_FUNC (server_edit_favs_up),
		      favs_data );
  gtk_widget_show (button);
 
  button = gtk_button_new ();
  gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, TRUE, 3);
  arrow = gtk_arrow_new (GTK_ARROW_DOWN, GTK_SHADOW_OUT);
  gtk_container_add (GTK_CONTAINER (button), arrow);
  gtk_widget_show (arrow);
  gtk_signal_connect( GTK_OBJECT (button), "clicked",
                      GTK_SIGNAL_FUNC (server_edit_favs_down),
		      favs_data );
  gtk_widget_show (button);

  button = gtk_button_new_with_label ("Ok");
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(favs_data->dialog)->action_area), button, TRUE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (server_edit_favs_ok),
		      favs_data);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (button);
  gtk_widget_show (button);

  button = gtk_button_new_with_label ("Cancel");
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(favs_data->dialog)->action_area), button, TRUE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT (button), "clicked",
		     GTK_SIGNAL_FUNC (server_edit_favs_cancel),
		     favs_data);
  gtk_widget_show (button);

  gtk_grab_add(favs_data->dialog);
  gtk_widget_show(favs_data->dialog);

  bs_function_leave();
  return;
}

void server_edit_destroy(GtkWidget *widget, ServerEditData *server_data)
{
  bs_function_enter();

  if (!server_data) {
    bs_function_leave();
    return;
  }

  if (server_data->negative_func) {
    server_data->negative_func(server_data->user_data);
  }

  g_free(server_data);

  bs_function_leave();
  return;
}

void server_edit_ok(GtkWidget *widget, ServerEditData *server_data)
{
  gchar **server_strs;

  bs_function_enter();

  if (server_data->positive_func) {
    server_strs = g_malloc(sizeof(gchar *) *6);
    server_strs[0] = g_strdup(gtk_entry_get_text( GTK_ENTRY(GTK_COMBO(server_data->combo)->entry)) );
    server_strs[1] = g_strdup(gtk_entry_get_text( GTK_ENTRY(server_data->name) ));
    server_strs[2] = g_strdup(gtk_entry_get_text( GTK_ENTRY(server_data->low_port) ));
    server_strs[3] = g_strdup(gtk_entry_get_text( GTK_ENTRY(server_data->high_port) ));
    if (GTK_TOGGLE_BUTTON(server_data->fav)->active) {
      server_strs[4] = g_strdup("0");
    } else {
      server_strs[4] = g_strdup("");
    }
    server_strs[5] = g_strdup(gtk_entry_get_text( GTK_ENTRY(server_data->descr) ));
    if (!strlen(server_strs[0]) || !strlen(server_strs[1]) ||
	!strlen(server_strs[2]) || !strlen(server_strs[5])) {
      gdk_beep();
      return;
    }
    server_data->positive_func(server_strs, server_data->user_data);
    server_to_strs_free(server_strs);
  }

  server_data->negative_func = NULL;
  gtk_widget_destroy(server_data->dialog);

  bs_function_leave();
  return;
}

void server_edit_cancel(GtkWidget *widget, ServerEditData *server_data)
{

  bs_function_enter();

  if (server_data->negative_func) {
    server_data->negative_func(server_data->user_data);
    server_data->negative_func = NULL;
  }

  gtk_widget_destroy(server_data->dialog);
  
  bs_function_leave();
  return;
}

void server_edit_dialog(void   *user_data,
			gchar **server_strs,
			void  (*positive_func)(gchar **, void *),
			void  (*negative_func)(void *))
{
  ServerEditData *server_data;
  GtkWidget *sub_hbox;
  GtkWidget *hbox;
  GtkWidget *label;
  GtkWidget *button;
  GList *cbitems = NULL;

  bs_function_enter();

  server_data = g_malloc(sizeof(ServerEditData));
  server_data->dialog = gtk_dialog_new();
  server_data->user_data = user_data;
  server_data->positive_func = positive_func;
  server_data->negative_func = negative_func;

  gtk_window_set_title (GTK_WINDOW (server_data->dialog), "Edit Server");
  gtk_window_position (GTK_WINDOW (server_data->dialog), GTK_WIN_POS_CENTER);
  gtk_signal_connect (GTK_OBJECT (server_data->dialog), "destroy",
		      GTK_SIGNAL_FUNC (server_edit_destroy), server_data);

  cbitems = networks_get_list();

  server_data->combo = gtk_combo_new ();
  gtk_combo_disable_activate (GTK_COMBO (server_data->combo));
  gtk_combo_set_popdown_strings (GTK_COMBO (server_data->combo), cbitems);
  hbox = create_labelled_widget("Network:*", 70, server_data->combo, 0);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(server_data->dialog)->vbox), hbox, FALSE, FALSE, 10);
  gtk_widget_show (server_data->combo);

  server_data->name = gtk_entry_new();
  hbox = create_labelled_widget("Server:*", 70, server_data->name, 0);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(server_data->dialog)->vbox), hbox, FALSE, FALSE, 10);
  gtk_widget_show (server_data->name);

  sub_hbox = gtk_hbox_new(FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER (sub_hbox), 4);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(server_data->dialog)->vbox), sub_hbox, FALSE, FALSE, 10);
  gtk_widget_show (sub_hbox);

  server_data->low_port = gtk_entry_new();
  hbox = create_labelled_widget("Port:*", 60, server_data->low_port, 50);
  gtk_box_pack_start (GTK_BOX (sub_hbox), hbox, FALSE, FALSE, 10);
  gtk_widget_show (server_data->low_port);

  server_data->high_port = gtk_entry_new();
  hbox = create_labelled_widget("to", 0, server_data->high_port, 50);
  gtk_box_pack_start (GTK_BOX (sub_hbox), hbox, FALSE, FALSE, 10);
  gtk_widget_show (server_data->name);

  server_data->descr = gtk_entry_new_with_max_length(25);
  hbox = create_labelled_widget("Description:*", 70, server_data->descr, 0);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(server_data->dialog)->vbox), hbox, FALSE, FALSE, 10);
  gtk_widget_show (server_data->name);
  
  server_data->fav = gtk_check_button_new_with_label ( "Add to favourites" );
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(server_data->dialog)->vbox), server_data->fav, FALSE, FALSE, 3);
  GTK_WIDGET_UNSET_FLAGS(server_data->fav, GTK_CAN_FOCUS);
  gtk_widget_show (server_data->fav);

  label = gtk_label_new("* - required field");
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(server_data->dialog)->vbox), label, FALSE, FALSE, 10);
  gtk_widget_show (label);

  button = gtk_button_new_with_label ("Ok");
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(server_data->dialog)->action_area), button, TRUE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (server_edit_ok),
		      server_data);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (button);
  gtk_widget_show (button);

  button = gtk_button_new_with_label ("Cancel");
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(server_data->dialog)->action_area), button, TRUE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT (button), "clicked",
		     GTK_SIGNAL_FUNC (server_edit_cancel),
		     server_data);
  gtk_widget_show (button);

  if (server_strs) {
    gtk_entry_set_text( GTK_ENTRY(GTK_COMBO(server_data->combo)->entry), server_strs[0] );
    gtk_entry_set_text( GTK_ENTRY(server_data->name), server_strs[1] );
    gtk_entry_set_text( GTK_ENTRY(server_data->low_port), server_strs[2] );
    gtk_entry_set_text( GTK_ENTRY(server_data->high_port), server_strs[3] );
    gtk_entry_set_text( GTK_ENTRY(server_data->descr), server_strs[5] );
    if ( strlen(server_strs[4]) ) {
      gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(server_data->fav), TRUE);
    }
  }

  gtk_grab_add (server_data->dialog);
  gtk_widget_show (server_data->dialog);

  bs_function_leave();
  return;
}

void server_prefs_add_ok(gchar **server_strs, void *data)
{
  ServersPrefsData *servers_prefs = data;
  gint position;

  position = servers_find_insert_position(server_strs[0], server_strs[1]);
  server_prefs_clist_add_entry(servers_prefs, server_strs, position);
  server_prefs_clist_set_column_widths(servers_prefs);
  servers_prefs->changed = TRUE;
  bs_function_leave();
  return;
}

void servers_prefs_add(GtkWidget *widget, ServersPrefsData *servers_prefs)
{

  bs_function_enter();

  server_edit_dialog(servers_prefs, NULL, server_prefs_add_ok, NULL);

  bs_function_leave();
  return;
}

void server_prefs_edit_ok(gchar **server_strs, void *data)
{
  ServersPrefsData *servers_prefs = data;
  gint position;
  char *fav;

  if (strlen(server_strs[4]) ) {
    g_free(server_strs[4]);
    gtk_clist_get_text( GTK_CLIST(servers_prefs->clist), servers_prefs->selected_row, 4, &fav);
    server_strs[4] = g_strdup(fav);
  }

  gtk_clist_freeze(GTK_CLIST(servers_prefs->clist));
  gtk_clist_remove(GTK_CLIST(servers_prefs->clist), servers_prefs->selected_row);
  position = servers_find_insert_position(server_strs[0], server_strs[1]);
  server_prefs_clist_add_entry(servers_prefs, server_strs, position);
  server_prefs_clist_set_column_widths(servers_prefs);
  gtk_clist_select_row( GTK_CLIST(servers_prefs->clist), position, -1);
  gtk_clist_thaw(GTK_CLIST(servers_prefs->clist));
  servers_prefs->changed = TRUE;

  bs_function_leave();
  return;
}

void servers_prefs_edit(GtkWidget *widget, ServersPrefsData *servers_prefs)
{
  gchar *server_strs[6];

  bs_function_enter();

  gtk_clist_get_text( GTK_CLIST(servers_prefs->clist), servers_prefs->selected_row, 0,
		      &server_strs[0]);
  gtk_clist_get_text( GTK_CLIST(servers_prefs->clist), servers_prefs->selected_row, 1,
		      &server_strs[1]);
  gtk_clist_get_text( GTK_CLIST(servers_prefs->clist), servers_prefs->selected_row, 2,
		      &server_strs[2]);
  gtk_clist_get_text( GTK_CLIST(servers_prefs->clist), servers_prefs->selected_row, 3,
		      &server_strs[3]);
  gtk_clist_get_text( GTK_CLIST(servers_prefs->clist), servers_prefs->selected_row, 4,
		      &server_strs[4]);
  gtk_clist_get_text( GTK_CLIST(servers_prefs->clist), servers_prefs->selected_row, 5,
		      &server_strs[5]);
  
  server_edit_dialog(servers_prefs, server_strs, server_prefs_edit_ok, NULL);

  bs_function_leave();
  return;
}

void servers_prefs_delete(GtkWidget *widget, ServersPrefsData *servers_prefs)
{
  bs_function_enter();

  gtk_clist_remove(GTK_CLIST(servers_prefs->clist), servers_prefs->selected_row);
  servers_prefs->changed = TRUE;

  bs_function_leave();
  return;
}

void servers_prefs_select_callback (GtkWidget       *widget,
				    gint              row,
				    gint              column, 
				    GdkEventButton   *bevent,
				    ServersPrefsData *servers_prefs)
{
  bs_function_enter();

  servers_prefs->selected_row = row;

  bs_function_leave();
  return;
}

GtkWidget *servers_prefs_create(void **pref_data)
{
  ServersPrefsData *servers_prefs;
/*   GtkWidget *vbox; */
/*   GtkWidget *hbox; */
  GtkWidget *button;

  servers_prefs = g_malloc(sizeof(ServersPrefsData));
  servers_prefs->changed = FALSE;

  servers_prefs->vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER (servers_prefs->vbox), 4);
  gtk_widget_show (servers_prefs->vbox);

  servers_prefs->clist = gtk_clist_new_with_titles(6, column_title);
  gtk_clist_column_titles_passive ( GTK_CLIST(servers_prefs->clist) );
  gtk_clist_set_selection_mode (GTK_CLIST (servers_prefs->clist), GTK_SELECTION_BROWSE);
  gtk_clist_set_policy (GTK_CLIST (servers_prefs->clist),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_signal_connect (GTK_OBJECT (servers_prefs->clist),
		      "select_row",
		      (GtkSignalFunc) servers_prefs_select_callback,
		      servers_prefs);
  gtk_box_pack_start(GTK_BOX(servers_prefs->vbox), servers_prefs->clist, TRUE, TRUE, 0);
  gtk_widget_show (servers_prefs->clist);

  servers_prefs->hbox = gtk_hbox_new(FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER (servers_prefs->hbox), 4);
  gtk_box_pack_start (GTK_BOX (servers_prefs->vbox), servers_prefs->hbox, FALSE, TRUE, 3);
  gtk_widget_show (servers_prefs->hbox);

  button = gtk_button_new_with_label("Add");
  gtk_box_pack_start (GTK_BOX (servers_prefs->hbox), button, FALSE, TRUE, 3);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (servers_prefs_add),
		      servers_prefs);
  gtk_widget_show (button);

  button = gtk_button_new_with_label("Edit");
  gtk_box_pack_start (GTK_BOX (servers_prefs->hbox), button, FALSE, TRUE, 3);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (servers_prefs_edit),
		      servers_prefs);
  gtk_widget_show (button);

  button = gtk_button_new_with_label("Delete");
  gtk_box_pack_start (GTK_BOX (servers_prefs->hbox), button, FALSE, TRUE, 3);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (servers_prefs_delete),
		      servers_prefs);
  gtk_widget_show (button);

  button = gtk_button_new_with_label("Favourites");
  gtk_box_pack_end (GTK_BOX (servers_prefs->hbox), button, FALSE, TRUE, 3);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (servers_edit_favourites),
		      servers_prefs);
  gtk_widget_show (button);

  *pref_data = servers_prefs;

  bs_function_leave();
  return(servers_prefs->vbox);
}

gint servers_prefs_save(void *pref_data)
{
  ServersPrefsData *servers_prefs = pref_data;
  Network *network=NULL;
  gchar *current_network;
  gchar *server_strs[6];
  guint row;

  bs_function_leave();

  current_network = g_strdup("");

  if (servers_prefs->changed == TRUE) {
    servers_clear();
    networks_clear();
    for (row = 0 ; row < GTK_CLIST(servers_prefs->clist)->rows ; row++) {
      gtk_clist_get_text( GTK_CLIST(servers_prefs->clist), row, 0, &server_strs[0]);
      gtk_clist_get_text( GTK_CLIST(servers_prefs->clist), row, 1, &server_strs[1]);
      gtk_clist_get_text( GTK_CLIST(servers_prefs->clist), row, 2, &server_strs[2]);
      gtk_clist_get_text( GTK_CLIST(servers_prefs->clist), row, 3, &server_strs[3]);
      gtk_clist_get_text( GTK_CLIST(servers_prefs->clist), row, 4, &server_strs[4]);
      gtk_clist_get_text( GTK_CLIST(servers_prefs->clist), row, 5, &server_strs[5]);
      if ( strcmp(current_network, server_strs[0]) ) {
	g_free(current_network);
	current_network = g_strdup(server_strs[0]);
	network = network_add(current_network);
      }
      servers_add(network, server_strs[1], server_strs[2], server_strs[3],
		 ( (strlen(server_strs[4]) > 0) ? server_strs[4] : "-1" ),
		 server_strs[5]);
    }
    g_free(current_network);
    servers_save();
  }

  return(TRUE);
}

void servers_prefs_reset(void *pref_data)
{
  ServersPrefsData *servers_prefs = pref_data;
  GSList *list_entry;
  Server *server;
  gchar **server_strs;

  servers_prefs->network_width = gdk_string_width(servers_prefs->clist->style->font, column_title[0]);
  servers_prefs->server_width = gdk_string_width(servers_prefs->clist->style->font, column_title[1]);
  servers_prefs->lp_width = gdk_string_width(servers_prefs->clist->style->font, column_title[2]);
  servers_prefs->hp_width = gdk_string_width(servers_prefs->clist->style->font, column_title[3]);
  servers_prefs->fav_width = gdk_string_width(servers_prefs->clist->style->font, column_title[4]);
  servers_prefs->descr_width = gdk_string_width(servers_prefs->clist->style->font, column_title[5]);

  gtk_widget_set_usize(servers_prefs->hbox, servers_prefs->network_width+servers_prefs->server_width+
		       servers_prefs->lp_width+servers_prefs->hp_width+servers_prefs->fav_width+
		       servers_prefs->descr_width, -1);

  gtk_clist_clear( GTK_CLIST(servers_prefs->clist) );

  list_entry = servers;
  while (list_entry) {
    server = (Server *) list_entry->data;
    server_strs = server_to_strs(server, servers_prefs);
    server_prefs_clist_add_entry(servers_prefs, server_strs, -1);
    server_to_strs_free(server_strs);
    list_entry = g_slist_next(list_entry);
  }

  server_prefs_clist_set_column_widths(servers_prefs);

  bs_function_leave();
  return;
}
