/** 
 * @namespace   biewlib
 * @file        biewlib/bbio.c
 * @brief       This file contains implementation of Buffering binary input/ouput routines.
 * @version     -
 * @remark      this source file is part of Binary vIEW project (BIEW).
 *              The Binary vIEW (BIEW) is copyright (C) 1995 Nick Kurshev.
 *              All rights reserved. This software is redistributable under the
 *              licence given in the file "Licence.en" ("Licence.ru" in russian
 *              translation) distributed in the BIEW archive.
 * @note        Requires POSIX compatible development system
 *
 * @author      Nick Kurshev
 * @since       1995
 * @note        Development, fixes and improvements
 * @todo        Support for preemptive memory allocation and multiple buffers for
 *              one opened stream
**/
#include <sys/stat.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <limits.h>

#include "biewlib/bbio.h"
#include "biewlib/pmalloc.h"

#define MK_FPTR(x) (x)

#define IS_CACHE_VALID(obj) ((obj)->MBuffer && !((obj)->optimize & BIO_OPT_NOCACHE))
/*
   This struct is ordered as it documented in Athlon manual
   Publication # 22007 Rev: D
*/
typedef struct tagBFILE
{
 unsigned long   FilePos;   /**< current logical position in file */
 unsigned long   FLength;   /**< real length of the file */
 unsigned long   FBufStart; /**< logical position of mirror the buffer onto file */
 char *          MBuffer;   /**< NULL - not buffered i/o */
 char *          FileName;  /**< Real file name of opened file */
 unsigned        openmode;  /**< mode,that OsOpen this file */
 unsigned        MBufLen;   /**< length data, actually contains in buffer */
 unsigned        MBufSize;  /**< real size of buffer */
 int             optimize;  /**< seek optimization */
 int             handle;    /**< file handle */
 tBool           updated;   /**< True if buffer contains data, that not pesent in file */
}BFILE;

struct tagBFILE bNull =
{
  0L, 0L,
  0L, NULL,
  NULL, UINT_MAX,
  0, 0, 0,
  2, True
};

/* notes: all function with prefix=>__ assume, that buffer present */

#define __isOutOfBuffer(obj,pos)\
        (int)((unsigned long)pos < ((BFILE *)obj)->FBufStart ||\
              (unsigned long)pos >= ((BFILE *)obj)->FBufStart +\
               ((BFILE *)obj)->MBufSize)

#define __isOutOfContents(obj,pos)\
        (int)((unsigned long)pos < ((BFILE *)obj)->FBufStart ||\
              (unsigned long)pos >= ((BFILE *)obj)->FBufStart +\
               ((BFILE *)obj)->MBufLen)

static tBool __NEAR__ __FASTCALL__ __fill(BFILE  *obj,long pos)
{
  void * mbuff;
  unsigned long remaind;
  tBool ret;
  if(pos < 0) pos = 0;
  if((unsigned)pos > obj->FLength) { pos = obj->FLength; obj->MBufLen = 0; ret = False; }
  else
  {
    obj->FBufStart = pos;
    remaind = obj->FLength - pos;
    obj->MBufLen = (unsigned long)obj->MBufSize < remaind ? obj->MBufSize : (unsigned)remaind;
    mbuff = MK_FPTR(obj->MBuffer);
    __OsSeek(obj->handle,pos,SEEKF_START);
    ret = (unsigned)__OsRead(obj->handle,mbuff,obj->MBufLen) == obj->MBufLen;
  }
  return ret;
}

static tBool __NEAR__ __FASTCALL__ __flush(BFILE  *obj)
{
  void * mbuff;
  tBool ret;
  ret = True;
  if(!obj->updated)
  {
    mbuff = MK_FPTR(obj->MBuffer);
    __OsSeek(obj->handle,obj->FBufStart,SEEKF_START);
    if(obj->MBufLen) ret = (unsigned)__OsWrite(obj->handle,mbuff,obj->MBufLen) == obj->MBufLen;
    if(ret)          obj->updated = True;
  }
  return ret;
}

static tBool __NEAR__ __FASTCALL__ __seek(BFILE  *obj,long pos,int origin)
{
 tBool ret,rret;
 ret = True;
 switch(origin)
 {
    case BIO_SEEK_SET: break;
    case BIO_SEEK_CUR: pos += obj->FilePos; break;
    default:           pos += obj->FLength;
 }
 if((unsigned long)pos > (unsigned long)obj->FLength)
 {
    pos = obj->FLength;
    ret = False;
 }
 obj->FilePos = pos;
 if(__isOutOfBuffer(obj,pos))
 {
    __flush(obj);
    switch(obj->optimize & BIO_OPT_DIRMASK)
    {
      default:
      case BIO_OPT_DB:         break;
      case BIO_OPT_RANDOM:     pos -= obj->MBufSize / 2;
                               break;
      case BIO_OPT_BACKSCAN:   pos -= (obj->MBufSize - 1);
                               break;
      case BIO_OPT_RFORWARD:   pos -= obj->MBufSize / 10;
                               break;
      case BIO_OPT_RBACKSCAN:  pos -= (obj->MBufSize/10)*9;
                               break;
    }
    if(obj->FilePos < obj->FLength)
    {
      rret = __fill(obj,pos);
      if(ret) ret = rret;
    }
    else
    {
      obj->FBufStart = obj->FilePos;
      obj->MBufLen = 0;
    }
 }
 return ret;
}

static tBool founderr;

static unsigned char __NEAR__ __FASTCALL__ __getc(BFILE  *obj)
{
  char * buff = MK_FPTR(obj->MBuffer);
  unsigned buffpos;
  tBool ret;
  unsigned char ch;
  ret = True;
  founderr = False;
  if(__isOutOfContents(obj,obj->FilePos)) ret = __seek(obj,obj->FilePos,SEEKF_START);
  ch = -1;
  if(obj->MBufLen && ret && obj->FilePos <= obj->FLength)
  {
    buffpos = (unsigned)(obj->FilePos - obj->FBufStart);
    if(buffpos < obj->MBufLen)  ch = buff[buffpos];
    if(obj->FilePos < obj->FLength) obj->FilePos++;
  }
  else { errno = EACCES; ret = False; }
  if(ret == False) founderr = True;
  return ch;
}

static tBool __NEAR__ __FASTCALL__ __putc(BFILE  *obj,unsigned char ch)
{
  char *  buff;
  unsigned buffpos;
  tBool ret;
  if(((obj->openmode & (FO_READWRITE | FO_WRITEONLY)) == FO_WRITEONLY) ||
     ((obj->openmode & (FO_READWRITE | FO_WRITEONLY)) == FO_READWRITE)
    )
  {
    ret = True;
    buff = MK_FPTR(obj->MBuffer);
    if(__isOutOfBuffer(obj,obj->FilePos)) __seek(obj,obj->FilePos,SEEKF_START);
    buffpos = (unsigned)(obj->FilePos - obj->FBufStart);
    buff[buffpos++] = ch;
    obj->updated = False;
    if(obj->MBufLen < buffpos) if(obj->MBufLen < obj->MBufSize) obj->MBufLen = buffpos;
    if(obj->FLength <= obj->FBufStart + buffpos) obj->FLength = obj->FBufStart + buffpos;
    obj->FilePos++;
  }
  else { errno = EACCES; ret = False; }
  return ret;
}

static tBool __NEAR__ __FASTCALL__ __getbuff(BFILE  *obj,char * buff,unsigned cbBuff)
{
#if 1 /* most optimized, but large method */
  unsigned diffsize;
  unsigned MBufStart,MBufRem;
  int optimize = obj->optimize;
  tBool ret;
  ret = True;
  obj->optimize = (optimize & ~BIO_OPT_DIRMASK) | BIO_OPT_DB;
  while(cbBuff)
  {
    MBufStart = (unsigned)(obj->FilePos - obj->FBufStart);
    MBufRem = obj->MBufLen - MBufStart;
    if(!MBufRem || __isOutOfContents(obj,obj->FilePos))
    {
     ret = __seek(obj,obj->FilePos,SEEKF_START);
     if(!ret) break;
     if(obj->FilePos >= obj->FLength) break;
     MBufStart = (unsigned)(obj->FilePos - obj->FBufStart);
     MBufRem = obj->MBufLen - MBufStart;
    }
    if(!MBufRem)
    {
       errno = E2BIG; /* fault: internal error */
       ret = False;
       break;
    }
    diffsize = min(MBufRem,cbBuff);
    memcpy(buff,&obj->MBuffer[MBufStart],diffsize);
    obj->FilePos += diffsize;
    if(obj->FilePos > obj->FLength) obj->FilePos = obj->FLength;
    cbBuff -= diffsize;
    buff += diffsize;
  }
  obj->optimize = optimize;
  return ret;
#else /* look more safety */
  unsigned i;
  tBool ret;
  ret = True;
  for(i = 0;i < cbBuff;i++)
  {
    buff[i] = __getc(obj);
    if(founderr) { ret = False; break; }
  }
  return ret;
#endif
}

static tBool __NEAR__ __FASTCALL__ __putbuff(BFILE  *obj,const char * buff,unsigned cbBuff)
{
#if 1 /* most optimized, but large method */
  unsigned diffsize;
  unsigned MBufStart,MBufRem;
  int optimize = obj->optimize;
  tBool ret;
  obj->optimize = (optimize & ~BIO_OPT_DIRMASK) | BIO_OPT_DB;
  if(((obj->openmode & (FO_READWRITE | FO_WRITEONLY)) == FO_WRITEONLY) ||
     ((obj->openmode & (FO_READWRITE | FO_WRITEONLY)) == FO_READWRITE)
    )
  {
    ret = True;
    while(cbBuff)
    {
      MBufStart = (unsigned)(obj->FilePos - obj->FBufStart);
      MBufRem = obj->MBufSize - MBufStart;
      if(!MBufRem || __isOutOfBuffer(obj,obj->FilePos))
      {
       ret = __seek(obj,obj->FilePos,SEEKF_START);
       if(!ret) break;
       MBufStart = (unsigned)(obj->FilePos - obj->FBufStart);
       MBufRem = obj->MBufSize - MBufStart;
      }
      if(!MBufRem)
      {
         errno = E2BIG; /* fault: internal error */
         ret = False;
         break;
      }
      diffsize = min(MBufRem,cbBuff);
      memcpy(&obj->MBuffer[MBufStart],buff,diffsize);
      obj->updated = False;
      if(obj->MBufLen < MBufStart + diffsize) obj->MBufLen = MBufStart + diffsize;
      if(obj->FLength < obj->FilePos + diffsize) obj->FLength = obj->FilePos + diffsize;
      obj->FilePos += diffsize;
      cbBuff -= diffsize;
      buff += diffsize;
    }
  }
  else { errno = EACCES; ret = False; }
  obj->optimize = optimize;
  return ret;
#else /* look more safety */
  unsigned i;
  tBool ret = False;
  for(i = 0;i < cbBuff;i++)
  {
    ret = __putc(obj,buff[i]);  if(!ret) break;
  }
  return ret;
#endif
}

BGLOBAL  __FASTCALL__ bioOpen(const char * fname,unsigned openmode,unsigned bSize)
{
 BFILE  * bFile;
 BGLOBAL ret = NULL;
 int handle;
 unsigned len;
 ret = PMalloc(sizeof(BFILE));
 if(ret)
 {
   bFile = MK_FPTR(ret);
   bFile->FilePos = 0;
   bFile->FBufStart = 0L;
   bFile->openmode = openmode;
   bFile->updated  = True;
   bFile->optimize = BIO_OPT_DB;
   if(!(bFile->FileName = PMalloc(strlen(fname)+1)))
   {
     PFREE(bFile);
     return &bNull;
   }
   strcpy(bFile->FileName,fname);
   handle = __OsOpen(fname,openmode);
   if(handle == -1)
   {
     PFREE(bFile->FileName);
     PFREE(ret);
     return &bNull;
   }
   bFile->handle = handle;
   __OsSeek(handle,0L,SEEKF_END);
   bFile->FLength = __OsTell(handle);
   len = bSize;
   if(bSize == UINT_MAX) len = (unsigned)bFile->FLength;
   if(len)
   {
     if((bFile->MBuffer = PMalloc(len)) != NULL)
     {
       bFile->MBufLen = 0;
       bFile->MBufSize = len;
     }
     else
     {
       PFREE(bFile->FileName);
       __OsClose(bFile->handle);
       PFREE(ret);
       return &bNull;
     }
     __fill(bFile,0L);
   }
   else bFile->MBuffer = NULL;
 }
 else ret = &bNull;
 return ret;
}

tBool  __FASTCALL__ bioClose(BGLOBAL handle)
{
  BFILE * bFile = MK_FPTR(handle);
  if(bFile->handle != -1)
  {
    if(((bFile->openmode & FO_READWRITE) == FO_WRITEONLY ||
       (bFile->openmode & FO_READWRITE) == FO_READWRITE))
               __flush(bFile);
    if(bFile->handle != 2) __OsClose(bFile->handle);
  }
  if(bFile->MBuffer) { PFREE(bFile->FileName); PFREE(bFile->MBuffer); }
  PFREE(handle);
  return True;
}


tBool   __FASTCALL__ bioSeek(BGLOBAL bioFile,long pos,int orig)
{
 BFILE  *obj =MK_FPTR(bioFile);
 tBool ret;
 if(IS_CACHE_VALID(obj)) ret = __seek(obj,pos,orig);
 else
 {
   ret = __OsSeek(obj->handle,pos,orig) != 0;
   if(pos == 0L && orig == SEEKF_START) ret = True;
 }
 return ret;
}

unsigned long  __FASTCALL__ bioTell(BGLOBAL bioFile)
{
  BFILE  *obj = MK_FPTR(bioFile);
  return IS_CACHE_VALID(obj) ? obj->FilePos : __OsTell(obj->handle);
}

uint8_t __FASTCALL__ bioReadByte(BGLOBAL bioFile)
{
  BFILE  *obj = MK_FPTR(bioFile);
  uint8_t ret;
  if(IS_CACHE_VALID(obj)) ret = __getc(obj);
  else
    if(__OsRead(obj->handle,&ret,sizeof(uint8_t)) != sizeof(uint8_t)) ret = -1;
  return ret;
}

uint16_t __FASTCALL__ bioReadWord(BGLOBAL bioFile)
{
  BFILE  *obj = MK_FPTR(bioFile);
  uint16_t ret;
  if(IS_CACHE_VALID(obj))
  {
    if(!__getbuff(obj,(void *)&ret,sizeof(uint16_t))) ret = -1;
  }
  else
    if(__OsRead(obj->handle,&ret,sizeof(uint16_t)) != sizeof(uint16_t)) ret = -1;
  return ret;
}

uint32_t __FASTCALL__ bioReadDWord(BGLOBAL bioFile)
{
  BFILE  *obj =MK_FPTR(bioFile);
  uint32_t ret;
  if(IS_CACHE_VALID(obj))
  {
    if(!__getbuff(obj,(void *)&ret,sizeof(uint32_t))) ret = -1;
  }
  else
    if(__OsRead(obj->handle,&ret,sizeof(uint32_t)) != sizeof(uint32_t)) ret = -1;
  return ret;
}

tBool __FASTCALL__  bioReadBuffer(BGLOBAL bioFile,void * buffer,unsigned cbBuffer)
{
  BFILE  *obj = MK_FPTR(bioFile);
  return IS_CACHE_VALID(obj) ? __getbuff(obj,buffer,cbBuffer) :
                              ((unsigned)__OsRead(obj->handle,buffer,cbBuffer)) == cbBuffer;
}

tBool __FASTCALL__  bioWriteByte(BGLOBAL bioFile,uint8_t bVal)
{
  BFILE  *obj = MK_FPTR(bioFile);
  return IS_CACHE_VALID(obj) ? __putc(obj,bVal) :
                               __OsWrite(obj->handle,&bVal,sizeof(uint8_t)) == sizeof(uint8_t);
}

tBool __FASTCALL__  bioWriteWord(BGLOBAL bioFile,uint16_t wVal)
{
  BFILE  *obj = MK_FPTR(bioFile);
  return IS_CACHE_VALID(obj) ? __putbuff(obj,(void *)&wVal,sizeof(uint16_t)) :
                               __OsWrite(obj->handle,&wVal,sizeof(uint16_t)) == sizeof(uint16_t);
}

tBool __FASTCALL__  bioWriteDWord(BGLOBAL bioFile,uint32_t dwVal)
{
  BFILE  *obj = MK_FPTR(bioFile);
  return IS_CACHE_VALID(obj) ? __putbuff(obj,(void *)&dwVal,sizeof(uint32_t)) :
                               __OsWrite(obj->handle,&dwVal,sizeof(uint32_t)) == sizeof(uint32_t);
}

tBool __FASTCALL__  bioWriteBuffer(BGLOBAL bioFile,const void * buffer,unsigned cbBuffer)
{
  BFILE  *obj = MK_FPTR(bioFile);
  return IS_CACHE_VALID(obj) ? __putbuff(obj,buffer,cbBuffer) :
                               ((unsigned)__OsWrite(obj->handle,buffer,cbBuffer)) == cbBuffer;
}

tBool __FASTCALL__  bioFlush(BGLOBAL bioFile)
{
  BFILE  * obj = MK_FPTR(bioFile);
  return IS_CACHE_VALID(obj) ? __flush(obj) : True;
}

tBool __FASTCALL__  bioReRead(BGLOBAL bioFile)
{
  unsigned long fpos;
  BFILE  * obj = MK_FPTR(bioFile);
  fpos = __OsTell(obj->handle);
  __OsSeek(obj->handle,0L,SEEKF_END);
  obj->FLength = __OsTell(obj->handle);
  __OsSeek(obj->handle,fpos,SEEKF_START);
  return __fill(obj,obj->FBufStart);
}

unsigned long  __FASTCALL__  bioFLength(BGLOBAL bioFile)
{
  BFILE  * bFile = MK_FPTR(bioFile);
  return bFile->FLength;
}

tBool __FASTCALL__  bioChSize(BGLOBAL bioFile,unsigned long newsize)
{
    unsigned long length, fillsize;
    char * buf;
    BFILE  *obj = MK_FPTR(bioFile);
    unsigned  bufsize, numtowrite;
    tBool ret;

    length = obj->FLength;
    if(length >= newsize) /* truncate size */
    {
        __seek(obj, newsize, SEEKF_START);
        length = newsize;
        ret = __OsTruncFile(obj->handle, length) != 0;
        obj->FLength = newsize;
        if(obj->FBufStart > obj->FLength) { obj->FBufStart = obj->FLength; obj->MBufLen = 0; }
        if(obj->FBufStart + obj->MBufLen > obj->FLength) obj->MBufLen = (unsigned)(obj->FLength - obj->FBufStart);
        ret = ret ? False : True;
    }
    else
    {
      fillsize=newsize-length;  /* increase size */
      ret = False;
      bufsize = obj->MBuffer ? obj->MBufSize : 8192;
      bufsize = (unsigned) min(fillsize,bufsize);
      if((buf = PMalloc(bufsize)) != NULL)
      {
        ret = True;
        memset(buf, 0, bufsize);   /* write zeros to pad file */
        bioSeek(bioFile,0L,SEEKF_END);
        do
        {
          numtowrite = (unsigned)min(bufsize,fillsize);
          if(!bioWriteBuffer(bioFile, (void * )buf, numtowrite)) { ret = False; break; }
          fillsize-=numtowrite;
        } while(fillsize);
        PFREE(buf);
      }
   }
   return ret;
}

unsigned  __FASTCALL__ bioSetOptimization(BGLOBAL bioFile,unsigned flags)
{
  BFILE  *obj = MK_FPTR(bioFile);
  unsigned ret;
  ret = obj->optimize;
  obj->optimize = flags;
  return ret;
}

unsigned  __FASTCALL__ bioGetOptimization(BGLOBAL bioFile)
{
  BFILE  *obj = MK_FPTR(bioFile);
  return obj->optimize;
}

int  __FASTCALL__ bioHandle(BGLOBAL bioFile)
{
   BFILE *obj = MK_FPTR(bioFile);
   return obj->handle;
}

char * __FASTCALL__ bioFileName(BGLOBAL bioFile)
{
  BFILE *obj = MK_FPTR(bioFile);
  return obj->FileName;
}

void * __FASTCALL__ bioBuffer(BGLOBAL bioFile)
{
  BFILE *obj = MK_FPTR(bioFile);
  return obj->MBuffer;
}

unsigned __FASTCALL__ bioBuffLen(BGLOBAL bioFile)
{
  BFILE *obj = MK_FPTR(bioFile);
  return obj->MBufLen;
}

unsigned __FASTCALL__ bioBuffPos(BGLOBAL bioFile)
{
  BFILE *obj = MK_FPTR(bioFile);
  return obj->MBufLen - (unsigned)(obj->FilePos - obj->FBufStart);
}

BGLOBAL __FASTCALL__ bioDupEx(BGLOBAL bioFile,unsigned buff_Size)
{
 BGLOBAL ret = NULL;
 int handle;
 unsigned len;
 ret = PMalloc(sizeof(BFILE));
 if(ret)
 {
   BFILE * bFile,* fromFile;
   bFile = MK_FPTR(ret);
   fromFile = MK_FPTR(bioFile);
   bFile->openmode = fromFile->openmode;
   bFile->updated  = fromFile->updated;
   bFile->optimize = fromFile->optimize;
   if(!(bFile->FileName = PMalloc(strlen(fromFile->FileName)+1)))
   {
     PFREE(bFile);
     return &bNull;
   }
   strcpy(bFile->FileName,fromFile->FileName);

   handle = __OsDupHandle(fromFile->handle);
   if(handle == -1)
   {
     PFREE(bFile->FileName);
     PFREE(ret);
     return &bNull;
   }
   bFile->handle = handle;
   bFile->FLength = fromFile->FLength;
   len = buff_Size;
   if(len)
   {
     if((bFile->MBuffer = PMalloc(len)) != NULL)
     {
       if(IS_CACHE_VALID(fromFile))
       {
         bFile->MBufLen = min(len,fromFile->MBufLen);
         bFile->MBufSize = len;
         bFile->FilePos = min(fromFile->FilePos,fromFile->FBufStart+(len/2));
         bFile->FBufStart = fromFile->FBufStart;
         memcpy(bFile->MBuffer,fromFile->MBuffer,bFile->MBufLen);
       }
       else
       {
         bFile->MBufLen = 0;
         bFile->MBufSize = 0;
         bFile->FilePos = 0L;
         bFile->FBufStart = 0;
       }
     }
     else
     {
       PFREE(bFile->FileName);
       __OsClose(bFile->handle);
       PFREE(ret);
       return &bNull;
     }
   }
   else bFile->MBuffer = 0;
 }
 else ret = &bNull;
 return ret;
}

BGLOBAL __FASTCALL__ bioDup(BGLOBAL bHandle)
{
  BFILE *fromFile = MK_FPTR(bHandle);
  return bioDupEx(bHandle,fromFile->MBufSize);
}

tBool __FASTCALL__ bioEOF(BGLOBAL bHandle)
{
  BFILE  *obj = MK_FPTR(bHandle);
  unsigned long fpos;
  if(IS_CACHE_VALID(obj)) fpos = obj->FilePos;
  else                    fpos = __OsTell(obj->handle);
  return fpos >= obj->FLength;
}
