/** 
 * @namespace   biew_plugins_auto
 * @file        plugins/bin/elf386.h
 * @brief       This file defines standard ELF types, structures, and macros.
 * @version     -
 * @remark      Copyright (C) 1995, 1996, 1997, 1998 Free Software Foundation,
 *              Inc. This file is part of the GNU C Library.
 *              The GNU C Library is free software; you can redistribute it and/or
 *              modify it under the terms of the GNU Library General Public License as
 *              published by the Free Software Foundation; either version 2 of the
 *              License, or (at your option) any later version.
 *              The GNU C Library is distributed in the hope that it will be useful,
 *              but WITHOUT ANY WARRANTY; without even the implied warranty of
 *              MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *              Library General Public License for more details.
 *              You should have received a copy of the GNU Library General Public
 *              License along with the GNU C Library; see the file COPYING.LIB.  If not,
 *              write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 *              Boston, MA 02111-1307, USA.
 * @note        Requires POSIX compatible development system
 *
 * @author      GNU FSF
 * @since       1995
**/
#ifndef __ELF_INC
#define __ELF_INC

#ifndef __SYS_DEP_H
#include "_sys_dep.h"
#endif

#ifdef __HAVE_PRAGMA_PACK__
#pragma pack(1)
#endif

#define EI_NIDENT	16		/**< Size of e_ident[] */
/** Fields in e_ident[] */

#define EI_MAG0		0		/**< File identification byte 0 index */
#define ELFMAG0		0x7F		/**< Magic number byte 0 */

#define EI_MAG1		1		/**< File identification byte 1 index */
#define ELFMAG1		'E'		/**< Magic number byte 1 */

#define EI_MAG2		2		/**< File identification byte 2 index */
#define ELFMAG2		'L'		/**< Magic number byte 2 */

#define EI_MAG3		3		/**< File identification byte 3 index */
#define ELFMAG3		'F'		/**< Magic number byte 3 */

#define EI_CLASS	4		/**< File class */
#define ELFCLASSNONE	0		/**< Invalid class */
#define ELFCLASS32	1		/**< 32-bit objects */
#define ELFCLASS64	2		/**< 64-bit objects */

#define EI_DATA		5		/**< Data encoding */
#define ELFDATANONE	0		/**< Invalid data encoding */
#define ELFDATA2LSB	1		/**< 2's complement, little endian */
#define ELFDATA2MSB	2		/**< 2's complement, big endian */

#define EI_VERSION	6		/**< File version */

#define EI_PAD		7		/**< Start of padding bytes */


/** Values for e_type, which identifies the object file type */

#define ET_NONE		0		/**< No file type */
#define ET_REL		1		/**< Relocatable file */
#define ET_EXEC		2		/**< Executable file */
#define ET_DYN		3		/**< Shared object file */
#define ET_CORE		4		/**< Core file */
#define	ET_NUM		5		/**< Number of defined types.  */
#define ET_LOPROC	0xFF00U		/**< Processor-specific */
#define ET_HIPROC	0xFFFFU		/**< Processor-specific */

/** Values for e_machine, which identifies the architecture */

#define EM_NONE		0	/**< No machine */
#define EM_M32		1	/**< AT&T WE 32100 */
#define EM_SPARC	2	/**< SUN SPARC */
#define EM_386		3	/**< Intel 80386 */
#define EM_68K		4	/**< Motorola m68k family */
#define EM_88K		5	/**< Motorola m88k family */
#define EM_860		7	/**< Intel 80860 */
#define EM_MIPS		8	/**< MIPS R3000 (officially, big-endian only) */

#define EM_MIPS_RS4_BE 10	/**< MIPS R4000 big-endian */

#define EM_SPARC64     11	/**< SPARC v9 (not official) 64-bit */

#define EM_PARISC      15	/**< HPPA */

#define EM_SPARC32PLUS 18	/**< Sun's "v8plus" */

#define EM_PPC	       20	/**< PowerPC */

#define EM_SH	       42	/**< Hitachi SH */

/** If it is necessary to assign new unofficial EM_* values, please pick large
   random numbers (0x8523, 0xa7f2, etc.) to minimize the chances of collision
   with official or non-GNU unofficial values.

   NOTE: Do not just increment the most recent number by one.
   Somebody else somewhere will do exactly the same thing, and you
   will have a collision.  Instead, pick a random number.  */

/** Cygnus PowerPC ELF backend.  Written in the absence of an ABI.  */
#define EM_CYGNUS_POWERPC 0x9025U

/** Old version of PowerPC, this should be removed shortly. */
#define EM_PPC_OLD	17


/** Cygnus M32R ELF backend.  Written in the absence of an ABI.  */
#define EM_CYGNUS_M32R 0x9041U

/** Alpha backend magic number.  Written in the absence of an ABI.  */
#define EM_ALPHA	0x9026U

/** D10V backend magic number.  Written in the absence of an ABI.  */
#define EM_CYGNUS_D10V	0x7650U



/** mn10200 and mn10300 backend magic numbers.
   Written in the absense of an ABI.  */
#define EM_CYGNUS_MN10200	0xdeadL
#define EM_CYGNUS_MN10300	0xbeefL

/** See the above comment before you add a new EM_* value here.  */

/** Values for e_version */

#define EV_NONE		0		/**< Invalid ELF version */
#define EV_CURRENT	1		/**< Current version */

/** Values for program header, p_type field */

#define	PT_NULL		0		/**< Program header table entry unused */
#define PT_LOAD		1		/**< Loadable program segment */
#define PT_DYNAMIC	2		/**< Dynamic linking information */
#define PT_INTERP	3		/**< Program interpreter */
#define PT_NOTE		4		/**< Auxiliary information */
#define PT_SHLIB	5		/**< Reserved, unspecified semantics */
#define PT_PHDR		6		/**< Entry for header table itself */
#define	PT_NUM		7		/**< Number of defined types.  */
#define PT_LOPROC	0x70000000UL	/**< Processor-specific */
#define PT_HIPROC	0x7FFFFFFFUL	/**< Processor-specific */

/** Program segment permissions, in program header p_flags field */

#define PF_X		(1 << 0)	/**< Segment is executable */
#define PF_W		(1 << 1)	/**< Segment is writable */
#define PF_R		(1 << 2)	/**< Segment is readable */
#define PF_MASKPROC	0xF0000000UL	/**< Processor-specific reserved bits */

/** Values for section header, sh_type field */

#define SHT_NULL	0		/**< Section header table entry unused */
#define SHT_PROGBITS	1		/**< Program specific (private) data */
#define SHT_SYMTAB	2		/**< Link editing symbol table */
#define SHT_STRTAB	3		/**< A string table */
#define SHT_RELA	4		/**< Relocation entries with addends */
#define SHT_HASH	5		/**< A symbol hash table */
#define SHT_DYNAMIC	6		/**< Information for dynamic linking */
#define SHT_NOTE	7		/**< Information that marks file */
#define SHT_NOBITS	8		/**< Section occupies no space in file */
#define SHT_REL		9		/**< Relocation entries, no addends */
#define SHT_SHLIB	10		/**< Reserved, unspecified semantics */
#define SHT_DYNSYM	11		/**< Dynamic linking symbol table */
#define	SHT_NUM		12		/**< Number of defined types.  */

#define SHT_LOSUNW	0x6ffffffdUL	/**< Sun-specific low bound.  */

/** The next three section types are defined by Solaris, and are named
   SHT_SUNW*.  We use them in GNU code, so we also define SHT_GNU*
   versions.  */
#define SHT_SUNW_verdef	0x6ffffffdUL	/**< Versions defined by file */
#define SHT_SUNW_verneed 0x6ffffffeUL	/**< Versions needed by file */
#define SHT_SUNW_versym	0x6fffffffUL	/**< Symbol versions */

#define SHT_GNU_verdef	SHT_SUNW_verdef
#define SHT_GNU_verneed	SHT_SUNW_verneed
#define SHT_GNU_versym	SHT_SUNW_versym

#define SHT_LOPROC	0x70000000UL	/**< Processor-specific semantics, lo */
#define SHT_HIPROC	0x7FFFFFFFUL	/**< Processor-specific semantics, hi */
#define SHT_LOUSER	0x80000000UL	/**< Application-specific semantics */
#define SHT_HIUSER	0x8FFFFFFFUL	/**< Application-specific semantics */

/** Values for section header, sh_flags field */

#define SHF_WRITE	(1 << 0)	/**< Writable data during execution */
#define SHF_ALLOC	(1 << 1)	/**< Occupies memory during execution */
#define SHF_EXECINSTR	(1 << 2)	/**< Executable machine instructions */
#define SHF_MASKPROC	0xF0000000UL	/**< Processor-specific semantics */

/** Values of note segment descriptor types for core files. */

#define NT_PRSTATUS	1		/**< Contains copy of prstatus struct */
#define NT_FPREGSET	2		/**< Contains copy of fpregset struct */
#define NT_PRPSINFO	3		/**< Contains copy of prpsinfo struct */

/** Values of note segment descriptor types for object files.  
   (Only for hppa right now.  Should this be moved elsewhere?)  */

#define NT_VERSION	1		/**< Contains a version string.  */

/** These three macros disassemble and assemble a symbol table st_info field,
    which contains the symbol binding and symbol type.  The STB_ and STT_
    defines identify the binding and type. */

#define ELF_ST_BIND(val)		(((uint8_t)(val)) >> 4)
#define ELF_ST_TYPE(val)		(((uint8_t)(val)) & 0xF)
#define ELF_ST_INFO(bind,type)		((((uint8_t)(bind)) << 4) + ((((uint8_t)type)) & 0xF))

#define STN_UNDEF	0		/**< undefined symbol index */

#define STB_LOCAL	0		/**< Symbol not visible outside obj */
#define STB_GLOBAL	1		/**< Symbol visible outside obj */
#define STB_WEAK	2		/**< Like globals, lower precedence */
#define	STB_NUM		3		/**< Number of defined types.  */
#define STB_LOPROC	13		/**< Application-specific semantics */
#define STB_HIPROC	15		/**< Application-specific semantics */

#define STT_NOTYPE	0		/**< Symbol type is unspecified */
#define STT_OBJECT	1		/**< Symbol is a data object */
#define STT_FUNC	2		/**< Symbol is a code object */
#define STT_SECTION	3		/**< Symbol associated with a section */
#define STT_FILE	4		/**< Symbol gives a file name */
#define	STT_NUM		5		/**< Number of defined types.  */
#define STT_LOPROC	13		/**< Application-specific semantics */
#define STT_HIPROC	15		/**< Application-specific semantics */

/** Special section indices, which may show up in st_shndx fields, among
    other places. */

#define SHN_UNDEF	0		/**< Undefined section reference */
#define SHN_LORESERVE	0xFF00U		/**< Begin range of reserved indices */
#define SHN_LOPROC	0xFF00U		/**< Begin range of appl-specific */
#define SHN_HIPROC	0xFF1FU		/**< End range of appl-specific */
#define SHN_ABS		0xFFF1U		/**< Associated symbol is absolute */
#define SHN_COMMON	0xFFF2U		/**< Associated symbol is in common */
#define SHN_HIRESERVE	0xFFFFU		/**< End range of reserved indices */

/** relocation info handling macros */

#define ELF32_R_SYM(i)		((i) >> 8)
#define ELF32_R_TYPE(i)		((i) & 0xff)
#define ELF32_R_INFO(s,t)	(((s) << 8) + ((t) & 0xff))

#define ELF64_R_SYM(i)		((i) >> 32)
#define ELF64_R_TYPE(i)		((i) & 0xffffffff)
#define ELF64_R_INFO(s,t)	(((bfd_vma) (s) << 32) + (bfd_vma) (t))

/** Legal values for d_tag (dynamic entry type).  */

#define DT_NULL		0		/**< Marks end of dynamic section */
#define DT_NEEDED	1		/**< Name of needed library */
#define DT_PLTRELSZ	2		/**< Size in bytes of PLT relocs */
#define DT_PLTGOT	3		/**< Processor defined value */
#define DT_HASH		4		/**< Address of symbol hash table */
#define DT_STRTAB	5		/**< Address of string table */
#define DT_SYMTAB	6		/**< Address of symbol table */
#define DT_RELA		7		/**< Address of Rela relocs */
#define DT_RELASZ	8		/**< Total size of Rela relocs */
#define DT_RELAENT	9		/**< Size of one Rela reloc */
#define DT_STRSZ	10		/**< Size of string table */
#define DT_SYMENT	11		/**< Size of one symbol table entry */
#define DT_INIT		12		/**< Address of init function */
#define DT_FINI		13		/**< Address of termination function */
#define DT_SONAME	14		/**< Name of shared object */
#define DT_RPATH	15		/**< Library search path */
#define DT_SYMBOLIC	16		/**< Start symbol search here */
#define DT_REL		17		/**< Address of Rel relocs */
#define DT_RELSZ	18		/**< Total size of Rel relocs */
#define DT_RELENT	19		/**< Size of one Rel reloc */
#define DT_PLTREL	20		/**< Type of reloc in PLT */
#define DT_DEBUG	21		/**< For debugging; unspecified */
#define DT_TEXTREL	22		/**< Reloc might modify .text */
#define DT_JMPREL	23		/**< Address of PLT relocs */
#define	DT_NUM		24		/**< Number used */
#define DT_LOPROC	0x70000000UL	/**< Start of processor-specific */
#define DT_HIPROC	0x7fffffffUL	/**< End of processor-specific */
#define	DT_PROCNUM	DT_MIPS_NUM	/**< Most used by any processor */

/** The versioning entry types.  The next are defined as part of the
   GNU extension.  */
#define DT_VERSYM	0x6ffffff0UL

/** These were chosen by Sun.  */
#define	DT_VERDEF	0x6ffffffcUL	/**< Address of version definition
					   table */
#define	DT_VERDEFNUM	0x6ffffffdUL	/**< Number of version definitions */
#define	DT_VERNEED	0x6ffffffeUL	/**< Address of table with needed
					   versions */
#define	DT_VERNEEDNUM	0x6fffffffUL	/**< Number of needed versions */
#define DT_VERSIONTAGIDX(tag)	(DT_VERNEEDNUM - (tag))	/**< Reverse order! */
#define DT_VERSIONTAGNUM 16

/** Sun added these machine-independent extensions in the "processor-specific"
   range.  Be compatible.  */
#define DT_AUXILIARY    0x7ffffffdUL      /**< Shared object to load before self */
#define DT_FILTER       0x7fffffffUL      /**< Shared object to get values from */
#define DT_EXTRATAGIDX(tag)	((tByte)-((Elf386_Sword) (tag) <<1>>1)-1)
#define DT_EXTRANUM	3


/** These constants are used for the version number of a Elf386_Verdef
   structure.  */

#define VER_DEF_NONE		0
#define VER_DEF_CURRENT		1
#define VER_DEF_NUM	        2		/**< Given version number */

/** These constants appear in the vd_flags field of a Elf386_Verdef
   structure.  */

#define VER_FLG_BASE		0x1
#define VER_FLG_WEAK		0x2

/** These special constants can be found in an Elf386_Versym field.  */

#define VER_NDX_LOCAL		0
#define VER_NDX_GLOBAL		1

/** These constants are used for the version number of a Elf386_Verneed
   structure.  */

#define VER_NEED_NONE		0
#define VER_NEED_CURRENT	1
#define VER_NEED_NUM	        2		/**< Given version number */

/** This flag appears in a Versym structure.  It means that the symbol
   is hidden, and is only visible with an explicit version number.
   This is a GNU extension.  */

#define VERSYM_HIDDEN		0x8000U

/** This is the mask for the rest of the Versym information.  */

#define VERSYM_VERSION		0x7fffU

/** This is a special token which appears as part of a symbol name.  It
   indictes that the rest of the name is actually the name of a
   version node, and is not part of the actual name.  This is a GNU
   extension.  For example, the symbol name `stat@ver2' is taken to
   mean the symbol `stat' in version `ver2'.  */

#define ELF_VER_CHR	'@'


typedef struct {
  uint8_t	e_ident[16];		/**< ELF "magic number" */
  uint8_t	e_type[2];		/**< Identifies object file type */
  uint8_t	e_machine[2];		/**< Specifies required architecture */
  uint8_t	e_version[4];		/**< Identifies object file version */
  uint8_t	e_entry[4];		/**< Entry point virtual address */
  uint8_t	e_phoff[4];		/**< Program header table file offset */
  uint8_t	e_shoff[4];		/**< Section header table file offset */
  uint8_t	e_flags[4];		/**< Processor-specific flags */
  uint8_t	e_ehsize[2];		/**< ELF header size in bytes */
  uint8_t	e_phentsize[2];		/**< Program header table entry size */
  uint8_t	e_phnum[2];		/**< Program header table entry count */
  uint8_t	e_shentsize[2];		/**< Section header table entry size */
  uint8_t	e_shnum[2];		/**< Section header table entry count */
  uint8_t	e_shstrndx[2];		/**< Section header string table index */
} Elf386_External_Ehdr;

typedef struct {
  uint8_t	e_ident[16];		/**< ELF "magic number" */
  uint8_t	e_type[2];		/**< Identifies object file type */
  uint8_t	e_machine[2];		/**< Specifies required architecture */
  uint8_t	e_version[4];		/**< Identifies object file version */
  uint8_t	e_entry[8];		/**< Entry point virtual address */
  uint8_t	e_phoff[8];		/**< Program header table file offset */
  uint8_t	e_shoff[8];		/**< Section header table file offset */
  uint8_t	e_flags[4];		/**< Processor-specific flags */
  uint8_t	e_ehsize[2];		/**< ELF header size in bytes */
  uint8_t	e_phentsize[2];		/**< Program header table entry size */
  uint8_t	e_phnum[2];		/**< Program header table entry count */
  uint8_t	e_shentsize[2];		/**< Section header table entry size */
  uint8_t	e_shnum[2];		/**< Section header table entry count */
  uint8_t	e_shstrndx[2];		/**< Section header string table index */
} Elf64_External_Ehdr;

/** Program header */

typedef struct {
  uint8_t	p_type[4];		/**< Identifies program segment type */
  uint8_t	p_offset[4];		/**< Segment file offset */
  uint8_t	p_vaddr[4];		/**< Segment virtual address */
  uint8_t	p_paddr[4];		/**< Segment physical address (ignored on SystemV) */
  uint8_t	p_filesz[4];		/**< Segment size in file */
  uint8_t	p_memsz[4];		/**< Segment size in memory */
  uint8_t	p_flags[4];		/**< Segment flags */
  uint8_t	p_align[4];		/**< Segment alignment, file & memory */
} Elf386_External_Phdr;

typedef struct {
  uint8_t	p_type[4];		/**< Identifies program segment type */
  uint8_t	p_flags[4];		/**< Segment flags */
  uint8_t	p_offset[8];		/**< Segment file offset */
  uint8_t	p_vaddr[8];		/**< Segment virtual address */
  uint8_t	p_paddr[8];		/**< Segment physical address (ignored on SystemV)*/
  uint8_t	p_filesz[8];		/**< Segment size in file */
  uint8_t	p_memsz[8];		/**< Segment size in memory */
  uint8_t	p_align[8];		/**< Segment alignment, file & memory */
} Elf64_External_Phdr;

/** Section header */

typedef struct {
  uint8_t	sh_name[4];		/**< Section name, index in string tbl */
  uint8_t	sh_type[4];		/**< Type of section */
  uint8_t	sh_flags[4];		/**< Miscellaneous section attributes */
  uint8_t	sh_addr[4];		/**< Section virtual addr at execution */
  uint8_t	sh_offset[4];		/**< Section file offset */
  uint8_t	sh_size[4];		/**< Size of section in bytes */
  uint8_t	sh_link[4];		/**< Index of another section */
  uint8_t	sh_info[4];		/**< Additional section information */
  uint8_t	sh_addralign[4];	/**< Section alignment */
  uint8_t	sh_entsize[4];		/**< Entry size if section holds table */
} Elf386_External_Shdr;

typedef struct {
  uint8_t	sh_name[4];		/**< Section name, index in string tbl */
  uint8_t	sh_type[4];		/**< Type of section */
  uint8_t	sh_flags[8];		/**< Miscellaneous section attributes */
  uint8_t	sh_addr[8];		/**< Section virtual addr at execution */
  uint8_t	sh_offset[8];		/**< Section file offset */
  uint8_t	sh_size[8];		/**< Size of section in bytes */
  uint8_t	sh_link[4];		/**< Index of another section */
  uint8_t	sh_info[4];		/**< Additional section information */
  uint8_t	sh_addralign[8];	/**< Section alignment */
  uint8_t	sh_entsize[8];		/**< Entry size if section holds table */
} Elf64_External_Shdr;

/** Symbol table entry */

typedef struct {
  uint8_t	st_name[4];		/**< Symbol name, index in string tbl */
  uint8_t	st_value[4];		/**< Value of the symbol */
  uint8_t	st_size[4];		/**< Associated symbol size */
  uint8_t	st_info[1];		/**< Type and binding attributes */
  uint8_t	st_other[1];		/**< No defined meaning, 0 */
  uint8_t	st_shndx[2];		/**< Associated section index */
} Elf386_External_Sym;

typedef struct {
  uint8_t	st_name[4];		/**< Symbol name, index in string tbl */
  uint8_t	st_info[1];		/**< Type and binding attributes */
  uint8_t	st_other[1];		/**< No defined meaning, 0 */
  uint8_t	st_shndx[2];		/**< Associated section index */
  uint8_t	st_value[8];		/**< Value of the symbol */
  uint8_t	st_size[8];		/**< Associated symbol size */
} Elf64_External_Sym;

/** Note segments */

typedef struct {
  uint8_t	namesz[4];		/**< Size of entry's owner string */
  uint8_t	descsz[4];		/**< Size of the note descriptor */
  uint8_t	type[4];		/**< Interpretation of the descriptor */
  int8_t	name[1];		/**< Start of the name+desc data */
} Elf_External_Note;

/** Relocation Entries */
typedef struct {
  uint8_t r_offset[4];	/**< Location at which to apply the action */
  uint8_t	r_info[4];	/**< index and type of relocation */
} Elf386_External_Rel;

typedef struct {
  uint8_t r_offset[4];	/**< Location at which to apply the action */
  uint8_t	r_info[4];	/**< index and type of relocation */
  uint8_t	r_addend[4];	/**< Constant addend used to compute value */
} Elf386_External_Rela;

typedef struct {
  uint8_t r_offset[8];	/**< Location at which to apply the action */
  uint8_t	r_info[8];	/**< index and type of relocation */
} Elf64_External_Rel;

typedef struct {
  uint8_t r_offset[8];	/**< Location at which to apply the action */
  uint8_t	r_info[8];	/**< index and type of relocation */
  uint8_t	r_addend[8];	/**< Constant addend used to compute value */
} Elf64_External_Rela;

/** dynamic section structure */

typedef struct {
  uint8_t	d_tag[4];		/**< entry tag value */
  union {
    uint8_t	d_val[4];
    uint8_t	d_ptr[4];
  } d_un;
} Elf386_External_Dyn;

typedef struct {
  uint8_t	d_tag[8];		/**< entry tag value */
  union {
    uint8_t	d_val[8];
    uint8_t	d_ptr[8];
  } d_un;
} Elf64_External_Dyn;

/** The version structures are currently size independent.  They are
   named without a 32 or 64.  If that ever changes, these structures
   will need to be renamed.  */

/** This structure appears in a SHT_GNU_verdef section.  */

typedef struct {
  uint8_t		vd_version[2];
  uint8_t		vd_flags[2];
  uint8_t		vd_ndx[2];
  uint8_t		vd_cnt[2];
  uint8_t		vd_hash[4];
  uint8_t		vd_aux[4];
  uint8_t		vd_next[4];
} Elf_External_Verdef;

/** This structure appears in a SHT_GNU_verdef section.  */

typedef struct {
  uint8_t		vda_name[4];
  uint8_t		vda_next[4];
} Elf_External_Verdaux;

/** This structure appears in a SHT_GNU_verneed section.  */

typedef struct {
  uint8_t		vn_version[2];
  uint8_t		vn_cnt[2];
  uint8_t		vn_file[4];
  uint8_t		vn_aux[4];
  uint8_t		vn_next[4];
} Elf_External_Verneed;

/** This structure appears in a SHT_GNU_verneed section.  */

typedef struct {
  uint8_t		vna_hash[4];
  uint8_t		vna_flags[2];
  uint8_t		vna_other[2];
  uint8_t		vna_name[4];
  uint8_t		vna_next[4];
} Elf_External_Vernaux;

/** This structure appears in a SHT_GNU_versym section.  This is not a
   standard ELF structure; ELF just uses Elf386_Half.  */

typedef struct {
  uint8_t		vs_vers[2];
} Elf_External_Versym;

/** Auxiliary vector.  */

/** This vector is normally only used by the program interpreter.  The
   usual definition in an ABI supplement uses the name auxv_t.  The
   vector is not usually defined in a standard <elf.h> file, but it
   can't hurt.  We rename it to avoid conflicts.  The sizes of these
   types are an arrangement between the exec server and the program
   interpreter, so we don't fully specify them here.  */

typedef struct
{
  int32_t a_type;		/**< Entry type */
  union
    {
      int32_t a_val;		/**< Integer value */
      void *a_ptr;		/**< Pointer value */
      void (*a_fcn) (void);	/**< Function pointer value */
    } a_un;
} Elf386_auxv_t;

typedef struct
{
  int32_t a_type;		/**< Entry type */
  union
    {
      int32_t a_val;		/**< Integer value */
      void *a_ptr;		/**< Pointer value */
      void (*a_fcn) (void);	/**< Function pointer value */
    } a_un;
} Elf64_auxv_t;

/** Legal values for a_type (entry type).  */

#define AT_NULL		0		/**< End of vector */
#define AT_IGNORE	1		/**< Entry should be ignored */
#define AT_EXECFD	2		/**< File descriptor of program */
#define AT_PHDR		3		/**< Program headers for program */
#define AT_PHENT	4		/**< Size of program header entry */
#define AT_PHNUM	5		/**< Number of program headers */
#define AT_PAGESZ	6		/**< System page size */
#define AT_BASE		7		/**< Base address of interpreter */
#define AT_FLAGS	8		/**< Flags */
#define AT_ENTRY	9		/**< Entry point of program */
#define AT_NOTELF	10		/**< Program is not ELF */
#define AT_UID		11		/**< Real uid */
#define AT_EUID		12		/**< Effective uid */
#define AT_GID		13		/**< Real gid */
#define AT_EGID		14		/**< Effective gid */

/** Motorola 68k specific definitions.  */

/** m68k relocs.  */

#define R_68K_NONE	0		/**< No reloc */
#define R_68K_32	1		/**< Direct 32 bit  */
#define R_68K_16	2		/**< Direct 16 bit  */
#define R_68K_8		3		/**< Direct 8 bit  */
#define R_68K_PC32	4		/**< PC relative 32 bit */
#define R_68K_PC16	5		/**< PC relative 16 bit */
#define R_68K_PC8	6		/**< PC relative 8 bit */
#define R_68K_GOT32	7		/**< 32 bit PC relative GOT entry */
#define R_68K_GOT16	8		/**< 16 bit PC relative GOT entry */
#define R_68K_GOT8	9		/**< 8 bit PC relative GOT entry */
#define R_68K_GOT32O	10		/**< 32 bit GOT offset */
#define R_68K_GOT16O	11		/**< 16 bit GOT offset */
#define R_68K_GOT8O	12		/**< 8 bit GOT offset */
#define R_68K_PLT32	13		/**< 32 bit PC relative PLT address */
#define R_68K_PLT16	14		/**< 16 bit PC relative PLT address */
#define R_68K_PLT8	15		/**< 8 bit PC relative PLT address */
#define R_68K_PLT32O	16		/**< 32 bit PLT offset */
#define R_68K_PLT16O	17		/**< 16 bit PLT offset */
#define R_68K_PLT8O	18		/**< 8 bit PLT offset */
#define R_68K_COPY	19		/**< Copy symbol at runtime */
#define R_68K_GLOB_DAT	20		/**< Create GOT entry */
#define R_68K_JMP_SLOT	21		/**< Create PLT entry */
#define R_68K_RELATIVE	22		/**< Adjust by program base */
#define R_68K_NUM	23

/** Intel 80386 specific definitions.  */

/** 386 relocs.  */

#define R_386_NONE	0		/**< No reloc */
#define R_386_32	1		/**< Direct 32 bit  */
#define R_386_PC32	2		/**< PC relative 32 bit */
#define R_386_GOT32	3		/**< 32 bit GOT entry */
#define R_386_PLT32	4		/**< 32 bit PLT address */
#define R_386_COPY	5		/**< Copy symbol at runtime */
#define R_386_GLOB_DAT	6		/**< Create GOT entry */
#define R_386_JMP_SLOT	7		/**< Create PLT entry */
#define R_386_RELATIVE	8		/**< Adjust by program base */
#define R_386_GOTOFF	9		/**< 32 bit offset to GOT */
#define R_386_GOTPC	10		/**< 32 bit PC relative offset to GOT */
#define R_386_NUM	11
#define R_386_GNU_16    20              /**< Direct 16 bit  */
#define R_386_GNU_PC16  21              /**< PC relative 16 bit */
#define R_386_GNU_8     22              /**< Direct 8 bit  */
#define R_386_GNU_PC8   23              /**< PC relative 8 bit */
#define R_386_GNU_max   24


/** SUN SPARC specific definitions.  */

/** SPARC relocs.  */

#define R_SPARC_NONE	0		/**< No reloc */
#define R_SPARC_8	1		/**< Direct 8 bit */
#define R_SPARC_16	2		/**< Direct 16 bit */
#define R_SPARC_32	3		/**< Direct 32 bit */
#define R_SPARC_DISP8	4		/**< PC relative 8 bit */
#define R_SPARC_DISP16	5		/**< PC relative 16 bit */
#define R_SPARC_DISP32	6		/**< PC relative 32 bit */
#define R_SPARC_WDISP30	7		/**< PC relative 30 bit shifted */
#define R_SPARC_WDISP22	8		/**< PC relative 22 bit shifted */
#define R_SPARC_HI22	9		/**< High 22 bit */
#define R_SPARC_22	10		/**< Direct 22 bit */
#define R_SPARC_13	11		/**< Direct 13 bit */
#define R_SPARC_LO10	12		/**< Truncated 10 bit */
#define R_SPARC_GOT10	13		/**< Truncated 10 bit GOT entry */
#define R_SPARC_GOT13	14		/**< 13 bit GOT entry */
#define R_SPARC_GOT22	15		/**< 22 bit GOT entry shifted */
#define R_SPARC_PC10	16		/**< PC relative 10 bit truncated */
#define R_SPARC_PC22	17		/**< PC relative 22 bit shifted */
#define R_SPARC_WPLT30	18		/**< 30 bit PC relative PLT address */
#define R_SPARC_COPY	19		/**< Copy symbol at runtime */
#define R_SPARC_GLOB_DAT 20		/**< Create GOT entry */
#define R_SPARC_JMP_SLOT 21		/**< Create PLT entry */
#define R_SPARC_RELATIVE 22		/**< Adjust by program base */
#define R_SPARC_UA32	23		/**< Direct 32 bit unaligned */
#define R_SPARC_NUM	24

/** MIPS R3000 specific definitions.  */

/** Legal values for e_flags field of Elf386_Ehdr.  */

#define EF_MIPS_NOREORDER 1		/**< A .noreorder directive was used */
#define EF_MIPS_PIC	  2		/**< Contains PIC code */
#define EF_MIPS_CPIC	  4		/**< Uses PIC calling sequence */
#define EF_MIPS_ARCH	  0xf0000000U	/**< MIPS architecture level */

/** Legal values for MIPS architecture level.  */

#define E_MIPS_ARCH_1	  0x00000000UL	/**< -mips1 code.  */
#define E_MIPS_ARCH_2	  0x10000000UL	/**< -mips2 code.  */
#define E_MIPS_ARCH_3	  0x20000000UL	/**< -mips3 code.  */

/** Special section indices.  */

#define SHN_MIPS_ACOMMON 0xff00		/**< Allocated common symbols */
#define SHN_MIPS_TEXT	 0xff01		/**< Allocated test symbols.  */
#define SHN_MIPS_DATA	 0xff02		/**< Allocated data symbols.  */
#define SHN_MIPS_SCOMMON 0xff03		/**< Small common symbols */
#define SHN_MIPS_SUNDEFINED 0xff04	/**< Small undefined symbols */

/** Legal values for sh_type field of Elf386_Shdr.  */

#define SHT_MIPS_LIBLIST  0x70000000UL	/**< Shared objects used in link */
#define SHT_MIPS_CONFLICT 0x70000002UL	/**< Conflicting symbols */
#define SHT_MIPS_GPTAB	  0x70000003UL	/**< Global data area sizes */
#define SHT_MIPS_UCODE	  0x70000004UL	/**< Reserved for SGI/MIPS compilers */
#define SHT_MIPS_DEBUG	  0x70000005UL	/**< MIPS ECOFF debugging information */
#define SHT_MIPS_REGINFO  0x70000006UL	/**< Register usage information */
#define SHT_MIPS_OPTIONS  0x7000000dUL	/**< Miscellaneous options.  */
#define SHT_MIPS_DWARF    0x7000001eUL	/**< DWARF debugging information.  */
#define SHT_MIPS_EVENTS	  0x70000021UL	/**< Event section.  */

/** Legal values for sh_flags field of Elf386_Shdr.  */

#define SHF_MIPS_GPREL	0x10000000UL	/**< Must be part of global data area */

/** Entries found in sections of type SHT_MIPS_GPTAB.  */

typedef union
{
  struct
    {
      uint8_t gt_current_g_value[4];	/**< -G value used for compilation */
      uint8_t gt_unused[4];		/**< Not used */
    } gt_header;        	   	/**< First entry in section */
  struct
    {
      uint8_t gt_g_value[4];		/**< If this value were used for -G */
      uint8_t gt_bytes[4];		/**< This many bytes would be used */
    } gt_entry;				/**< Subsequent entries in section */
} Elf386_gptab;

/** Entry found in sections of type SHT_MIPS_REGINFO.  */

typedef struct
{
  uint8_t	ri_gprmask[4];		/**< General registers used */
  uint8_t	ri_cprmask[4][4];	/**< Coprocessor registers used */
  uint8_t	ri_gp_value[4];		/**< $gp register value */
} Elf386_RegInfo;

/** MIPS relocs.  */

#define R_MIPS_NONE	0		/**< No reloc */
#define R_MIPS_16	1		/**< Direct 16 bit */
#define R_MIPS_32	2		/**< Direct 32 bit */
#define R_MIPS_REL32	3		/**< PC relative 32 bit */
#define R_MIPS_26	4		/**< Direct 26 bit shifted */
#define R_MIPS_HI16	5		/**< High 16 bit */
#define R_MIPS_LO16	6		/**< Low 16 bit */
#define R_MIPS_GPREL16	7		/**< GP relative 16 bit */
#define R_MIPS_LITERAL	8		/**< 16 bit literal entry */
#define R_MIPS_GOT16	9		/**< 16 bit GOT entry */
#define R_MIPS_PC16	10		/**< PC relative 16 bit */
#define R_MIPS_CALL16	11		/**< 16 bit GOT entry for function */
#define R_MIPS_GPREL32	12		/**< GP relative 32 bit */
#define R_MIPS_NUM	13

/** Legal values for p_type field of Elf386_Phdr.  */

#define PT_MIPS_REGINFO	0x70000000	/**< Register usage information */

/** Legal values for d_tag field of Elf386_Dyn.  */

#define DT_MIPS_RLD_VERSION  0x70000001UL /**< Runtime linker interface version */
#define DT_MIPS_TIME_STAMP   0x70000002UL /**< Timestamp */
#define DT_MIPS_ICHECKSUM    0x70000003UL /**< Checksum */
#define DT_MIPS_IVERSION     0x70000004UL /**< Version string (string tbl index) */
#define DT_MIPS_FLAGS	     0x70000005UL /**< Flags */
#define DT_MIPS_BASE_ADDRESS 0x70000006UL /**< Base address */
#define DT_MIPS_CONFLICT     0x70000008UL /**< Address of CONFLICT section */
#define DT_MIPS_LIBLIST	     0x70000009UL /**< Address of LIBLIST section */
#define DT_MIPS_LOCAL_GOTNO  0x7000000aUL /**< Number of local GOT entries */
#define DT_MIPS_CONFLICTNO   0x7000000bUL /**< Number of CONFLICT entries */
#define DT_MIPS_LIBLISTNO    0x70000010UL /**< Number of LIBLIST entries */
#define DT_MIPS_SYMTABNO     0x70000011UL /**< Number of DYNSYM entries */
#define DT_MIPS_UNREFEXTNO   0x70000012UL /**< First external DYNSYM */
#define DT_MIPS_GOTSYM	     0x70000013UL /**< First GOT entry in DYNSYM */
#define DT_MIPS_HIPAGENO     0x70000014UL /**< Number of GOT page table entries */
#define DT_MIPS_RLD_MAP	     0x70000016UL /**< Address of run time loader map.  */
#define DT_MIPS_NUM	     0x17

/** Legal values for DT_MIPS_FLAG Elf386_Dyn entry.  */

#define RHF_NONE		   0		/**< No flags */
#define RHF_QUICKSTART		   (1 << 0)	/**< Use quickstart */
#define RHF_NOTPOT		   (1 << 1)	/**< Hash size not power of 2 */
#define RHF_NO_LIBRARY_REPLACEMENT (1 << 2)	/**< Ignore LD_LIBRARY_PATH */

/** Entries found in sections of type SHT_MIPS_LIBLIST.  */

typedef struct
{
  uint8_t	l_name[4];		/**< Name (string table index) */
  uint8_t	l_time_stamp[4];	/**< Timestamp */
  uint8_t	l_checksum[4];		/**< Checksum */
  uint8_t	l_version[4];		/**< Interface version */
  uint8_t	l_flags[4];		/**< Flags */
} Elf386_Lib;

/** Legal values for l_flags.  */

#define LL_EXACT_MATCH	  (1 << 0)	/**< Require exact match */
#define LL_IGNORE_INT_VER (1 << 1)	/**< Ignore interface version */

/** HPPA specific definitions.  */

/** Legal values for sh_type field of Elf386_Shdr.  */

#define SHT_PARISC_GOT		0x70000000UL /**< GOT for external data.  */
#define SHT_PARISC_ARCH		0x70000001UL /**< Architecture extensions.  */
#define SHT_PARISC_GLOBAL	0x70000002UL /**< Definition of $global$.  */
#define SHT_PARISC_MILLI	0x70000003UL /**< Millicode routines.  */
#define SHT_PARISC_UNWIND	0x70000004UL /**< Unwind information.  */
#define SHT_PARISC_PLT		0x70000005UL /**< Procedure linkage table.  */
#define SHT_PARISC_SDATA	0x70000006UL /**< Short initialized data.  */
#define SHT_PARISC_SBSS		0x70000007UL /**< Short uninitialized data.  */
#define SHT_PARISC_SYMEXTN	0x70000008UL /**< Argument/relocation info.  */
#define SHT_PARISC_STUBS	0x70000009UL /**< Linker stubs.  */

/** Legal values for sh_flags field of Elf386_Shdr.  */

#define SHF_PARISC_SHORT	0x20000000UL /**< Section with short addressing. */

/** Legal values for ST_TYPE subfield of st_info (symbol type).  */

#define STT_PARISC_MILLICODE	13	/**< Millicode function entry point.  */


/** Alpha specific declarations.  */

/** Alpha relocs.  */

#define R_ALPHA_NONE		0	/**< No reloc */
#define R_ALPHA_REFLONG		1	/**< Direct 32 bit */
#define R_ALPHA_REFQUAD		2	/**< Direct 64 bit */
#define R_ALPHA_GPREL32		3	/**< GP relative 32 bit */
#define R_ALPHA_LITERAL		4	/**< GP relative 16 bit w/optimization */
#define R_ALPHA_LITUSE		5	/**< Optimization hint for LITERAL */
#define R_ALPHA_GPDISP		6	/**< Add displacement to GP */
#define R_ALPHA_BRADDR		7	/**< PC+4 relative 23 bit shifted */
#define R_ALPHA_HINT		8	/**< PC+4 relative 16 bit shifted */
#define R_ALPHA_SREL16		9	/**< PC relative 16 bit */
#define R_ALPHA_SREL32		10	/**< PC relative 32 bit */
#define R_ALPHA_SREL64		11	/**< PC relative 64 bit */
#define R_ALPHA_OP_PUSH		12	/**< OP stack push */
#define R_ALPHA_OP_STORE	13	/**< OP stack pop and store */
#define R_ALPHA_OP_PSUB		14	/**< OP stack subtract */
#define R_ALPHA_OP_PRSHIFT	15	/**< OP stack right shift */
#define R_ALPHA_GPVALUE		16
#define R_ALPHA_GPRELHIGH	17
#define R_ALPHA_GPRELLOW	18
#define R_ALPHA_IMMED_GP_16	19
#define R_ALPHA_IMMED_GP_HI32	20
#define R_ALPHA_IMMED_SCN_HI32	21
#define R_ALPHA_IMMED_BR_HI32	22
#define R_ALPHA_IMMED_LO32	23
#define R_ALPHA_COPY		24	/**< Copy symbol at runtime */
#define R_ALPHA_GLOB_DAT	25	/**< Create GOT entry */
#define R_ALPHA_JMP_SLOT	26	/**< Create PLT entry */
#define R_ALPHA_RELATIVE	27	/**< Adjust by program base */
#define R_ALPHA_NUM		28

#ifdef __HAVE_PRAGMA_PACK__
#pragma pack()
#endif

#endif
