/*=====================================================================*/
/*    serrano/prgm/project/bigloo/runtime/Clib/cports.c                */
/*    -------------------------------------------------------------    */
/*    Author      :  Manuel Serrano                                    */
/*    Creation    :  Thu Jul 23 15:34:53 1992                          */
/*    Last change :  Thu Sep  7 06:02:04 2006 (serrano)                */
/*    -------------------------------------------------------------    */
/*    Input ports handling                                             */
/*=====================================================================*/
#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#ifndef _MSC_VER
#   include <dirent.h>
#   include <sys/time.h>
#else
#   include <io.h>
#endif
#include <string.h>
#if( !(defined( NeXT ) && (defined( mc68000 ) || defined( i386 ))) )
#   ifdef HAVE_TERMIO
#      include <termio.h>
#   endif
#endif
#if !defined( sony_news ) && \
    !(defined( NeXT ) && (defined( mc68000 ) || defined( i386 ))) && \
    !defined( _MSC_VER )
#   include <unistd.h>
#endif
#ifndef _MSC_VER
#  include <sys/file.h>
#endif
#include <bigloo.h>
#if( defined( sony_news ) || (defined( NeXT ) && defined( mc68000 )) )
#   include <ctype.h>
#endif

#if POSIX_FILE_OPS
#   include <unistd.h>
#endif

#ifdef _MSC_VER
#   include <io.h>
#   include <windows.h>
#   define pclose _pclose
#   define popen _popen
#   define S_ISDIR( mode ) ((mode & _S_IFDIR) != 0)
#endif

/*---------------------------------------------------------------------*/
/*    isascii                                                          */
/*---------------------------------------------------------------------*/
#if( !defined( isascii ) )
#   define isascii( c ) (!((c) & ~0177))
#endif

/*---------------------------------------------------------------------*/
/*    Global variables                                                 */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF long default_io_bufsiz;

/*---------------------------------------------------------------------*/
/*    External definitions.                                            */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DECL obj_t make_input_port(char *, FILE *, obj_t, long);
extern obj_t make_string();
extern long bgl_file_size(char *);

#if( BGL_HAVE_SENDFILE )
extern ssize_t sendfile(int, int, off_t *, size_t);
#endif

/*---------------------------------------------------------------------*/
/*    Prototypes                                                       */
/*---------------------------------------------------------------------*/
static bool_t pipe_name_p(char *);
static char *pipe_name(char *);

static size_t strwrite(void *, size_t, size_t, obj_t port );
static int strputc(int, obj_t );

static size_t closedwrite(void *, size_t, size_t, obj_t port );
static int closedputc(int, obj_t );

/*---------------------------------------------------------------------*/
/*    struct timeout_t ...                                             */
/*---------------------------------------------------------------------*/
struct timeout_t {
   long timeout;
   long (*sysread)();
   int (*sysputc)();
   size_t (*syswrite)();
};

/*---------------------------------------------------------------------*/
/*    static int                                                       */
/*    bgl_fputc ...                                                    */
/*---------------------------------------------------------------------*/
static int
bgl_fputc( int c, obj_t port ) {
   FILE *stream = PORT( port ).stream;
   
   return putc( c, stream );
}

/*---------------------------------------------------------------------*/
/*    static size_t                                                    */
/*    bgl_fwrite ...                                                   */
/*---------------------------------------------------------------------*/
static size_t
bgl_fwrite( void *ptr, size_t size, size_t nmemb, obj_t port ) {
   return fwrite( ptr, size, nmemb, PORT( port ).stream );
}

/*---------------------------------------------------------------------*/
/*    int                                                              */
/*    bgl_feof ...                                                     */
/*    -------------------------------------------------------------    */
/*    We explicitly define this function because on some system        */
/*    feof is a macro.                                                 */
/*---------------------------------------------------------------------*/
int
bgl_feof( obj_t obj ) {
   return feof( ((FILE *)PORT( obj ).stream) );
}

/*---------------------------------------------------------------------*/
/*    static int                                                       */
/*    proc_eof ...                                                     */
/*---------------------------------------------------------------------*/
static int
proc_eof( obj_t obj ) {
   return 1;
}

/*---------------------------------------------------------------------*/
/*    static int                                                       */
/*    bgl_fclose ...                                                   */
/*---------------------------------------------------------------------*/
static int
bgl_fclose( obj_t obj ) {
   return fclose( PORT( obj ).stream );
}

/*---------------------------------------------------------------------*/
/*    static int                                                       */
/*    bgl_pclose ...                                                   */
/*---------------------------------------------------------------------*/
static int
bgl_pclose( obj_t obj ) {
   return pclose( PORT( obj ).stream );
}

/*---------------------------------------------------------------------*/
/*    static long                                                      */
/*    plain_fread ...                                                  */
/*---------------------------------------------------------------------*/
static long
plain_fread( char *ptr, long size, long nmemb, obj_t port ) {
   FILE *stream = PORT( port ).stream;
   
#ifdef POSIX_FILE_OPS
   long num = size * nmemb;
   long n;

 loop:
   if( (n = read( fileno( stream ), ptr, num ) ) < 0 )
      if (errno == EINTR)
	 goto loop;

   return n;
#else
   long n;

 loop:
   if( !(n = fread( ptr, size, nmemb, stream )) ) {
      if( ferror( stream ) == EINTR ) {
	 clearerr( stream );
	 goto loop;
      } else {
	 C_SYSTEM_FAILURE( BGL_IO_READ_ERROR, "read", strerror(errno), port );
      }
   }

   return n;
#endif
}

/*---------------------------------------------------------------------*/
/*    static long                                                      */
/*    console_fread ...                                                */
/*    -------------------------------------------------------------    */
/*    In constrast to fread, this function does not block on input, if */
/*    not enough characters are available.                             */
/*    -------------------------------------------------------------    */
/*    This function is prefered to bgl_pipe_fread for consoles because */
/*    it automatically flushes stdout before reading (see getc).       */
/*---------------------------------------------------------------------*/
static long
console_fread( char *ptr, long size, long nmemb, obj_t port ) {
   FILE *stream = PORT( port ).stream;
   long num = size * nmemb;
   char *buf = ptr;
   int c;

   while( ((c = getc( stream )) != EOF) ) {
      *buf++ = c;

      if( c == '\n' ) break;
      if( --num <= 0 ) break;
   }

   return (long)(buf - ptr);
}

/*---------------------------------------------------------------------*/
/*    long                                                             */
/*    bgl_pipe_fread ...                                               */
/*    -------------------------------------------------------------    */
/*    In constrast to fread, this function does not block on input if  */
/*    insufficient characters are available.                           */
/*---------------------------------------------------------------------*/
long
bgl_pipe_fread( char *ptr, long size, long nmemb, obj_t port ) {
#ifdef POSIX_FILE_OPS
   FILE *stream = PORT( port ).stream;
   
   long num = size * nmemb;
   long n;

 loop:
   if( (n = read( fileno( stream ), ptr, num) ) < 0 ) {
      if (errno == EINTR) goto loop;
      C_SYSTEM_FAILURE( BGL_IO_READ_ERROR, "read", strerror( errno ), port );
   }

   return n;
#else
   return console_fread( ptr, size, nmemb, port );
#endif
}

/*---------------------------------------------------------------------*/
/*    static long                                                      */
/*    proc_read ...                                                    */
/*---------------------------------------------------------------------*/
static long
proc_read( char *b, long s, long n, obj_t port ) {
   long l = s * n;
   obj_t buf = CREF(port)->input_port_t.pbuffer;

 loop:

   if( STRINGP( buf ) ) {
      /* won't read because the proc buffer is already filled */
      char *s = BSTRING_TO_STRING( buf );
      long p = CREF( port )->input_port_t.pbufpos;
      long r = STRING_LENGTH( buf ) - p;

      if( r <= l ) {
	 memmove( b, &s[ p ], r );
	 CREF( port )->input_port_t.pbuffer = BFALSE;
	 CREF( port )->input_port_t.pbufpos = 0;
	 return r;
      } else {
	 memmove( b, &s[ p ], l );
	 CREF( port )->input_port_t.pbufpos += l;
	 return l;
      }
   } else {
      /* invoke the procedure to fill the buffer */
      obj_t proc = CREF( port )->input_port_t.port.name;
      obj_t nbuf = PROCEDURE_ENTRY( proc )( proc, BEOA );

      if( STRINGP( nbuf ) ) {
	 buf = CREF( port )->input_port_t.pbuffer = nbuf;
	 goto loop;
      } else {
	 if( nbuf == BFALSE ) {
	    /* eof has been reached */
	    CREF( port )->input_port_t.eof = 1;
	    return 0;
	 } else {
	    C_SYSTEM_FAILURE( BGL_IO_PORT_ERROR,
			      "input-procedure-port",
			      "Procedure result must be #f, or a string",
			      nbuf);
	 }
      }
   }
}

/*---------------------------------------------------------------------*/
/*    static void                                                      */
/*    timeout_set_port_blocking ...                                    */
/*---------------------------------------------------------------------*/
#if( BGL_HAVE_FCNTL )
static void
timeout_set_port_blocking( char *fun, int fd, int bool ) {
   int val;

   if( (val = fcntl( fd, F_GETFL, 0 ) ) < 0 ) {
      C_SYSTEM_FAILURE( BGL_IO_ERROR, fun, strerror( errno ), BINT( fd ) );
   }

   if( !bool ) 
      val |= O_NONBLOCK;
   else 
      val &= ~O_NONBLOCK;

   if( fcntl( fd, F_SETFL, val ) < 0 ) {
      C_SYSTEM_FAILURE( BGL_IO_ERROR, fun, strerror( errno ), BINT( fd ) );
   }
}
#endif 

/*---------------------------------------------------------------------*/
/*    static int                                                       */
/*    sysputc_with_timeout ...                                         */
/*---------------------------------------------------------------------*/
static int
sysputc_with_timeout( int c, obj_t port ) {
#if( defined( POSIX_FILE_OPS ) && BGL_HAVE_SELECT && BGL_HAVE_FCNTL )
   FILE *stream = PORT( port ).stream;
   struct timeout_t *tmt = PORT( port ).timeout;
   int (*sysputc)() = tmt->sysputc;
   int n;
   
   long to = tmt->timeout;

   fd_set writefds;
   struct timeval timeout;
   int fd = fileno( stream );

   FD_ZERO( &writefds );
   FD_SET( fd, &writefds );

   timeout.tv_sec = 0;
   timeout.tv_usec = to;

   if( (n = select( fd + 1, NULL, &writefds, NULL, &timeout )) <= 0 ) {
      if( n == 0 ) {
	 C_SYSTEM_FAILURE( BGL_IO_TIMEOUT_ERROR,
			   "putc/timeout",
			   "time limit exceeded",
			   port );
      } else {
	 C_SYSTEM_FAILURE( BGL_IO_WRITE_ERROR,
			   "putc/timeout",
			   strerror( errno ),
			   port );
	 
      }
   } else {
      return sysputc( c, port );
   }
#else
   return OUTPUT_PORT( port ).sysputc( c, port );
#endif
}

/*---------------------------------------------------------------------*/
/*    static size_t                                                    */
/*    syswrite_with_timeout ...                                        */
/*---------------------------------------------------------------------*/
static size_t
syswrite_with_timeout( void *ptr, size_t size, size_t nmemb, obj_t port ) {
#if( defined( POSIX_FILE_OPS ) && BGL_HAVE_SELECT && BGL_HAVE_FCNTL )
   FILE *stream = PORT( port ).stream;
   struct timeout_t *tmt = PORT( port ).timeout;
   size_t (*syswrite)() = tmt->syswrite;
   
   long to = tmt->timeout;
   long count = 0;
   long sum = nmemb * size;

   fd_set writefds;
   struct timeval timeout;
   int fd = fileno( stream );

   FD_ZERO( &writefds );
   FD_SET( fd, &writefds );

   timeout.tv_sec = 0;
   timeout.tv_usec = to;

   while( count < sum ) {
      int n;
      
      if( n = select( fd + 1, NULL, &writefds, NULL, &timeout ) <= 0 ) {
	 if( n == 0 ) {
	    C_SYSTEM_FAILURE( BGL_IO_TIMEOUT_ERROR,
			      "write/timeout",
			      "time limit exceeded",
			      port );
	 } else {
	    C_SYSTEM_FAILURE( BGL_IO_WRITE_ERROR,
			      "write/timeout",
			      strerror( errno ),
			      port );
	 }
      } else {
	 int n = syswrite( &((char *) ptr)[ count ], 1, sum - count, port );

	 if (n >= 0) {
	    count += n;
	 } else {
	    C_SYSTEM_FAILURE( BGL_IO_WRITE_ERROR,
			      "write/timeout",
			      strerror( errno ),
			      port );
	 }
      }
   }

   return count;
#else
   return OUTPUT_PORT( port ).syswrite( ptr, size, nmemb, port );
#endif
}

/*---------------------------------------------------------------------*/
/*    static long                                                      */
/*    sysread_with_timeout ...                                         */
/*    -------------------------------------------------------------    */
/*    In constrast to fread, this function does not block on input if  */
/*    insufficient characters are available.                           */
/*---------------------------------------------------------------------*/
static long
sysread_with_timeout( char *ptr, long size, long nmemb, obj_t port ) {
#if( defined( POSIX_FILE_OPS ) && BGL_HAVE_SELECT && BGL_HAVE_FCNTL )
   FILE *stream = PORT( port ).stream;
   struct timeout_t *tmt = PORT( port ).timeout;
   long (*sysread)() = tmt->sysread;
   long n;

   long to = tmt->timeout;

   fd_set readfds;
   struct timeval timeout;
   int fd = fileno( stream );

   FD_ZERO( &readfds );
   FD_SET( fd, &readfds );

   timeout.tv_sec = 0;
   timeout.tv_usec = to;

   if( (n = select( fd + 1, &readfds, NULL, NULL, &timeout )) <= 0 ) {
      if( n == 0 ) {
	 C_SYSTEM_FAILURE( BGL_IO_TIMEOUT_ERROR,
			   "read/timeout",
			   "time limit exceeded",
			   port );
      } else {
	 C_SYSTEM_FAILURE( BGL_IO_READ_ERROR,
			   "read/timeout",
			   strerror( errno ),
			   port );
      }
   } else {
      n = sysread( ptr, size, nmemb, port );
      
      return n;
   }
#else
   return INPUT_PORT( port ).sysread( ptr, size, nmemb, port );
#endif
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    fileflush ...                                                    */
/*---------------------------------------------------------------------*/
static obj_t
fileflush( obj_t o ) {
   return ((fflush( PORT( o ).stream ) == EOF) ? BFALSE : BTRUE);
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    closedflush ...                                                  */
/*---------------------------------------------------------------------*/
static obj_t
closedflush( obj_t o ) {
   ;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    strflush ...                                                     */
/*---------------------------------------------------------------------*/
static obj_t
strflush( obj_t port ) {
   obj_t res;
   
   if( OUTPUT_STRING_PORT( port ).buffer ) {
      OUTPUT_STRING_PORT( port ).buffer[ OUTPUT_STRING_PORT( port ).offset ] = 0;
      res = string_to_bstring( OUTPUT_STRING_PORT( port ).buffer );

      return res;
   } else {
      return string_to_bstring( "" );
   }
}

/*---------------------------------------------------------------------*/
/*     make_output_port ...                                            */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF obj_t
make_output_port( char *name, FILE * file, obj_t kindof ) {
   obj_t new_output_port;

   new_output_port = GC_MALLOC( OUTPUT_PORT_SIZE );

   new_output_port->output_port_t.port.header = MAKE_HEADER( OUTPUT_PORT_TYPE, 0 );
   new_output_port->port_t.stream = file;
   new_output_port->port_t.name = string_to_bstring( name );
   new_output_port->port_t.kindof = kindof;
   new_output_port->port_t.sysseek = 0;
   new_output_port->port_t.chook = BUNSPEC;
   new_output_port->port_t.userdata = BUNSPEC;
   new_output_port->port_t.timeout = 0L;
   new_output_port->output_port_t.sysputc = bgl_fputc;
   new_output_port->output_port_t.syswrite = bgl_fwrite;
   new_output_port->output_port_t.sysflush = fileflush;

   switch( (long) kindof ) {
      case (long) KINDOF_FILE:
      case (long) KINDOF_PROCPIPE:
	 new_output_port->port_t.sysclose = bgl_fclose;
	 break;

      case (long) KINDOF_PIPE:
	 new_output_port->port_t.sysclose = bgl_pclose;
	 break;

      case (long) KINDOF_CONSOLE:
	 new_output_port->port_t.sysclose = 0L;
	 break;

      default:
	 new_output_port->port_t.sysclose = 0L;
   }

   return BREF( new_output_port );
}

/*---------------------------------------------------------------------*/
/*    bool_t                                                           */
/*    bgl_output_port_timeout_set ...                                  */
/*---------------------------------------------------------------------*/
bool_t
bgl_output_port_timeout_set( obj_t port, long timeout ) {
#if defined( POSIX_FILE_OPS ) && BGL_HAVE_SELECT && BGL_HAVE_FCNTL
   if( (PORT( port ).kindof == KINDOF_FILE) ||
       (PORT( port ).kindof == KINDOF_PIPE) ||
       (PORT( port ).kindof == KINDOF_PROCPIPE) ||
       (PORT( port ).kindof == KINDOF_CONSOLE) ||
       (PORT( port ).kindof == KINDOF_SOCKET) ) {

      if( timeout == 0 ) {
	 struct timeout_t *to = PORT( port ).timeout;

	 if( to ) {
	    OUTPUT_PORT( port ).syswrite = to->syswrite;
	    OUTPUT_PORT( port ).sysputc = to->sysputc;
	    PORT( port ).timeout = 0L;

	    timeout_set_port_blocking( "output-port-timeout-set!",
				       fileno( (FILE *)(PORT( port ).stream) ),
				       1 );
	 }

	 return 0;
      } else {
	 struct timeout_t *to =
	    (struct timeout_t *)GC_MALLOC( sizeof( struct timeout_t ) );

	 to->timeout = timeout;

	 if( PORT( port ).timeout ) {
	    to->syswrite =
	       ((struct timeout_t *)PORT( port ).timeout)->syswrite;
	    to->sysputc =
	       ((struct timeout_t *)PORT( port ).timeout)->sysputc;
	    
	    PORT( port ).timeout = (FILE *)to;
	 } else {
	    to->syswrite = OUTPUT_PORT( port ).syswrite;
	    to->sysputc = OUTPUT_PORT( port ).sysputc;
	    
	    OUTPUT_PORT( port ).syswrite = &syswrite_with_timeout;
	    OUTPUT_PORT( port ).sysputc = &sysputc_with_timeout;
	    
	    PORT( port ).timeout = (FILE *)to;
	    
	    timeout_set_port_blocking( "output-port-timeout-set!",
				       fileno( (FILE *)(PORT( port ).stream) ),
				       0 );

	 }
	 
	 return 1;
      }
   }
#endif

   return 0;
}

/*---------------------------------------------------------------------*/
/*    open_output_file ...                                             */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF obj_t
open_output_file( obj_t name ) {
   FILE *file;
   char *cname = BSTRING_TO_STRING( name );

#if( HAVE_PIPE )
   if( pipe_name_p( cname ) ) {
      if( !(file = popen( pipe_name( cname ), "w" )) )
         return BFALSE;

       return make_output_port( cname, file, KINDOF_PIPE );
   } else
#endif
   {
      if (strcmp( cname, "null:" ) == 0)
#        ifndef _MSC_VER
            cname= "/dev/null";
#        else
            cname= "NUL:";
#        endif

      if( !(file = fopen( cname, "wb" )) )
         return BFALSE;

      return make_output_port( cname, file, KINDOF_FILE );
   }
}

/*---------------------------------------------------------------------*/
/*    append_output_file ...                                           */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF obj_t
append_output_file( obj_t name ) {
   FILE *file;

   if( !(file = fopen( BSTRING_TO_STRING(name), "a+b ")) )
      return BFALSE;

   return make_output_port( BSTRING_TO_STRING( name ),
			    file,
			    KINDOF_FILE );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    open_output_string ...                                           */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF obj_t
open_output_string() {
   char *buffer;
   obj_t port;

   port = GC_MALLOC( OUTPUT_STRING_PORT_SIZE );

   /* We allocate a buffer filled of zero */
   buffer = (char *)(GC_MALLOC_ATOMIC( OUTPUT_STRING_PORT_BUFFER_SIZE + 1 ));
   bzero( buffer, OUTPUT_STRING_PORT_BUFFER_SIZE + 1 );
   
   port->port_t.header = MAKE_HEADER(OUTPUT_STRING_PORT_TYPE, 0);
   port->port_t.kindof = KINDOF_STRING;
   port->port_t.sysclose = 0L;
   port->port_t.sysseek = 0L;
   port->port_t.chook = BUNSPEC;
   port->port_t.userdata = BUNSPEC;
   port->port_t.stream = port;
   port->output_port_t.sysputc = strputc;
   port->output_port_t.syswrite = strwrite;
   port->output_port_t.sysflush = strflush;
   
   port->output_string_port_t.buffer = buffer;
   port->output_string_port_t.size = OUTPUT_STRING_PORT_BUFFER_SIZE;
   port->output_string_port_t.offset = 0;

   return BREF( port );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    get_output_string ...                                            */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF obj_t
get_output_string( obj_t port ) {
   if( OUTPUT_STRING_PORTP( port ) ) {
      if( OUTPUT_STRING_PORT (port ).size == 0 )
	 return make_string( 0, ' ' );
      else
	 return string_to_bstring_len( OUTPUT_STRING_PORT( port ).buffer,
				       OUTPUT_STRING_PORT( port ).offset);
   } else {
      C_SYSTEM_FAILURE( BGL_IO_PORT_ERROR,
			"get-output-string",
			"Not a string port",
			port );
      return BUNSPEC;
   }
}

/*---------------------------------------------------------------------*/
/*    close_output_port ...                                            */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF obj_t
close_output_port( obj_t port ) {
   obj_t res = port; 
   if( OUTPUT_PORTP( port ) && (PORT( port ).kindof != KINDOF_CLOSED) ) {
      obj_t chook = PORT_CHOOK( port );
      
      PORT( port ).kindof = KINDOF_CLOSED;
      
      if( OUTPUT_STRING_PORTP( port ) ) {
	 res = get_output_string( port );
	 GC_free( (obj_t)(CREF( port )->output_string_port_t.buffer) );
	 CREF( port )->output_string_port_t.buffer = 0;
      }

      OUTPUT_PORT( port ).sysputc = closedputc;
      OUTPUT_PORT( port ).syswrite = closedwrite;
      
      if( PORT( port ).sysclose ) PORT( port ).sysclose( port );

      OUTPUT_PORT( port ).sysflush = closedflush;

      PORT( port ).sysclose = 0L;
      
      if( PROCEDUREP( chook ) ) {
	 if( PROCEDURE_ARITY( chook ) == 1 ) {
	    PROCEDURE_ENTRY( chook )( chook, port, BEOA );
	 } else {
	    C_SYSTEM_FAILURE( BGL_IO_PORT_ERROR,
			      "close-output-port",
			      "illegal close hook arity",
			      chook );
	 }
      }
   }
   
   return res;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    make_input_port ...                                              */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF obj_t
make_input_port( char *name, FILE *file, obj_t kindof, long bufsiz ) {
   obj_t new_input_port;

   /* An input port cannot be allocated as an atomic object    */
   /* because it holds a buffer and a name that are GC objects */
   new_input_port = GC_MALLOC( INPUT_PORT_SIZE );

   new_input_port->port_t.header = MAKE_HEADER( INPUT_PORT_TYPE, 0 );
   new_input_port->port_t.kindof = kindof;
   new_input_port->port_t.name = string_to_bstring( name );
   new_input_port->port_t.stream = file;
   new_input_port->port_t.timeout = 0L;
   new_input_port->port_t.sysseek = 0L;
   new_input_port->port_t.chook = BUNSPEC;
   new_input_port->port_t.userdata = BUNSPEC;
   new_input_port->input_port_t.filepos = 0;
   new_input_port->input_port_t.fillbarrier = -1;
   new_input_port->input_port_t.bufsiz = bufsiz;
   new_input_port->input_port_t.eof = 0;
   new_input_port->input_port_t.matchstart = 0;
   new_input_port->input_port_t.matchstop = 0;
   new_input_port->input_port_t.forward = 0;
   new_input_port->input_port_t.bufpos = 1;
   new_input_port->input_port_t.lastchar = '\n';
   new_input_port->input_port_t.syseof = bgl_feof;
   new_input_port->input_port_t.pbuffer = BFALSE;
   new_input_port->input_port_t.pbufpos = 0;
   new_input_port->input_port_t.gzip = BFALSE;

   switch( (long) kindof ) {
      case (long)KINDOF_CONSOLE:
	 new_input_port->port_t.sysclose = 0;
	 new_input_port->input_port_t.sysread = console_fread;
	 break;

#if( HAVE_PIPE )
      case (long)KINDOF_PIPE:
	 new_input_port->port_t.sysclose = bgl_pclose;
	 new_input_port->input_port_t.sysread = bgl_pipe_fread;
	 break;

      case (long)KINDOF_PROCPIPE:
#endif
      case (long)KINDOF_SOCKET:
	 new_input_port->port_t.sysclose = bgl_fclose;
	 new_input_port->input_port_t.sysread = bgl_pipe_fread;
	 break;

      case (long)KINDOF_FILE:
	 new_input_port->port_t.sysclose = bgl_fclose;
	 new_input_port->input_port_t.sysread = plain_fread;
	 break;

      case (long)KINDOF_PROCEDURE:
      case (long)KINDOF_GZIP:
	 new_input_port->port_t.sysclose = 0;
	 new_input_port->input_port_t.syseof = proc_eof;
	 new_input_port->input_port_t.sysread = proc_read;
	 break;

      default:
	 new_input_port->port_t.sysclose = 0;
	 new_input_port->input_port_t.sysread = plain_fread;
   }


   if( bufsiz > 0 ) {
      new_input_port->input_port_t.buffer =
	 (unsigned char *) GC_MALLOC_ATOMIC( bufsiz + 1 );
      new_input_port->input_port_t.buffer[ 0 ] = '\0';
   } else {
      new_input_port->input_port_t.buffer = '\0';
   }

   return BREF( new_input_port );
}

/*---------------------------------------------------------------------*/
/*    bool_t                                                           */
/*    bgl_input_port_timeout_set ...                                   */
/*---------------------------------------------------------------------*/
bool_t
bgl_input_port_timeout_set( obj_t port, long timeout ) {
#if defined( POSIX_FILE_OPS ) && BGL_HAVE_SELECT && BGL_HAVE_FCNTL
   if( (PORT(port).kindof == KINDOF_FILE) ||
       (PORT(port).kindof == KINDOF_PIPE) ||
       (PORT(port).kindof == KINDOF_PROCPIPE) ||
       (PORT(port).kindof == KINDOF_CONSOLE) ||
       (PORT(port).kindof == KINDOF_SOCKET) ) {
      if( timeout == 0 ) {
	 struct timeout_t *to = PORT( port ).timeout;

	 if( to ) {
	    INPUT_PORT( port ).sysread = to->sysread;
	    PORT( port ).timeout = 0L;

	    timeout_set_port_blocking( "input-port-timeout-set!",
				       fileno( (FILE *)(PORT( port ).stream) ),
				       1 );
	 }

	 return 0;
      } else {
	 struct timeout_t *to =
	    (struct timeout_t *)GC_MALLOC( sizeof( struct timeout_t ) );

	 to->timeout = timeout;
	 
	 if( PORT( port ).timeout ) {
	    to->sysread =
	       ((struct timeout_t *)PORT( port ).timeout)->sysread; 
	    
	    PORT( port ).timeout = (FILE *)to;
	 } else {
	    to->sysread = INPUT_PORT( port ).sysread;

	    if( fileno( (FILE *)(PORT( port ).stream) ) == -1 ) {
	       C_SYSTEM_FAILURE( BGL_IO_ERROR,
				 "input-port-timeout-set!",
				 "Illegal input-port",
				 port );
	    }

	    INPUT_PORT( port ).sysread = &sysread_with_timeout;

	    PORT( port ).timeout = (FILE *)to;
	    
	    timeout_set_port_blocking( "input-port-timeout-set!",
				       fileno( (FILE *)(PORT( port ).stream) ),
				       0);
	 }

	 return 1;
      }
   }
#endif

   return 0;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    open_input_pipe ...                                              */
/*---------------------------------------------------------------------*/
obj_t
open_input_pipe( obj_t name, obj_t bbufsiz ) {
#if( HAVE_PIPE )
   FILE *file;
   char *cname = BSTRING_TO_STRING( name );

   if( !(file = popen( cname, "r" )) )
      return BFALSE;

   /* we use our own buffer */
   setvbuf( file, NULL, _IONBF, 0 );  

   return make_input_port( cname, file, KINDOF_PIPE, CINT( bbufsiz ) );
#else
   return BFLASE;
#endif
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    open_input_resource ...                                          */
/*    -------------------------------------------------------------    */
/*    Resources are only meaning in Java.                              */
/*---------------------------------------------------------------------*/
obj_t
open_input_resource( obj_t name, obj_t bbufsiz ) {
   return BFALSE;
}

/*---------------------------------------------------------------------*/
/*    open_input_file ...                                              */
/*    -------------------------------------------------------------    */
/*    !!! WARNING !!! WARNING !!! WARNING !!! WARNING !!! WARNING !!!  */
/*    We fill up its associated buffer when opening an input port.     */
/*    -------------------------------------------------------------    */
/*    This function open two kind of files. Regular file and Unix      */
/*    like pipes when the file name is something like "| ...".         */
/*---------------------------------------------------------------------*/
obj_t
open_input_file( obj_t name, obj_t bbufsiz ) {
   FILE *file;
   char *cname = BSTRING_TO_STRING( name );

#if( HAVE_PIPE )
   if( pipe_name_p( cname ) ) {
      if( !(file = popen( pipe_name( cname ), "r" ) ) )
	 return BFALSE;

      /* we use our own buffer */
      setvbuf( file, NULL, _IONBF, 0 );  

      return make_input_port( cname, file, KINDOF_PIPE, CINT( bbufsiz ) );
   } else
#endif
   {
      if (strcmp( cname, "null:" ) == 0)
#        ifndef _MSC_VER
            cname= "/dev/null";
#        else
            cname= "NUL:";
#        endif

       if( !(file = fopen( cname, "rb" )) ) {
	  return BFALSE;
       } else {
	  long sz = bgl_file_size( cname );
	  long bufsiz = CINT( bbufsiz );

	  switch( sz ) {
	     case 0:
		break;
	     case 1:
		bufsiz = 3;
		break;

	     default:
		if( (sz < bufsiz) ) bufsiz = (long)sz + 1;
	  }

	  /* we use our own buffer */
	  setvbuf( file, NULL, _IONBF, 0 );  

	  return make_input_port( cname, file, KINDOF_FILE, bufsiz );
       }
   }
}

/*---------------------------------------------------------------------*/
/*    open_input_console ...                                           */
/*---------------------------------------------------------------------*/
obj_t
open_input_console() {
   return make_input_port("[stdin]", stdin, KINDOF_CONSOLE, default_io_bufsiz);
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    file_to_buffered_input_port ...                                  */
/*---------------------------------------------------------------------*/
obj_t
file_to_buffered_input_port( FILE *file, long bufsize ) {
   if( file == stdin )
      return open_input_console();
   else {
      return make_input_port( "[file]", file, KINDOF_FILE, bufsize );
   }
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    file_to_input_port ...                                           */
/*---------------------------------------------------------------------*/
obj_t
file_to_input_port( FILE * file ) {
   return file_to_buffered_input_port( file, -1 );
}

/*---------------------------------------------------------------------*/
/*    open_input_string ...                                            */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF obj_t
open_input_string( obj_t string ) {
   obj_t port;
   long bufsiz = STRING_LENGTH( string );

   port = make_input_port( "[string]", 0L, KINDOF_STRING, bufsiz + 1 );

   CREF(port)->input_port_t.eof = 1;
   CREF(port)->input_port_t.bufpos = bufsiz + 1;
   memcpy( &RGC_BUFFER(port)[ 0 ], BSTRING_TO_STRING( string ), bufsiz );
   RGC_BUFFER(port)[ bufsiz ] = '\0';

   return port;
}

/*---------------------------------------------------------------------*/
/*    open_input_procedure ...                                         */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF obj_t
open_input_procedure( obj_t fun, obj_t bfz ) {
   if( PROCEDURE_CORRECT_ARITYP( fun, 0 ) ) {
      obj_t port = make_input_port( "", 0L, KINDOF_PROCEDURE, CINT( bfz )+1 );

      CREF( port )->port_t.stream = port;
      CREF( port )->port_t.name = fun;
      CREF( port )->input_port_t.pbuffer = BUNSPEC;
      CREF( port )->input_port_t.pbufpos = 0;

      return port;
   } else {
      C_SYSTEM_FAILURE( BGL_IO_PORT_ERROR,
			"open-input-procedure",
			"Illegal procedure arity",
			fun );
   }
}

/*---------------------------------------------------------------------*/
/*    open_input_gzip_port ...                                         */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF obj_t
open_input_gzip_port( obj_t fun, obj_t in, obj_t bfz ) {
   if( PROCEDURE_CORRECT_ARITYP( fun, 0 ) ) {
      obj_t port = make_input_port( "", 0L, KINDOF_GZIP, CINT( bfz )+1 );

      CREF( port )->port_t.stream = port;
      CREF( port )->port_t.name = fun;
      CREF( port )->input_port_t.pbuffer = BUNSPEC;
      CREF( port )->input_port_t.pbufpos = 0;
      CREF( port )->input_port_t.gzip = in;

      return port;
   } else {
      C_SYSTEM_FAILURE( BGL_IO_PORT_ERROR,
			"open-input-gzip-port",
			"Illegal procedure arity",
			fun );
   }
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    open_input_c_string ...                                          */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF obj_t
open_input_c_string( char *c_string ) {
   obj_t port;
   long bufsiz = (long)strlen( c_string );
   char *new_string = 0;

   port = make_input_port( "[c_string]", 0L, KINDOF_STRING, 0 );

   CREF(port)->input_port_t.eof = 1;
   CREF(port)->input_port_t.bufsiz = bufsiz + 1;
   CREF(port)->input_port_t.bufpos = bufsiz + 1;
   CREF(port)->input_port_t.buffer = (unsigned char *)c_string;

   return port;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    reopen_input_c_string ...                                        */
/*    -------------------------------------------------------------    */
/*    Simply changes the input buffer of an input string. Does not     */
/*    re-allocate a brand new input-port.                              */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF obj_t
reopen_input_c_string( obj_t port, char *c_string ) {
   long bufsiz = (long)strlen(c_string);

   if( CREF(port)->input_port_t.bufsiz < (bufsiz + 1) ) {
      CREF(port)->input_port_t.bufsiz = bufsiz + 1;
      CREF(port)->input_port_t.buffer =
	 (unsigned char *)GC_MALLOC_ATOMIC(bufsiz + 1);
   }

   CREF(port)->input_port_t.bufpos = bufsiz + 1;
   CREF(port)->input_port_t.matchstart = 0;
   CREF(port)->input_port_t.matchstop = 0;
   CREF(port)->input_port_t.forward = 0;
   CREF(port)->input_port_t.lastchar = '\n';
   strcpy( (char *)(CREF(port)->input_port_t.buffer), (char *)c_string );

   return port;
}

/*---------------------------------------------------------------------*/
/*    close_input_port ...                                             */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF obj_t
close_input_port( obj_t port ) {
   if( INPUT_PORTP( port ) ) {
      if( (PORT( port ).kindof != KINDOF_CLOSED) &&
	  (PORT( port ).kindof != KINDOF_CONSOLE) ) {
	 obj_t chook = PORT_CHOOK( port );
	 
	 if( PORT( port ).sysclose ) PORT( port ).sysclose( port );
	 
	 /* added by MS on 11oct2005 */
	 if( (PORT( port ).kindof != KINDOF_STRING) )
	    GC_free( (obj_t)(INPUT_PORT( port ).buffer) );
	 INPUT_PORT( port ).buffer = 0L;
	 
	 INPUT_PORT( port ).eof = 1;
      	 PORT( port ).kindof = KINDOF_CLOSED;
	 PORT( port ).sysclose = 0L;
	 
	 if( PROCEDUREP( chook ) ) {
	    if( PROCEDURE_ARITY( chook ) == 1 ) {
	       PROCEDURE_ENTRY( chook )( chook, port, BEOA );
	    } else {
	       C_SYSTEM_FAILURE( BGL_IO_PORT_ERROR,
				 "close-input-port",
				 "illegal close hook arity",
				 chook );
	    }
	 }
      }

   }

   return port;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    bgl_input_port_seek ...                                          */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF obj_t
bgl_input_port_seek( obj_t port, long pos ) {
   if( INPUT_PORT_ON_FILEP( port ) ) {
      if( fseek( PORT( port ).stream, pos, SEEK_SET ) )
	 return BFALSE;
      INPUT_PORT( port ).filepos = pos;
      INPUT_PORT( port ).eof = 0;
      INPUT_PORT( port ).matchstart = 0;
      INPUT_PORT( port ).matchstop = 0;
      INPUT_PORT( port ).forward = 0;
      INPUT_PORT( port ).bufpos = 1;
      INPUT_PORT( port ).lastchar = '\n';
      INPUT_PORT( port ).buffer[0] = '\0';

      return BTRUE;
   }

   if( INPUT_PORT_ON_STRINGP( port ) && (pos < INPUT_PORT( port ).bufsiz) ) {
      INPUT_PORT( port ).filepos = pos;
      INPUT_PORT( port ).matchstart = pos;
      INPUT_PORT( port ).matchstop = pos;
      INPUT_PORT( port ).forward = pos;

      return BTRUE;
   }

   return BFALSE;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    bgl_output_port_seek ...                                         */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF obj_t
bgl_output_port_seek( obj_t port, long pos ) {
   if( OUTPUT_FILE_PORTP( port ) ) {
      if( fseek( PORT( port ).stream, pos, SEEK_SET ) )
	 return BFALSE;
      return BTRUE;
   }
   if( OUTPUT_STRING_PORTP( port ) ) {
      if( pos >= OUTPUT_STRING_PORT( port ).size )
	 return BFALSE;

      OUTPUT_STRING_PORT( port ).offset = pos;
      return BTRUE;
   }
   return BFALSE;
}
   
/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    bgl_input_port_reopen ...                                        */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF obj_t
bgl_input_port_reopen( obj_t port ) {
   FILE *nf;
   
   if( !INPUT_PORT_ON_FILEP( port ) ) {
      if( INPUT_STRING_PORTP( port ) ) {
	 INPUT_PORT( port ).matchstart = 0;
	 INPUT_PORT( port ).matchstop = 0;
	 INPUT_PORT( port ).forward = 0;
	 INPUT_PORT( port ).lastchar = '\n';
	 INPUT_PORT( port ).filepos = 0;
	 INPUT_PORT( port ).eof = 0;
      } else {
	 return BFALSE;
      }
   }
   
   nf = freopen( BSTRING_TO_STRING( PORT( port ).name ),
		 "r",
		 (FILE *)PORT( port ).stream );

   if( !nf ) return BFALSE;
   
   PORT( port ).stream = (void *)nf;
      
   /* we use our own buffer */
   setvbuf( (FILE *)PORT( port ).stream, NULL, _IONBF, 0 );

   INPUT_PORT( port ).filepos = 0;
   INPUT_PORT( port ).eof = 0;
   INPUT_PORT( port ).matchstart = 0;
   INPUT_PORT( port ).matchstop = 0;
   INPUT_PORT( port ).forward = 0;
   INPUT_PORT( port ).bufpos = 1;
   INPUT_PORT( port ).lastchar = '\n';
   INPUT_PORT( port ).buffer[ 0 ] = '\0';

   return BTRUE;
}

/*---------------------------------------------------------------------*/
/*    obj                                                              */
/*    reset_console ...                                                */
/*    -------------------------------------------------------------    */
/*    We flush input port, for ^C to work correctly within the         */
/*    interpreter. The only place this function is called is in the    */
/*    REPL (see Eval/eval.scm).                                        */
/*---------------------------------------------------------------------*/
obj_t
reset_console( obj_t port ) {
   if( PORT( port ).kindof == KINDOF_CONSOLE ) {
      INPUT_PORT( port ).matchstart = 0;
      INPUT_PORT( port ).matchstop = 0;
      INPUT_PORT( port ).bufpos = 1;
      INPUT_PORT( port ).buffer[ 0 ] = '\0';
      INPUT_PORT( port ).lastchar = '\n';
   }

   return BUNSPEC;
}

/*---------------------------------------------------------------------*/
/*     bgl_init_io ...                                                 */
/*---------------------------------------------------------------------*/
void
bgl_init_io() {
#if( !defined( _SBFSIZ ) )
#   define _SBFSIZ 1
#endif

   default_io_bufsiz = BUFSIZ * _SBFSIZ;

   BGL_CURRENT_OUTPUT_PORT_SET(
      make_output_port( "stdout", stdout, KINDOF_CONSOLE ) );
   BGL_CURRENT_ERROR_PORT_SET(
      make_output_port( "stderr", stderr, KINDOF_CONSOLE ) );
   BGL_CURRENT_INPUT_PORT_SET(
      open_input_console() );
}

/*---------------------------------------------------------------------*/
/*    fexists ...                                                      */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF bool_t
fexists( char *name ) {
#if( HAVE_PIPE )
   if( pipe_name_p( name ) ) {
      return 1;
   }
#else
   if( pipe_name_p(name) ) {
      return 0;
   }
#endif

# ifndef _MSC_VER
    return !access( name, F_OK );
# else
    /* !!!!! verify semantics of Unix' access */
    return !_access( name, 0 );        
# endif
}

/*---------------------------------------------------------------------*/
/*    bool_t                                                           */
/*    reset_eof ...                                                    */
/*    -------------------------------------------------------------    */
/*    The function erase the end-of-file of input console port.        */
/*    This allows, restart reading after a ^D.                         */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF bool_t
reset_eof( obj_t port ) {
   if( PORT( port ).kindof == KINDOF_CONSOLE ) {
      /* we forget about EOF */
      INPUT_PORT( port ).eof = 0;

      /* we cleanup buffer   */
      reset_console( port );

      /* we clear errors.    */
      clearerr( stdin );

      return 1;
   }
   else
      return 0;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    bgl_strport_grow ...                                             */
/*---------------------------------------------------------------------*/
obj_t
bgl_strport_grow( obj_t p ) {
   char *buffer = OUTPUT_STRING_PORT(p).buffer;
   long old_size = OUTPUT_STRING_PORT(p).size;
   long new_size;

   if( old_size == 0 ) {
      C_SYSTEM_FAILURE( BGL_IO_PORT_ERROR, "string-port", "port closed", p );

      return p;
   } else {
      new_size = old_size * 2;
      buffer = (char *)GC_realloc( (obj_t)buffer, new_size + 1 );
      buffer[ old_size ] ='\0';
      OUTPUT_STRING_PORT(p).buffer = buffer;

      OUTPUT_STRING_PORT(p).size = new_size;

      return p;
   }
}

/*---------------------------------------------------------------------*/
/*    static bool_t                                                    */
/*    pipe_name_p ...                                                  */
/*    -------------------------------------------------------------    */
/*    Is a file name a pipe name ? A pipe name start by the            */
/*    sequence "| ".                                                   */
/*---------------------------------------------------------------------*/
static bool_t
pipe_name_p( char *name ) {
   return( (name[ 0 ] == '|') && (name[ 1 ] == ' ') );
}

/*---------------------------------------------------------------------*/
/*    char *                                                           */
/*    pipe_name ...                                                    */
/*    -------------------------------------------------------------    */
/*    Pipe name to name translation.                                   */
/*---------------------------------------------------------------------*/
static char *
pipe_name( char *pipe_name ) {
   return (pipe_name + 1);
}

/*---------------------------------------------------------------------*/
/*    bool_t                                                           */
/*    directoryp ...                                                   */
/*    -------------------------------------------------------------    */
/*    Is a file a directory?                                           */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF bool_t
directoryp( char *name ) { 
   struct stat buf;

   if( stat( name, &buf ) == -1 )
      return 0;

   return S_ISDIR( buf.st_mode & S_IFMT );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    directory_to_list ...                                            */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF obj_t
directory_to_list( char *name ) {
   obj_t res = BNIL;
#ifndef _MSC_VER
   DIR *dir;
   struct dirent *dirent;

   if( (dir = opendir( name )) ) {
      while( (dirent = readdir( dir )) ) {
	 char *fname = dirent->d_name;

	 if( strcmp( fname, "." ) && strcmp( fname, ".." ) )
	    res = MAKE_PAIR( string_to_bstring( fname ), res );
      }
      closedir( dir );
   }
#else
   char *const path = (char *)malloc( strlen( name ) + 2 + 1 );

   strcpy( path, name );
   strcat( path, "\\*" );

   {
      WIN32_FIND_DATA find_data;
      HANDLE hSearch = FindFirstFile( path, &find_data );

      if( hSearch != INVALID_HANDLE_VALUE ) {
         BOOL keep_going;

         do {
            if( (strcmp( find_data.cFileName, "." ) != 0)
                && (strcmp( find_data.cFileName, ".." ) != 0) )
               res = MAKE_PAIR( string_to_bstring( find_data.cFileName ),
				res );
            keep_going= FindNextFile( hSearch, &find_data );
         } while( keep_going );

         FindClose( hSearch );
      }
   }
#endif
   return res;
}

/*---------------------------------------------------------------------*/
/*    static long                                                      */
/*    copyfile ...                                                     */
/*    -------------------------------------------------------------    */
/*    This function is a replacement for sendfile. In particular       */
/*    it is used when the INPUT-PORT is a socket descriptor or when    */
/*    the host system does not support the sendfile system call.       */
/*---------------------------------------------------------------------*/
static long
copyfile( long on, long in, long sz ) {
   long rsz = 0;
   if( sz < 0 ) {
#ifdef __GNUC__
      char buf[ default_io_bufsiz + 1 ];
#else
      char *buf = alloca( default_io_bufsiz + 1 );
#endif
      long n, m;

loopr:
      while( (n = read( in, buf, default_io_bufsiz )) > 0 ) {
loopw:
	 if( (m = write( on, buf, n )) < 0 ) {
	    if( errno == EINTR ) {
	       goto loopw;
	    } else {
	       return -1;
	    }
	 }

	 rsz += n;
      }

      if( (n < 0) && (errno == EINTR) ) {
	 goto loopr;
      }

      return rsz;
   } else {
      long n = 0;
      long s = (default_io_bufsiz > sz) ? sz : default_io_bufsiz;
#ifdef __GNUC__
      char buf[ s + 1 ];
#else
      char *buf = alloca( s + 1 );
#endif

loopr2:      
      while( (sz > 0) && (n = read( in, buf, s )) > 0 ) {
loopw2:
	 if( write( on, buf, n ) < 0 ) {
	    if( errno == EINTR )
	       goto loopw2;
	    else 
	       return -1;
	 }
	 
	 rsz += n;
	 sz -= n;

	 if( sz < s ) s = sz;
      }
      
      if( (n < 0) && (errno == EINTR) ) {
	 goto loopr2;
      } 

      return rsz;
   }
}   
   
/*---------------------------------------------------------------------*/
/*    static long                                                      */
/*    copyfile_with_timeout ...                                        */
/*    -------------------------------------------------------------    */
/*    This function is equivalent to COPYFILE. The only difference     */
/*    comes from that is does not inline the call to READ and WRITE.   */
/*---------------------------------------------------------------------*/
static long
copyfile_with_timeout( obj_t op, obj_t ip, long sz ) {
   size_t (*syswrite)() = OUTPUT_PORT( op ).syswrite;
   long (*sysread)() = INPUT_PORT( ip ).sysread;
   long rsz = 0;
   if( sz < 0 ) {
#ifdef __GNUC__
      char buf[ default_io_bufsiz + 1 ];
#else
      char *buf = alloca( default_io_bufsiz + 1 );
#endif
      long n;
      size_t m;

loopr:
      while( (n = sysread( buf, default_io_bufsiz, 1, ip )) > 0 ) {
loopw:
	 if( (m = syswrite( buf, n, 1, op )) < 0 ) {
	    if( errno == EINTR ) {
	       goto loopw;
	    } else {
	       return -1;
	    }
	 }

	 rsz += n;
      }

      if( (n < 0) && (errno == EINTR) ) {
	 goto loopr;
      }

      return rsz;
   } else {
      long n = 0;
      long s = (default_io_bufsiz > sz) ? sz : default_io_bufsiz;
#ifdef __GNUC__
      char buf[ s + 1 ];
#else
      char *buf = alloca( s + 1 );
#endif

loopr2:      
      while( (sz > 0) && (n = sysread( buf, s, 1, ip )) > 0 ) {
loopw2:
	 if( syswrite( buf, n, 1, op ) < 0 ) {
	    if( errno == EINTR )
	       goto loopw2;
	    else 
	       return -1;
	 }
	 
	 rsz += n;
	 sz -= n;

	 if( sz < s ) s = sz;
      }
      
      if( (n < 0) && (errno == EINTR) ) {
	 goto loopr2;
      } 

      return rsz;
   }
}   
   
/*---------------------------------------------------------------------*/
/*    bgl_sendchars ...                                                */
/*    -------------------------------------------------------------    */
/*    uses sendfile to "copy" the input-port to the output-port        */
/*    flushes output-port!                                             */
/*---------------------------------------------------------------------*/
obj_t
bgl_sendchars( obj_t ip, obj_t op, long sz, long offset ) {
#define inp INPUT_PORT( ip )
#define outp OUTPUT_PORT( op )
   long dsz;
   long ws = 0;
   struct stat in;
   struct stat out;
   long n;

   switch( (long)(inp.port.kindof) ) {
      case (long)KINDOF_STRING:
      case (long)KINDOF_CLOSED:
      case (long)KINDOF_PROCEDURE:
      case (long)KINDOF_GZIP:
	 return BFALSE;
   }

   if( OUTPUT_STRING_PORTP( op ) )
       return BFALSE;
       
   switch( (long)(outp.port.kindof) ) {
      case (long)KINDOF_CLOSED:
      case (long)KINDOF_PROCEDURE:
	 return BFALSE;
   }

   if( offset >= 0 ) {
      bgl_input_port_seek( ip, offset );
   } else {
      dsz = inp.bufpos - inp.matchstop - 1;
   
      outp.sysflush( op );

      if( dsz > 0 ) {
	 /* flush the buffer when it contains characters */
	 size_t w;

	 ws = ((sz > 0) && (dsz > sz)) ? sz : dsz;
	 w = outp.syswrite( &(inp.buffer[ inp.matchstop ]), 1, ws, op );

	 inp.matchstop += (long)w;
	 inp.forward = inp.matchstop;

	 if( w < ws ) {
	    if( ferror( (FILE *)outp.port.stream ) || outp.sysflush( op ) ) {
	       C_SYSTEM_FAILURE( BGL_IO_SIGPIPE_ERROR,
				 "sendchars",
				 strerror( errno ),
				 MAKE_PAIR( ip, op ) );
	       return BINT( w );
	    }
	 }

	 outp.sysflush( op );

	 if( sz > 0 ) {
	    
	    if( dsz > sz ) {
	       return BINT( ws );
	    }

	    sz -= ws;
	 }
      }
   }
   
#if( BGL_HAVE_SENDFILE )
   if( 
      /* Some operating systems (such as Linux 2.6.10) are demanding */
      /* on the input and output ports. These requirements are set   */
      /* in the configuration files and used to determine what as    */
      /* to be checked before invoking the actual sendfile sys call  */
#  if( BGL_SENDFILE_REQUIRE_INPUT_FILE )      
       (inp.port.kindof != KINDOF_FILE)
       || fstat( fileno( (FILE *)inp.port.stream ), &in )
       || !S_ISREG( in.st_mode )
#  endif
#  if( BGL_SENDFILE_REQUIRE_OUTPUT_SOCKET )
       || (outp.port.kindof != KINDOF_FILE)
       || fstat( fileno( (FILE *)outp.port.stream ), &out )
       || ((out.st_mode & S_IFSOCK) != S_IFSOCK)
#  endif       
       || 0 ) {
#endif
      if( PORT( ip ).timeout || PORT( op ).timeout ) {
	 n = copyfile_with_timeout( op, ip, sz );
      } else {
	 n = copyfile( fileno( (FILE *)outp.port.stream ),
		       fileno( (FILE *)inp.port.stream ),
		       sz );
      }
#if( BGL_HAVE_SENDFILE )
   } else {
      if( sz == -1 ) sz = in.st_size;

/*    fprintf( stderr, "bgl_sendchars: IP=%s OP=%s --> SENDFILE\n",    */
/* 	    BSTRING_TO_STRING( PORT( ip ).name ),                      */
/* 	    BSTRING_TO_STRING( PORT( op ).name ) );                    */
	    
      bgl_gc_start_blocking();

      if( offset < 0 ) offset = ftell( (FILE *)inp.port.stream );

      n = sendfile( fileno( (FILE *)outp.port.stream ),
		    fileno( (FILE*)inp.port.stream ),
		    (off_t *)(&offset),
		    sz );

      bgl_gc_stop_blocking();

      fseek( (FILE *)inp.port.stream, offset, SEEK_SET );
	    
      if( n < 0 ) {
	 C_SYSTEM_FAILURE( BGL_IO_PORT_ERROR,
			   "send-chars",
			   strerror( errno ),
			   MAKE_PAIR( ip, op ) );
      }
   }
#endif
   if( n == -1 ) {
      C_SYSTEM_FAILURE( BGL_IO_ERROR,
			"sendchars",
			strerror( errno ),
			MAKE_PAIR( ip, op ) );
      return BINT( 0 );
   }
     
   inp.filepos += ws + n;
   fseek( (FILE *)inp.port.stream, inp.filepos, SEEK_SET );

   return BINT( ws + n );
}

/*---------------------------------------------------------------------*/
/*    static size_t                                                    */
/*    strwrite ...                                                     */
/*---------------------------------------------------------------------*/
static size_t
strwrite( void *ptr, size_t size, size_t nmemb, obj_t p ) {
   long len = (long)(size == 1 ? nmemb : size * nmemb);
   long offset = OUTPUT_STRING_PORT( p ).offset;

   while( (OUTPUT_STRING_PORT( p ).offset+len) > OUTPUT_STRING_PORT( p ).size )
      bgl_strport_grow( p );

   memcpy( &(OUTPUT_STRING_PORT(p).buffer[ offset ] ), ptr, len );

   OUTPUT_STRING_PORT( p ).offset = offset + len;

   return len;
}

/*---------------------------------------------------------------------*/
/*    static int                                                       */
/*    strputc ...                                                      */
/*---------------------------------------------------------------------*/
static int
strputc( int c, obj_t p ) {
   long offset = OUTPUT_STRING_PORT( p ).offset;
   
   if( END_OF_STRING_PORTP( p ) )
      bgl_strport_grow( p );

   OUTPUT_STRING_PORT( p ).buffer[ offset ] = c;
   OUTPUT_STRING_PORT( p ).offset = offset + 1;

   return c;
}

/*---------------------------------------------------------------------*/
/*    static size_t                                                    */
/*    closedwrite ...                                                  */
/*---------------------------------------------------------------------*/
static size_t
closedwrite( void *ptr, size_t size, size_t nmemb, obj_t port ) {
   C_SYSTEM_FAILURE( BGL_IO_CLOSED_ERROR,
		     "write",
		     "Illegal write to a closed port",
		     port );
   return -1;
}

/*---------------------------------------------------------------------*/
/*    static int                                                       */
/*    closedputc ...                                                   */
/*---------------------------------------------------------------------*/
static int
closedputc( int c, obj_t port ) {
   C_SYSTEM_FAILURE( BGL_IO_CLOSED_ERROR,
		     "write",
		     "Illegal write to a closed port",
		     port );
   return -1;
}
